#!/usr/bin/perl

##  Copyright 2010-2022 Carnegie Mellon University
##  See license information in LICENSE.txt.

##  Reads the NEWS file on stdin and writes the releases.xml file to stdout.
##  Requires one argument: the name of this project.

use warnings;
use strict;

my $project = $ARGV[0] or die "Expected package name as first argument\n";

# This table determines whether to include a download link and the
# license to use.  Releases older than the oldest date in this table
# do not include a download link.  Other dates indicate when the
# release license changed and what it changed to.
#
my %licenses = (
    '2015-07-01' => 'lgpl',     # fixbuf 1.7.0
    '2018-05-01' => 'lgpl3',    # fixbuf 2.0.0
    '2022-01-01' => 'gpl',      # fixbuf 3.0.0
);

# Sort the license change dates from most recent to oldest
my @license_change_dates = sort {$b cmp $a} keys %licenses;

print <<HEAD;
<?xml version="1.0"?>
<p:project xmlns:p="http://netsa.cert.org/xml/project/1.0"
           xmlns="http://www.w3.org/1999/xhtml"
           xmlns:xi="http://www.w3.org/2001/XInclude">
HEAD

my $ul = 0;
my $li = 0;

# slurp in all of the standard input
my $content;
{
    local $/ = undef;
    $content = <STDIN>;
}


# This regexp is pretty liberal, so as to be able to grok most NEWS formats.
while($content =~ /^Version (\d[^:]*?):?\s+\(?([^\n]+?)\)?\s*\n\s*=+\s*((?:.(?!^Version))+)/msg)
{
    my ($ver, $date, $notes) = ($1, $2, $3);

    print <<RELHEAD1;
 <p:release>
  <p:version>$ver</p:version>
  <p:date>$date</p:date>
RELHEAD1

    for my $lcd (@license_change_dates) {
        if ($date ge $lcd) {
            my $license = $licenses{$lcd};
            print <<RELHEAD2;
  <p:file href="../releases/$project-$1.tar.gz" license="$license"/>
RELHEAD2
            last;
        }
    }

    print <<RELHEAD3;
  <p:changelog>
   <ul>
RELHEAD3

    # html escape the notes
    $notes =~ s/&/&amp;/g;
    $notes =~ s/</&lt;/g;
    $notes =~ s/>/&gt;/g;

    # split notes into individual items: assumes items are double spaced
    $notes =~ s/\n+$//s;
    my @items = split /\n\n+/, $notes;

    for my $item (@items) {
        print "    <li>\n$item\n    </li>\n";
    }

    print <<RELTAIL;
   </ul>
  </p:changelog>
 </p:release>
RELTAIL
;

}
print <<TAIL;
</p:project>
TAIL


##  @DISTRIBUTION_STATEMENT_BEGIN@
##  libfixbuf 3.0.0
##
##  Copyright 2022 Carnegie Mellon University.
##
##  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
##  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
##  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
##  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
##  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
##  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
##  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
##  INFRINGEMENT.
##
##  Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
##  contact permission@sei.cmu.edu for full terms.
##
##  [DISTRIBUTION STATEMENT A] This material has been approved for public
##  release and unlimited distribution.  Please see Copyright notice for
##  non-US Government use and distribution.
##
##  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
##  Trademark Office by Carnegie Mellon University.
##
##  This Software includes and/or makes use of the following Third-Party
##  Software subject to its own license:
##
##  1. GLib-2.0 (https://gitlab.gnome.org/GNOME/glib/-/blob/main/COPYING)
##     Copyright 1995 GLib-2.0 Team.
##
##  2. Doxygen (http://www.gnu.org/licenses/old-licenses/gpl-2.0.html)
##     Copyright 2021 Dimitri van Heesch.
##
##  DM22-0006
##  @DISTRIBUTION_STATEMENT_END@
