/*
 ** aggregate.h
 ** NAF second-stage flow table (aggregates and filters flushed flows)
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#ifndef _NAF_AGGREGATE_H_
#define _NAF_AGGREGATE_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include <naf/match.h>

/**
 * @file
 *
 * NAF Core Library Flow Aggregation API. Used by nafalize. Defines a single
 * entry point for aggregating flows from a multibin managed by the match API 
 * (see match.h). This function should be called periodically, and to 
 * completely aggregate and flush the multibin at process or aggregation 
 * termination.
 *
 * The proper use of the match and aggregate APIs is as follows:
 *
 *  - naf_exparse() to initialize a NAFAggConfig from an expression
 *  - naf_mtab_init() to initialize a NAFMultiBin
 *  - to process each flow:
 *    - naf_mtab_add_raw() to add each raw flow to the NAFMultiBin, matching it
 *    - naf_aggregate() to aggregate and flush flows as they are dequeueable.
 *      aggregate does not neet to be called for each flow, but it should be
 *      called periodically to flush the multibin.
 */

/**
 * Aggregate dequeueable bin tables from a given multibin according to an 
 * aggregation configuration, writing the resulting aggregates. This is a 
 * single entry point to the NAF aggregation apparatus, and should be called
 * periodically with flush set to FALSE, then once at process or aggregation 
 * termination with flush set TRUE.
 *
 * @param flush TRUE to aggregate every bintable, not just dequeueable ones
 *              beyond the horizon.
 * @param mtab  NAFMultiBin to take bintables from, from naf_mtab_init().
 * @param conf  Aggregation configuration from naf_exparse().
 * @param fbuf_ary  Array of IPFIX message buffers; must have conf->fanout 
                    fBuf_t members.
 * @param err   An error description.
 * @return TRUE on success, FALSE otherwise.
 */
 
 gboolean naf_aggregate(
    gboolean        flush,
    NAFMultiBin     *mtab,
    NAFAggConfig    *conf,
    fBuf_t          **fbuf_ary,
    GError          **err);
   
#endif
