/*
 ** dynflow.h
 ** NAF dynamic flow allocation and manipulation
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * NAF Flow Dynamics routines. Provides dynamic allocation of flow structures 
 * and utility routines for manipulating and modifying flows.  
 */

/* idem hack */
#ifndef _NAF_DYNFLOW_H_
#define _NAF_DYNFLOW_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>

/**
 * Allocate a new flow key in a given memory chunk. Optionally copy the given 
 * key into the new key.
 *
 * @param keychunk GMemChunk to allocate key in
 * @param key optional key to copy into new key, or NULL.
 * @return a new flow key
 */
 
NAFlowKey       *naf_flowkey_alloc(
    GMemChunk           *keychunk,
    NAFlowKey           *key);
  
/**
 * Allocate a new flow value in a given memory chunk. Optionally copy the given 
 * value into the new value. Does not deep copy value unique counters; VUC
 * should be NULL.
 *
 * @param valchunk GMemChunk to allocate value in
 * @param val optional value to copy into new value, or NULL.
 * @return a new flow value
 */

NAFlowVal       *naf_flowval_alloc(
    GMemChunk           *valchunk,
    NAFlowVal           *val);

/**
 * Allocate a new flow value unique counter structure in a given memory chunk. 
 * The returned VUC is initialized to zero.
 *
 * @param vucchunk GMemChunk to allocate value unique counter structure in
 * @return a new flow value unique counter structure
 */

NAFlowVUC       *naf_flowvuc_alloc(
    GMemChunk           *vucchunk);

/**
 * Allocate a new full flow in a given memory chunk. Optionally copy the given
 * key and value into the new value. Does not deep copy value unique counters; 
 * VUC should be NULL.
 *
 * @param flowchunk GMemChunk to allocate full flow in
 * @param key key to copy into full flow, or NULL.
 * @param val value to copy into full flow, or NULL.
 * @return a new full flow
 */
 
NAFlow          *naf_flow_alloc(
    GMemChunk           *flowchunk,
    NAFlowKey           *key,
    NAFlowVal           *val);

/**
 * Hash a flow key. Pass this to g_hash_table_new() or g_hash_table_new_full() 
 * to store flows in a hash table.
 *
 * @param key flow key to hash
 * @return hash of the given flow key.
 */
 
uint32_t naf_flowkey_hash(
    NAFlowKey       *key);

/**
 * Compare two flow keys for equality. Pass this to g_hash_table_new() or 
 * g_hash_table_new_full() to store flows in a hash table.
 *
 * @param a flow key to compare
 * @param b another flow key to compare
 * @return TRUE if the keys are equal, FALSE otherwise.
 */ 

gboolean naf_flowkey_equal(
    NAFlowKey       *a,
    NAFlowKey       *b);

/**
 * Compare two flow keys for order. This function enforces standard nafalize 
 * output order - in ascending order by bin, source ID, source IP, source IP 
 * mask length, destination IP, destination IP mask length, protocol, source
 * transport port, and destination transport port. Pass this to g_tree_new(),
 * g_tree_new_full(), or g_tree_new_with_data() to store flows in balanced
 * binary tree.
 *
 * @param a flow key to compare
 * @param b another flow key to compare
 * @param u comparison context; ignored
 * @return Positive if a > b, negative if a < b, 0 if a == b.
 */ 

int naf_flowkey_cmp(
    const void          *a, 
    const void          *b, 
    void                *u); 

/**
 * Copy and mask a flow key. Uses the field mask and sip and dip masks of the 
 * given mask; all masked out fields in the destination are set to 0. Forces 
 * all IP masks in the destination to an upper bound of the ip masks in the 
 * mask. May modify the mask to fill in mask bits. May be used to mask a key
 * in place if in and out point to the same key. 
 *
 * @param in pointer to key to mask
 * @param out pointer to key buffer to copy masked key into
 * @param mask mask to use
 */

void naf_flowkey_mask(
    NAFlowKey       *in,
    NAFlowKey       *out,
    NAFlowMask      *mask);

/**
 * Copy and reverse a flow key. Swaps source and destination linked fields; 
 * does not swap ports if protocol is ICMP, because NAF defines the source port 
 * to 0 and the destination port to contain the ICMP type and code for ICMP
 * aggregate flows. May be used to reverse a key in place if in and out point 
 * to the same key. 
 *
 * @param in pointer to key to reverse
 * @param out pointer to key buffer to copy reversed key into
 */

void naf_flowkey_reverse(
    NAFlowKey       *in,
    NAFlowKey       *out);

/**
 * Copy and reverse a flow value. Swaps source and destination linked fields; 
 * must be used on a value without value unique counters. May be used to 
 * reverse a value in place if in and out point to the same value. 
 *
 * @param in pointer to value to reverse
 * @param out pointer to value buffer to copy reversed value into
 */
 
void naf_flowval_reverse(
    NAFlowVal       *in,
    NAFlowVal       *out);

/**
 * Convert a time to a bin number given a bin size. A bin number is simply the 
 * largest whole multiple of the bin size in seconds less than the given time.
 *
 * @param tv time to bin
 * @param size bin size
 * @return binned time
 */
 
NAFTimeSec naf_bin_time(
    NAFTimeSec          tv,
    NAFTimeSec          size);

#endif
