/*
 ** exparse.c
 ** NAF normalizer and aggregator configuration
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/* idem hack */
#ifndef _NAF_EXPARSE_H_
#define _NAF_EXPARSE_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include <naf/filter.h>
#include <naf/sort.h>

/** 
 * @file
 *
 * NAF aggregation expression parser and aggregation configuration API.
 */

/** Maximum fanout supported by the NAFAggConfig. */
#define NAFZ_MAX_FANOUT      16

/** 
 * An aggregation configuration structure. Filled in by naf_exparse() and by 
 * the application; used by the aggregation and match APIs.
 */
typedef struct _NAFAggConfig {
    /** 
     * Horizon in seconds. The bintable will store flows for this long; 
     * this should be larger than the active timeout interval for the flow 
     * sensor from which raw flows are generated. This should be an even
     * multiple of binsize, and is NOT filled in by naf_exparse().
     */
    NAFTimeSec          horizon;
    /** 
     * observationDomainID of exported records. 
     * NOT filled in by naf_exparse() 
     */
    uint32_t            srcid;
    /** Bin size in seconds. */
    NAFTimeSec          binsize;
    /** Bin algorithm. One of the NAF_BA_* constants from nafcore.h. */
    uint32_t            binalg;
    /** 
     * TRUE to disable biflow matching in naf_mtab_add(). 
     * Not filled in by naf_exparse(). 
     */
    gboolean            uniflow_mode;
    /** Rangelist defining the perimeter for perimeter biflow mode. */
    GArray              *perimeter;
    /** Filter applied to binned flows in the matching stage. */
    NAFilter            prefilter;
    /** 
     * Fanout. Aggregations to perform simultaneously; this is the count
     * of valid entries in each of the filter, sorter, mask, and label arrays.
     */
    uint32_t            fanout;
    /** Array of filters to apply to flows before aggregation, for each mask */
    NAFilter            filter[NAFZ_MAX_FANOUT];
    /** Array of sorters to apply after aggregation, for each mask */
    NAFSorter           sorter[NAFZ_MAX_FANOUT];
    /** Array of aggregation masks */
    NAFlowMask          mask[NAFZ_MAX_FANOUT];
    /** Array of aggregation labels for each mask */
    char                *label[NAFZ_MAX_FANOUT];  
} NAFAggConfig;

/**
 * Parse an aggregation expression from the command line into an aggregation 
 * configuration structure. See NAFAggConfig for information on the fields 
 * filled in by this call.
 * 
 * @param argc argument count
 * @param argv argument list; must contain only valid aggregation 
 *             expression tokens
 * @param conf configuration to fill in
 * @return TRUE on success; FALSE (and logs an error) on failure.
 */
 
gboolean naf_exparse(
    int             argc,
    char            *argv[],
    NAFAggConfig    *conf);

/**
 * Dump a filled configuration to the log.
 *
 * @param conf configuration to dump
 */

void naf_config_dump(
    NAFAggConfig    *conf);

#endif
