/*
 ** filter.h
 ** NAF flow filter support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * NAF Core Library Flow Filter API. Used by nafalize and nafilter. Defines an 
 * API for handling rangelist-based filters for flows. The flow filter API also
 * provides a facility for parsing filter expressions from user input.
 */

/* idem hack */
#ifndef _NAF_FILTER_H_
#define _NAF_FILTER_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include <naf/lexcore.h>

/** A rangelist-based filter for NAF raw and aggregate flows. */
typedef struct _NAFilter {
    /** Time rangelist (epoch seconds) */
    GArray                  *binrl;
    /** Set to TRUE to invert the range for source IPs. */
    gboolean                sipnot;
    /** Source IP rangelist */
    GArray                  *siprl;
    /** Set to TRUE to invert the range for destination IPs. */
    gboolean                dipnot;
    /** Destination IP rangelist */
    GArray                  *diprl;
    /** Set to TRUE to invert the range for source ports. */
    gboolean                spnot;
    /** Source port rangelist */
    GArray                  *sprl;
    /** Set to TRUE to invert the range for destination ports. */
    gboolean                dpnot;
    /** Destination port rangelist */
    GArray                  *dprl;
    /** Set to TRUE to invert the range for protocols. */
    gboolean                protonot;
    /** Protocol rangelist */
    GArray                  *protorl;
    /** Forward active flow count rangelist */
    GArray                  *florl;
    /** Reverse active flow count rangelist */
    GArray                  *rflorl;
    /** Forward packet count rangelist */
    GArray                  *pktrl;
    /** Reverse packet count rangelist */
    GArray                  *rpktrl;
    /** Forward octet count rangelist */
    GArray                  *octrl;
    /** Reverse octet count rangelist */
    GArray                  *roctrl;
} NAFilter;

/** Convenience macro to determine if a filter has any rangelists */
#define naf_filter_active(_filter_) (((_filter_).binrl) || \
                                     ((_filter_).siprl) || \
                                     ((_filter_).diprl) || \
                                     ((_filter_).sprl) || \
                                     ((_filter_).dprl) || \
                                     ((_filter_).protorl) || \
                                     ((_filter_).florl) || \
                                     ((_filter_).rflorl) || \
                                     ((_filter_).pktrl) || \
                                     ((_filter_).rpktrl) || \
                                     ((_filter_).octrl) || \
                                     ((_filter_).roctrl))

/**
 * Add a range to a rangelist. All rangelist ranges are treated as inclusive.
 *
 * @param rl pointer to rangelist to add to. naf_filter_rl_add will allocate a 
 *           new GArray if *rl is NULL.
 * @param a start of range to add
 * @param b end of range to add
 */
 
void naf_filter_rl_add(
    GArray                  **rl,
    uint32_t                a,
    uint32_t                b);

/**
 * Determine if a rangelist contains a given value.
 *
 * @param rl rangelist to check
 * @param v value to check for
 * @return TRUE if the rangelist contains the value, FALSE otherwise.
 */

gboolean naf_filter_rl_contains(
    GArray                  *rl,
    uint32_t                v);

/**
 * Determine if a filter matches a given flow key. Filters are evaluated
 * as AND-of-OR; that is, for each rangelist present in a filter, a flow 
 * must match one of the ranges in the list to match the filter as a whole.
 *
 * @param filter filter to check flow against
 * @param key flow key to check
 * @return TRUE if the filter matches the flow, FALSE otherwise.
 */

gboolean naf_filter_key(
    NAFilter                *filter,
    NAFlowKey               *key);

/**
 * Determine if a filter matches a given flow value. Filters are evaluated
 * as AND-of-OR; that is, for each rangelist present in a filter, a flow 
 * must match one of the ranges in the list to match the filter as a whole.
 *
 * @param filter filter to check flow against
 * @param val flow value to check
 * @return TRUE if the filter matches the flow, FALSE otherwise.
 */

gboolean naf_filter_val(
    NAFilter                *filter,
    NAFlowVal               *val);

/**
 * Initialize a filter by setting all its rangelists to empty. Call this to 
 * prepare a new filter for parsing or other construction.
 *
 * @param filter filter to initialize
 */

void naf_filter_init(
    NAFilter                *filter);

/**
 * Reinitialize a filter by freeing all its rangelists. Call this to 
 * prepare a previously constructed filter for parsing or other construction.
 *
 * @param filter filter to reinitialize
 */

void naf_filter_reinit(
    NAFilter                *filter);

/**
 * Append a textual description of an unsigned rangelist to a GString.
 *
 * @param rl rangelist to print
 * @param str string to append to
 */
 
void naf_filter_rl_print(
    GArray                  *rl,
    GString                 *str);

/**
 * Append a textual description of an IPv4 address rangelist to a GString.
 *
 * @param rl rangelist to print
 * @param str string to append to
 */

void naf_filter_rl_print_ipaddr(
    GArray                  *rl,
    GString                 *str);

/**
 * Append a textual description of a filter to a GString.
 *
 * @param filter filter to print
 * @param str string to append to
 */

void naf_filter_print(
    NAFilter                *filter,
    GString                 *str);

/**
 * Parse an IPv4 address rangelist. Requires a scanner initialized with 
 * naf_lex_init() or naf_lex_init_argv(). 
 * 
 * @param scanner   scanner to take rangelist tokens from
 * @param rl        pointer to GArray to store new rangelist in.
 * @return          TRUE on success, FALSE on error. Will use glib logging 
 *                  facility to note parse errors; ensure that log routing is 
 *                  configured before calling this routine.
 */

gboolean naf_filter_rl_parse_ipaddr(
    GScanner            *scanner,
    GArray              **rl);

/**
 * Parse an unsigned rangelist. Requires a scanner initialized with 
 * naf_lex_init() or naf_lex_init_argv(). 
 * 
 * @param scanner   scanner to take rangelist tokens from
 * @param rl        pointer to GArray to store new rangelist in.
 * @param max       Maximum value to accept. Pass 255 for uint8 rangelists, 
 *                  65535 for uint16 rangelists.
 * @return          TRUE on success, FALSE on error. Will use glib logging 
 *                  facility to note parse errors; ensure that log routing is 
 *                  configured before calling this routine.
 */
 
gboolean naf_filter_rl_parse_uint(
    GScanner            *scanner,
    GArray              **rl,
    uint32_t            max);

/**
 * Parse a filter expression. Requires a scanner initialized with 
 * naf_lex_init() or naf_lex_init_argv(). The scanner should be positioned such 
 * that the next token is NAF_SYM_FILTER.
 * 
 * @param scanner   scanner to take filter tokens from
 * @param filter    structure to store parsed filter in.
 * @return          TRUE on success, FALSE on error. Will use glib logging 
 *                  facility to note parse errors; ensure that log routing is 
 *                  configured before calling this routine.
 */

gboolean naf_filter_parse(
    GScanner            *scanner,
    NAFilter            *filter);

#endif
