/*
 ** match.h
 ** NAF first-stage flow table (matches and prefilters biflows)
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#ifndef _NAF_MATCH_H_
#define _NAF_MATCH_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include <naf/exparse.h>

/**
 * @file
 *
 * NAF Core Library Flow Match API. Used by nafalize. Defines an API for 
 * manipulating NAFMultiBin structures, which bin and match flows before 
 * aggregation. This structure allows NAF to build biflow aggregates from
 * uniflow data sources. See aggregate.h or the nafalize source code for
 * guidelines on using these calls.
 */

/**
 * An indexed, time-ordered queue of flow bins; the basic unit of matching.
 */
typedef struct _NAFMultiBin {
    /** Queue of bintables. Head is most recent, tail is least recent. */
    GQueue          *binqueue;
    /** Hash table mapping bin number in seconds to a given bintable. */
    GHashTable      *bintables;
    /** Reference to aggregation configuration for this multibin. */
    NAFAggConfig    *conf;
    /** Bin number in seconds of least recent bintable in queue. */
    NAFTimeSec      minbin;
    /** Bin number in seconds of most recent bintable in queue. */
    NAFTimeSec      maxbin;
    /** Count of raw flows added to multibin */
    uint32_t        raw_count;
    /** Count of binned flows added matching flows in the forward direction */
    uint32_t        hit_count;
    /** Count of binned flows added matching flows in the reverse direction */
    uint32_t        match_count;
    /** Count of binned flows added which did not match previous flows */
    uint32_t        miss_count;
    /** Count of binned flows dropped by the prefilter */
    uint32_t        predrop_count;
    /** Count of binned flows not crossing the perimeter */
    uint32_t        peridrop_count;
    /** Count of binned flows crossing the perimeter outbound */
    uint32_t        perirev_count;
    /** Count of binned flows dropped for horizon violations */
    uint32_t        horizon_count;
    /** Current count of active flow records in the multibin */
    uint32_t        active_count;
    /** Maximum count of active flow records in the multibin */
    uint32_t        active_max;
} NAFMultiBin;

/**
 * A single flow bin. 
 */
typedef struct _NAFBinTable {
    /** Reference to the containing multibin */
    NAFMultiBin     *mtab;
    /** Bin number in seconds of this bin */
    NAFTimeSec      bin;
    /** Table mapping flow keys to flow values */
    GHashTable      *table;
    /** Memory chunk for flow key storage in the table */
    GMemChunk       *keychunk;
    /** Memory chunk for flow value storage in the table */
    GMemChunk       *valchunk;
} NAFBinTable;


/** 
 * Initialize a given multibin for use with a given aggregation configuration.
 * mtab should be a pointer to an uninitialized, allocated NAFMultiBin 
 * structure.
 *
 * @param mtab multibin to initialize
 * @param conf configuration to initialize with, from naf_exparse()
 */
 
void naf_mtab_init(
    NAFMultiBin         *mtab,
    NAFAggConfig        *conf);

/**
 * Add a binned flow to a multibin. Ensures the multibin contains a bin table
 * corresponding to the flow's bin. 
 *
 * @param mtab multibin to add flow to
 * @param key flow key to add
 * @param val flow value to add
 * @param err an error descriptor
 * @return FALSE if add failed (e.g., on a horizon violation).
 */

gboolean naf_mtab_add(
    NAFMultiBin         *mtab,
    NAFlowKey           *key,
    NAFlowVal           *val,
    GError              **err);

/**
 * Add a raw flow to a multibin. Splits the flow into binned flows according
 * to the bin size and binning algorithm in the multibin's configuration.
 *
 * @param mtab multibin to add flow to
 * @param raw raw flow to add
 * @param err an error descriptor
 * @return FALSE if add failed (e.g., on a horizon violation).
 */

gboolean naf_mtab_add_raw(
    NAFMultiBin         *mtab,
    NAFlowRaw           *raw,
    GError              **err);

/**
 * Dequeue the next bintable from the multibin if it is over the given horizon. 
 * Used by naf_aggregate() to retrieve bintables during aggregation. 
 * Applications should have no reason to call this routine.
 *
 * @param mtab multibin to get next bintable from.
 * @param horizon horizon in seconds, or 0 to always return the next bintable.
 * @return the next bintable, or NULL if none is available.
 */
 
NAFBinTable *naf_mtab_dequeue(
    NAFMultiBin         *mtab,
    NAFTimeSec          horizon);

/**
 * Free storage used by a bintable, after dequeueing it with naf_mtab_dequeue.
 * Used by naf_aggregate().  Applications should have no reason to call this
 * routine.
 *
 * @param bintab bintable to free
 */
 
void naf_mtab_bintable_complete(
    NAFBinTable      *bintab);

#endif
