/*
 ** nafcore.h
 ** NetSA Aggregated Flow (NAF) core ADT support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/** 
 * @mainpage NAF Core Library
 *
 * @section Introduction
 *
 * The NAF Core Library consists of all code common to the NAF tools: the NAF 
 * read/write routines used by all of the NAF tools to handle NAF I/O, the NAF 
 * flow filter and sorting functionality used in nafalize and nafilter, 
 * and the aggregation and matching code used by nafalize. These 
 * routines are split out into a separately installed library to allow other
 * applications to easily interoperate with the NAF tools. 
 *
 * @section Building
 *
 * The NAF Core Library is automatically built and installed as part of the NAF 
 * installation process. 
 *
 * @section Copyright
 *
 * NAF is copyright 2005-2006 Carnegie Mellon University, and is released under 
 * the GNU General Public License. See the COPYING file in the 
 * distribution for details.
 *
 * NAF was developed at the CERT Network Situational Awareness Group
 * by Brian Trammell <bht@cert.org>.
 */

/**
 * @file
 *
 * NAF Core Library. Defines API for reading and writing NAF files, and the
 * aggregated flow data structures.
 */
 
/* idem hack */
#ifndef _NAF_NAFCORE_H_
#define _NAF_NAFCORE_H_

#include <naf/autoinc.h>

/* NAF core integral type definitions */

/** A time in epoch seconds, or a bin number */
typedef uint32_t NAFTimeSec;

/** 
 * An aggregated flow key. NAF flows are stored and handled internally as a 
 * split data structure - a key which identifies the aggregated flow, and a 
 * value which counts octets, packets, flows, and hosts associated with it.
 */
typedef struct _NAFlowKey {
    /** Source ID. User or flow source assigned observation domain ID. */
    uint32_t            srcid;
    /** Time bin start in epoch seconds. key.bin mod mask.binsize must be 0. */
    NAFTimeSec          bin;
    /** Source IP address */
    uint32_t         sip;
    /** Destination IP address */
    uint32_t         dip;
    /** Source transport port */
    uint16_t            sp;
    /** Destination transport port. Contains type and code for ICMP */
    uint16_t            dp;
    /** Source IP address CIDR mask length. */
    uint8_t             sipmask;
    /** Destination IP address CIDR mask length. */
    uint8_t             dipmask;
    /** IP protocol */
    uint8_t             proto;
} NAFlowKey;    

/**
 * Flow Value Unique Counters. Used internally by nafalize to count 
 * unique hosts during aggregation. You shouldn't need this structure 
 * outside nafalize.
 */
typedef struct _NAFlowVUC {
    /** Source host IP address table */
    GHashTable          *htab;
    /** Destination host IP address table */
    GHashTable          *rhtab;
    /** Source port table */
    GHashTable          *ptab;
    /** Destination port table */
    GHashTable          *rptab;
} NAFlowVUC;

/**
 * An aggregated flow value. Together with an aggregated flow key, 
 * stores an aggregated flow.
 */
typedef struct _NAFlowVal {
    /** Forward direction octet count */
    uint64_t            oct;
    /** Reverse direction octet count */
    uint64_t            roct;
    /** Forward direction packet count */
    uint64_t            pkt;
    /** Reverse direction packet count */
    uint64_t            rpkt;
    /** Forward direction flow count */
    uint32_t            flo;
    /** Reverse direction flow count */
    uint32_t            rflo;
    /** Forward direction (source) host count */
    uint32_t            host;
    /** Reverse direction (destination) host count */
    uint32_t            rhost;
    /** Forward direction (source) port count */
    uint16_t            port;
    /** Reverse direction (destination) port count */
    uint16_t            rport;
    /** Value unique counter pointer. Should be NULL outside nafalize. */
    NAFlowVUC           *vuc;
} NAFlowVal;

/** 
 * A full aggregated flow; the unification of a flow key and value. This is
 * convenient for postprocessing and sorting, as in nafilter.
 */
typedef struct _NAFlow {
    /** Flow key */
    NAFlowKey   k;
    /** Flow value */
    NAFlowVal   v;
} NAFlow;


/**
 * A raw flow record. This is the interface between the various NAF aggregation
 * input facilities and the NAF aggregation core.
 */
typedef struct _NAFlowRaw {
    /** Forward direction octet count */
    uint64_t        oct;
    /** Reverse direction octet count */
    uint64_t        roct;
    /** Forward direction packet count */
    uint64_t        pkt;
    /** Reverse direction packet count */
    uint64_t        rpkt;
    /** Forward direction flow count */
    uint32_t        flo;
    /** Reverse direction flow count */
    uint32_t        rflo;
    /** 
     * Flow start time in epoch seconds. Sub-second resolution is not 
     * supported by NAF, as sub-second binning is not supported.
     */
    uint32_t        stime;
    /** 
     * Flow end time in epoch seconds. Sub-second resolution is not 
     * supported by NAF, as sub-second binning is not supported.
     */
    uint32_t        etime;
    /** Source ID. User or flow source assigned observation domain ID. */
    uint32_t        srcid;
    /** Source IP address */
    uint32_t     sip;
    /** Destination IP address */
    uint32_t     dip;
    /** Source transport port */
    uint16_t        sp;
    /** Destination transport port. Contains type and code for ICMP */
    uint16_t        dp;
    /** 
     * Source IP address CIDR mask length. 
     * Caller must set this to 32 if record is host-addressed. 
     */
    uint8_t         sipmask;
    /** 
     * Destination IP address CIDR mask length. 
     * Caller must set this to 32 if record is host-addressed. 
     */
    uint8_t         dipmask;
    /** IP protocol */
    uint8_t         proto;
    /** Structure alignment padding for fixbuf transcoder. */
    char            pad0[5];
} NAFlowRaw;    

/** Fieldmask bit for source identifier. */
#define NAF_FM_SRCID    0x00000001
/** Fieldmask bit for source IP address */
#define NAF_FM_SIP      0x00000002
/** Fieldmask bit for source IP address CIDR mask length */
#define NAF_FM_SIPMASK  0x00000004
/** Fieldmask bit for destination IP address */
#define NAF_FM_DIP      0x00000008
/** Fieldmask bit for destination IP address CIDR mask length */
#define NAF_FM_DIPMASK  0x00000010
/** Fieldmask bit for IP protocol. */
#define NAF_FM_PROTO    0x00000020
/** Fieldmask bit for source transport port. */
#define NAF_FM_SP       0x00000040
/** Fieldmask bit for destination transport port and ICMP type/code. */
#define NAF_FM_DP       0x00000080
/** Fieldmask bit for forward octet count. */
#define NAF_FM_OCT      0x00000100
/** Fieldmask bit for reverse octet count. */
#define NAF_FM_ROCT     0x00000200
/** Fieldmask bit for forward packet count. */
#define NAF_FM_PKT      0x00000400
/** Fieldmask bit for reverse packet count. */
#define NAF_FM_RPKT     0x00000800
/** Fieldmask bit for forward flow count. */
#define NAF_FM_FLO      0x00001000
/** Fieldmask bit for reverse flow count. */
#define NAF_FM_RFLO     0x00002000
/** Fieldmask bit for source host count. */
#define NAF_FM_SHOSTC   0x00004000
/** Fieldmask bit for destination host count. */
#define NAF_FM_DHOSTC   0x00008000
/** Fieldmask bit for source port count. */
#define NAF_FM_SPORTC   0x00010000
/** Fieldmask bit for destination port count. */
#define NAF_FM_DPORTC   0x00020000
/** Internal fieldmask pseudofield bit for reverse (descending) sort */
#define NAF_FM_REVSORT  0x10000000
/** Internal fieldmask pseudofield bit for totaling counts */
#define NAF_FM_MTOTAL   0x20000000
/** Internal fieldmask pseudofield bit for in-memory padding */
#define NAF_FM_VPAD     0x40000000
/** Internal fieldmask pseudofield bit for bin */
#define NAF_FM_VSTIME   0x80000000

/** Mask for real field bits */
#define NAF_FM_MASKF    0x0003FFFF
/** Mask for pseudofield bits */
#define NAF_FM_MASKINT  0xF0000000


/** Binning algorithm number for no binning algorithm selected. */
#define NAF_BA_NONE     0x00000000
/** Binning algorithm number for uniform binning. */
#define NAF_BA_UNIFORM  0x00000001
/** Binning algorithm number for start-time binning. */
#define NAF_BA_START    0x00000002
/** Binning algorithm number for end-time binning. */
#define NAF_BA_END      0x00000003

/** Convenience macro for ICMP IP protocol number */
#define NAF_IP_ICMP     1
/** Convenience macro for TCP IP protocol number */
#define NAF_IP_TCP      6
/** Convenience macro for UDP IP protocol number */
#define NAF_IP_UDP      17

/**
 * A flow mask. Flow masks are used to describe aggregation operations, as well
 * as which values are present in a NAF file.
 */
typedef struct _NAFlowMask {
    /** Source IP address CIDR mask length. Used during aggregation. */
    uint16_t        sipmask;
    /** Destination IP address CIDR mask length. Used during aggregation. */
    uint16_t        dipmask;
    /** Source IP address CIDR mask bits. Used during aggregation. */
    uint32_t     sipmaskbits;
    /** Destination IP address CIDR mask bits. Used during aggregation. */
    uint32_t     dipmaskbits;
    /** Field mask bits. Composed of a bitwise OR of NAF_FM_* constants. */
    uint32_t        fieldmask;
    /** Time bin size in seconds. */
    NAFTimeSec      binsize;
    /** Binning algorithm number. Used during aggregation. */
    uint32_t        binalg;
} NAFlowMask;

/* error handling */

/** 
 * GError domain for NAF errors. All NAF tool errors belong to this domain, 
 * not just those returned by the NAF Core Library. In addition, NAF core 
 * library routines can return libfixbuf errors if reading or writing fails.
 */
#define NAF_ERROR_DOMAIN        (g_quark_from_string("certNAFError"))
/** A NAF file header was malformed. The file is probably not a NAF file. */
#define NAF_ERROR_HEADER        1
/** Illegal argument error. */
#define NAF_ERROR_ARGUMENT      2
/** General I/O error */
#define NAF_ERROR_IO            3
/** Horizon violation. A flow was dropped because its bin was flushed. */
#define NAF_ERROR_HORIZON       4
/** Multiple errors were encountered and logged. */
#define NAF_ERROR_MULTIPLE      5
/** 
 * End of file on read from driver layer. Driver-mediated raw flow input 
 * must set this error on read EOF.
 */
#define NAF_ERROR_EOF           6

/**
 * Get the NAF application global IPFIX information model, initializing it
 * if necessary.
 *
 * @return the NAF application global IPFIX information model.
 */
fbInfoModel_t *nfInfoModel();

/**
 * Begin reading a NAF file. Reads the file header and returns a flow mask
 * describing the fields available in the file, and returns a libfixbuf message
 * reader for reading NAF records from the file with naf_read().
 *
 * @param fbuf  an optional IPFIX message buffer returned from a previous 
 *              nfReaderForFP() call. If supplied, nfReaderForFP() will 
 *              reuse the buffer instead of allocating a new one. If supplied 
 *              and an error occurs, nfReaderForFP() will free the supplied 
 *              buffer. Pass NULL to allocate a new buffer.
 * @param fp    file pointer to read from.
 * @param mask  pointer to a mask structure to store bin size and fieldmask 
 *              into. Those fields of the supplied mask will be overwritten.
 * @param err   an error description; required.
 * @return      an IPFIX message buffer ready to read NAF records from, 
 *              or NULL on error.  
 */
 
fBuf_t *nfReaderForFP(
    fBuf_t          *fbuf,
    FILE            *fp,
    NAFlowMask      *mask,
    GError          **err);

/**
 * Read a single NAF record from a NAF file. Takes a libfixbuf message reader 
 * previously opened with naf_read_start(), and copies flow key and value into
 * supplied buffers.
 *
 * @param fbuf  IPFIX message buffer to read message from.
 * @param mask  mask structure filled in by nfReaderForFP(). 
 * @param key   pointer to key to fill in with next record in file.
 * @param val   pointer to value to fill in with next record in file.
 * @param err   an error description; required.
 * @return      TRUE on success, FALSE otherwise. Check error against 
 *              FIX_ERROR_EOF to determine if the buffer is at end of 
 *              file. Check error against FIX_ERROR_EOM to determine if the
 *              buffer is not in automatic mode and at end of message.
 */
 
 gboolean nfRead(
    fBuf_t                  *fbuf,
    NAFlowMask              *mask,
    NAFlowKey               *key,
    NAFlowVal               *val,
    GError                  **err);

/**
 * Begin writing NAF data to an open file. Writes a file header for a given 
 * file mask and prepares an IPFIX message to write to the file with 
 * nfWrite().
 *
 * @param fbuf  an optional IPFIX message buffer returned from a previous 
 *              nfWriterForFP() call. If supplied, nfWriterForFP() will 
 *              reuse the writer instead of allocating a new one. If supplied 
 *              and an error occurs, nfWriterForFP() will free the supplied 
 *              writer. Pass NULL to allocate a new reader.
 * @param fp    file pointer to write to.
 * @param domain observation domain to export flows in
 * @param mask  mask structure to get bin size and fieldmask 
 *              from. The mask defines which fields will appear in the output.
 * @param err   an error description; required.
 * @return      an IPFIX message buffer ready to write NAF records to, 
 *              or NULL on error.  
 */

fBuf_t *nfWriterForFP(
    fBuf_t                  *fbuf,
    FILE                    *fp,
    uint32_t                domain,
    NAFlowMask              *mask,
    GError                  **err);

/**
 * Write a single NAF record to an IPFIX message buffer returned by 
 * nfWriterForFP(). Copies and encodes flow key and value into the buffer.
 *
 * @param fbuf  IPFIX message buffer to write message to.
 * @param mask  mask structure used to open file with nfWriterForFP(). 
 * @param key   pointer to key to write to file.
 * @param val   pointer to value to write to file.
 * @param err   an error description; required.
 * @return      TRUE on success, FALSE otherwise. 
 */

gboolean nfWrite(
    fBuf_t                  *fbuf,
    NAFlowMask              *mask,
    NAFlowKey               *key,
    NAFlowVal               *val,
    GError                  **err);

/**
 * Finish writing a NAF file. The underlying file pointer passed to
 * nfWriterForFP() is not automatically closed by this call.
 *
 * @param fbuf  IPFIX message buffer from nfWriterForFP() to close.
 * @param err   an error description, set on failure
 * @return TRUE on success, FALSE otherwise.
 */

gboolean nfWriterClose(
    fBuf_t          *fbuf,
    GError          **err);

/**
 * Internal debugging routine to dump a flow to standard error.
 *
 * @param key key to dump
 * @param val value to dump
 * @param inf short string describing type of flow dump
 */
 
void nfDumpFlow(
    NAFlowKey           *key,
    NAFlowVal           *val,
    const char          *inf);

/* end idem */
#endif
