/*
 ** sort.h
 ** NAF flow sorting support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * NAF Core Library Flow Sorting API. Used by nafalize and nafilter. Defines an 
 * API for sorting collections of flows. The flow sorting API also
 * provides a facility for parsing sort expressions from user input.
 */

/* idem hack */
#ifndef _NAF_SORT_H_
#define _NAF_SORT_H_

#include <naf/autoinc.h>
#include <naf/nafcore.h>

/** 
 * Count of fields subject to sorting. When new fields are added to 
 * the NAFlow structure and the sort facility, update this count.
 */
#define NAF_SORT_FIELDCOUNT 18

/**
 * A sorter flush function. Called by naf_sort_flush to write a NAFlow 
 * to a private context. Used by nafalize and nafilter to write sorted
 * NAFlow tree to output.
 */
typedef gboolean (*NAFSortFlushFn) (
    void            *ctx,
    NAFlow          *flow,
    GError          **err);

/** A sort context for sorting NAFlows */
typedef struct _NAFSorter {
    /** Flow storage memory chunk */
    GMemChunk       *chunk;
    /** Flow array to sort */
    GPtrArray       *array;
    /** Sort descriptor; array of NAF_FM_* bits describing order of fields */
    uint32_t        desc[NAF_SORT_FIELDCOUNT];
    /** Limit of records to output per bin; 0 for unlimited. */
    uint32_t        limit;
    /** Internal flag used to track whether the descriptor is valid */
    gboolean        defined;
    /** Internal flag used to track whether sorter is active.
     *  If TRUE, array and chunk are valid. */
    gboolean        active;
} NAFSorter;

/**
 * Initialize a new sorter. Call this once to zero the fields of an 
 * allocated sorter structure; use naf_sort_reinit() after flush to zero 
 * an active sorter, instead.
 *
 * @param sorter NAFSorter to initialize
 */
 
void naf_sort_init(
    NAFSorter           *sorter);

/**
 * Reinitialize a sorter for sorting a new bin after a naf_sort_flush() call. 
 * Clears out the sorter tree and flow chunk, but leaves the descriptor 
 * prepared by naf_sort_parse() intact.
 *
 * @param sorter NAFSorter to reinitialize
 */
 
void naf_sort_reinit(
    NAFSorter           *sorter);

/**
 * Begin sorting flows with an initalized or reinitialized NAFSorter.
 *
 * @param sorter NAFSorter to start sorting flows with.
 */

void naf_sort_begin(
    NAFSorter           *sorter);

/**
 * Add a flow to a sorter. Inserts the flow into the sort tree. 
 * It is assumed this flow is in the same bin as the other flows in the sorter. 
 * The caller is responsible for the storage of the given NAFlow instance.
 *
 * @param sorter NAFSorter to add flow to.
 * @param flow NAFlow to sort. This flow is added to the sorter by reference.
 */
 
void naf_sort_flow(
    NAFSorter           *sorter,
    NAFlow              *flow);

/**
 * Add a flow to a sorter. Copies the flow into the sort tree. 
 * It is assumed this flow is in the same bin as the other flows in the sorter. 
 * It is safe to reuse or free the key and val parameters after this call.
 *
 * @param sorter NAFSorter to add flow to.
 * @param key NAFlowKey of flow to sort. This flow key is copied.
 * @param val NAFlowVal of flow to sort. This flow value is copied.
 */
 
void naf_sort_flow_kv(
    NAFSorter           *sorter,
    NAFlowKey           *key,
    NAFlowVal           *val);

/**
 * Flush a sorter. Calls the supplied sort flush function once per sorted 
 * flow, in sort order. Call this after adding all the flows in a given bin to
 * the sorter.
 *
 * @param sorter NAFSorter to flush
 * @param ffn Flush function to call for each NAFlow in the sorter
 * @param fctx Context to pass to flush function
 * @param err An error description
 * @return TRUE if flush function succeeded for every flow, FALSE otherwise.
 */

gboolean naf_sort_flush(
    NAFSorter           *sorter,
    NAFSortFlushFn      ffn,
    void                *fctx,
    GError              **err);

/**
 * Print a parseable sort descriptor for a given sorter.
 *
 * @param sorter NAFSorter to print descriptor from
 * @param str GString to append parseable sort descriptor to
 */

void naf_sort_print(
    NAFSorter           *sorter,
    GString             *str);

/**
 * Parse a sort descriptor. 
 * 
 * @param scanner GScanner (from naf_lex_init()) to parse sort descriptor from.
 * @param sorter NAFSorter to fill descriptor in on; must be newly intialized.
 * @return FALSE if parsing failed
 */

gboolean naf_sort_parse(
    GScanner            *scanner,
    NAFSorter           *sorter);

/* end idem */
#endif
