/*
 ** aggregate.c
 ** NAF second-stage flow table (aggregates and filters flushed flows)
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/aggregate.h>
#include <naf/dynflow.h>
#include <naf/filter.h>
#include <naf/sort.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: aggregate.c 6585 2007-03-08 00:08:38Z bht $";

/* Debug flags */
#define NF_DEBUG_AGGREGATE 0

typedef struct _NAFAggregator {
    NAFlowMask      *mask;
    NAFilter        *filter;
    GHashTable      *table;
    NAFSorter       *sorter;
    GMemChunk       *keychunk;
    GMemChunk       *valchunk;
    GMemChunk       *vucchunk;
} NAFAggregator;

typedef struct _NAFAggregatorFlushContext {
    NAFAggregator   *agg;
    fBuf_t          *fbuf;
    gboolean        *rc;
    GError          **err;
} NAFAggregatorFlushContext;

static void    naf_aggregator_create_entry(
    NAFlowKey       *key,
    NAFlowVal       *val,
    NAFAggregator   *agg)
{
    NAFlowKey   mkey, *tkey;
    NAFlowVal   *tval;
        
    /* Check for filter match */
    if (!naf_filter_key(agg->filter, key)) return;
    if (!naf_filter_val(agg->filter, val)) return;
    
    /* Mask the key */
    naf_flowkey_mask(key, &mkey, agg->mask);
    
    /* Look up flow key in table */
    tval = g_hash_table_lookup(agg->table, &mkey);
    
    if (tval) {
        /* node exists; increment counters */
        tval->oct += val->oct;
        tval->roct += val->roct;
        tval->pkt += val->pkt;
        tval->rpkt += val->rpkt;
        tval->flo += val->flo;
        tval->rflo += val->rflo;
        
        /* add to VUC lists if necessary */
        if (tval->vuc) {
            if (tval->vuc->htab) {
                g_hash_table_insert(tval->vuc->htab, 
                                    GUINT_TO_POINTER(key->sip), 
                                    GUINT_TO_POINTER(1));
            }
            if (tval->vuc->rhtab) {
                g_hash_table_insert(tval->vuc->rhtab, 
                                    GUINT_TO_POINTER(key->dip), 
                                    GUINT_TO_POINTER(1));

            }
            if (tval->vuc->ptab) {
                g_hash_table_insert(tval->vuc->ptab, 
                                    GUINT_TO_POINTER((uint32_t)key->sp), 
                                    GUINT_TO_POINTER(1));
            }
            if (tval->vuc->rptab) {
                g_hash_table_insert(tval->vuc->rptab, 
                                    GUINT_TO_POINTER((uint32_t)key->dp), 
                                    GUINT_TO_POINTER(1));

            }
        }
#if NF_DEBUG_AGGREGATE
        nfDumpFlow(&mkey, tval, "ahit");
#endif
    } else {
        /* node does not exist; create new node */
        tkey = naf_flowkey_alloc(agg->keychunk, &mkey);
        tval = naf_flowval_alloc(agg->valchunk, val);
        
        /* create VUC lists if necessary */
        if (agg->mask->fieldmask & NAF_FM_SHOSTC) {
            if (!tval->vuc) tval->vuc = naf_flowvuc_alloc(agg->vucchunk);
            tval->vuc->htab = g_hash_table_new(g_direct_hash, g_direct_equal);
            g_hash_table_insert(tval->vuc->htab, 
                                GUINT_TO_POINTER(key->sip), 
                                GUINT_TO_POINTER(1));
        }

        if (agg->mask->fieldmask & NAF_FM_DHOSTC) {
            if (!tval->vuc) tval->vuc = naf_flowvuc_alloc(agg->vucchunk);        
            tval->vuc->rhtab = g_hash_table_new(g_direct_hash, g_direct_equal);
            g_hash_table_insert(tval->vuc->rhtab, 
                                GUINT_TO_POINTER(key->dip), 
                                GUINT_TO_POINTER(1));
        }
        
        if (agg->mask->fieldmask & NAF_FM_SPORTC) {
            if (!tval->vuc) tval->vuc = naf_flowvuc_alloc(agg->vucchunk);
            tval->vuc->ptab = g_hash_table_new(g_direct_hash, g_direct_equal);
            g_hash_table_insert(tval->vuc->ptab, 
                                GUINT_TO_POINTER((uint32_t)key->sp), 
                                GUINT_TO_POINTER(1));
        }

        if (agg->mask->fieldmask & NAF_FM_DPORTC) {
            if (!tval->vuc) tval->vuc = naf_flowvuc_alloc(agg->vucchunk);        
            tval->vuc->rptab = g_hash_table_new(g_direct_hash, g_direct_equal);
            g_hash_table_insert(tval->vuc->rptab, 
                                GUINT_TO_POINTER((uint32_t)key->dp), 
                                GUINT_TO_POINTER(1));
        }

        /* insert new node in table */
        g_hash_table_insert(agg->table, tkey, tval);
#if NF_DEBUG_AGGREGATE
        nfDumpFlow(tkey, tval, "aggr");
#endif
    }
}

static NAFAggregator *naf_aggregator_create(
    NAFBinTable     *bintable,
    NAFlowMask      *mask,
    NAFilter        *filter,
    NAFSorter       *sorter)
{
    NAFAggregator   *agg = NULL;
    
    /* Create a new aggregator */
    agg = g_new0(NAFAggregator, 1);
    agg->mask = mask;
    agg->filter = filter;
    agg->sorter = sorter;
    agg->table = g_hash_table_new((GHashFunc)naf_flowkey_hash, 
                                  (GEqualFunc)naf_flowkey_equal);
    agg->keychunk = g_mem_chunk_new("akey", sizeof(NAFlowKey),
                                        4096 * sizeof(NAFlowKey),
                                        G_ALLOC_ONLY);
    agg->valchunk = g_mem_chunk_new("aval", sizeof(NAFlowVal),
                                        4096 * sizeof(NAFlowVal),
                                        G_ALLOC_ONLY);
    agg->vucchunk = g_mem_chunk_new("avuc", sizeof(NAFlowVUC),
                                        4096 * sizeof(NAFlowVUC),
                                        G_ALLOC_ONLY);
    
    /* Build the tree from the table */
    g_hash_table_foreach(bintable->table, 
                         (GHFunc)naf_aggregator_create_entry, agg);
    
/*
    {
        char binbuf[AIR_TIME_BUF_MINSZ];
        air_time_buf_print(binbuf, bintable->bin, AIR_TIME_ISO8601);
        g_debug("Aggregated bin %s (%u entries)", binbuf, 
                g_hash_table_size(agg->table));
    }
*/
    
    /* Return the tree */
    return agg;
}

static gboolean naf_aggregator_flush_entry(
    NAFlowKey                   *key,
    NAFlowVal                   *val,
    NAFAggregatorFlushContext   *fctx)
{
    /* move VUC lists to counts */
    if (val->vuc) {
        if (val->vuc->htab) {
            val->host = g_hash_table_size(val->vuc->htab);
            g_hash_table_destroy(val->vuc->htab);
            val->vuc->htab = NULL;
        }
        if (val->vuc->rhtab) {
            val->rhost = g_hash_table_size(val->vuc->rhtab);
            g_hash_table_destroy(val->vuc->rhtab);
            val->vuc->rhtab = NULL;
        }
        if (val->vuc->ptab) {
            val->port = g_hash_table_size(val->vuc->ptab);
            g_hash_table_destroy(val->vuc->ptab);
            val->vuc->ptab = NULL;
        }
        if (val->vuc->rptab) {
            val->rport = g_hash_table_size(val->vuc->rptab);
            g_hash_table_destroy(val->vuc->rptab);
            val->vuc->rptab = NULL;
        }
    }
        
    /* write aggregated flow to output */
    (*fctx->rc) = nfWrite(fctx->fbuf, fctx->agg->mask, key, val, fctx->err);
    
    /* we're done if write fails */
    if (!(*(fctx->rc))) return TRUE;        

    /* otherwise, keep going */
    return FALSE;
}

static gboolean naf_aggregator_flush_unsorted(
    NAFAggregator               *agg,
    fBuf_t                      *fbuf,
    GError                      **err)
{
    NAFAggregatorFlushContext   fctx;
    gboolean                    rc = TRUE;
    
    /* create flush context */
    fctx.agg = agg;
    fctx.fbuf = fbuf;
    fctx.rc = &rc;
    fctx.err = err;
    
    /* flush aggregated records to disk */
    g_hash_table_foreach(agg->table, 
                         (GHFunc)naf_aggregator_flush_entry, &fctx);
    
    /* nuke aggregator */
    g_hash_table_destroy(agg->table);
    g_mem_chunk_destroy(agg->keychunk);
    g_mem_chunk_destroy(agg->valchunk);
    g_mem_chunk_destroy(agg->vucchunk);
    g_free(agg);
    
    /* return flush result */
    return rc;
}

static void naf_aggregator_sort_entry(
    NAFlowKey                   *key,
    NAFlowVal                   *val,
    NAFAggregatorFlushContext   *fctx)
{
    /* just sort the flow */
    naf_sort_flow_kv(fctx->agg->sorter, key, val);
}

static gboolean naf_aggregator_flush_sorted_inner(
    NAFAggregatorFlushContext   *fctx,
    NAFlow                      *flow,
    GError                      **err)
{
    naf_aggregator_flush_entry(&(flow->k), &(flow->v), fctx);
    return *(fctx->rc);
}

static gboolean naf_aggregator_flush_sorted(
    NAFAggregator               *agg,
    fBuf_t                      *fbuf,
    GError                      **err)
{
    NAFAggregatorFlushContext   fctx;
    gboolean                    rc = TRUE;
    
    /* create flush context */
    fctx.agg = agg;
    fctx.fbuf = fbuf;
    fctx.rc = &rc;
    fctx.err = err;

    /* begin sort */
    naf_sort_begin(agg->sorter);

    /* flush aggregated records to sorter */
    g_hash_table_foreach(agg->table, 
                         (GHFunc)naf_aggregator_sort_entry, &fctx);
    
    /* then flush the sorter */
    rc = naf_sort_flush(agg->sorter, 
                        (NAFSortFlushFn)naf_aggregator_flush_sorted_inner, 
                        &fctx, err);
    naf_sort_reinit(agg->sorter);
    
    /* nuke aggregator */
    g_hash_table_destroy(agg->table);
    g_mem_chunk_destroy(agg->keychunk);
    g_mem_chunk_destroy(agg->valchunk);
    g_mem_chunk_destroy(agg->vucchunk);
    g_free(agg);
    
    /* return flush result */
    return rc;
}

gboolean naf_aggregate(
    gboolean        flush,
    NAFMultiBin     *mtab,
    NAFAggConfig    *conf,
    fBuf_t          **fbuf_ary,
    GError          **err)
{
    NAFBinTable     *bintable = NULL;
    NAFAggregator   *agg = NULL;
    uint32_t        i;

    while ((bintable = naf_mtab_dequeue(mtab, flush ? 0 : conf->horizon))) {
        for (i = 0; i < conf->fanout; i++) {
            agg = naf_aggregator_create(bintable, &(conf->mask[i]),
                                         &(conf->filter[i]),
                                         &(conf->sorter[i]));
            if (conf->sorter[i].defined) {
                if (!naf_aggregator_flush_sorted(agg, fbuf_ary[i], err)) 
                    return FALSE;
            } else {
                if (!naf_aggregator_flush_unsorted(agg, fbuf_ary[i], err)) 
                    return FALSE;
            }
        }
        naf_mtab_bintable_complete(bintable);
    }
    return TRUE;
}
