/*
 ** drv_argus.c
 ** NAF RA (Argus) v2.0.6 input driver
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include "nafalize.h"
#include "nafz_indrv.h"

static char *RCSID __attribute__ ((unused)) = 
    "$Id: drv_argus.c 6585 2007-03-08 00:08:38Z bht $";

#define NAF_RA_HDR_SZ 16
#define NAF_RA_MIN_SZ 84
#define NAF_RA_MAX_SZ 4096

#define NAF_RA_FARTYPE      1
#define NAF_RA_ETHERMASK    0x0000FFFF
#define NAF_RA_ETHERIP      0x0800
#define NAF_RA_ETH_SZ       14

#define buf_ra_type                (*((uint8_t  *)( buf +  0 )))
#define buf_ra_length       g_ntohs(*((uint16_t *)( buf +  2 )))
#define buf_ra_status       g_ntohl(*((uint32_t *)( buf +  4 )))
#define buf_ra_st_sec       g_ntohl(*((uint32_t *)( buf + 24 )))
#define buf_ra_st_usec      g_ntohl(*((uint32_t *)( buf + 28 )))
#define buf_ra_et_sec       g_ntohl(*((uint32_t *)( buf + 32 )))
#define buf_ra_et_usec      g_ntohl(*((uint32_t *)( buf + 36 )))
#define buf_ip_sip          g_ntohl(*((uint32_t *)( buf + 40 )))
#define buf_ip_dip          g_ntohl(*((uint32_t *)( buf + 44 )))
#define buf_ip_proto               (*((uint8_t  *)( buf + 48 )))
#define buf_ip_sp           g_ntohs(*((uint16_t *)( buf + 50 )))
#define buf_ip_dp           g_ntohs(*((uint16_t *)( buf + 52 )))
#define buf_ip_it                  (*((uint8_t  *)( buf + 50 )))
#define buf_ip_ic                  (*((uint8_t  *)( buf + 51 )))
#define buf_met_sp          g_ntohl(*((uint32_t *)( buf + 64 )))
#define buf_met_so          g_ntohl(*((uint32_t *)( buf + 68 )))
#define buf_met_dp          g_ntohl(*((uint32_t *)( buf + 76 )))
#define buf_met_do          g_ntohl(*((uint32_t *)( buf + 80 )))


static gboolean nafz_argus_open(
    MIOSource           *source,
    void                **dctx,
    GError              **err) 
{        
    *dctx = NULL;
    return TRUE;
}

static gboolean nafz_argus_read(
    MIOSource           *source,
    void                *dctx,
    NAFlowRaw           *flow,
    GError              **err)
{
    static char         buf[NAF_RA_MAX_SZ];
    gboolean            ok = TRUE;
    int                 rc;
    
    while (1) {
        rc = fread(buf, NAF_RA_HDR_SZ, 1, mio_fp(source));
        if (rc == 0) {
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_EOF, "end of file");
            ok = FALSE;
            goto end;
        } else if (rc < 0) {
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO, 
                        "I/O error reading from %s: %s", 
                        source->name, strerror(errno));
            ok = FALSE;
            goto end;
        }
    
        /* die if record length is out of bounds */
        if (buf_ra_length <  NAF_RA_HDR_SZ ||
            buf_ra_length > NAF_RA_MAX_SZ) {
            /* create error */
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_HEADER,
                        "Bad RA record length: %u", buf_ra_length);
            /* and stop */
            ok = FALSE;
            goto end;
        }

        /* at least the record length is okay. read the rest of the record. */
        rc = fread(buf + NAF_RA_HDR_SZ, buf_ra_length - NAF_RA_HDR_SZ, 
                   1, mio_fp(source));
        if (rc == 0) {
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_EOF, "end of file");
            ok = FALSE;
            goto end;
        } else if (rc < 0) {
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO, 
                        "I/O error reading from %s: %s", 
                        source->name, strerror(errno));
            ok = FALSE;
            goto end;
        }
                
        /* here begins the decoding */

        /* skip record if not IP FAR */
        if (buf_ra_type != NAF_RA_FARTYPE || 
            (buf_ra_status & NAF_RA_ETHERMASK) != NAF_RA_ETHERIP) continue;

        /* copy data into raw output record */
        flow->srcid = 0; /* FIXME we could use argus' source ID */
        flow->stime = buf_ra_st_sec;
        flow->etime = buf_ra_et_sec;
        flow->sip = buf_ip_sip;
        flow->dip = buf_ip_dip;
        flow->sipmask = 32;
        flow->dipmask = 32;
        flow->proto = buf_ip_proto;
        switch (flow->proto) {
        case NAF_IP_TCP:
        case NAF_IP_UDP:
            flow->sp = buf_ip_sp;
            flow->dp = buf_ip_dp;
            break;
        case NAF_IP_ICMP:
            flow->sp = 0;
            flow->dp = ((buf_ip_it << 8) | buf_ip_ic);
            break;
        default:
            flow->sp = flow->dp = 0;
            break;
        }
        
        flow->oct = buf_met_so;
        flow->roct = buf_met_do;
        flow->pkt = buf_met_sp;
        flow->rpkt = buf_met_dp;
        flow->flo = 1;
        flow->rflo = (flow->rpkt) ? 1 : 0;
        
        /* HACK - Remove 14 octets per packet for ethernet header */
        flow->oct -= (NAF_RA_ETH_SZ * flow->pkt);
        flow->roct -= (NAF_RA_ETH_SZ * flow->rpkt);

        /* if we're here, we're done. */
        break;
    }    

 end:
    return ok;
}

static gboolean nafz_argus_close(
    MIOSource           *source,
    void                **dctx,
    GError              **err)
{        
    return TRUE;
}

static NAFDriver argus_drv =   {"argus2", 
                                MIO_T_FP,
                                (NAFDriverFn)nafz_argus_open, 
                                (NAFDriverReadFn)nafz_argus_read, 
                                (NAFDriverFn)nafz_argus_close };

void nafz_argus_register() 
{
    nafz_drv_register(&argus_drv);
}
