/*
 ** drv_renaf.c
 ** NAF NAF input driver
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/autoinc.h>
#include <naf/nafcore.h>
#include "nafalize.h"
#include "nafz_indrv.h"

static char *RCSID __attribute__ ((unused)) = 
    "$Id: drv_renaf.c 6585 2007-03-08 00:08:38Z bht $";

static fBuf_t *fbuf = NULL;

static gboolean nafz_renaf_open(
    MIOSource           *source,
    void                **dctx,
    GError              **err) {
    
    /* allocate mask */
    NAFlowMask      *mask = g_new0(NAFlowMask, 1);
    
    /* read NAF header and mask */
    fbuf = nfReaderForFP(fbuf, mio_fp(source), mask, err);
    if (fbuf == NULL) {
        g_free(mask);
        return FALSE;
    }
    
    /* return mask */
    *dctx = mask;
    
    return TRUE;
}

static gboolean nafz_renaf_read(
    MIOSource           *source,
    void                *dctx,
    NAFlowRaw           *flow,
    GError              **err) {

    NAFlowKey       key;
    NAFlowVal       val;
    NAFlowMask      *mask = (NAFlowMask *)dctx;
    
    /* read a record */
    if (!nfRead(fbuf, mask, &key, &val, err)) {
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_EOF, "end of file");            
        } 
        return FALSE;
    }
    
    /* copy it into the flow */
    flow->srcid = key.srcid;
    flow->stime = key.bin;
    flow->etime = key.bin + mask->binsize - 1;
    flow->sip = key.sip;
    flow->dip = key.dip;
    flow->sp = key.sp;
    flow->dp = key.dp;
    flow->sipmask = key.sipmask;
    flow->dipmask = key.dipmask;
    flow->proto = key.proto;
    flow->oct = val.oct;
    flow->roct = val.roct;
    flow->pkt = val.pkt;
    flow->rpkt = val.rpkt;
    flow->flo = val.flo;
    flow->rflo = val.rflo;
    
    return TRUE;
}


static gboolean nafz_renaf_close(
    MIOSource           *source,
    void                **dctx,
    GError              **err) {

    /* free mask */
    if (*dctx) g_free(*dctx);
    
    /* all done */
    return TRUE;
}

static NAFDriver renaf_drv =   {"naf",
                                MIO_T_FP, 
                                (NAFDriverFn)nafz_renaf_open, 
                                (NAFDriverReadFn)nafz_renaf_read, 
                                (NAFDriverFn)nafz_renaf_close };

void nafz_renaf_register() {
    nafz_drv_register(&renaf_drv);
}
