/*
 ** dynflow.c
 ** NAF dynamic flow allocation and manipulation
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#include <airframe/airutil.h>
#include <naf/dynflow.h>

#define NAF_IPM_MASK    0x003F
#define NAF_IPM_F_VALID 0x8000

static char *RCSID __attribute__ ((unused)) = 
    "$Id: dynflow.c 6585 2007-03-08 00:08:38Z bht $";

NAFlowKey       *naf_flowkey_alloc(
    GMemChunk           *keychunk,
    NAFlowKey           *key) 
{
    NAFlowKey           *nkey = g_chunk_new0(NAFlowKey, keychunk);
    if (key) memcpy(nkey, key, sizeof(NAFlowKey));
    return nkey;
}

NAFlowVal       *naf_flowval_alloc(
    GMemChunk           *valchunk,
    NAFlowVal           *val) 
{
    NAFlowVal           *nval = g_chunk_new0(NAFlowVal, valchunk);
    if (val) memcpy(nval, val, sizeof(NAFlowVal));
    return nval;
}

NAFlowVUC       *naf_flowvuc_alloc(
    GMemChunk           *vucchunk) 
{
    return g_chunk_new0(NAFlowVUC, vucchunk);
}

NAFlow          *naf_flow_alloc(
    GMemChunk           *flowchunk,
    NAFlowKey           *key,
    NAFlowVal           *val)
{
    NAFlow              *nflow = g_chunk_new0(NAFlow, flowchunk);
    if (key) memcpy(&(nflow->k), key, sizeof(NAFlowKey));
    if (val) memcpy(&(nflow->v), val, sizeof(NAFlowVal));
    return nflow;
}

uint32_t naf_flowkey_hash(
    NAFlowKey       *key)
{
    return  key->srcid ^ key->sip ^ key->dip ^ 
            (key->sp << 16) ^ key->dp ^ 
            (key->sipmask << 19) ^ (key->dipmask << 14) ^ (key->proto << 24);
}

gboolean naf_flowkey_equal(
    NAFlowKey       *a,
    NAFlowKey       *b)
{
    if (a->srcid == b->srcid &&
    a->sip == b->sip &&
    a->dip == b->dip &&
    a->sp == b->sp &&
    a->dp == b->dp &&
    a->proto == b->proto &&
    a->sipmask == b->sipmask &&
    a->dipmask == b->dipmask) return TRUE;
    else return FALSE;
}

int naf_flowkey_cmp(
    const void          *a, 
    const void          *b, 
    void                *u)
{
    NAFlowKey   *ak = (NAFlowKey *)a;
    NAFlowKey   *bk = (NAFlowKey *)b;

    if (ak->bin > bk->bin) return 1;
    if (ak->bin < bk->bin) return -1;
    if (ak->srcid > bk->srcid) return 1;
    if (ak->srcid < bk->srcid) return -1;
    if (ak->sip > bk->sip) return 1;
    if (ak->sip < bk->sip) return -1;
    if (ak->sipmask > bk->sipmask) return 1;
    if (ak->sipmask < bk->sipmask) return -1;
    if (ak->dip > bk->dip) return 1;
    if (ak->dip < bk->dip) return -1;
    if (ak->dipmask > bk->dipmask) return 1;
    if (ak->dipmask < bk->dipmask) return -1;
    if (ak->proto > bk->proto) return 1;
    if (ak->proto < bk->proto) return -1;
    if (ak->sp > bk->sp) return 1;
    if (ak->sp < bk->sp) return -1;
    if (ak->dp > bk->dp) return 1;
    if (ak->dp < bk->dp) return -1;
    return 0;
}

void naf_flowkey_mask(
    NAFlowKey       *in,
    NAFlowKey       *out,
    NAFlowMask      *mask)
{
    /* lazy calculation of sip/dip mask bits */
    if (!(mask->sipmask & NAF_IPM_F_VALID)) {
        mask->sipmaskbits = air_mask_from_prefix(mask->sipmask & NAF_IPM_MASK);
        mask->sipmask |= NAF_IPM_F_VALID;
    }

    if (!(mask->dipmask & NAF_IPM_F_VALID)) {
        mask->dipmaskbits = air_mask_from_prefix(mask->dipmask & NAF_IPM_MASK);
        mask->dipmask |= NAF_IPM_F_VALID;
    }
    
    /* mask fields */
    out->bin = in->bin;
    out->srcid = (mask->fieldmask & NAF_FM_SRCID) ? in->srcid : 0;
    out->sp = (mask->fieldmask & NAF_FM_SP) ? in->sp : 0;
    out->dp = (mask->fieldmask & NAF_FM_DP) ? in->dp : 0;
    out->proto = (mask->fieldmask & NAF_FM_PROTO) ? in->proto : 0;
    
    /* mask addresses */
    if (in->sipmask > (mask->sipmask & NAF_IPM_MASK)) {
        out->sip = in->sip & mask->sipmaskbits;
        out->sipmask = mask->sipmask & NAF_IPM_MASK;
    } else {
        out->sip = in->sip;
        out->sipmask = in->sipmask;
    }

    if (in->dipmask > (mask->dipmask & NAF_IPM_MASK)) {
        out->dip = in->dip & mask->dipmaskbits;
        out->dipmask = mask->dipmask & NAF_IPM_MASK;
    } else {
        out->dip = in->dip;
        out->dipmask = in->dipmask;
    }
    
}

void naf_flowkey_reverse(
    NAFlowKey       *in,
    NAFlowKey       *out)
{    
    out->srcid = in->srcid;
    out->bin = in->bin;
    out->sip = in->dip;
    out->sipmask = in->dipmask;
    out->dip = in->sip;
    out->dipmask = in->sipmask;
    out->proto = in->proto;
    if (out->proto == NAF_IP_ICMP) {
        out->sp = in->sp;
        out->dp = in->dp;
    } else {
        out->sp = in->dp;
        out->dp = in->sp;
    }
}

void naf_flowval_reverse(
    NAFlowVal       *in,
    NAFlowVal       *out)
{    
    out->oct = in->roct;
    out->roct = in->oct;
    out->pkt = in->rpkt;
    out->rpkt = in->pkt;
    out->flo = in->rflo;
    out->rflo = in->flo;
    out->host = in->rhost;
    out->rhost = in->host;
    out->port = in->rport;
    out->rport = in->port;
    g_assert(!in->vuc);
    out->vuc = NULL;
}

NAFTimeSec naf_bin_time(
    NAFTimeSec                  tv,
    NAFTimeSec                  size)
{
    NAFTimeSec                  out;
    out = tv / size;
    out *= size;
    return out;
}
