/*
 ** exparse.c
 ** NAF normalizer and aggregator configuration
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/exparse.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: exparse.c 6585 2007-03-08 00:08:38Z bht $";

static gboolean naf_exparse_bin(
    GScanner            *scanner,
    uint32_t            *binsize,
    uint32_t            *binalg)
{
    /* First token must be "bin" */
    if (g_scanner_get_next_token(scanner) != (GTokenType)NAF_SYM_BIN) {
        g_scanner_error(scanner, "Missing required bin specifier");
        return FALSE;
    }

    /* Parse bin algorithm if present */
    *binalg = NAF_BA_UNIFORM;
    g_scanner_get_next_token(scanner);
    if (scanner->token == (GTokenType)NAF_SYM_START) {
        *binalg = NAF_BA_START;
        g_scanner_get_next_token(scanner);
    } else if (scanner->token == (GTokenType)NAF_SYM_UNIFORM) {
        g_scanner_get_next_token(scanner);
    } else if (scanner->token == (GTokenType)NAF_SYM_END) {
        *binalg = NAF_BA_END;
        g_scanner_get_next_token(scanner);
    }
    
    /* The next thing out of the stream must be a number */
    if (scanner->token != G_TOKEN_INT) {
        g_scanner_error(scanner, "Missing required bin time");
        return FALSE;
    }
    
    *binsize = scanner->value.v_int;
    
    /* And the next must be a time specifier */
    g_scanner_get_next_token(scanner);
    if (scanner->token == (GTokenType)NAF_SYM_HR) {
        *binsize *= 3600;
    } else if (scanner->token == (GTokenType)NAF_SYM_MIN) {
        *binsize *= 60;
    } else if (scanner->token != (GTokenType)NAF_SYM_SEC) {
        g_scanner_error(scanner, "Missing required bin time unit");
    }
    
    /* All done */
    return TRUE;
}

static gboolean naf_exparse_aggregate(
    GScanner            *scanner,
    NAFTimeSec          binsize,
    uint32_t            binalg,
    NAFilter            *filter,
    NAFSorter           *sorter,
    NAFlowMask          *mask,
    char                **label)
{
    /* consume the aggregate token */
    g_assert(g_scanner_get_next_token(scanner) == (GTokenType)NAF_SYM_AGGREGATE);

    /* parse the aggregate list */
    while (1) {
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_SIP) {
            /* Aggregate source IP */
            mask->fieldmask |= NAF_FM_SIP;
            g_scanner_get_next_token(scanner);
            /* Look for a masklength */
            if (g_scanner_peek_next_token(scanner) == '/') {
                g_scanner_get_next_token(scanner);
                if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                    g_scanner_error(scanner, 
                                    "Missing sip aggregation mask length");
                    return FALSE;
                }
                if (scanner->value.v_int < 0 || scanner->value.v_int > 32) {
                                     g_scanner_error(scanner, 
                                    "Bad sip aggregation mask length");
                    return FALSE;
                }
                mask->sipmask = scanner->value.v_int;
                mask->fieldmask |= NAF_FM_SIPMASK;
            } else {
                /* sip mask is implicitly 32 */
                mask->sipmask = 32;
            }
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DIP) {
            /* Aggregate destination IP */
            mask->fieldmask |= NAF_FM_DIP;
            g_scanner_get_next_token(scanner);
            /* Look for a masklength */
            if (g_scanner_peek_next_token(scanner) == '/') {
                g_scanner_get_next_token(scanner);
                if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                    g_scanner_error(scanner, 
                                    "Missing dip aggregation mask length");
                    return FALSE;
                }
                if (scanner->value.v_int < 0 || scanner->value.v_int > 32) {
                    g_scanner_error(scanner, 
                                    "Bad dip aggregation mask length");
                    return FALSE;
                }
                mask->dipmask = scanner->value.v_int;
                mask->fieldmask |= NAF_FM_DIPMASK;
            } else {
                /* dip mask is implicitly 32 */
                mask->dipmask = 32;
            }
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SP) {
            /* Aggregate source port */
            mask->fieldmask |= NAF_FM_SP;
            g_scanner_get_next_token(scanner);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DP) {
            /* Aggregate destination port */
            mask->fieldmask |= NAF_FM_DP;
            g_scanner_get_next_token(scanner);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PROTO) {
            /* Aggregate protocol */
            mask->fieldmask |= NAF_FM_PROTO;
            g_scanner_get_next_token(scanner);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SRCID) {
            /* Aggregate source ID */
            mask->fieldmask |= NAF_FM_SRCID;
            g_scanner_get_next_token(scanner);
        } else {
            /* We're moving on. */
            break;
        }
    }
    
    /* parse the count list */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_COUNT) {
        g_scanner_get_next_token(scanner);
        
        while (1) {
            g_scanner_peek_next_token(scanner);
            if (scanner->next_token == (GTokenType)NAF_SYM_TOTAL) {
                mask->fieldmask |= NAF_FM_MTOTAL;
                g_scanner_get_next_token(scanner);
            } else if (scanner->next_token == (GTokenType)NAF_SYM_HOSTS) {
                mask->fieldmask |= (NAF_FM_SHOSTC | NAF_FM_DHOSTC);
                g_scanner_get_next_token(scanner);
            } else if (scanner->next_token == (GTokenType)NAF_SYM_PORTS) {
                mask->fieldmask |= (NAF_FM_SPORTC | NAF_FM_DPORTC);
                g_scanner_get_next_token(scanner);
            } else if (scanner->next_token == (GTokenType)NAF_SYM_FLOWS) {
                mask->fieldmask |= (NAF_FM_FLO | NAF_FM_RFLO);
                g_scanner_get_next_token(scanner);
            } else if (scanner->next_token == (GTokenType)NAF_SYM_PACKETS) {
                mask->fieldmask |= (NAF_FM_PKT | NAF_FM_RPKT);
                g_scanner_get_next_token(scanner);
            } else if (scanner->next_token == (GTokenType)NAF_SYM_OCTETS) {
                mask->fieldmask |= (NAF_FM_OCT | NAF_FM_ROCT);
                g_scanner_get_next_token(scanner);
            } else {
                break;
            }
        }
    } 
    
    /* parse filter */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_FILTER) {
        if (!naf_filter_parse(scanner, filter)) {
            return FALSE;
        }
    }    
    
    /* parse sorter */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_SORT) {
        if (!naf_sort_parse(scanner, sorter)) {
            return FALSE;
        }
    }
        
    /* parse label */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_LABEL) {
        g_scanner_get_next_token(scanner);
        if (g_scanner_get_next_token(scanner) != G_TOKEN_IDENTIFIER) {
            g_scanner_error(scanner, "Bad aggregation label");
            return FALSE;
        }
        *label = g_strdup(scanner->value.v_identifier);
    }
    
    /* set bin information */
    mask->binsize = binsize;
    mask->binalg = binalg;
    
    /* All done */
    return TRUE;
}

static void naf_exparse_init(
    NAFAggConfig  *conf)
{
    uint32_t        i;
    
    conf->horizon = 0;
    conf->binsize = 0;
    conf->binalg = 0;
    conf->uniflow_mode = FALSE;
    conf->perimeter = NULL;
    naf_filter_init(&(conf->prefilter));
    conf->fanout = 0;
    for (i = 0; i < NAFZ_MAX_FANOUT; i++) {
        naf_filter_init(&(conf->filter[i]));
        naf_sort_init(&(conf->sorter[i]));
        conf->mask[i].sipmask = 0;
        conf->mask[i].dipmask = 0;
        conf->mask[i].sipmaskbits = 0;
        conf->mask[i].dipmaskbits = 0;
        conf->mask[i].fieldmask = 0;
        conf->mask[i].binsize = 0;
        conf->mask[i].binalg = 0;
        conf->label[i] = NULL;
    }
}

static gboolean naf_exparse_expr(
    GScanner            *scanner,
    NAFAggConfig      *conf)
{
    /* Clear configuration */
    naf_exparse_init(conf);

    /* Parse time bin */
    if (!naf_exparse_bin(scanner, &(conf->binsize), &(conf->binalg))) {
        return FALSE;
    }
    
    /* Parse modifiers */
    while (1) {
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_UNIFLOW) {
            g_scanner_get_next_token(scanner);
            conf->uniflow_mode = TRUE;
        } else {
            break;
        }
    }
    
    /* Check for perimeter */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_PERIMETER) {
        g_scanner_get_next_token(scanner);
        naf_filter_rl_parse_ipaddr(scanner, &(conf->perimeter));
    }
    
    /* Check for prefilter */
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_FILTER) {
        if (!naf_filter_parse(scanner, &(conf->prefilter))) {
            return FALSE;
        }
    }
    
    /* Parse aggregations */
    while (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_AGGREGATE) {
        if (conf->fanout >= NAFZ_MAX_FANOUT) {
            g_scanner_error(scanner, 
                            "Maximum fanout (%u) exceeded",
                            NAFZ_MAX_FANOUT);
            return FALSE;
        }
        if (!naf_exparse_aggregate(scanner,
                                    conf->binsize, conf->binalg,
                                    &(conf->filter[conf->fanout]),
                                    &(conf->sorter[conf->fanout]), 
                                    &(conf->mask[conf->fanout]), 
                                    &(conf->label[conf->fanout]))) {
            return FALSE;
        }
        ++(conf->fanout);
    }
    
    /* naf_exparse_aggregate should eat everything 
       to the end of the stream. */
    if (g_scanner_get_next_token(scanner) != G_TOKEN_EOF) {
        g_scanner_error(scanner, 
                        "Malformed expression");
        return FALSE;
    }

    /* All done */
    return TRUE;
}

gboolean naf_exparse(
    int             argc,
    char            *argv[],
    NAFAggConfig  *conf)
{
    /* parse expression from top */
    return naf_exparse_expr(naf_lex_init_argv(argc, argv), conf);
}

static void naf_config_dump_mask(
    NAFlowMask     *mask,
    GString            *str)
{
    if (mask->fieldmask & NAF_FM_SIP) 
        g_string_append_printf(str, "sip/%u ", mask->sipmask);
    if (mask->fieldmask & NAF_FM_DIP) 
        g_string_append_printf(str, "dip/%u ", mask->dipmask);
    if (mask->fieldmask & NAF_FM_SP) g_string_append_printf(str, "sp ");
    if (mask->fieldmask & NAF_FM_DP) g_string_append_printf(str, "dp ");
    if (mask->fieldmask & NAF_FM_PROTO) g_string_append_printf(str, "proto ");
    if (mask->fieldmask & NAF_FM_SRCID) g_string_append_printf(str, "srcid ");
    g_string_append_printf(str, "count ");
    if (mask->fieldmask & NAF_FM_MTOTAL) g_string_append_printf(str, "total ");
    if (mask->fieldmask & NAF_FM_SHOSTC ||
        mask->fieldmask & NAF_FM_DHOSTC) g_string_append_printf(str, "hosts ");
    if (mask->fieldmask & NAF_FM_FLO || 
        mask->fieldmask & NAF_FM_RFLO) g_string_append_printf(str, "flows ");
    if (mask->fieldmask & NAF_FM_PKT || 
        mask->fieldmask & NAF_FM_RPKT) g_string_append_printf(str, "packets ");
    if (mask->fieldmask & NAF_FM_OCT || 
        mask->fieldmask & NAF_FM_ROCT) g_string_append_printf(str, "octets ");
}

void naf_config_dump(
    NAFAggConfig      *conf)
{
    GString             *dump = NULL;
    char                *algstr;
    uint32_t            i;
    
    dump = g_string_new("Normalized aggregation expression: ");
    
    /* Print bin phrase */
    switch (conf->binalg) {
        case NAF_BA_UNIFORM: algstr = NAF_SYM_UNIFORM; break;
        case NAF_BA_START: algstr = NAF_SYM_START; break;
        case NAF_BA_END: algstr = NAF_SYM_END; break;
        default: algstr = "unknown"; break;
    }
    g_string_append_printf(dump, "bin %s %u sec ", algstr, conf->binsize);

    /* Print modifiers */
    if (conf->uniflow_mode) g_string_append(dump, "uniflow ");
    
    /* Print perimeter */
    if (conf->perimeter) {
        g_string_append(dump, "perimeter ");
        naf_filter_rl_print_ipaddr(conf->perimeter, dump);
    }
    
    /* Print prefilter */
    if (naf_filter_active(conf->prefilter)) {
        g_string_append(dump, "filter ");
        naf_filter_print(&(conf->prefilter), dump);
    }
    
    /* Print masks and filters */
    for (i = 0; i < conf->fanout; i++) { 
        g_string_append_printf(dump, "aggregate ");
        naf_config_dump_mask(&(conf->mask[i]), dump);
        if (naf_filter_active(conf->filter[i])) {
            g_string_append(dump, "filter ");
            naf_filter_print(&(conf->filter[i]), dump);
        }
        if (conf->sorter[i].defined) {
            g_string_append(dump, "sort ");
            naf_sort_print(&(conf->sorter[i]), dump);
        }
        if (conf->label[i]) {
            g_string_append_printf(dump, "label %s ", conf->label[i]);
        }
    }
    
    g_warning("%s", dump->str);
    g_string_free(dump, TRUE);
}
    
    
