/*
 ** filter.c
 ** NAF flow filter support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/filter.h>
#include <airframe/airutil.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: filter.c 6585 2007-03-08 00:08:38Z bht $";

typedef struct naf_range_st {
    uint32_t                a;
    uint32_t                b;
} naf_range_t;

static int naf_filter_rl_compare(
    naf_range_t             *x,
    naf_range_t             *y)
{
    return (x->a > y->a) ? 1 : ((y->a > x->a) ? -1 : 0);
}

static void naf_filter_rl_coalesce(
    GArray                  *rl)
{
    naf_range_t             *x, *y;
    uint32_t                i;
    
    /* First, sort the rangelist by range start */
    g_array_sort(rl, (GCompareFunc)naf_filter_rl_compare);
    
    /* Iterate over the rangelist combining overlapping ranges */
    for (i = 0; i < rl->len - 1; i++) {
        x = &g_array_index(rl, naf_range_t, i);
        y = &g_array_index(rl, naf_range_t, i+1);
        if (y->a <= (x->b + 1)) {
            /* Ranges overlap or abut and should be coalesced. */
            /* Extend range if necessary. */
            if (y->b > x->b) x->b = y->b;    
            /* Delete subsequent range */
            g_array_remove_index(rl, i + 1);
            i--;
        }
    }
}

void naf_filter_rl_add(
    GArray                  **rl,
    uint32_t                a,
    uint32_t                b)
{
    naf_range_t             r;
    
    /* create rangelist if necessary */
    if (!*rl)
        *rl = g_array_new(FALSE, TRUE, sizeof(naf_range_t));
    
    /* add range to rangelist */
    if (a > b) { r.a = b; r.b = a; } else { r.a = a; r.b = b; }
    g_array_append_vals(*rl, &r, 1);
}


gboolean naf_filter_rl_contains(
    GArray                  *rl,
    uint32_t                v)
{
    int32_t                si, ei, mi;
    naf_range_t             *m;
    
    /* Consider the entire array to start */
    si = 0;
    ei = rl->len - 1;
    
    while (ei >= si) {
        /* Find middle element in array */
        mi = (si + ei) / 2;
        m = &g_array_index(rl, naf_range_t, mi);
        
        if (v < m->a) {
            /* Too high. Search left half of array. */
            ei = mi - 1;
        } else if (v > m->b) {
            /* Too low. Search right half of array. */
            si = mi + 1;
        } else {
            /* Just right. Done. */
            return TRUE;
        }
    }
    
    /* No more array to search. Done. */
    return FALSE;
}

static gboolean naf_filter_rl_contains_u64(
    GArray                  *rl,
    uint64_t                v)
{
    if (v > UINT32_MAX) v = UINT32_MAX;
    return naf_filter_rl_contains(rl, (uint32_t)v);
}

gboolean naf_filter_key(
    NAFilter                *filter,
    NAFlowKey               *key)
{
    gboolean                rlc;
    
    /* Filter on time ranges if present */
    if (filter->binrl) {
        if (!naf_filter_rl_contains(filter->binrl, key->bin)) {
            return FALSE;
        }
    }
    
    /* Filter on source IP ranges if present */
    if (filter->siprl) {
        rlc = naf_filter_rl_contains(filter->siprl, key->sip);
        if (filter->sipnot ? rlc : !rlc) return FALSE;
    }
    
    /* Filter on destination IP ranges if present */
    if (filter->diprl) {
        rlc = naf_filter_rl_contains(filter->diprl, key->dip);
        if (filter->dipnot ? rlc : !rlc) return FALSE;
    }

    /* Filter on source port ranges if present */
    if (filter->sprl) {
        rlc = naf_filter_rl_contains(filter->sprl, key->sp);
        if (filter->spnot ? rlc : !rlc) return FALSE;
    }

    /* Filter on destination port ranges if present */
    if (filter->dprl) {
        rlc = naf_filter_rl_contains(filter->dprl, key->dp);
        if (filter->dpnot ? rlc : !rlc) return FALSE;
    }

    /* Filter on protocol ranges if present */
    if (filter->protorl) {
        rlc = naf_filter_rl_contains(filter->protorl, key->proto);
        if (filter->protonot ? rlc : !rlc) return FALSE;
    }
    
    
    /* Pass by default */
    return TRUE;
}

gboolean naf_filter_val(
    NAFilter                *filter,    
    NAFlowVal               *val)
{
    if (filter->florl) {
        if (!naf_filter_rl_contains(filter->florl, val->flo))
            return FALSE;
    }

    if (filter->rflorl) {
        if (!naf_filter_rl_contains(filter->rflorl, val->rflo))
            return FALSE;
    }

    if (filter->pktrl) {
        if (!naf_filter_rl_contains_u64(filter->pktrl, val->pkt))
            return FALSE;
    }

    if (filter->rpktrl) {
        if (!naf_filter_rl_contains_u64(filter->rpktrl, val->rpkt))
            return FALSE;
    }

    if (filter->octrl) {
        if (!naf_filter_rl_contains_u64(filter->octrl, val->oct))
            return FALSE;
    }

    if (filter->roctrl) {
        if (!naf_filter_rl_contains_u64(filter->roctrl, val->roct))
            return FALSE;
    }

    /* Pass by default */
    return TRUE;
}


void naf_filter_init(
    NAFilter            *filter)
{
    filter->binrl = NULL;
    filter->siprl = NULL;
    filter->sipnot = FALSE;
    filter->diprl = NULL;
    filter->dipnot = FALSE;
    filter->sprl = NULL;
    filter->spnot = FALSE;
    filter->dprl = NULL;
    filter->dpnot = FALSE;
    filter->protorl = NULL;
    filter->protonot = FALSE;
    filter->florl = NULL;
    filter->rflorl = NULL;
    filter->pktrl = NULL;
    filter->rpktrl = NULL;
    filter->octrl = NULL;
    filter->roctrl = NULL;
}

void naf_filter_reinit(
    NAFilter            *filter)
{
    if (filter->binrl)      g_array_free(filter->binrl, TRUE);
    if (filter->siprl)      g_array_free(filter->siprl, TRUE);
    if (filter->diprl)      g_array_free(filter->diprl, TRUE);
    if (filter->sprl)       g_array_free(filter->sprl, TRUE);
    if (filter->dprl)       g_array_free(filter->dprl, TRUE);
    if (filter->protorl)    g_array_free(filter->protorl, TRUE);
    if (filter->florl)      g_array_free(filter->florl, TRUE);
    if (filter->rflorl)     g_array_free(filter->rflorl, TRUE);
    if (filter->pktrl)      g_array_free(filter->pktrl, TRUE);
    if (filter->rpktrl)     g_array_free(filter->rpktrl, TRUE);
    if (filter->octrl)      g_array_free(filter->octrl, TRUE);
    if (filter->roctrl)     g_array_free(filter->roctrl, TRUE);
    naf_filter_init(filter);
}

static void naf_filter_rl_print_time(
    GArray                  *rl,
    GString                 *str)
{
    uint32_t                i;
    naf_range_t             *r;

    for (i = 0; i < rl->len; i++) {
        r = &g_array_index(rl, naf_range_t, i);
        air_time_g_string_append(str, r->a, AIR_TIME_ISO8601);
        if (r->a != r->b) {
            g_string_append(str, " - ");
            air_time_g_string_append(str, r->b, AIR_TIME_ISO8601);
        }
    }
}

void naf_filter_rl_print_ipaddr(
    GArray                  *rl,
    GString                 *str)
{    
    uint32_t                i;

    naf_range_t             *r;
    char                    ipbufa[16], ipbufb[16];
    
    for (i = 0; i < rl->len; i++) {
        r = &g_array_index(rl, naf_range_t, i);
        if (r->a == r->b) {
            air_ipaddr_buf_print(ipbufa, r->a);
            g_string_append_printf(str, "%s ", ipbufa);
        } else {
            air_ipaddr_buf_print(ipbufa, r->a);
            air_ipaddr_buf_print(ipbufb, r->b);
            g_string_append_printf(str, "%s-%s ", ipbufa, ipbufb);
        }
    }
}

void naf_filter_rl_print(
    GArray                  *rl,
    GString                 *str)
{
    naf_range_t             *r;
    uint32_t                i;
    
    for (i = 0; i < rl->len; i++) {
        r = &g_array_index(rl, naf_range_t, i);
        if (r->a == r->b) {
            g_string_append_printf(str, "%u ", r->a);
        } else {
            g_string_append_printf(str, "%u-%u ", r->a, r->b);
        }
    }
}

void naf_filter_print(
    NAFilter                *filter,
    GString                 *str)
{
    if (filter->binrl) {
        g_string_append(str, "bin ");
        naf_filter_rl_print_time(filter->binrl, str);
    }

    if (filter->siprl) {
        g_string_append(str, "sip ");
        if (filter->sipnot) {
            g_string_append(str, "not ");
        }
        naf_filter_rl_print_ipaddr(filter->siprl, str);
    }

    if (filter->diprl) {
        g_string_append(str, "dip ");
        if (filter->dipnot) {
            g_string_append(str, "not ");
        }
        naf_filter_rl_print_ipaddr(filter->diprl, str);
    }
    
    if (filter->sprl) {
        g_string_append(str, "sp ");
        if (filter->spnot) {
            g_string_append(str, "not ");
        }
        naf_filter_rl_print(filter->sprl, str);
    }

    if (filter->dprl) {
        g_string_append(str, "dp ");
        if (filter->dpnot) {
            g_string_append(str, "not ");
        }
        naf_filter_rl_print(filter->dprl, str);
    }

    if (filter->protorl) {
        g_string_append(str, "proto ");
        if (filter->protonot) {
            g_string_append(str, "not ");
        }
        naf_filter_rl_print(filter->protorl, str);
    }
    
    if (filter->florl) {
        g_string_append(str, "flows ");
        naf_filter_rl_print(filter->florl, str);
    }

    if (filter->rflorl) {
        g_string_append(str, "rev flows ");
        naf_filter_rl_print(filter->rflorl, str);
    }

    if (filter->pktrl) {
        g_string_append(str, "packets ");
        naf_filter_rl_print(filter->pktrl, str);
    }

    if (filter->rpktrl) {
        g_string_append(str, "rev packets ");
        naf_filter_rl_print(filter->rpktrl, str);
    }

    if (filter->octrl) {
        g_string_append(str, "octets ");
        naf_filter_rl_print(filter->octrl, str);
    }

    if (filter->roctrl) {
        g_string_append(str, "rev octets ");
        naf_filter_rl_print(filter->roctrl, str);
    }
}

gboolean naf_filter_rl_parse_time(
    GScanner            *scanner,
    GArray              **rl)
{
    uint32_t            yra, moa, dya, hra, mna, sca;
    uint32_t            yrb, mob, dyb, hrb, mnb, scb;
    time_t              ta, tb;
    char                *errdesc = NULL;
    
    while (1) {    
        /* grab the first date */
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
            errdesc = "missing year in range start"; goto err;
        }
        if ((scanner->value.v_int < 1970) || (scanner->value.v_int > 2038)) {
            errdesc = "year out of range in range start"; goto err;
        }
        yra = scanner->value.v_int;
        if (g_scanner_get_next_token(scanner) != '-') {
            errdesc = "missing - in range start"; goto err; 
        }
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
            errdesc = "missing month in range start"; goto err;
        }
        if (scanner->value.v_int < 1 || scanner->value.v_int > 12) {
            errdesc = "month out of range in range start"; goto err;
        }
        moa = scanner->value.v_int;
        if (g_scanner_get_next_token(scanner) != '-') { 
            errdesc = "missing - in range start"; goto err; 
        }
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
            errdesc = "missing day in range start"; goto err;
        }
        if (scanner->value.v_int < 1 || scanner->value.v_int > 31) { 
            errdesc = "day out of range in range start"; goto err;
        }
        dya = scanner->value.v_int;
     
        /* do we have an hour? */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == G_TOKEN_INT) {
            /* Yep. Parse time. */
            g_scanner_get_next_token(scanner);
            if (scanner->value.v_int < 0 || scanner->value.v_int > 23) {
                errdesc = "hour out of range in range start"; goto err;
            }
            hra = scanner->value.v_int;
            if (g_scanner_get_next_token(scanner) != ':') {
                errdesc = "missing : in range start"; goto err;
            }
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                errdesc = "missing minute in range start"; goto err;
            }
            if (scanner->value.v_int < 0 || scanner->value.v_int > 59) {
                errdesc = "minute out of range in range start"; goto err;
            }
            mna = scanner->value.v_int;
            if (g_scanner_get_next_token(scanner) != ':') {
                errdesc = "missing : in range start"; goto err; 
            }
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                errdesc = "missing second in range start"; goto err;
            }
            if (scanner->value.v_int < 0 || scanner->value.v_int > 59) {
                errdesc = "second out of range in range start"; goto err;
            }
            sca = scanner->value.v_int;
        } else {
            /* nope - start at beginning of day */
            hra = mna = sca = 0;
        }
        
        /* do we have a range? */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == '-') {
            /* eat the dash */
            g_scanner_get_next_token(scanner);
        
            /* grab the second date */
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                errdesc = "missing year in range end"; goto err; 
            }
            if (scanner->value.v_int < 1970 || scanner->value.v_int > 2038) {
                errdesc = "year out of range in range end"; goto err;
            }
            yrb = scanner->value.v_int;
            if (g_scanner_get_next_token(scanner) != '-') {
                errdesc = "missing - in range end"; goto err; 
            }
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                errdesc = "missing month in range end"; goto err;
            }
            if (scanner->value.v_int < 1 || scanner->value.v_int > 12) {
                errdesc = "month out of range in range end"; goto err;
            }
            mob = scanner->value.v_int;
            if (g_scanner_get_next_token(scanner) != '-') {
                errdesc = "missing - in range end"; goto err; 
            }
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                errdesc = "missing day in range end"; goto err;
            }
            if (scanner->value.v_int < 1 || scanner->value.v_int > 31) {
                errdesc = "day out of range in range end"; goto err;
            }
            dyb = scanner->value.v_int;

            /* do we have an hour? */
            g_scanner_peek_next_token(scanner);
            if (scanner->next_token == G_TOKEN_INT) {
                /* Yep. Parse time. */
                g_scanner_get_next_token(scanner);
                if (scanner->value.v_int < 0 || scanner->value.v_int > 23) {
                    errdesc = "hour out of range in range end"; goto err;
                }
                hrb = scanner->value.v_int;
                if (g_scanner_get_next_token(scanner) != ':') {
                    errdesc = "missing : in range end"; goto err; 
                }
                if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                    errdesc = "missing minute in range end"; goto err;
                }
                if (scanner->value.v_int < 0 || scanner->value.v_int > 59) {
                    errdesc = "minute out of range in range end"; goto err;
                }
                mnb = scanner->value.v_int;
                if (g_scanner_get_next_token(scanner) != ':') {
                    errdesc = "missing : in range end"; goto err; 
                }
                if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
                    errdesc = "missing second in range end"; goto err;
                }
                if (scanner->value.v_int < 0 || scanner->value.v_int > 59) {
                    errdesc = "second out of range in range end"; goto err;
                }
                scb = scanner->value.v_int;
            } else {
                /* Nope. End of day. */
                hrb = 23;
                mnb = 59;
                scb = 59;
            }
        } else {
            /* No range. tmb is end of tma day. */
            yrb = yra;
            mob = moa;
            dyb = dya;
            hrb = 23;
            mnb = 59;
            scb = 59;
        }
        
        /* If we are here, tma, tmb are valid; add them to the range list */
        ta = air_time_gm(yra, moa, dya, hra, mna, sca);
        tb = air_time_gm(yrb, mob, dyb, hrb, mnb, scb);

        naf_filter_rl_add(rl, ta, tb);
    
        /* Check for a comma to see if we're still going. */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token != ',') break;
        
        /* Still going. Eat the comma. */
        g_scanner_get_next_token(scanner);
    }
    
    /* All done. Coalesce range list. */
    naf_filter_rl_coalesce(*rl);

    return TRUE;
    
  err:
    g_scanner_error(scanner, "Malformed time range (%s)", errdesc);
    return FALSE;
}


gboolean naf_filter_rl_parse_ipaddr(
    GScanner            *scanner,
    GArray              **rl)
{
    uint32_t            a;
    uint32_t            b;
    uint32_t            mask;
    
    while (1) {
        /* grab first address in range */
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
        if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
        a = ((scanner->value.v_int & 0x000000FF) << 24);
        if (g_scanner_get_next_token(scanner) != '.') goto err;
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
        if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
        a |= ((scanner->value.v_int & 0x000000FF) << 16);
        if (g_scanner_get_next_token(scanner) != '.') goto err;
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
        if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
        a |= ((scanner->value.v_int & 0x000000FF) << 8);
        if (g_scanner_get_next_token(scanner) != '.') goto err;
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
        if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
        a |= (scanner->value.v_int & 0x000000FF);

        /* EOA for first address. Peek for next token. */
        g_scanner_peek_next_token(scanner);
        
        /* Slash means CIDR, dash means explicit inclusive range. */
        if (scanner->next_token == '/') {
            /* This range is CIDRized. Eat the slash. */
            g_scanner_get_next_token(scanner);
            /* Calculate A and B from CIDR notation. */
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > 32) goto err;
            mask = air_mask_from_prefix(scanner->value.v_int);
            a &= mask;
            b = a | (~mask);
        } else if (scanner->next_token == '-') {
            /* Explicit range. Eat the dash. */
            g_scanner_get_next_token(scanner);
            /* Grab second address in range */
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
            b = ((scanner->value.v_int & 0x000000FF) << 24);
            if (g_scanner_get_next_token(scanner) != '.') goto err;
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
            b |= ((scanner->value.v_int & 0x000000FF) << 16);
            if (g_scanner_get_next_token(scanner) != '.') goto err;
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
            b |= ((scanner->value.v_int & 0x000000FF) << 8);
            if (g_scanner_get_next_token(scanner) != '.') goto err;
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > 255) goto err;
            b |= (scanner->value.v_int & 0x000000FF);
        } else {
            /* Presume single address in range. */
            b = a;
        }
            
        /* If we are here, A and B are valid; add them to the range list */
        naf_filter_rl_add(rl, a, b);
        
        /* Check for a comma to see if we're still going. */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token != ',') break;
        
        /* Still going. Eat the comma. */
        g_scanner_get_next_token(scanner);
    }
    
    /* All done. Coalesce range list. */
    naf_filter_rl_coalesce(*rl);

    return TRUE;
    
  err:
    g_scanner_error(scanner, "Malformed IP address range");
    return FALSE;
}

gboolean naf_filter_rl_parse_uint(
    GScanner            *scanner,
    GArray              **rl,
    uint32_t            max)
{
    uint32_t            a = 0;
    uint32_t            b = 0;
    gboolean            ineq = FALSE, ineq_gt = FALSE, needb = FALSE;
    
    while (1) {
        
        /* Get first number in range */
        g_scanner_get_next_token(scanner);
        if (scanner->token == '<') {
            /* < means implicit first number of 0 */
            a = 0;
            ineq = TRUE;
            ineq_gt = FALSE;
        } else if (scanner->token == '>') {
            /* > means implicit first number of max */
            a = max;
            ineq = TRUE;
            ineq_gt = TRUE;
        } else if (scanner->token == G_TOKEN_INT) {
            if (scanner->value.v_int < 0 || scanner->value.v_int > max) goto err;
            a = scanner->value.v_int;
            ineq = FALSE;
        }
    
        /* EON for first number. Get the second. */
        needb = TRUE;
        if (!ineq) {
            /* Look for a dash */
            g_scanner_peek_next_token(scanner);
            if (scanner->next_token == '-') {
                /* Dash present. Eat it. */
                g_scanner_get_next_token(scanner);
            } else {
                /* Single number range, and we're done */
                b = a;
                needb = FALSE;
            }
        }
        
        if (needb) {
            /* Next token must be a number. */
            if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) 
                goto err;
            if (scanner->value.v_int < 0 || scanner->value.v_int > max) 
                goto err;
            b = scanner->value.v_int;
        }
        
        /* Adjust value of second number for inequalities */
        b += (ineq ? (ineq_gt ? 1 : -1) : 0);
        
        /* If we are here, A and B are valid; add them to the range list */
        naf_filter_rl_add(rl, a, b);
        
        /* Check for a comma to see if we're still going. */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token != ',') break;
        
        /* Still going. Eat the comma. */
        g_scanner_get_next_token(scanner);
    }
        
    /* All done. Coalesce range list. */
    naf_filter_rl_coalesce(*rl);

    return TRUE;
    
  err:
    g_scanner_error(scanner, "Malformed number range");
    return FALSE;
}

static gboolean naf_filter_rl_parse_uint32(
    GScanner            *scanner,
    GArray              **rl)
{
    return naf_filter_rl_parse_uint(scanner, rl, UINT32_MAX);
}

static gboolean naf_filter_rl_parse_uint16(
    GScanner            *scanner,
    GArray              **rl)
{
    return naf_filter_rl_parse_uint(scanner, rl, UINT16_MAX);
}

static gboolean naf_filter_rl_parse_uint8(
    GScanner            *scanner,
    GArray              **rl)
{
    return naf_filter_rl_parse_uint(scanner, rl, UINT8_MAX);
}

gboolean naf_filter_parse(
    GScanner            *scanner,
    NAFilter            *filter)
{
    gboolean            is_reverse = FALSE;
    gboolean            *notp;
    GArray              **rlp;
    gboolean            (*rpfn)(GScanner *, GArray **);
    
    /* consume the filter token */
    g_assert(g_scanner_get_next_token(scanner) == (GTokenType)NAF_SYM_FILTER);
    
    /* do filter range parsing until we're done */
    while (1) {
    
        /* check for "reverse" */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_REV) {
            is_reverse = TRUE;
            g_scanner_get_next_token(scanner);
        }
    
        /* get next range */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_BIN) {
            notp = NULL;
            rlp = &filter->binrl;
            rpfn = naf_filter_rl_parse_time;
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SIP) {
            notp = &filter->sipnot;
            rlp = &filter->siprl;
            rpfn = naf_filter_rl_parse_ipaddr;
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DIP) {
            notp = &filter->dipnot;
            rlp = &filter->diprl;
            rpfn = naf_filter_rl_parse_ipaddr;     
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SP) {
            notp = &filter->spnot;
            rlp = &filter->sprl;
            rpfn = naf_filter_rl_parse_uint16;     
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DP) {
            notp = &filter->dpnot;
            rlp = &filter->dprl;
            rpfn = naf_filter_rl_parse_uint16;     
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PROTO) {
            notp = &filter->protonot;
            rlp = &filter->protorl;
            rpfn = naf_filter_rl_parse_uint8;    
        } else if (scanner->next_token == (GTokenType)NAF_SYM_FLOWS) {
            notp = NULL;
            rlp = is_reverse ? &filter->rflorl : &filter->florl;
            rpfn = naf_filter_rl_parse_uint32;
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PACKETS) {
            notp = NULL;
            rlp = is_reverse ? &filter->rpktrl : &filter->pktrl;
            rpfn = naf_filter_rl_parse_uint32;
        } else if (scanner->next_token == (GTokenType)NAF_SYM_OCTETS) {
            notp = NULL;
            rlp = is_reverse ? &filter->roctrl : &filter->octrl;
            rpfn = naf_filter_rl_parse_uint32;
        } else {
            return TRUE;
        }
        
        /* clear reverse flag */
        is_reverse = FALSE;
        
        /* eat the token */
        g_scanner_get_next_token(scanner);
        
        /* check for negation */
        if (notp) {
            if (g_scanner_peek_next_token(scanner) == 
                    (GTokenType)NAF_SYM_NOT) {
                *notp = TRUE;
                g_scanner_get_next_token(scanner);
            } else {
                *notp = FALSE;
            }
        }
        
        /* and dispatch */
        if (!rpfn(scanner, rlp)) return FALSE;
    }
}
