/*
 ** lexcore.c
 ** NAF common parser support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/lexcore.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: lexcore.c 6585 2007-03-08 00:08:38Z bht $";

char *NAF_SYM_BIN =         "bin";
char *NAF_SYM_UNIFORM =     "uniform";
char *NAF_SYM_START =       "start";
char *NAF_SYM_END =         "end";
char *NAF_SYM_SEC =         "sec";
char *NAF_SYM_MIN =         "min";
char *NAF_SYM_HR =          "hr";
char *NAF_SYM_UNIFLOW =     "uniflow";
char *NAF_SYM_REVEPH =      "reveph";
char *NAF_SYM_PERIMETER =   "perimeter";
char *NAF_SYM_FILTER =      "filter";
char *NAF_SYM_NOT =         "not";
char *NAF_SYM_IN =          "in";
char *NAF_SYM_AGGREGATE =   "aggregate";
char *NAF_SYM_SRCID =       "srcid";
char *NAF_SYM_SIP =         "sip";
char *NAF_SYM_DIP =         "dip";
char *NAF_SYM_SP =          "sp";
char *NAF_SYM_DP =          "dp";
char *NAF_SYM_PROTO =       "proto";
char *NAF_SYM_COUNT =       "count";
char *NAF_SYM_TOTAL =       "total";
char *NAF_SYM_HOSTS =       "hosts";
char *NAF_SYM_PORTS =       "ports";
char *NAF_SYM_FLOWS =       "flows";
char *NAF_SYM_PACKETS =     "packets";
char *NAF_SYM_OCTETS =      "octets";
char *NAF_SYM_LABEL =       "label";
char *NAF_SYM_REV =     "rev";
char *NAF_SYM_SRC =         "src";
char *NAF_SYM_DEST =        "dest";
char *NAF_SYM_SORT =        "sort";
char *NAF_SYM_ASC =         "asc";
char *NAF_SYM_DESC =        "desc";
char *NAF_SYM_LIMIT =       "limit";

static void naf_lex_msg(
    GScanner        *scanner,
    char            *message,
    gboolean        error)
{
    if (error) {
        g_critical("Error at line %u, position %u in expression: %s.", 
                    scanner->line, scanner->position, message);
    } else {
        g_warning("Warning at line %u, position %u in expression: %s.", 
                    scanner->line, scanner->position, message);
    }
}

GScanner *naf_lex_init(
   char             *str)
{
    GScannerConfig  scancfg;
    GScanner        *scanner;
    
    /* configure scanner */
    scancfg.cset_skip_characters = " \t\r\n";
    scancfg.cset_identifier_first = G_CSET_a_2_z 
                                    G_CSET_A_2_Z 
                                    "_";
    scancfg.cset_identifier_nth =   G_CSET_a_2_z 
                                    G_CSET_A_2_Z 
                                    G_CSET_LATINS 
                                    G_CSET_LATINC 
                                    "_0123456789";
    scancfg.cpair_comment_single =  "#\n";
    scancfg.case_sensitive =        0;
    scancfg.skip_comment_multi =    1;
    scancfg.skip_comment_single =   1;
    scancfg.scan_comment_multi =    0;
    scancfg.scan_identifier =       1;
    scancfg.scan_identifier_1char = 0;
    scancfg.scan_identifier_NULL =  0;
    scancfg.scan_symbols =          1;
    scancfg.scan_binary =           0;
    scancfg.scan_octal =            0;
    scancfg.scan_float =            0;
    scancfg.scan_hex =              1;
    scancfg.scan_hex_dollar =       0;
    scancfg.scan_string_sq =        0;
    scancfg.scan_string_dq =        0;
    scancfg.numbers_2_int =         1;
    scancfg.int_2_float =           0;
    scancfg.identifier_2_string =   0;
    scancfg.char_2_token =          1;
    scancfg.symbol_2_token =        1;
    scancfg.scope_0_fallback =      0;
    scancfg.store_int64 =           0;
    
    /* create scanner */
    scanner = g_scanner_new(&scancfg);
    
    /* set error handler */
    scanner->msg_handler = naf_lex_msg;
    
    /* add symbols */
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_BIN, NAF_SYM_BIN);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_UNIFORM, NAF_SYM_UNIFORM);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_START, NAF_SYM_START);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_END, NAF_SYM_END);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SEC, NAF_SYM_SEC);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_MIN, NAF_SYM_MIN);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_HR, NAF_SYM_HR);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_UNIFLOW, NAF_SYM_UNIFLOW);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_REVEPH, NAF_SYM_REVEPH);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_PERIMETER, NAF_SYM_PERIMETER);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_FILTER, NAF_SYM_FILTER);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_NOT, NAF_SYM_NOT);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_IN, NAF_SYM_IN);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_AGGREGATE, NAF_SYM_AGGREGATE);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SRCID, NAF_SYM_SRCID);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SIP, NAF_SYM_SIP);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_DIP, NAF_SYM_DIP);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SP, NAF_SYM_SP);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_DP, NAF_SYM_DP);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_PROTO, NAF_SYM_PROTO);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_COUNT, NAF_SYM_COUNT);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_TOTAL, NAF_SYM_TOTAL);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_HOSTS, NAF_SYM_HOSTS);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_PORTS, NAF_SYM_PORTS);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_FLOWS, NAF_SYM_FLOWS);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_PACKETS, NAF_SYM_PACKETS);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_OCTETS, NAF_SYM_OCTETS);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_LABEL, NAF_SYM_LABEL);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_REV, NAF_SYM_REV);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SRC, NAF_SYM_SRC);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_DEST, NAF_SYM_DEST);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_SORT, NAF_SYM_SORT);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_ASC, NAF_SYM_ASC);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_DESC, NAF_SYM_DESC);
    g_scanner_scope_add_symbol(scanner, 0, NAF_SYM_LIMIT, NAF_SYM_LIMIT);

    /* attach expression */
    g_scanner_input_text(scanner, str, strlen(str));

    /* Ready to go. */
    return scanner;
}

GScanner *naf_lex_init_argv(
    int              argc,
    char             *argv[])
{
    GScanner        *scanner;
    GString         *expr;
    uint32_t        i;
    
    /* join arguments into a single string, skipping program name. */
    expr = g_string_new("");
    for (i = 1; i < argc; i++) {
        g_string_append(expr, argv[i]);
        g_string_append_c(expr, ' ');
    }

    scanner = naf_lex_init(expr->str);
    g_string_free(expr, FALSE);
    return scanner;
}
