/*
 ** match.c
 ** NAF first-stage flow table (matches and prefilters biflows)
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/dynflow.h>
#include <naf/match.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: match.c 6585 2007-03-08 00:08:38Z bht $";

/* Debug flags */
#define NF_DEBUG_MATCH 0

void naf_mtab_init(
    NAFMultiBin         *mtab,
    NAFAggConfig        *conf)
{
    /* Allocate queue and table */
    mtab->binqueue = g_queue_new();
    mtab->bintables = g_hash_table_new(g_direct_hash, g_direct_equal);
    
    /* Store configuration */
    mtab->conf = conf;
    
    /* Initialize minimum and maximum time bins */
    mtab->minbin = 0;
    mtab->maxbin = 0;
    
    /* Zero counters */
    mtab->raw_count = 0;
    mtab->hit_count = 0;
    mtab->match_count = 0;
    mtab->miss_count = 0;
    mtab->predrop_count = 0;
    mtab->peridrop_count = 0;
    mtab->perirev_count = 0;
    mtab->horizon_count = 0;
    mtab->active_count = 0;
    mtab->active_max = 0;
}

static NAFBinTable *naf_mtab_bintable(
    NAFMultiBin     *mtab,
    NAFTimeSec      bin,
    GError          **err)
{
    NAFBinTable  *bintable = NULL;
    
    /* Detect horizon violation by bin underrun */
    if (bin < mtab->minbin) {
        char bsbuf[AIR_TIME_BUF_MINSZ], mbsbuf[AIR_TIME_BUF_MINSZ];
        air_time_buf_print(bsbuf, bin, AIR_TIME_ISO8601);
        air_time_buf_print(mbsbuf, mtab->minbin, AIR_TIME_ISO8601);
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_HORIZON,
                    "Horizon violation; bin %s already flushed, minbin is %s",
                    bsbuf, mbsbuf);
        mtab->horizon_count++;
        return NULL;
    }
    
    /* Enqueue new bins until one is available */
    while (mtab->maxbin < bin) {
        /* Determine bin number of next bin to enqueue */
        if (mtab->maxbin == 0) {
            mtab->maxbin = bin - mtab->conf->horizon;
            mtab->minbin = mtab->maxbin;
        } else {
            mtab->maxbin += mtab->conf->binsize;
        }
        
        /* Create a new bin */
        bintable = g_new0(NAFBinTable, 1);
        bintable->mtab = mtab;
        bintable->bin = mtab->maxbin;
        bintable->table = g_hash_table_new((GHashFunc)naf_flowkey_hash, 
                                           (GEqualFunc)naf_flowkey_equal);
        bintable->keychunk = g_mem_chunk_new("mkey", sizeof(NAFlowKey), 
                                             32768 * sizeof(NAFlowKey),
                                             G_ALLOC_ONLY);
        bintable->valchunk = g_mem_chunk_new("mval", sizeof(NAFlowVal), 
                                             32768 * sizeof(NAFlowVal),
                                             G_ALLOC_ONLY);

        /* Place it in the bin table... */
        g_hash_table_insert(mtab->bintables,
                            GUINT_TO_POINTER(mtab->maxbin), bintable);
        /* ...then place it in the queue */
        g_queue_push_head(mtab->binqueue, bintable);

/*        
        {
            char binbuf[AIR_TIME_BUF_MINSZ];
            air_time_buf_print(binbuf, bintable->bin, AIR_TIME_ISO8601);
            g_debug("Enqueued bin %s", binbuf);
        }
*/
    }

    /* At this point the bin is guaranteed to be in the queue. Look it up. */
    bintable = (NAFBinTable *)g_hash_table_lookup(mtab->bintables, 
                                                     GUINT_TO_POINTER(bin));
    /* Paranoia. */
    g_assert(bintable);
    return bintable;
}

gboolean naf_mtab_add(
    NAFMultiBin         *mtab,
    NAFlowKey           *key,
    NAFlowVal           *val,
    GError              **err)
{
    NAFBinTable         *bintable;
    NAFlowKey           rkey;
    NAFlowVal           rval, *tval;
    
    /* Verify no value unique counter data for incoming flows */
    g_assert(!val->vuc);
    
    /* Get the hashtable for this bin; detect horizon violations */
    bintable = naf_mtab_bintable(mtab, key->bin, err);
    if (!bintable) return FALSE;
    
    /* Look for the flow in the table */
    tval = g_hash_table_lookup(bintable->table, key);
    
    /* Check for hit, filter value if so */
    if (tval) {
        if (!naf_filter_val(&(mtab->conf->prefilter), val)) {
            ++mtab->predrop_count;
#if NF_DEBUG_MATCH
            nfDumpFlow(&rkey, &rval, "pffv");
#endif
            return TRUE;
        }
        tval->oct += val->oct;
        tval->roct += val->roct;
        tval->pkt += val->pkt;
        tval->rpkt += val->rpkt;
        tval->flo += val->flo;
        tval->rflo += val->rflo;
        ++mtab->hit_count;
#if NF_DEBUG_MATCH
        nfDumpFlow(key, tval, "hit");
#endif
        return TRUE;
    }
    
    /* Check for reverse hit, filter value if so */
    if (!mtab->conf->uniflow_mode) {
        naf_flowkey_reverse(key, &rkey);
        tval = g_hash_table_lookup(bintable->table, &rkey);
        if (tval) {
            if (!naf_filter_val(&(mtab->conf->prefilter), val)) {
                ++mtab->predrop_count;
#if NF_DEBUG_MATCH
                nfDumpFlow(&rkey, &rval, "pfrv");
#endif
                return TRUE;
            }
            tval->oct += val->roct;
            tval->roct += val->oct;
            tval->pkt += val->rpkt;
            tval->rpkt += val->pkt;
            tval->flo += val->rflo;
            tval->rflo += val->flo;
            ++mtab->match_count;
#if NF_DEBUG_MATCH
            nfDumpFlow(&rkey, tval, "match");
#endif
            return TRUE;
        }
    }

    /* Handle permieter mode */
    if (mtab->conf->perimeter) {
        /* Drop flow if it doesn't cross the perimeter */
        if (naf_filter_rl_contains(mtab->conf->perimeter, key->sip) ==
            naf_filter_rl_contains(mtab->conf->perimeter, key->dip)) {
                ++mtab->peridrop_count;
#if NF_DEBUG_MATCH
                nfDumpFlow(key, val, "peri");
#endif
                return TRUE;
        }
        /* Reverse flow if it originates inside the perimeter */
        if (naf_filter_rl_contains(mtab->conf->perimeter, key->sip)) {
            ++mtab->perirev_count;
            naf_flowkey_reverse(key, &rkey);
            naf_flowval_reverse(val, &rval);
        } else {
            /* Null reverse flows */
            memcpy(&rkey, key, sizeof(NAFlowKey));
            memcpy(&rval, val, sizeof(NAFlowVal));
        }
    } else {
        /* Null reverse flows */
        memcpy(&rkey, key, sizeof(NAFlowKey));
        memcpy(&rval, val, sizeof(NAFlowVal));
    }

    /* Filter the key */
    if (!naf_filter_key(&(mtab->conf->prefilter), &rkey)) {
        ++mtab->predrop_count;
#if NF_DEBUG_MATCH
        nfDumpFlow(&rkey, &rval, "pfk");
#endif
        return TRUE;
    }
        
    /* Insert the flow into the bin table */
    g_hash_table_insert(bintable->table,
                        naf_flowkey_alloc(bintable->keychunk, &rkey),
                        naf_flowval_alloc(bintable->valchunk, &rval));
    ++mtab->miss_count;
#if NF_DEBUG_MATCH
    nfDumpFlow(&rkey, &rval, "miss");
#endif    

    /* Count it; track maximum table size. */
    if (++mtab->active_count > mtab->active_max) {
        mtab->active_max = mtab->active_count;
    }
    
    return TRUE;
}

gboolean naf_mtab_add_raw(
    NAFMultiBin         *mtab,
    NAFlowRaw           *raw,
    GError              **err)
{
    NAFlowKey           key;
    NAFlowVal           val;
    uint32_t            sbin, ebin, bincount;
    uint64_t            octq, roctq, pktq, rpktq;
    uint64_t            octr, roctr, pktr, rpktr;
    
    /* count flow */
    ++mtab->raw_count;
    
    /* copy key from raw */
    key.srcid = raw->srcid;
    key.sip = raw->sip;
    key.dip = raw->dip;
    key.sipmask = raw->sipmask;
    key.dipmask = raw->dipmask;
    key.sp = raw->sp;
    key.dp = raw->dp;
    key.proto = raw->proto;

    /* calculate start and end bin numbers */
    sbin = naf_bin_time(raw->stime, mtab->conf->binsize);
    ebin = naf_bin_time(raw->etime, mtab->conf->binsize);

    /* add flow to bin(s) */
    if (sbin == ebin || mtab->conf->binalg == NAF_BA_START) {
        /* single bin is equivalent to start bin algorithm */
        key.bin = sbin;
        val.oct = raw->oct;
        val.roct = raw->roct;
        val.pkt = raw->pkt;
        val.rpkt = raw->rpkt;
        val.flo = raw->flo;
        val.rflo = raw->rflo;
        val.host = 0;
        val.rhost = 0;
        val.port = 0;
        val.rport = 0;
        val.vuc = NULL;
        return naf_mtab_add(mtab, &key, &val, err);
    } else if (mtab->conf->binalg == NAF_BA_END) {
        key.bin = ebin;
        val.oct = raw->oct;
        val.roct = raw->roct;
        val.pkt = raw->pkt;
        val.rpkt = raw->rpkt;
        val.flo = raw->flo;
        val.rflo = raw->rflo;
        val.host = 0;
        val.rhost = 0;
        val.port = 0;
        val.rport = 0;
        val.vuc = NULL;
        return naf_mtab_add(mtab, &key, &val, err);
    } else switch (mtab->conf->binalg) {
        case NAF_BA_UNIFORM:
        /* Fill in unchanging flow value fields */
        val.flo = raw->flo;
        val.rflo = raw->rflo;
        val.host = 0;
        val.rhost = 0;
        val.port = 0;
        val.rport = 0;
        val.vuc = NULL;
        /* Set up uniform bin algorithm */
        bincount = (ebin / mtab->conf->binsize) -
                   (sbin / mtab->conf->binsize) + 1;
        octq = raw->oct / bincount;
        roctq = raw->roct / bincount;
        pktq = raw->pkt / bincount;
        rpktq = raw->rpkt / bincount;
        octr = raw->oct % bincount;
        roctr = raw->roct % bincount;
        pktr = raw->pkt % bincount;
        rpktr = raw->rpkt % bincount;
        key.bin = sbin;
        /* Iteratively distribute uniformly, spacing out the remainder */
        while (bincount) {
            val.oct = octq;
            val.roct = roctq;
            val.pkt = pktq;
            val.rpkt = rpktq;
            if (octr) { val.oct++; octr--; }
            if (roctr) { val.roct++; roctr--; }
            if (pktr) { val.pkt++; pktr--; }
            if (rpktr) { val.rpkt++; rpktr--; }
            if ((val.oct || val.pkt || val.roct || val.rpkt) &&
                !naf_mtab_add(mtab, &key, &val, err)) return FALSE;
            --bincount;
            key.bin += mtab->conf->binsize;
        }
        break;
    default:
        g_assert_not_reached();
    }
    
    /* All done. Flow(s) added. */
    return TRUE;
}

NAFBinTable *naf_mtab_dequeue(
    NAFMultiBin          *mtab,
    NAFTimeSec          horizon)
{
    NAFBinTable      *nextbin = NULL;

    /* check for no dequeuable */
    if (horizon && ((mtab->maxbin - mtab->minbin) <= horizon)) return NULL;
    
    /* dequeue next table */
    nextbin = g_queue_pop_tail(mtab->binqueue);
    
    /* No dequeue means empty queue; reset min/max. */
    if (!nextbin) {
        mtab->minbin = 0;
        mtab->maxbin = 0;
        return NULL;
    }
    
    /* Increment minbin */
    mtab->minbin += mtab->conf->binsize;
    
    /* Done. */
    return nextbin;
}

void naf_mtab_bintable_complete(
    NAFBinTable      *bintable)
{
    bintable->mtab->active_count -= g_hash_table_size(bintable->table);
    g_hash_table_remove(bintable->mtab->bintables,
                        GUINT_TO_POINTER(bintable->bin));
    g_hash_table_destroy(bintable->table);
    g_mem_chunk_destroy(bintable->keychunk);
    g_mem_chunk_destroy(bintable->valchunk);
    g_free(bintable);
}
