/*
 ** nafalize.c
 ** NAF normalizer and aggregator
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/nafcore.h>
#include <naf/filter.h>
#include <naf/dynflow.h>
#include <naf/match.h>
#include <naf/aggregate.h>
#include <naf/exparse.h>
#include "nafalize.h"
#if NAF_PCAP_ENABLE
#include <yaf/defrag.h>
#include "nafz_inpcap.h"
#endif
#include "nafz_inipfix.h"
#include "nafz_indrv.h"
#include "nafz_outfile.h"
#include "drv_renaf.h"
#include "drv_argus.h"
#if HAVE_LIBRW
#include "drv_silk.h"
#endif

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafalize.c 6585 2007-03-08 00:08:38Z bht $";

/* GOption managed options */
static char     *nafz_intype =      "ipfix";
static uint32_t nafz_horizon =      3600;
static uint32_t nafz_srcid =        0;
static gboolean nafz_syncheck =     FALSE;
static gboolean nafz_ipfix_live =   FALSE;

/* Global GOption managed options */
uint32_t        nafz_liverotate =   300;

/* MIO command-line configuration */
static uint32_t         nafz_cliflags = MIO_F_CLI_FILE_IN |
                                        MIO_F_CLI_DIR_IN |
#if NAF_PCAP_ENABLE
                                        MIO_F_CLI_PCAP_IN |
#endif
                                        MIO_F_CLI_DEF_STDIN |
                                        MIO_F_CLI_UDP_IN |
                                        MIO_F_CLI_TCP_IN |
                                        MIO_F_CLI_FILE_OUT |
                                        MIO_F_CLI_DIR_OUT |
                                        MIO_F_CLI_DEF_STDOUT |
                                        MIO_F_CLI_UDP_OUT |
                                        MIO_F_CLI_TCP_OUT;

/* GOption command-line configuration */
static GOptionEntry nafz_optentries[] = {
    { "intype", 't', 0, G_OPTION_ARG_STRING, &nafz_intype,
      "Input type (ipfix, pcap, naf, silk, argus2) "
      "[ipfix]", "type" },
    { "horizon", 'h', 0, G_OPTION_ARG_INT, &nafz_horizon,
      "Flow table horizon [3600, 1h]", "sec" },
    { "rotate-delay", 'R', 0, G_OPTION_ARG_INT, &nafz_liverotate,
      "Output autorotate delay [300, 5m]", "sec" },
    { "sid", 's', 0, G_OPTION_ARG_INT, &nafz_srcid,
      "Override event source ID", "int"}, 
    { "check-syntax", (char)0, 0, G_OPTION_ARG_NONE, &nafz_syncheck,
      "Check aggregation expression syntax and exit", NULL}, 
    { NULL }
};

static void nafz_opt_parse(
    int             *argc,
    char            **argv[]) {

    GOptionContext  *octx = NULL;
    GError          *oerr = NULL;

    octx = g_option_context_new("[aggregation-expression]");
    g_option_context_add_main_entries(octx, nafz_optentries, NULL);
    g_option_context_add_group(octx, mio_option_group(nafz_cliflags));
    g_option_context_add_group(octx, daec_option_group());
    g_option_context_add_group(octx, logc_option_group("nafalize",VERSION));
    
    g_option_context_parse(octx, argc, argv, &oerr);

    if (oerr) {
        g_error("Couldn't parse command line: %s\nUse --help for usage.", 
                oerr->message);
    }
}

static void nafz_cfg_init(
    NAFAggConfig      *cfg)
{
    uint32_t            mi = 0;

    /* Default to rwcount style output */
    if (cfg->fanout == 0) {
        g_warning("No output specified; defaulting to 5-minute total count.");
        cfg->fanout = 1;
        cfg->mask[0].sipmask =     0;
        cfg->mask[0].dipmask =     0;
        cfg->mask[0].fieldmask =   NAF_FM_FLO |
                                   NAF_FM_PKT |
                                   NAF_FM_OCT |
                                   NAF_FM_SHOSTC |
                                   NAF_FM_DHOSTC |
                                   NAF_FM_MTOTAL;
        cfg->mask[0].binsize = cfg->binsize;
        cfg->mask[0].binalg = cfg->binalg;
    }
    
    /* ensure horizon and bin make sense together */
    if (cfg->horizon % cfg->binsize) {
        cfg->horizon += (cfg->binsize - 
                        (cfg->horizon % cfg->binsize));
        g_warning("Forcing horizon to %u for compatibility with bin.",
                    cfg->horizon);
    }

    for (mi = 0; mi < cfg->fanout; mi++) {
        /* Verify label */
        if (cfg->fanout > 1 && cfg->label[mi] == NULL) {
            g_warning("Missing label on aggregate statement %u;"
                      "using a default numeric label.",mi);
            cfg->label[mi] = g_new0(char, 3);
            snprintf(cfg->label[mi], 3, "%02u", mi);
        }
    
        /* Don't count hosts if the host count will always be one */
        if (cfg->mask[mi].sipmask == 32) 
            cfg->mask[mi].fieldmask &= (~NAF_FM_SHOSTC);
        if (cfg->mask[mi].dipmask == 32) 
            cfg->mask[mi].fieldmask &= (~NAF_FM_DHOSTC);
            
        /* Don't count ports if the port count will always be one */
        if (cfg->mask[mi].fieldmask & NAF_FM_SP)
            cfg->mask[mi].fieldmask &= (~NAF_FM_SPORTC);
        if (cfg->mask[mi].fieldmask & NAF_FM_DP)
            cfg->mask[mi].fieldmask &= (~NAF_FM_DPORTC);

        /* Always total in uniflow mode */
        if (cfg->uniflow_mode) {
            cfg->mask[mi].fieldmask |= NAF_FM_MTOTAL;
        }

        /* Modify counts if total */
        if (cfg->mask[mi].fieldmask & NAF_FM_MTOTAL) {
            if (cfg->mask[mi].fieldmask & NAF_FM_FLO) {
                cfg->mask[mi].fieldmask &= (~NAF_FM_RFLO);
            }
            if (cfg->mask[mi].fieldmask & NAF_FM_PKT) {
                cfg->mask[mi].fieldmask &= (~NAF_FM_RPKT);
            }
            if (cfg->mask[mi].fieldmask & NAF_FM_OCT) {
                cfg->mask[mi].fieldmask &= (~NAF_FM_ROCT);
            }
        }
    }
}

static void nafz_appin_init(
    NafalizeContext     *zctx,
    MIOAppDriver        *adrv)
{
    if (strcmp(nafz_intype, "ipfix") == 0) {
        nafz_cliflags &= ~(MIO_F_CLI_PCAP_IN |
                           MIO_F_CLI_UDP_IN |
                           MIO_F_CLI_TCP_IN);
        adrv->app_open_source = nafz_open_ipfix_file;
        adrv->app_process = nafz_process_ipfix;
        adrv->app_close_source = nafz_close_ipfix;
/* IPFIX TCP/UDP collection have been removed for now */
#if 0
    } else if (strcmp(nafz_intype, "ipfix-tcp") == 0) {
        nafz_cliflags &= ~(MIO_F_CLI_FILE_IN |
                           MIO_F_CLI_DIR_IN |
                           MIO_F_CLI_PCAP_IN |
                           MIO_F_CLI_DEF_STDIN |
                           MIO_F_CLI_UDP_IN);
        adrv->app_open_source = nafz_open_ipfix_tcp;
        adrv->app_process = nafz_process_ipfix;
        adrv->app_close_source = nafz_close_ipfix;
        nafz_ipfix_live = TRUE;
    } else if (strcmp(nafz_intype, "ipfix-udp") == 0) {
        nafz_cliflags &= ~(MIO_F_CLI_FILE_IN |
                           MIO_F_CLI_DIR_IN |
                           MIO_F_CLI_PCAP_IN |
                           MIO_F_CLI_DEF_STDIN |
                           MIO_F_CLI_TCP_IN);
        adrv->app_open_source = nafz_open_ipfix_udp;
        adrv->app_process = nafz_process_ipfix;
        adrv->app_close_source = nafz_close_ipfix;
        nafz_ipfix_live = TRUE;
#endif
#if NAF_PCAP_ENABLE
    } else if (strcmp(nafz_intype, "pcap") == 0) {
        nafz_cliflags &= ~(MIO_F_CLI_FILE_IN |
                           MIO_F_CLI_DIR_IN |
                           MIO_F_CLI_DEF_STDIN |
                           MIO_F_CLI_UDP_IN |
                           MIO_F_CLI_TCP_IN);
        adrv->app_open_source = nafz_open_pcap;
        adrv->app_process = nafz_process_pcap;
        adrv->app_close_source = nafz_close_pcap;
#endif
    } else {
        nafz_cliflags &= ~(MIO_F_CLI_PCAP_IN |
                           MIO_F_CLI_UDP_IN |
                           MIO_F_CLI_TCP_IN);
        /* No special handling; use driver input. Select a driver. */
        zctx->driver = nafz_drv_lookup(nafz_intype);
        if (!zctx->driver) {
            g_error("Unknown or unsupported input type %s", nafz_intype);
        }
        adrv->app_open_source = nafz_open_drv;
        adrv->app_process = nafz_process_drv;
        adrv->app_close_source = nafz_close_drv;
    }
}

static void nafz_appout_init(
    NafalizeContext     *zctx,
    MIOAppDriver        *adrv)
{   
    nafz_cliflags &= ~(MIO_F_CLI_UDP_OUT | MIO_F_CLI_TCP_OUT);
    if (zctx->cfg->fanout == 1) {
        adrv->app_open_sink = nafz_open_file_out;
    } else {
        adrv->app_open_sink = nafz_open_multifile_out;
    }
    adrv->app_close_sink = nafz_close_file_out;
}

int main (
    int                 argc,
    char                *argv[])
{
    GError              *err = NULL;
    
    NAFAggConfig        cfg;
    NAFMultiBin         mtab;
    fBuf_t              *fbuf_ary[NAFZ_MAX_FANOUT];
    NafalizeContext     zctx = {&cfg, &mtab, fbuf_ary, NULL, NULL};
    
    MIOSource           source;
    MIOSink             sink;
    MIOAppDriver        adrv = {NULL, NULL, NULL, NULL, NULL};
    uint32_t            miodflags = MIO_F_OPT_SINKLINK;
    char                *basepat = NULL;
    GTimer              *runtimer = NULL;
    uint32_t            mi = 0;
    
    /* register input drivers */
    nafz_renaf_register();
    nafz_argus_register();
#if HAVE_LIBSILK
    nafz_silk_register();
#endif    

    /* initialize fBuf_t array */
    for (mi = 0; mi < NAFZ_MAX_FANOUT; mi++) {
        zctx.fbuf_ary[mi] = NULL;
    }

    /* parse options - this leaves the aggregation expression in argv */
    nafz_opt_parse(&argc, &argv);

    /* set up logging */
    if (!logc_setup(&err)) {
        g_error("Cannot set up logging: %s", err->message);
    }

    /* parse aggregation expression */
    if (!naf_exparse(argc, argv, &cfg)) {
        g_error("Failed to parse aggregation expression.");
    }

    /* stuff horizon, rotate delay from command-line into configuration */
    cfg.horizon = nafz_horizon;
    cfg.srcid = nafz_srcid;
    
    /* default to IPFIX input */
    if (!nafz_intype) nafz_intype = "ipfix";

    /* dump configuration if necessary */
    if (nafz_syncheck) {
        naf_config_dump(&cfg);
        exit(0);
    }

    /* fork if necessary */
    if (!daec_setup(&err)) {
        g_error("Cannot set up daemon: %s", err->message);
    }

    /* initialize naf configuration */
    nafz_cfg_init(&cfg);
    
    /* set MIO private variables */
    mio_ov_pcaplen = 65535;
    mio_ov_pcapto = 1000;
    mio_ov_port = "4739";

    /* Preconfigure source and app driver based on input type */
    nafz_appin_init(&zctx, &adrv);
    
    /* Preconfigure sink and app driver based on output type 
       (FIXME: no such thing as output type yet) */
    nafz_appout_init(&zctx, &adrv);
    
    /* select base pattern for output */
    if (mio_ov_live || nafz_ipfix_live) {
        basepat = "%s-%T.naf";
    } else {
        basepat = "%s.naf";
    }

    /* set up source */
    if (!mio_config_source(&source, nafz_cliflags, &miodflags, &err)) {
        g_error("Cannot set up input: %s", err->message);
    }

    /* set up sink based on fanout */
    if (cfg.fanout == 1) {
        if (!mio_config_sink(&source, &sink, basepat, nafz_cliflags, 
                             &miodflags, &err)) {
            g_error("Cannot set up output: %s", err->message);
        }
    } else if (cfg.fanout > 1) {
        if (!mio_config_multisink_file(&source, &sink, basepat, 
                                       cfg.fanout, cfg.label, 
                                       nafz_cliflags, &miodflags, &err)) {
            g_error("Cannot set up output for fanout %u: %s", 
                    cfg.fanout, err->message);
        }
    } else {
        g_assert_not_reached();
    }
    
    /* initialize flow table */
    naf_mtab_init(&mtab, &cfg);
        
    /* Initialization complete. Dispatch. */
    runtimer = g_timer_new();
    g_timer_start(runtimer);
    mio_dispatch_loop(&source, &sink, &adrv, (void *)&zctx, 
                      miodflags, mio_ov_poll, 1, mio_ov_poll);
    g_timer_stop(runtimer);
    
    /* log statistics */
    g_message("nafalize terminating");
    g_message("processed %u input flows (mean flow rate %.2f/s)",
              mtab.raw_count, 
              (double)mtab.raw_count / g_timer_elapsed(runtimer, NULL));

    g_debug("Flow table statistics:");
    g_debug("%u total flows", mtab.miss_count);
    g_debug("%u uniflow hits", mtab.hit_count);
    g_debug("%u biflow hits", mtab.match_count);
    if (mtab.predrop_count) 
        g_debug("%u prefilter rejections", mtab.predrop_count);
    if (mtab.peridrop_count) 
        g_debug("%u perimeter rejections", mtab.peridrop_count);
    if (mtab.perirev_count) 
        g_debug("%u perimeter reversals", mtab.perirev_count);
    g_debug("Maximum flow table size %u", mtab.active_max);
    
    if (mtab.horizon_count)
        g_warning("%u horizon violations detected.", mtab.horizon_count);

#if NAF_PCAP_ENABLE    
    if (adrv.app_process == nafz_process_pcap)
        yfDefragDumpStats();
#endif    

    /* all done */
    return 0;
}

