/*
 ** nafcore.c
 ** NetSA Aggregated Flow (NAF) core ADT support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/nafcore.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafcore.c 6585 2007-03-08 00:08:38Z bht $";

/* Debug flags */
#define NF_DEBUG_WRITE 0
#define NF_DEBUG_READ 0

/* Full record template */

#define NAF_OPT_TID 0x7AEF
#define NAF_FULL_TID 0x7AF0
#define NAF_MASKED_TID 0x7AF1
#define NAF_MASKED_TID_BASE 0x7AF1

/* IPFIX information elements in 6871 space for NAF */
static fbInfoElement_t naf_info_elements[] = {
    FB_IE_INIT("activeFlowCount", 6871, 1, 4, FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("binLengthSeconds", 6871, 8, 4, FB_IE_F_ENDIAN),
    FB_IE_INIT("sourceActiveHostCount", 6871, 9, 4, FB_IE_F_ENDIAN),
    FB_IE_INIT("destinationActiveHostCount", 6871, 10, 4, FB_IE_F_ENDIAN),
    FB_IE_INIT("sourceActivePortCount", 6871, 25, 2, FB_IE_F_ENDIAN),
    FB_IE_INIT("destinationActivePortCount", 6871, 26, 2, FB_IE_F_ENDIAN),
    FB_IE_NULL
};


static fbInfoElementSpec_t naf_full_spec[] = {
    { "observationDomainId",            0, NAF_FM_SRCID },
    { "flowStartSeconds",               0, NAF_FM_VSTIME },
    { "sourceIPv4Address",              0, NAF_FM_SIP },
    { "destinationIPv4Address",         0, NAF_FM_DIP },
    { "sourceTransportPort",            0, NAF_FM_SP },
    { "destinationTransportPort",       0, NAF_FM_DP },
    { "sourceIPv4PrefixLength",         0, NAF_FM_SIPMASK },
    { "destinationIPv4PrefixLength",    0, NAF_FM_DIPMASK },
    { "paddingOctets",                  1, NAF_FM_VPAD },
    { "protocolIdentifier",             0, NAF_FM_PROTO },
    { "octetTotalCount",                0, NAF_FM_OCT },
    { "reverseOctetTotalCount",         0, NAF_FM_ROCT },
    { "packetTotalCount",               0, NAF_FM_PKT },
    { "reversePacketTotalCount",        0, NAF_FM_RPKT },
    { "activeFlowCount",                0, NAF_FM_FLO },
    { "reverseActiveFlowCount",         0, NAF_FM_RFLO },
    { "sourceActiveHostCount",          0, NAF_FM_SHOSTC },
    { "destinationActiveHostCount",     0, NAF_FM_DHOSTC },
    { "sourceActivePortCount",          0, NAF_FM_SPORTC },
    { "destinationActivePortCount",     0, NAF_FM_DPORTC },
    FB_IESPEC_NULL
};

typedef struct _NAFRecord {
    uint32_t    sourceId;
    uint32_t    flowStartSeconds;
    uint32_t    sourceIPv4Address;
    uint32_t    destinationIPv4Address;
    uint16_t    sourceTransportPort;
    uint16_t    destinationTransportPort;
    uint8_t     sourceIPv4Mask;
    uint8_t     destinationIPv4Mask;
    uint8_t     paddingOctets;
    uint8_t     protocolIdentifier;
    uint64_t    octetTotalCount;
    uint64_t    reverseOctetTotalCount;
    uint64_t    packetTotalCount;
    uint64_t    reversePacketTotalCount;
    uint32_t    activeFlowCount;
    uint32_t    reverseActiveFlowCount;
    uint32_t    sourceActiveHostCount;
    uint32_t    destinationActiveHostCount;
    uint16_t    sourceActivePortCount;
    uint16_t    destinationActivePortCount;
} NAFRecord;

struct _NAFRecord  __attribute__ ((packed));

static fbInfoElementSpec_t naf_opt_spec[] = {
    { "observationDomainId", 0, 0 },
    { "binLengthSeconds", 0, 0 },
    FB_IESPEC_NULL
};

typedef struct _NAFOptRecord {
    uint32_t    sourceId;
    uint32_t    binLengthSeconds;
} NAFOptRecord;

struct _NAFOptRecord  __attribute__ ((packed));

fbInfoModel_t *nfInfoModel()
{
    static fbInfoModel_t *naf_model = NULL;
    
    if (!naf_model) {
        naf_model = fbInfoModelAlloc();
        fbInfoModelAddElementArray(naf_model, naf_info_elements);
    }
    
    return naf_model;
}

static fbSession_t *nfInitSession(
    gboolean            exportSession,
    uint32_t            domain,
    GError              **err)
{
    fbInfoModel_t       *model = nfInfoModel();
    fbTemplate_t        *tmpl = NULL;
    fbSession_t         *session = NULL;
    
    /* Allocate the session */
    session = fbSessionAlloc(model);

    /* set observation domain */
    fbSessionSetDomain(session, domain);

    /* Create the options template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, naf_opt_spec, 0, err))
        return NULL;
    fbTemplateSetOptionsScope(tmpl, 1);
    
    /* Add it to the session, internal and external. */
    if (exportSession) {
        if (!fbSessionAddTemplate(session, FALSE, NAF_OPT_TID, tmpl, err))
            return NULL;
    }
    if (!fbSessionAddTemplate(session, TRUE, NAF_OPT_TID, tmpl, err))
        return NULL;
        
    /* Create the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, naf_full_spec, 0xFFFFFFFF, err))
        return NULL;
    
    /* Add it to the session, internal only */
    if (!fbSessionAddTemplate(session, TRUE, NAF_FULL_TID, tmpl, err))
        return NULL;
        
    /* now we have a session, suitable for export or collection. */
    return session;
}

fBuf_t *nfWriterForFP(
    fBuf_t                  *fbuf,
    FILE                    *fp,
    uint32_t                domain,
    NAFlowMask              *mask,
    GError                  **err)
{
    fbExporter_t            *exporter = NULL;
    fbSession_t             *session = NULL;
    fbTemplate_t            *tmpl = NULL;
    NAFOptRecord            optrec;

    /* Reset open file, or open a new exporter. */
    if (fbuf) {
        session = fBufGetSession(fbuf);
        exporter = fBufGetExporter(fbuf);
        fbExporterSetFP(exporter, fp);
    } else {
        if (!(session = nfInitSession(TRUE, domain, err))) goto err;
        exporter = fbExporterAllocFP(fp);
        fbuf = fBufAllocForExport(session, exporter);
    }
    
    /* Export the options template */
    if (!fbSessionExportTemplate(session, NAF_OPT_TID, err))
        goto err;

    /* Create mask template and add it to the session */
    /* This will cause the template to be emitted */
    mask->fieldmask |= NAF_FM_VSTIME;
    tmpl = fbTemplateAlloc(nfInfoModel());
    if (!fbTemplateAppendSpecArray(tmpl, naf_full_spec, mask->fieldmask, err))
        goto err;
    if (!fbSessionAddTemplate(session, FALSE, NAF_MASKED_TID, tmpl, err))
        goto err;
        
    /* write options data for bin */
    /* FIXME this is not IPFIX compliant. what we want is session scope. */
    optrec.sourceId = 0;
    optrec.binLengthSeconds = mask->binsize;
    if (!fBufSetInternalTemplate(fbuf, NAF_OPT_TID, err)) goto err;
    if (!fBufSetExportTemplate(fbuf, NAF_OPT_TID, err)) goto err;
    if (!fBufAppend(fbuf, (uint8_t *)&optrec, sizeof(optrec), err)) goto err;
    
    /* set templates for NAF record writing */
    if (!fBufSetInternalTemplate(fbuf, NAF_FULL_TID, err)) goto err;
    if (!fBufSetExportTemplate(fbuf, NAF_MASKED_TID, err)) goto err;
    
    /* all done, return buffer */
    return fbuf;
       
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

gboolean nfWrite(
    fBuf_t                  *fbuf,
    NAFlowMask              *mask,
    NAFlowKey               *key,
    NAFlowVal               *val,
    GError                  **err)
{
    NAFRecord               rec;
    
    /* debug */
#if NF_DEBUG_WRITE
    nfDumpFlow(key, val, "write");
#endif
    
    /* copy key to full buffer */
    rec.sourceId                    = key->srcid;
    rec.flowStartSeconds            = key->bin;
    rec.sourceIPv4Address           = key->sip;
    rec.destinationIPv4Address      = key->dip;
    rec.sourceTransportPort         = key->sp;
    rec.destinationTransportPort    = key->dp;
    rec.sourceIPv4Mask              = key->sipmask;
    rec.destinationIPv4Mask         = key->dipmask;
    rec.paddingOctets               = 0;
    rec.protocolIdentifier          = key->proto;
    
    /* total values if required */
    if (mask->fieldmask & NAF_FM_MTOTAL) {
        val->oct += val->roct; val->roct = 0;
        val->pkt += val->rpkt; val->rpkt = 0;
        val->flo += val->rflo; val->rflo = 0;
    }
        
    /* copy values to full buffer */
    rec.octetTotalCount             = val->oct;
    rec.reverseOctetTotalCount      = val->roct;
    rec.packetTotalCount            = val->pkt;
    rec.reversePacketTotalCount     = val->rpkt;
    rec.activeFlowCount             = val->flo;
    rec.reverseActiveFlowCount      = val->rflo;
    rec.sourceActiveHostCount       = val->host;
    rec.destinationActiveHostCount  = val->rhost;
    rec.sourceActivePortCount       = val->port;
    rec.destinationActivePortCount  = val->rport;
    
    /* write next message as full buffer */
    return fBufAppend(fbuf, (uint8_t *)&rec, sizeof(rec), err);
}
    
gboolean nfWriterClose(
    fBuf_t          *fbuf,
    GError          **err)
{
    gboolean        ok = TRUE;
    
    ok = fBufEmit(fbuf, err);
    fbExporterClose(fBufGetExporter(fbuf));
    return ok;
}

fBuf_t *nfReaderForFP(
    fBuf_t          *fbuf,
    FILE            *fp,
    NAFlowMask      *mask,
    GError          **err)
{
    fbSession_t         *session;
    fbCollector_t       *collector;
    fbTemplate_t        *extTmpl;
    uint16_t            extTmplID;
    size_t              optlen;
    NAFOptRecord        optrec;
    fbInfoElementSpec_t *spec;
    
    /* initialize mask fields */
    mask->sipmask = 0;
    mask->dipmask = 0;
    mask->sipmaskbits = 0;
    mask->dipmaskbits = 0;
    mask->binalg = 0;

    /* Allocate a collector for the file */
    collector = fbCollectorAllocFP(NULL, fp);
    
    /* Allocate a buffer, or reset the collector */
    if (fbuf) {
        fbCollectorFree(fBufGetCollector(fbuf));
        fBufSetCollector(fbuf, collector);
    } else {        
        if (!(session = nfInitSession(FALSE, 0, err))) goto err;
        fbuf = fBufAllocForCollection(session, collector);
    }

    /* Verify the first record is a NAF options record. */
    if (!(extTmpl = fBufNextCollectionTemplate(fbuf, &extTmplID, err)))
        goto err;
        
    if (extTmplID != NAF_OPT_TID) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_HEADER,
                    "Input file missing initial NAF options record.");
        goto err;
    }

    /* Read the NAF options record */
    if (!fBufSetInternalTemplate(fbuf, NAF_OPT_TID, err)) goto err;
    
    optlen = sizeof(optrec);
    if (!fBufNext(fbuf, (uint8_t *)&optrec, &optlen, err))
        goto err;

    /* grab bin size */
    mask->binsize = optrec.binLengthSeconds;

    /* now derive a mask from the next template */
    if (!(extTmpl = fBufNextCollectionTemplate(fbuf, &extTmplID, err)))
        goto err;
    
    for (spec = naf_full_spec; spec->name; spec++) {
        if (fbTemplateContainsElementByName(extTmpl, spec)) {
            mask->fieldmask |= spec->flags;
        }
    }
    
    /* verify that information is binned */
    if (!(mask->fieldmask & NAF_FM_VSTIME)) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_HEADER,
                    "Input file missing start time field.");
        goto err;
    }
    
    /* set internal template for full record */
    if (!fBufSetInternalTemplate(fbuf, NAF_FULL_TID, err)) goto err;

    /* all done, return buffer */
    return fbuf;
       
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

gboolean nfRead(
    fBuf_t                  *fbuf,
    NAFlowMask              *mask,
    NAFlowKey               *key,
    NAFlowVal               *val,
    GError                  **err)
{
    size_t                  len;
    NAFRecord               rec;
    
    /* read next message as full buffer */
    len = sizeof(rec);
    if (!fBufNext(fbuf, (uint8_t *)&rec, &len, err))
        return FALSE;

    /* copy into key */
    key->srcid   = rec.sourceId;
    key->bin     = rec.flowStartSeconds;
    key->sip     = rec.sourceIPv4Address;
    key->dip     = rec.destinationIPv4Address;
    key->sp      = rec.sourceTransportPort;
    key->dp      = rec.destinationTransportPort;
    key->sipmask = rec.sourceIPv4Mask;
    key->dipmask = rec.destinationIPv4Mask;
    key->proto   = rec.protocolIdentifier;
    
    /* if IP present but mask not, mask default is 32 */
    if ((mask->fieldmask & NAF_FM_SIP) && 
        !(mask->fieldmask & NAF_FM_SIPMASK)) {
        key->sipmask = 32;
    }

    if ((mask->fieldmask & NAF_FM_DIP) &&
        !(mask->fieldmask & NAF_FM_DIPMASK)) {
        key->dipmask = 32;
    }

    /* copy into value */
    memset(val, 0, sizeof(NAFlowVal));
    val->oct  = rec.octetTotalCount;
    val->roct = rec.reverseOctetTotalCount;
    val->pkt  = rec.packetTotalCount;
    val->rpkt = rec.reversePacketTotalCount;
    val->flo  = rec.activeFlowCount;
    val->rflo = rec.reverseActiveFlowCount;
    val->host = rec.sourceActiveHostCount;
    val->rhost = rec.destinationActiveHostCount;
    val->port = rec.sourceActivePortCount;
    val->rport = rec.destinationActivePortCount;

    /* No unique counters available on read NAF records */
    val->vuc = NULL;
    
        
    /* debug */
#if NF_DEBUG_READ
    nfDumpFlow(key, val, "read");
#endif
    
    return TRUE;
}

void nfDumpFlow(
    NAFlowKey       *key,
    NAFlowVal       *val,
    const char      *inf)
{
    char            ipbuf[16];
    static GString   *dstr = NULL;
    
    if (!dstr) dstr = g_string_new("");
    g_string_printf(dstr, "%6s ", inf);
    air_time_g_string_append(dstr, key->bin, AIR_TIME_ISO8601);
    g_string_append_printf(dstr, " [%3u]", key->proto);
    air_ipaddr_buf_print(ipbuf, key->sip);
    g_string_append_printf(dstr, " %15s/%02u:%-5u", 
                           ipbuf, key->sipmask, key->sp);
    air_ipaddr_buf_print(ipbuf, key->dip);
    g_string_append_printf(dstr, " -> %15s/%02u:%-5u", 
                           ipbuf, key->dipmask, key->dp);
    g_string_append_printf(dstr, " %u-%u %u-%u %u-%u %llu-%llu %llu-%llu",
                           val->host, val->rhost,
                           val->port, val->rport,
                           val->flo, val->rflo,
                           val->pkt, val->rpkt,
                           val->oct, val->roct);
    fprintf(stderr, "%s\n", dstr->str);
}
