/*
 ** nafilter.c
 ** Simple NAF-to-NAF filter and sorter.
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/nafcore.h>
#include <naf/filter.h>
#include <naf/lexcore.h>
#include <naf/dynflow.h>
#include <naf/sort.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafilter.c 6585 2007-03-08 00:08:38Z bht $";

static uint32_t         nafistat_in = 0;
static uint32_t         nafistat_inrec = 0;
static uint32_t         nafistat_outrec = 0;

static fBuf_t           *fbuf_in = NULL;
static fBuf_t           *fbuf_out = NULL;

static NAFilter         filter;
static NAFSorter        sorter;
static NAFlowMask       mask;

/* MIO command-line configuration */
static uint32_t         nafi_cliflags = MIO_F_CLI_FILE_IN |
                                        MIO_F_CLI_DIR_IN |
                                        MIO_F_CLI_DEF_STDIN |
                                        MIO_F_CLI_FILE_OUT |
                                        MIO_F_CLI_DIR_OUT |
                                        MIO_F_CLI_DEF_STDOUT;

static void nafi_opt_parse(
    int             argc,
    char            *argv[]) {

    GOptionContext  *octx = NULL;
    GError          *oerr = NULL;
    GScanner        *scanner = NULL;
    
    octx = g_option_context_new("");
    g_option_context_add_group(octx, mio_option_group(nafi_cliflags));
    g_option_context_add_group(octx, daec_option_group());
    g_option_context_add_group(octx, logc_option_group("nafilter",VERSION));
    
    g_option_context_parse(octx, &argc, &argv, &oerr);

    if (oerr) {
        g_error("Couldn't parse command line: %s\nUse --help for usage.", 
                oerr->message);
    }
    
    /* parse remaining command line */
    scanner = naf_lex_init_argv(argc, argv);
    
    /* parse filter expression */
    naf_filter_init(&filter);
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_FILTER) {
        if (!naf_filter_parse(scanner, &filter)) {
            g_error("Couldn't parse filter expression.");
        }
    }
    
    /* parse sort expression */
    naf_sort_init(&sorter);
    if (g_scanner_peek_next_token(scanner) == (GTokenType)NAF_SYM_SORT) {
        if (!naf_sort_parse(scanner, &sorter)) {
            g_error("Couldn't parse sort expression.");
        }
    }
    
}

static gboolean nafi_open_in(
    MIOSource       *source,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    if ((fbuf_in = nfReaderForFP(fbuf_in, mio_fp(source), &mask, err)) == NULL) {
        *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    }
    
    ++nafistat_in;
    
    return TRUE;
}

static gboolean nafi_open_out(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    if ((fbuf_out = nfWriterForFP(fbuf_out, mio_fp(sink), 0, &mask, err)) == NULL) {
        *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    }
    
    return TRUE;
}

static gboolean nafi_process(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    NAFlowKey       key;
    NAFlowVal       val;

    if (nfRead(fbuf_in, &mask, &key, &val, err)) {
        ++nafistat_inrec;
        if (naf_filter_key(&filter, &key) && naf_filter_val(&filter, &val)) {
            if (!nfWrite(fbuf_out, &mask, &key, &val, err)) {
                *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_ERROR);
                return FALSE;
            }
            ++nafistat_outrec;
        }
    } else {
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            *flags |= MIO_F_CTL_SOURCECLOSE;
            return TRUE;
        } else {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
    }
    
    return TRUE;
}

static gboolean nafi_flush (
    void            *ctx,
    NAFlow          *flow,
    GError          **err)
{
    if (!nfWrite(fbuf_out, &mask, &(flow->k), &(flow->v), err)) return FALSE;
    ++nafistat_outrec;    
    return TRUE;
}

static gboolean nafi_process_sort(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    NAFlowKey       key;
    NAFlowVal       val;
    static uint32_t cbin = 0;
    gboolean        rc;
    
    if (!nfRead(fbuf_in, &mask, &key, &val, err)) {
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            *flags |= MIO_F_CTL_SOURCECLOSE;
            return TRUE;
        } else {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
    }
    
    ++nafistat_inrec;
    
    /* short circuit if record doesn't match filter */
    if (!naf_filter_key(&filter, &key) || !naf_filter_val(&filter, &val)) {
        return TRUE;
    }

    /* flush sorter if we've moved on to the next bin */
    if (!cbin) cbin = key.bin;
    if (cbin != key.bin) {                
        /* flush */
        rc = naf_sort_flush(&sorter, nafi_flush, NULL, err);
        
        /* reset the sorter */
        naf_sort_reinit(&sorter);
            
        /* check for error */
        if (!rc) return FALSE;
            
        /* update current bin */
        cbin = key.bin;
    }

    /* create a new tree and flow chunk if we need to */
    if (!sorter.active) {
        naf_sort_begin(&sorter);
    }
            
    /* insert flow into tree */
    naf_sort_flow_kv(&sorter, &key, &val);
    
    /* Done */
    return TRUE;
}

static gboolean nafi_close_out(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    return nfWriterClose(fbuf_out, err);
}

static gboolean nafi_close_out_sort(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    gboolean        rc = TRUE;

    /* do final flush */
    rc = naf_sort_flush(&sorter, nafi_flush, NULL, err);
            
    /* nuke */
    naf_sort_reinit(&sorter);
            
    /* check for error in flush */
    if (!rc) {
        g_warning("error flushing final bin: %s", (*err)->message);
        g_clear_error(err);
    }
    
    /* write result */
    return nfWriterClose(fbuf_out, err);
}

int main (
    int         argc,
    char        *argv[]) {

    GError              *err = NULL;

    MIOSource           source;
    MIOSink             sink;
    MIOAppDriver        adrv = {nafi_open_in, nafi_open_out, nafi_process, 
                                NULL, nafi_close_out};
    uint32_t            miodflags = MIO_F_OPT_SINKLINK;
    
    /* parse options */
    nafi_opt_parse(argc, argv);
    
    /* set up logging */
    if (!logc_setup(&err)) {
        g_error("Couldn't set up logging: %s", err->message);
    }
    
    /* fork if necessary */
    if (!daec_setup(&err)) {
        g_error("Couldn't set up daemon: %s", err->message);
    }
    
    /* set up source and sink */
    if (!mio_config_source(&source, nafi_cliflags, &miodflags, &err)) {
        g_error("Cannot set up input: %s", err->message);
    }

    if (!mio_config_sink(&source, &sink, "%s.naf", nafi_cliflags, 
                         &miodflags, &err)) {
        g_error("Cannot set up output: %s", err->message);
    }

    /* don't sort unless we need to */
    if (sorter.defined) {
        adrv.app_process = nafi_process_sort;
        adrv.app_close_sink = nafi_close_out_sort;
    }
    
    /* run dispatch loop */
    mio_dispatch_loop(&source, &sink, &adrv, NULL, miodflags, mio_ov_poll, 
                      1, mio_ov_poll);

    /* log statistics */
    g_message("nafilter terminating");
    if (nafistat_in) {
        g_message("Processed %u records (passed %u) from %u input file(s)",
                  nafistat_inrec, nafistat_outrec, nafistat_in);
    } else {
        g_warning("No input.");
    }
   
    return 0;
}
