/*
 ** nafload.c
 ** NAF RDBMS loader
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/autoinc.h>

#if HAVE_LIBAIRDBC

#include <naf/nafcore.h>

#include <airdbc/airdbc.h>
#include <airdbc/drv_pg.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafload.c 7222 2007-05-17 19:47:55Z bht $";

#define NAFL_RETDELAY_MIN   20
#define NAFL_RETDELAY_MAX   1800

#define NAFL_SQL_DEFAULT "SELECT air_naf_insert(:stime, :etime, :srcid, :sip, :sipmask, :dip, :dipmask, :sp, :dp, :proto, :host, :rhost, :port, :rport, :flo, :rflo, :pkt, :rpkt, :oct, :roct, :label)"


static char             *nafl_outspec = NULL;
static uint32_t         nafl_sid = 0;
static char             *nafl_label = NULL;
static gboolean         nafl_autolabel = FALSE;

static char             *nafl_sql = NULL;

static uint32_t         naflstat_in = 0;
static uint32_t         naflstat_flow = 0;
static GTimer           *nafl_fft = NULL;

static NAFlowMask       mask;
static AdbConnection    *nafl_conn = NULL;
static AdbStatement     *nafl_stmt = NULL;

static GOptionEntry nafl_optentries[] = {
    { "out", 'o', 0, G_OPTION_ARG_STRING, &nafl_outspec,
      "AirDBC URL of database to insert into", "url" },
    { "sid", 's', 0, G_OPTION_ARG_INT, &nafl_sid,
      "Override event source ID", "int"}, 
    { "label", 'b', 0, G_OPTION_ARG_STRING, &nafl_label,
      "Set record label", "label" },
    { "autolabel", 'B', 0, G_OPTION_ARG_NONE, &nafl_autolabel,
      "Use nafalize label as record label", NULL },
    { NULL }
};

static fBuf_t           *fbuf = NULL;

/* MIO command-line configuration */
static uint32_t         nafl_cliflags = MIO_F_CLI_FILE_IN |
                                        MIO_F_CLI_DIR_IN |
                                        MIO_F_CLI_DEF_STDIN;

static void nafl_opt_parse(
    int             argc,
    char            *argv[]) 
{
    GOptionContext  *octx = NULL;
    GError          *oerr = NULL;
    GString         *sql = NULL;
    uint32_t        i;
    
    octx = g_option_context_new("");
    g_option_context_add_main_entries(octx, nafl_optentries, NULL);
    g_option_context_add_group(octx, mio_option_group(nafl_cliflags));
    g_option_context_add_group(octx, daec_option_group());
    g_option_context_add_group(octx, logc_option_group("nafload",VERSION));
    
    g_option_context_parse(octx, &argc, &argv, &oerr);

    if (oerr) {
        g_error("Couldn't parse command line: %s\nUse --help for usage.", 
                oerr->message);
    }
    
    if (argc > 1) {
        sql = g_string_new("");
        for (i = 1; i < argc; i++) {
            g_string_append_printf(sql, "%s ", argv[i]);
        }
        nafl_sql = sql->str;
        g_string_free(sql, FALSE);
    } else {
        nafl_sql = NAFL_SQL_DEFAULT;
    }
}


#define NIR_BIND_ALWAYS(_fmt_, _val_, _name_) {                     \
    g_string_printf(str, (_fmt_), (_val_));                         \
    bstr = str->str;                                                \
    if (!adb_stmt_bind_named(stmt, (_name_), bstr, err)) {          \
        g_string_free(str, TRUE);                                   \
        return FALSE;                                               \
    }                                                               \
}

#define NIR_BIND(_flag_, _fmt_, _val_, _name_) {                    \
    if (mask->fieldmask & (_flag_)) {                               \
        g_string_printf(str, (_fmt_), (_val_));                     \
        bstr = str->str;                                            \
    } else {                                                        \
        bstr = NULL;                                                \
    }                                                               \
    if (!adb_stmt_bind_named(stmt, (_name_), bstr, err)) {          \
        g_string_free(str, TRUE);                                   \
        return FALSE;                                               \
    }                                                               \
}

static gboolean nafl_insert_record(
    AdbStatement        *stmt,
    NAFlowKey           *key,
    NAFlowVal           *val,
    NAFlowMask          *mask,
    GError              **err)
{
    GString             *str;
    NAFTimeSec          etime;
    uint32_t            srcid;
    char                *bstr = NULL;
    
    /* allocate string for printing */
    str = g_string_new("");

    /* bind time */
    air_time_g_string_append(str, key->bin, AIR_TIME_ISO8601_NS);
    if (!adb_stmt_bind_named(stmt, "stime", str->str, err)) { 
        g_string_free(str, TRUE); 
        return FALSE;
    }
    g_string_truncate(str, 0);
    etime = key->bin + mask->binsize;
    air_time_g_string_append(str, etime, AIR_TIME_ISO8601_NS);
    if (!adb_stmt_bind_named(stmt, "etime", str->str, err)) { 
        g_string_free(str, TRUE); 
        return FALSE;
    }
    
    /* default or force source ID */
    if (nafl_sid) {
        srcid = nafl_sid;
    } else {
        srcid = key->srcid;
    }
    
    /* bind key fields */
    NIR_BIND_ALWAYS("%u", srcid, "srcid");
    NIR_BIND(NAF_FM_SIP, "%u", key->sip, "sip");
    NIR_BIND(NAF_FM_SIPMASK, "%hhu", key->sipmask, "sipmask");
    NIR_BIND(NAF_FM_DIP, "%u", key->dip, "dip");
    NIR_BIND(NAF_FM_DIPMASK, "%hhu", key->dipmask, "dipmask");
    NIR_BIND(NAF_FM_SP, "%hu", key->sp, "sp");
    NIR_BIND(NAF_FM_DP, "%hu", key->dp, "dp");
    NIR_BIND(NAF_FM_PROTO, "%hhu", key->proto, "proto");

    /* bind value fields */
    NIR_BIND(NAF_FM_SHOSTC, "%u", val->host, "host");
    NIR_BIND(NAF_FM_DHOSTC, "%u", val->rhost, "rhost");
    NIR_BIND(NAF_FM_SPORTC, "%u", val->port, "port");
    NIR_BIND(NAF_FM_DPORTC, "%u", val->rport, "rport");
    NIR_BIND(NAF_FM_FLO,  "%u", val->flo, "flo");
    NIR_BIND(NAF_FM_RFLO, "%u", val->rflo, "rflo");
    NIR_BIND(NAF_FM_PKT,  "%llu", val->pkt, "pkt");
    NIR_BIND(NAF_FM_RPKT, "%llu", val->rpkt, "rpkt");
    NIR_BIND(NAF_FM_OCT,  "%llu", val->oct, "oct");
    NIR_BIND(NAF_FM_ROCT, "%llu", val->roct, "roct");
    
    /* bind label */
    if (!adb_stmt_bind_named(stmt, "label", nafl_label, err)) return FALSE;
        
    /* clean up before execution */
    g_string_free(str, TRUE);
    
    /* execute statement */
    return adb_stmt_execute(stmt, err);
}

static gboolean nafl_open_in(
    MIOSource       *source,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    /* read a naf header */
    if ((fbuf = nfReaderForFP(fbuf, mio_fp(source), &mask, err)) == NULL) {
        *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    }
    
    /* force srcid in mask if present */
    if (nafl_sid) {
        mask.fieldmask |= NAF_FM_SRCID;
    }
    
    /* get label for autolabel mode */
    if (nafl_autolabel) {
        char *cp0, *cp1;
        /* free previous label */
        if (nafl_label) g_free(nafl_label);
        /* build new label in place from input path */
        nafl_label = g_strdup(source->name);
        cp0 = strrchr(nafl_label, '.');
        if (!cp0) {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
        *cp0 = (char)0;
        cp1 = strrchr(nafl_label, '-');
        if (!cp1) {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
        *cp1 = (char)0;
        memmove(nafl_label, cp1 + 1, cp0 - cp1);
    }

    ++naflstat_in;

    return TRUE;
}

static gboolean nafl_open_out(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    /* ensure connection is open */
    if (!adb_conn_open(nafl_conn, err)) 
        goto err;
        
    /* attempt to prepare statement */
    if (!nafl_stmt && 
        !(nafl_stmt = adb_stmt_prepare(nafl_conn, nafl_sql, 0, err))) 
        goto err;
        
    /* begin transaction */
    if (!adb_transaction_begin(nafl_conn, err)) goto err;
    
    return TRUE;
    
err:
    /* shut down the connection */
    if (nafl_stmt) {
        adb_stmt_free(nafl_stmt);
        nafl_stmt = NULL;
    }
    adb_conn_close(nafl_conn, NULL);
    
    *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_TRANSIENT);
    return FALSE;
}

static gboolean nafl_process(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    NAFlowKey       key;
    NAFlowVal       val;

    if (nfRead(fbuf, &mask, &key, &val, err)) {
        if (nafl_sid) key.srcid = nafl_sid;
        if (!nafl_insert_record(nafl_stmt, &key, &val, &mask, err)) {
            *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_TRANSIENT);
            return FALSE;
        }
    } else {
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
        } else {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
    }
    
    return TRUE;
}

static gboolean nafl_close_out(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    /* close the transaction */
    if (*flags & (MIO_F_CTL_TRANSIENT | MIO_F_CTL_ERROR)) {
        if (!adb_transaction_rollback(nafl_conn, err)) goto err;
    } else {
        if (!adb_transaction_commit(nafl_conn, err)) goto err;
    }

    return TRUE;

err:
    /* transaction end failure - shut down connection */
    adb_stmt_free(nafl_stmt);
    nafl_stmt = NULL;
    adb_conn_close(nafl_conn, NULL);
    *flags |= MIO_F_CTL_TRANSIENT;
    return FALSE;
}

int main (
    int         argc,
    char        *argv[]) {

    GError      *err = NULL;

    MIOSource           source;
    MIOSink             sink;
    MIOAppDriver        adrv = {nafl_open_in, nafl_open_out, nafl_process, 
                                NULL, nafl_close_out};
    uint32_t            miodflags = MIO_F_OPT_SINKLINK;

    /* initialize file timer */
    nafl_fft = g_timer_new();

    /* parse options */
    nafl_opt_parse(argc, argv);
    
    /* check for autolabel inconsistency */
    if (nafl_label && nafl_autolabel) {
        g_warning("Ignoring --autolabel in presence of --label");
        nafl_autolabel = FALSE;
    }
    
    /* set up logging */
    if (!logc_setup(&err)) {
        g_error("Couldn't set up logging: %s", err->message);
    }
    
    /* fork if necessary */
    if (!daec_setup(&err)) {
        g_error("Couldn't set up daemon: %s", err->message);
    }
    
    /* set up source */
    if (!mio_config_source(&source, nafl_cliflags, &miodflags, &err)) {
        g_error("Cannot set up input: %s", err->message);
    }

    /* set up sink */
    if (!mio_sink_init_app(&sink, nafl_outspec, MIO_T_APP, NULL, &err)) {
        g_error("Cannot set up output: %s", err->message);
    }

    /* create connection */
    if (!(nafl_conn = adb_conn_create(nafl_outspec, &err))) {
        g_error("Cannot create DB connection: %s", err->message);
    }
                
    /* run dispatch loop */
    mio_dispatch_loop(&source, &sink, &adrv, NULL, miodflags, mio_ov_poll, 
                      1, mio_ov_poll);

    /* do final connection close */
    if (nafl_stmt) {
        adb_stmt_free(nafl_stmt);
        nafl_stmt = NULL;
    }
    adb_conn_close(nafl_conn, NULL);

    /* log statistics */
    g_message("nafload terminating");
    if (naflstat_in) {
        g_message("Processed %u records from %u input file(s)",
                  naflstat_flow, naflstat_in);
    } else {
        g_warning("No input.");
    }

    return 0;
}

#else

int main (
    int         argc,
    char        *argv[]) {

    fprintf(stderr, "nafload requires libairdbc, "
                    "but NAF was built without AirDBC support.\n");
    exit(1);
}

#endif

