/*
 ** nafscii.c
 ** NAF flow printer
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/nafcore.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafscii.c 6585 2007-03-08 00:08:38Z bht $";

static uint32_t         naftstat_in = 0;
static uint32_t         naftstat_flow = 0;

static fBuf_t           *fbuf = NULL;
static NAFlowMask       mask;

/* MIO command-line configuration */
static uint32_t         naft_cliflags = MIO_F_CLI_FILE_IN |
                                        MIO_F_CLI_DIR_IN |
                                        MIO_F_CLI_DEF_STDIN |
                                        MIO_F_CLI_FILE_OUT |
                                        MIO_F_CLI_DIR_OUT |
                                        MIO_F_CLI_DEF_STDOUT;

static void naft_opt_parse(
    int             argc,
    char            *argv[]) {

    GOptionContext  *octx = NULL;
    GError          *oerr = NULL;

    octx = g_option_context_new("");
    g_option_context_add_group(octx, mio_option_group(naft_cliflags));
    g_option_context_add_group(octx, daec_option_group());
    g_option_context_add_group(octx, logc_option_group("nafscii",VERSION));
    
    g_option_context_parse(octx, &argc, &argv, &oerr);

    if (oerr) {
        g_error("Couldn't parse command line: %s\nUse --help for usage.", 
                oerr->message);
    }
}

static gboolean naft_print_header(
    FILE                *out,
    NAFlowMask          *mask,
    GError              **err)
{
    GString             *rstr = NULL;
    int                 rc = 0;
        
    /* print date and time headers */
    rstr = g_string_new("      date     time");
    
    /* print source ID header if necessary */
    if (mask->fieldmask & NAF_FM_SRCID) {
        g_string_append(rstr,"   source");
    }
    
    /* print source IP header if necessary */
    if (mask->fieldmask & NAF_FM_SIP) {
        g_string_append(rstr,"             sip");
        if (mask->fieldmask & NAF_FM_SIPMASK) {
            g_string_append(rstr,"   ");
        }
    }

    /* print destination IP header if necessary */
    if (mask->fieldmask & NAF_FM_DIP) {
        g_string_append(rstr,"             dip");
        if (mask->fieldmask & NAF_FM_DIPMASK) {
            g_string_append(rstr,"   ");
        }
    }
    
    /* print port and protocol headers if necessary */
    if (mask->fieldmask & NAF_FM_SP) {
        g_string_append(rstr,"    sp");
    }
    if (mask->fieldmask & NAF_FM_DP) {
        g_string_append(rstr,"    dp");
    }        
    if (mask->fieldmask & NAF_FM_PROTO) {
        g_string_append(rstr," proto");
    }
    
    /* print counter headers as necessary */
    if (mask->fieldmask & NAF_FM_SHOSTC) {
        g_string_append(rstr," shosts");
    }
    if (mask->fieldmask & NAF_FM_DHOSTC) {
        g_string_append(rstr," dhosts");
    }
    if (mask->fieldmask & NAF_FM_SPORTC) {
        g_string_append(rstr," sports");
    }
    if (mask->fieldmask & NAF_FM_DPORTC) {
        g_string_append(rstr," dports");
    }
    if (mask->fieldmask & NAF_FM_FLO) {
        g_string_append(rstr,"        flo");
    }
    if (mask->fieldmask & NAF_FM_RFLO) {
        g_string_append(rstr,"       rflo");
    }
    if (mask->fieldmask & NAF_FM_PKT) {
        g_string_append(rstr,"        pkt");
    }
    if (mask->fieldmask & NAF_FM_RPKT) {
        g_string_append(rstr,"       rpkt");
    }
    if (mask->fieldmask & NAF_FM_OCT) {
        g_string_append(rstr,"        oct");
    }
    if (mask->fieldmask & NAF_FM_ROCT) {
        g_string_append(rstr,"       roct");
    }
     
    /* end header line */
    g_string_append(rstr,"\n");
    
    /* print it */
    rc = fprintf(out, "%s", rstr->str);
    
    /* free string */
    g_string_free(rstr, TRUE);
    
    /* check error */
    if (rc < 0) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO,
                    "couldn't print header: %s", strerror(errno));
        return FALSE;
    } else {
        return TRUE;
    }
}

static gboolean naft_print_record(
    FILE                *out,
    NAFlowKey           *key,
    NAFlowVal           *val,
    NAFlowMask          *mask,
    GError              **err)
{
    char                ipbuf[16];
    GString             *rstr = NULL;
    int                 rc = 0;
    
    /* print bin as date and time */
    rstr = g_string_new("");
    air_time_g_string_append(rstr, key->bin, AIR_TIME_ISO8601);
          
    /* print source ID if necessary */
    if (mask->fieldmask & NAF_FM_SRCID) {
        g_string_append_printf(rstr, " %08x", key->srcid);
    }
    
    /* print source IP if necessary */
    if (mask->fieldmask & NAF_FM_SIP) {
        air_ipaddr_buf_print(ipbuf, key->sip);
        g_string_append_printf(rstr, " %15s", ipbuf);
        if (mask->fieldmask & NAF_FM_SIPMASK) {
            g_string_append_printf(rstr, "/%-2u", key->sipmask);
        }
    }
        
    /* print destination IP if necessary */
    if (mask->fieldmask & NAF_FM_DIP) {
        air_ipaddr_buf_print(ipbuf, key->dip);
        g_string_append_printf(rstr, " %15s", ipbuf);
        if (mask->fieldmask & NAF_FM_DIPMASK) {
            g_string_append_printf(rstr, "/%-2u", key->dipmask);
        }
    }
    
    /* print source port if necessary */
    if (mask->fieldmask & NAF_FM_SP) {
        g_string_append_printf(rstr," %5u", key->sp);
    }
        
    /* print destination port if necessary */
    if (mask->fieldmask & NAF_FM_DP) {
        g_string_append_printf(rstr," %5u", key->dp);
    }
        
    /* print protocol if necessary */
    if (mask->fieldmask & NAF_FM_PROTO) {
        g_string_append_printf(rstr," %5u", key->proto);
    }
    
    /* print counters as necessary */
    if (mask->fieldmask & NAF_FM_SHOSTC) {
        g_string_append_printf(rstr," %6u", val->host);
    }
    if (mask->fieldmask & NAF_FM_DHOSTC) {
        g_string_append_printf(rstr," %6u", val->rhost);
    }
    if (mask->fieldmask & NAF_FM_SPORTC) {
        g_string_append_printf(rstr," %6u", val->port);
    }
    if (mask->fieldmask & NAF_FM_DPORTC) {
        g_string_append_printf(rstr," %6u", val->rport);
    }
    if (mask->fieldmask & NAF_FM_FLO) {
        g_string_append_printf(rstr," %10u", val->flo);
    }
    if (mask->fieldmask & NAF_FM_RFLO) {
        g_string_append_printf(rstr," %10u", val->rflo);
    }
    if (mask->fieldmask & NAF_FM_PKT) {
        g_string_append_printf(rstr," %10llu", val->pkt);
    }
    if (mask->fieldmask & NAF_FM_RPKT) {
        g_string_append_printf(rstr," %10llu", val->rpkt);
    }
    if (mask->fieldmask & NAF_FM_OCT) {
        g_string_append_printf(rstr," %10llu", val->oct);
    }
    if (mask->fieldmask & NAF_FM_ROCT) {
        g_string_append_printf(rstr," %10llu", val->roct);
    }

    /* finish line */
    g_string_append(rstr,"\n");
    
    /* print it */
    rc = fprintf(out, "%s", rstr->str);
    
    /* free string */
    g_string_free(rstr, TRUE);
    
    /* check error */
    if (rc < 0) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO,
                    "couldn't print header: %s", strerror(errno));
        return FALSE;
    } else {
        return TRUE;
    }
}

static gboolean naft_open_in(
    MIOSource       *source,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    /* read a naf header */
    if ((fbuf = nfReaderForFP(fbuf, mio_fp(source), &mask, err)) == NULL) {
        *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    }
    
    ++naftstat_in;

    return TRUE;
}

static gboolean naft_open_out(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    /* print the mask to the output file */
    if (!naft_print_header(mio_fp(sink), &mask, err)) {
        *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    }

    return TRUE;
}

static gboolean naft_process(
    MIOSource       *source,
    MIOSink         *sink,
    void            *vctx,
    uint32_t        *flags,
    GError          **err)
{
    NAFlowKey       key;
    NAFlowVal       val;

    if (nfRead(fbuf, &mask, &key, &val, err)) {
        if (!naft_print_record(mio_fp(sink), &key, &val, &mask, err)) {
            *flags |= (MIO_F_CTL_SINKCLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
        ++naftstat_flow;
    } else {
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            *flags |= MIO_F_CTL_SOURCECLOSE;
            return TRUE;
        } else {
            *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
            return FALSE;
        }
    }
    
    return TRUE;
}


int main (
    int                 argc,
    char                *argv[]) {

    GError              *err = NULL;
    
    MIOSource           source;
    MIOSink             sink;
    MIOAppDriver        adrv = {naft_open_in, naft_open_out, naft_process, 
                                NULL, NULL};
    uint32_t            miodflags = MIO_F_OPT_SINKLINK;

    /* parse options */
    naft_opt_parse(argc, argv);
    
    /* set up logging */
    if (!logc_setup(&err)) {
        g_error("Couldn't set up logging: %s", err->message);
    }
    
    /* fork if necessary */
    if (!daec_setup(&err)) {
        g_error("Couldn't set up daemon: %s", err->message);
    }
    
    /* set up source and sink */
    if (!mio_config_source(&source, naft_cliflags, &miodflags, &err)) {
        g_error("Cannot set up input: %s", err->message);
    }

    if (!mio_config_sink(&source, &sink, "%s.txt", naft_cliflags, 
                         &miodflags, &err)) {
        g_error("Cannot set up output: %s", err->message);
    }

    /* run dispatch loop */
    mio_dispatch_loop(&source, &sink, &adrv, NULL, miodflags, mio_ov_poll,
                      1, mio_ov_poll);

    /* log statistics */
    g_message("nafscii terminating");
    if (naftstat_in) {
        g_message("Processed %u records from %u input file(s)",
                  naftstat_flow, naftstat_in);
    } else {
        g_warning("No input.");
    }
   
    return 0;
}
