/*
 ** naf_indrv.c
 ** nafalize driver-based flow input support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include "nafz_indrv.h"
#include "nafz_iocom.h"
#include <naf/aggregate.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafz_indrv.c 6585 2007-03-08 00:08:38Z bht $";

static GSList *drivers = NULL;

void nafz_drv_register(
    NAFDriver           *driver)
{
    drivers = g_slist_prepend(drivers, driver);
}

NAFDriver *nafz_drv_lookup(
    const char         *name)
{
    GSList              *td;

    for (td = drivers; td; td = td->next) {
        if (strcmp(name, ((NAFDriver *)td->data)->name) == 0) {
            return (NAFDriver *)td->data;
        }
    }

    return NULL;
}

gboolean nafz_open_drv(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;

    return zctx->driver->open(source, &(zctx->rctx), err);
}

gboolean nafz_process_drv(
    MIOSource               *source,
    MIOSink                 *sink,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;
    NAFlowRaw               flow;
    
    /* Read a flow from the driver */
    if (zctx->driver->read(source, zctx->rctx, &flow, err)) {
        /* Okay. Process it. */
        return nafz_process_flow(zctx, &flow, err);
    }
    
    /* If we're here, a read error occurred. */
    if (g_error_matches(*err, NAF_ERROR_DOMAIN, NAF_ERROR_EOF)) {
        g_clear_error(err);
        if (daec_did_quit()) {
            /* Quitting. */
            *flags |= MIO_F_CTL_TERMINATE;
        } else if (!mio_ov_live) {
            /* End of file. */
            *flags |= MIO_F_CTL_SOURCECLOSE;
        } else {
            return TRUE;
        }
        
        /* At end of flow file or termination, do final table flush */
        if (naf_aggregate(TRUE, zctx->mtab, zctx->cfg, zctx->fbuf_ary, err)) {
            return TRUE;
        } else {
            *flags |= MIO_F_CTL_ERROR;
            return FALSE;
        }
    } else {
        /* real, actual error */
        *flags |= MIO_F_CTL_ERROR;
        return FALSE;
    }
}

gboolean nafz_close_drv(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;

    return zctx->driver->close(source, &(zctx->rctx), err);
}
