/*
 ** naf_ipfix.c
 ** NAF IPFIX input support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include "nafz_inipfix.h"
#include "nafz_iocom.h"
#include <naf/aggregate.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafz_inipfix.c 6585 2007-03-08 00:08:38Z bht $";
    
extern uint32_t nafz_liverotate;

#define NAF_EXT_FLOW_TID                    0xAFFE

/* IPFIX definition of the full NAF raw flow record (without flow count) */
static fbInfoElementSpec_t naf_rawflow_spec[] = {
    { "octetTotalCount",                8, 0 },
    { "reverseOctetTotalCount",         8, 0 },
    { "packetTotalCount",               8, 0 },
    { "reversePacketTotalCount",        8, 0 },
    { "activeFlowCount",                0, 0 },
    { "reverseActiveFlowCount",         0, 0 },
    { "flowStartSeconds",               0, 0 },
    { "flowEndSeconds",                 0, 0 },
    { "observationDomainId",            0, 0 },
    { "sourceIPv4Address",              0, 0 },
    { "destinationIPv4Address",         0, 0 },
    { "sourceTransportPort",            0, 0 },
    { "destinationTransportPort",       0, 0 },
    { "sourceIPv4PrefixLength",         0, 0 },
    { "destinationIPv4PrefixLength",    0, 0 },
    { "protocolIdentifier",             0, 0 },
    { "paddingOctets",                  5, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t naf_extflow_spec[] = {
    { "flowStartMilliseconds",              0, 0 },
    { "flowEndMilliseconds",                0, 0 },    
    { "flowStartMicroseconds",              0, 0 },
    { "flowEndMicroseconds",                0, 0 },    
    { "flowStartDeltaMicroseconds",         0, 0 },
    { "flowEndDeltaMicroseconds",           0, 0 },
    { "flowDurationMicroseconds",           0, 0 },
    { "flowDurationMilliseconds",           0, 0 },
    { "octetDeltaCount",                    0, 0 },
    { "packetDeltaCount",                   0, 0 },
    FB_IESPEC_NULL
};

typedef struct _NAFlowExt {
    NAFlowRaw               f;
    uint64_t                stms;
    uint64_t                etms;
    uint64_t                stus;
    uint64_t                etus;
    uint32_t                stdus;
    uint32_t                etdus;
    uint32_t                dus;
    uint32_t                dms;
    uint64_t                doct;
    uint64_t                dpkt;
} NAFlowExt;

/* Static rotation clock */
static uint32_t     nafz_ctime =    0;
static uint32_t     nafz_lrtime =   0;

static fbSession_t *nfInitCollectionSession(
    GError              **err)
{
    fbInfoModel_t       *model = nfInfoModel();
    fbTemplate_t        *tmpl = NULL;
    fbSession_t         *session = NULL;
    
    /* Allocate the session */
    session = fbSessionAlloc(model);

    /* Create the extended record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, naf_rawflow_spec, 0, err))
        return NULL;
    if (!fbTemplateAppendSpecArray(tmpl, naf_extflow_spec, 0, err))
        return NULL;
    
    /* Add it to the session, internal only */
    if (!fbSessionAddTemplate(session, TRUE, NAF_EXT_FLOW_TID, tmpl, err))
        return NULL;
        
    /* now we have a session, suitable for export or collection. */
    return session;
}

gboolean nafz_open_ipfix_file(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;
    fBuf_t                  *fbuf = (fBuf_t *)zctx->rctx;
    fbCollector_t           *collector = NULL;
    fbSession_t             *session = NULL;
    
    /* Allocate a collector for the file */
    collector = fbCollectorAllocFP(NULL, mio_fp(source));
    
    /* Allocate a buffer, or reset the collector */
    if (fbuf) {
        fbCollectorFree(fBufGetCollector(fbuf));
        fBufSetCollector(fbuf, collector);
    } else {        
        if (!(session = nfInitCollectionSession(err))) goto err;
        fbuf = fBufAllocForCollection(session, collector);
    }
    
    /* set internal template to extended flow */
    if (!fBufSetInternalTemplate(fbuf, NAF_EXT_FLOW_TID, err)) goto err;
    
    /* stash buffer */
    zctx->rctx = fbuf;

    return TRUE;
    
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return FALSE;
}

static void nafz_process_ipfix_ext(
    fBuf_t                  *fbuf,
    NAFlowExt               *flow)
{
    uint32_t                exsec;

    /* Run the Gauntlet of Time (adapted from yafcollect) */
    if (flow->f.stime) {
        /* Handle native case first */
        /* Get end time from duration if missing */
        if (flow->f.etime < flow->f.stime) {
            flow->f.etime = flow->f.stime + flow->dms;
        }
    } else if (flow->stms) {
        /* Handle epoch milliseconds. */
        flow->f.stime = flow->stms / 1000;
        if (flow->etms > flow->stms) {
            flow->f.etime = flow->etms / 1000;
        } else {
            flow->f.etime = flow->f.stime + (flow->dms / 1000);
        }
    } else if (flow->stus) {
        /* Decode NTP format microseconds */
        flow->f.stime = flow->stus >> 32;
        if (flow->etus >= flow->stus) {
            flow->f.etime = flow->etus >> 32;
        } else {
            flow->f.etime = flow->f.stime + (flow->dus / 1000000);
        }
    } else if (flow->stdus) {
        /* Decode delta microseconds */
        exsec = fBufGetExportTime(fbuf);
        flow->f.stime = exsec - (flow->stdus / 1000000);
        if (flow->etdus && flow->etdus <= flow->stdus) {
            flow->f.etime = exsec - (flow->etdus / 1000000);
        } else {
            flow->f.etime = flow->f.stime + (flow->dus / 1000000);
        }
    } else {
        /* No time. Use current. */
        struct timeval ct;
        g_assert(!gettimeofday(&ct, NULL));
        flow->f.stime = ct.tv_sec;
        flow->f.etime = flow->f.stime;
    }
    
    /* Handle delta counters */
    if (!flow->f.oct) flow->f.oct = flow->doct;
    if (!flow->f.pkt) flow->f.pkt = flow->dpkt;
}

gboolean nafz_process_ipfix(
    MIOSource               *source,
    MIOSink                 *sink,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;
    fBuf_t                  *fbuf = (fBuf_t *)zctx->rctx;
    fbTemplate_t            *tmpl = NULL;
    gboolean                rr = TRUE;
    NAFlowExt               flow;
    size_t                  len;

    /* read next IPFIX record; retry on missing template. */
    while (1) {
        len = sizeof(NAFlowExt);
            
        /* read a record */
        rr = fBufNext(fbuf, (uint8_t *)&flow, &len, err); 
                
        /* Handle periodic sink close */
        if ((source->vsp_type == MIO_T_SOCK_DGRAM ||
             source->vsp_type == MIO_T_SOCK_STREAM) && sink->iterative) {
            
            nafz_ctime = time(NULL);
            
            if (nafz_lrtime) {
                if (nafz_ctime - nafz_lrtime > nafz_liverotate) {
                    nafz_lrtime = nafz_ctime;
                    *flags |= MIO_F_CTL_SINKCLOSE;
                }
            } else if (rr) {
                nafz_lrtime = nafz_ctime;
            }
        }
        
        /* skip options records */
        if (rr) {
            tmpl = fBufGetCollectionTemplate(fbuf, NULL);
            if (fbTemplateGetOptionsScope(tmpl)) {
                continue;
            } else {
                break;
            }
        }
        
        /* if we're here, we've got nothing. */
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_NLREAD)) {
            /* try again on timeout */
            g_clear_error(err);
            continue;
        } else if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_TMPL)) {
            /* try again on missing template */
            g_debug("skipping IPFIX data set: %s", (*err)->message);
            g_clear_error(err);
            continue;
        } else if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            if (daec_did_quit()) {
                /* Quitting. */
                *flags |= MIO_F_CTL_TERMINATE;
            } else if (!mio_ov_live) {
                /* End of flow file. */
                *flags |= MIO_F_CTL_SOURCECLOSE;
            } else {
                return TRUE;
            }
                
            /* At end of flow file or termination, do final table flush */
            if (naf_aggregate(TRUE, zctx->mtab, zctx->cfg, zctx->fbuf_ary, err)){
                return TRUE;
            } else {
                *flags |= MIO_F_CTL_ERROR;
                return FALSE;
            }
            
        } else {
            /* real, actual error */
            *flags |= MIO_F_CTL_ERROR;
            return FALSE;
        }
    }
    
    /* Flow available and is valid. Process extended information elements. */
    nafz_process_ipfix_ext(fbuf, &flow);
    
    /* Force flow's address prefix lengths to /32 */
    flow.f.sipmask = 32;
    flow.f.dipmask = 32;
    
    /* Count the flow */
    flow.f.flo = 1;
    flow.f.rflo = flow.f.rpkt ? 1 : 0;
    
    /* Flow read and modified as necessary. Handle it. */
    return nafz_process_flow(zctx, &(flow.f), err);
}

gboolean nafz_close_ipfix(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NafalizeContext         *zctx = (NafalizeContext *)vctx;
    fBuf_t                  *fbuf = (fBuf_t *)zctx->rctx;

    /* force end of read */
    if (fbuf) fbCollectorClose(fBufGetCollector(fbuf));
    
    return TRUE;
}

