/*
 ** nafz_pcap.c
 ** NAF libpcap input support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include "nafz_inpcap.h"

#if NAF_PCAP_ENABLE
#include <naf/aggregate.h>
#include <yaf/defrag.h>
#include <yaf/datalink.h>
#include "nafz_iocom.h"

static char *RCSID __attribute__ ((unused)) = 
    "$Id: nafz_inpcap.c 7222 2007-05-17 19:47:55Z bht $";

extern uint32_t nafz_liverotate;

/* Static fragment table. Whee. */
static GHashTable *nafz_pcap_ft =   NULL;
static NAFTimeSec nafz_pcap_ftto =  0;

/* Static datalink info. Also whee. */
static int nafz_pcap_dl =           0;

/* Static rotation clock. */
static uint32_t     nafz_ctime =    0;
static uint32_t     nafz_lrtime =   0;

typedef struct nafz_pcap_hctx_st {
    NAFlowRaw                   *flow;
    uint32_t                    flow_ok;
} nafz_pcap_hctx_t;

typedef struct nafz_ports_st {
    uint16_t        sp;
    uint16_t        dp;
} nafz_ports_t;

static void nafz_pcap_handle(
    uint8_t                     *vhctx,
    const struct pcap_pkthdr    *hdr,
    const uint8_t               *pkt)
{
    nafz_pcap_hctx_t          *hctx = (nafz_pcap_hctx_t *)vhctx;
    const yfHdrIPv4_t         *iph;
    const nafz_ports_t        *l4h;
    const yfHdrTcp_t          *tcph;
    size_t                    caplen;

    /* Update rotation clock */
    nafz_ctime = hdr->ts.tv_sec;

    /* fill in flow metadata */
    hctx->flow->srcid = 0;
    hctx->flow->stime = hctx->flow->etime = hdr->ts.tv_sec;
    
    /* zero counters */
    hctx->flow->oct = 0;
    hctx->flow->roct = 0;
    hctx->flow->pkt = 0;
    hctx->flow->rpkt = 0;
    hctx->flow->flo = 0;
    hctx->flow->rflo = 0;
        
    /* Skip non-IPv4 or truncated packets */
    caplen = hdr->caplen;
    iph = yfDecodeIPv4(nafz_pcap_dl, &caplen, pkt, NULL);
    if (!iph) return;

    /* calculate position of layer 4 header */
    l4h = (const nafz_ports_t *)(((uint8_t *)iph) + 4 * iph->ip_hl);
   
    /* this is an IP datagram - we can count it as a packet */
    hctx->flow->pkt = 1;
        
    /* fill in IP header fields */
    hctx->flow->sip = g_ntohl(iph->ip_src);
    hctx->flow->dip = g_ntohl(iph->ip_dst);
    hctx->flow->sipmask = 32;
    hctx->flow->dipmask = 32;
    hctx->flow->proto = iph->ip_p;
    hctx->flow->oct = g_ntohs(iph->ip_len);
    
    /* handle layer 4 for headers and flow count */
    switch (hctx->flow->proto) {
    case NAF_IP_ICMP:
        if (caplen < 2) return;
        hctx->flow->sp = 0;
        hctx->flow->dp = g_ntohs(l4h->sp);
        break;
    case NAF_IP_TCP:
        if (caplen < 20) return;
        tcph = (struct tcphdr *)l4h;
        hctx->flow->sp = g_ntohs(l4h->sp);
        hctx->flow->dp = g_ntohs(l4h->dp);
        /* count SYNs and SYNACKs as flows */
        if ((tcph->th_flags & (YAF_TF_SYN | YAF_TF_RST | YAF_TF_FIN)) == 
                YAF_TF_SYN) {
            hctx->flow->flo = 1;
        }
        break;
    case NAF_IP_UDP:
        if (caplen < 4) return;
        hctx->flow->sp = g_ntohs(l4h->sp);
        hctx->flow->dp = g_ntohs(l4h->dp);
        break;
    default:
        /* no ports if we don't recognize the transport */
        hctx->flow->sp = 0;
        hctx->flow->dp = 0;
    }

    /* mark flow good */
    hctx->flow_ok = 1;
}

gboolean nafz_open_pcap(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    /* verify datalink */
    nafz_pcap_dl = pcap_datalink(mio_pcap(source));
    if (nafz_pcap_dl != DLT_EN10MB && nafz_pcap_dl != DLT_RAW &&
        nafz_pcap_dl != DLT_LOOP && nafz_pcap_dl != DLT_NULL) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO,
                    "Unsupported pcap datalink type %s",
                    pcap_datalink_val_to_name(nafz_pcap_dl));
        *flags |= (MIO_F_CTL_ERROR | MIO_F_CTL_SOURCECLOSE);
        return FALSE;
    }
    
    /* initialize fragment table */
    yfDefragInit(nafz_pcap_dl, 0, 0, nafz_pcap_handle, NULL);

    return TRUE;
}

gboolean nafz_process_pcap(
    MIOSource               *source,
    MIOSink                 *sink,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    NAFlowRaw               flow;
    nafz_pcap_hctx_t        hctx = {&flow, FALSE};
    NafalizeContext         *zctx = (NafalizeContext *)vctx;
    int                     pcrv = 0;
    
    /* Try to process a packet into a "raw flow" */
    pcrv = pcap_dispatch(mio_pcap(source), 1, 
                         yfDefragPacket, (uint8_t*)&hctx);

    /* Handle periodic sink close */
    if (mio_ov_live && sink->iterative) {
        if (nafz_lrtime) {
            if (nafz_ctime - nafz_lrtime > nafz_liverotate) {
                nafz_lrtime = nafz_ctime;
                *flags |= MIO_F_CTL_SINKCLOSE;
            }
        } else if (pcrv > 0) {
            nafz_lrtime = nafz_ctime;
        }
    }

    /* Check pcap return value */
    if (pcrv > 0) {
        /* If a flow was available, handle it */
        if (hctx.flow_ok) {
            if (!nafz_process_flow(zctx, &flow, err)) {
                return FALSE;
            }
        }
    } else if (pcrv == 0) {
        /* No packet available. Check for EOF or quit. */
        if (daec_did_quit()) {
            /* Quitting. */
            *flags |= MIO_F_CTL_TERMINATE;
        } else if (!mio_ov_live) {
            /* End of capture file. */
            *flags |= MIO_F_CTL_SOURCECLOSE;
        } else {
           return TRUE;
        }
        
        /* At end of dump file or termination, do final table flush */
        if (naf_aggregate(TRUE, zctx->mtab, zctx->cfg, zctx->fbuf_ary, err)) {
            return TRUE;
        } else {
            *flags |= MIO_F_CTL_ERROR;
            return FALSE;
        }
    } else if (pcrv < 0) {
        g_set_error(err, NAF_ERROR_DOMAIN, NAF_ERROR_IO,
                    "Couldn't read next pcap record from %s: %s",
                    source->name, pcap_geterr(mio_pcap(source)));
        *flags |= (MIO_F_CTL_SOURCECLOSE | MIO_F_CTL_ERROR);
        return FALSE;
    } 
    
    /* All done */
    return TRUE;
}

gboolean nafz_close_pcap(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    if (nafz_pcap_ft) {
        g_hash_table_destroy(nafz_pcap_ft);
        nafz_pcap_ft = NULL;
    }
    
    nafz_pcap_ftto = 0;
    return TRUE;
}

gboolean nafz_pcap_stats(
    pcap_t              *pcap,
    uint32_t            *recv,
    uint32_t            *drop)
{
    struct pcap_stat    ps;
    
    if (pcap_stats(pcap, &ps) != 0) return FALSE;
    *recv = ps.ps_recv;
    *drop = ps.ps_drop;
    return TRUE;
}

/* end NAF_PCAP_ENABLE */
#endif
