/*
 ** sort.c
 ** NAF flow sorting support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _NAF_SOURCE_
#include <naf/sort.h>
#include <naf/dynflow.h>
#include <naf/lexcore.h>
#include <airframe/airutil.h>

#define NAF_SORT_COMPARE_FIELD(_f_)         \
    if (a->_f_ > b->_f_) { x = 1; }         \
    else if (a->_f_ < b->_f_) { x = -1; }   \
    else x = 0;                             \
    break;

static int naf_sort_compare(
    const NAFlow        **ap,
    const NAFlow        **bp,
    uint32_t            *sd)
{
    const NAFlow        *a = *ap;
    const NAFlow        *b = *bp;
    uint32_t            i;
    int                 x;

    for (i = 0; i < NAF_SORT_FIELDCOUNT; i++) {
        switch (NAF_FM_MASKF & sd[i]) {
        case NAF_FM_SRCID:
        case NAF_FM_SIP:
            NAF_SORT_COMPARE_FIELD(k.sip);
        case NAF_FM_SIPMASK:
            NAF_SORT_COMPARE_FIELD(k.sipmask);
        case NAF_FM_DIP:
            NAF_SORT_COMPARE_FIELD(k.dip);
        case NAF_FM_DIPMASK:
            NAF_SORT_COMPARE_FIELD(k.dipmask);
        case NAF_FM_PROTO:
            NAF_SORT_COMPARE_FIELD(k.proto);
        case NAF_FM_SP:
            NAF_SORT_COMPARE_FIELD(k.sp);
        case NAF_FM_DP:
            NAF_SORT_COMPARE_FIELD(k.dp);
        case NAF_FM_OCT:
            NAF_SORT_COMPARE_FIELD(v.oct);
        case NAF_FM_ROCT:
            NAF_SORT_COMPARE_FIELD(v.roct);
        case NAF_FM_PKT:
            NAF_SORT_COMPARE_FIELD(v.pkt);
        case NAF_FM_RPKT:
            NAF_SORT_COMPARE_FIELD(v.rpkt);
        case NAF_FM_FLO:
            NAF_SORT_COMPARE_FIELD(v.flo);
        case NAF_FM_RFLO:
            NAF_SORT_COMPARE_FIELD(v.rflo); 
        case NAF_FM_SHOSTC:
            NAF_SORT_COMPARE_FIELD(v.host);
        case NAF_FM_DHOSTC:
            NAF_SORT_COMPARE_FIELD(v.rhost);
        case NAF_FM_SPORTC:
            NAF_SORT_COMPARE_FIELD(v.port);
        case NAF_FM_DPORTC:
            NAF_SORT_COMPARE_FIELD(v.rport);
        default:
            g_assert_not_reached();
        }
        
        /* check for inequality and reverse if necessary */
        if (x) {
            if (sd[i] & NAF_FM_REVSORT) x = -x;
            return x;
        }
   }
   
   /* Flows are equal; return zero. */
   return 0;
}

static void naf_sort_descriptor_prepare(
    uint32_t            *sd)
{
    uint32_t            needf = NAF_FM_MASKF;
    uint32_t            nextf = 1;
    uint32_t            i;
    
    for (i = 0; (i < NAF_SORT_FIELDCOUNT) && (sd[i] & NAF_FM_MASKF); i++) {
        /* Bit set in sort descriptor; verify we still need it. */
        g_assert(needf & sd[i]);
        /* Note that we don't any longer */
        needf &= ~sd[i];
    }
    
    for ( ; i < NAF_SORT_FIELDCOUNT; i++) {
        /* Empty sort descriptor bit. Get next needed field. */
        while (!(needf & nextf)) nextf <<= 1;
        g_assert(nextf & NAF_FM_MASKF);
        /* Set bit in sort descriptor */
        sd[i] = nextf;
        /* Note that we don't need this bit any longer */
        needf &= ~sd[i];
    }
}

void naf_sort_init(
    NAFSorter           *sorter)
{
    uint32_t            i;
    
    sorter->array = NULL;
    sorter->chunk = NULL;
    for (i = 0; i < NAF_SORT_FIELDCOUNT; i++) {
        sorter->desc[i] = 0;
    }
    sorter->limit = 0;
    sorter->active = FALSE;
    sorter->defined = FALSE;
}

void naf_sort_reinit(
    NAFSorter           *sorter)
{
    if (sorter->array) g_ptr_array_free(sorter->array, TRUE);
    sorter->array = NULL;
    if (sorter->chunk) g_mem_chunk_destroy(sorter->chunk);
    sorter->chunk = NULL;
    sorter->active = FALSE;
}

void naf_sort_begin(
    NAFSorter           *sorter)
{
    /* Allocate a new array for buffering flows to sort */
    sorter->array = g_ptr_array_new();

    /* Allocate a flow chunk */
    sorter->chunk = g_mem_chunk_new("flow", sizeof(NAFlow), 
                                    4096 * sizeof(NAFlow), 
                                    G_ALLOC_ONLY);
        
    /* Sorter is active */
    sorter->active = TRUE;
}

void naf_sort_flow(
    NAFSorter           *sorter,
    NAFlow              *flow)
{
    /* Just append the flow to the array */
    g_ptr_array_add(sorter->array, flow);
}
    
void naf_sort_flow_kv(
    NAFSorter           *sorter,
    NAFlowKey           *key,
    NAFlowVal           *val)
{
    /* create new flow in chunk and sort it */
    naf_sort_flow(sorter, naf_flow_alloc(sorter->chunk, key, val));
}

gboolean naf_sort_flush(
    NAFSorter           *sorter,
    NAFSortFlushFn      ffn,
    void                *fctx,
    GError              **err)
{
    uint32_t            i;
    
    /* sort the array */
    g_ptr_array_sort_with_data(sorter->array, 
                               (GCompareDataFunc)naf_sort_compare, 
                               sorter->desc);

    /* traverse the array, flushing it */
    for (i = 0; i < sorter->array->len; i++) {
        /* check limit */
        if (sorter->limit && (i >= sorter->limit)) return TRUE;

        /* flush this flow */
        if (!ffn(fctx, 
                 (NAFlow *)g_ptr_array_index(sorter->array, i), 
                 err)) 
            return FALSE;
    }

    /* End of flush, all is well */
    return TRUE;
}

void naf_sort_print(
    NAFSorter           *sorter,
    GString             *str)
{
    uint32_t            i;
    
    for (i = 0; i < NAF_SORT_FIELDCOUNT; i++) {
        if (sorter->desc[i]) {
            switch (sorter->desc[i] & NAF_FM_MASKF) {
            case NAF_FM_SRCID:
                g_string_append(str, "srcid ");
                break;
            case NAF_FM_SIP:
                g_string_append(str, "sip ");
                break;
            case NAF_FM_SIPMASK:
                g_string_append(str, "sipmask ");
                break;
            case NAF_FM_DIP:
                g_string_append(str, "dip ");
                break;
            case NAF_FM_DIPMASK:
                g_string_append(str, "dipmask ");
                break;
            case NAF_FM_PROTO:
                g_string_append(str, "proto ");
                break;
            case NAF_FM_SP:
                g_string_append(str, "sp ");
                break;
            case NAF_FM_DP:
                g_string_append(str, "dp ");
                break; 
            case NAF_FM_OCT:
                g_string_append(str, "octets ");
                break;
            case NAF_FM_ROCT:
                g_string_append(str, "rev octets ");
                break;
            case NAF_FM_PKT:
                g_string_append(str, "packets ");
                break;
            case NAF_FM_RPKT:
                g_string_append(str, "rev packets ");
                break;
            case NAF_FM_FLO:
                g_string_append(str, "flows ");
                break;
            case NAF_FM_RFLO:
                g_string_append(str, "rev flows ");
                break;
            case NAF_FM_SHOSTC:
                g_string_append(str, "src hosts ");
                break;
            case NAF_FM_DHOSTC:
                g_string_append(str, "dest hosts ");
                break;
            case NAF_FM_SPORTC:
                g_string_append(str, "src ports ");
                break;
            case NAF_FM_DPORTC:
                g_string_append(str, "dest ports ");
                break;
            default:
                g_assert_not_reached();
            }
                
            if (sorter->desc[i] & NAF_FM_REVSORT) {
                g_string_append(str, "desc ");
            }
        } else {
            break;
        }
    }
    
    if (sorter->limit) {
        g_string_append_printf(str, "limit %u ", sorter->limit);
    }
}

#define NAF_SORT_PARSE_FIELD(_f_) {                             \
    if (havef & _f_) {                                          \
        g_scanner_error(scanner, "Duplicate field in sort expression");  \
        return FALSE;                                           \
    }                                                           \
    sorter->desc[i] = _f_;                                      \
    havef |= _f_;                                               \
}

#define NAF_SORT_PARSE_FIELD_REV(_ff_, _rf_) {                  \
    if (is_reverse) {                                           \
        NAF_SORT_PARSE_FIELD(_rf_);                             \
    } else {                                                    \
        NAF_SORT_PARSE_FIELD(_ff_);                             \
    }                                                           \
}

gboolean naf_sort_parse(
    GScanner            *scanner,
    NAFSorter           *sorter)
{
    uint32_t            i = 0, havef = 0;
    gboolean            is_reverse = FALSE;
    
    /* consume the sort token */
    g_assert(g_scanner_get_next_token(scanner) == (GTokenType)NAF_SYM_SORT);
    
    /* do sort field parsing until we're done */
    for (i = 0; i < NAF_SORT_FIELDCOUNT; i++) {
    
        /* check for "rev", "src", and "dest" */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_REV ||
            scanner->next_token == (GTokenType)NAF_SYM_DEST) {
            is_reverse = TRUE;
            g_scanner_get_next_token(scanner);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SRC) {
            g_scanner_get_next_token(scanner);
        }
    
        /* get next field */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_SRCID) {
            NAF_SORT_PARSE_FIELD(NAF_FM_SRCID);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SIP) {
            NAF_SORT_PARSE_FIELD(NAF_FM_SIP);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DIP) {
            NAF_SORT_PARSE_FIELD(NAF_FM_DIP);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PROTO) {
            NAF_SORT_PARSE_FIELD(NAF_FM_PROTO);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_SP) {
            NAF_SORT_PARSE_FIELD(NAF_FM_SP);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_DP) {
            NAF_SORT_PARSE_FIELD(NAF_FM_DP);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_OCTETS) {
            NAF_SORT_PARSE_FIELD_REV(NAF_FM_OCT, NAF_FM_ROCT);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PACKETS) {
            NAF_SORT_PARSE_FIELD_REV(NAF_FM_PKT, NAF_FM_RPKT);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_FLOWS) {
            NAF_SORT_PARSE_FIELD_REV(NAF_FM_FLO, NAF_FM_RFLO);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_HOSTS) {
            NAF_SORT_PARSE_FIELD_REV(NAF_FM_SHOSTC, NAF_FM_DHOSTC);
        } else if (scanner->next_token == (GTokenType)NAF_SYM_PORTS) {
            NAF_SORT_PARSE_FIELD_REV(NAF_FM_SPORTC, NAF_FM_DPORTC);
        } else {
            break;
        }
        
        /* clear reverse flag */
        is_reverse = FALSE;
        
        /* eat the token */
        g_scanner_get_next_token(scanner);
        
        /* check for "asc" (which is a no-op) */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_ASC) {
            g_scanner_get_next_token(scanner);
        }

        /* check for "desc" */
        g_scanner_peek_next_token(scanner);
        if (scanner->next_token == (GTokenType)NAF_SYM_DESC) {
            sorter->desc[i] |= NAF_FM_REVSORT;
            g_scanner_get_next_token(scanner);
        }
    }
    
    /* check for limit */
    g_scanner_peek_next_token(scanner);
    if (scanner->next_token == (GTokenType)NAF_SYM_LIMIT) {
        g_scanner_get_next_token(scanner);
        if (g_scanner_get_next_token(scanner) != G_TOKEN_INT) {
            g_scanner_error(scanner, "Bad sort limit");
            return FALSE;
        }
        sorter->limit = scanner->value.v_int;
    }

    /* Complete the sort descriptor */
    naf_sort_descriptor_prepare(sorter->desc);

    /* Note that the sorter is now defined */
    sorter->defined = TRUE;

    /* All done. */
    return TRUE;
}
