#!/usr/bin/env python

import dircache
import distutils.util
import os
import os.path
import shutil
import sys
import traceback
from optparse import OptionParser, SUPPRESS_HELP

## Error Reporting #####################################################

def error(options, msg):
    raise Exception(msg)

def warn(options, msg):
    print >>sys.stderr, "WARNING:", msg

## Filesystem Manipulation #############################################

def clear_path(options, p):
    if not os.path.lexists(p):
        return
    if os.path.islink(p):
        if options.debug:
            print >>sys.stderr, "rm %s" % repr(p)
        os.unlink(p)
    elif os.path.isdir(p):
        for f in dircache.listdir(p):
            clear_path(options, os.path.join(p, f))
        if options.debug:
            print >>sys.stderr, "rmdir %s" % repr(p)
        os.rmdir(p)
    elif os.path.isfile(p):
        if options.debug:
            print >>sys.stderr, "rm %s" % repr(p)
        os.unlink(p)
    else:
        warn(options, "Unrecognized object in target area: %s" % repr(p))
        os.unlink(p)

def clear_parts(options, parts):
    for p in parts:
        t = os.path.join(options.prefix, p)
        if p == 'lib/python':
            t = os.path.join(options.prefix, 'lib')
        if p == 'htdocs':
            t = options.htdocs_prefix
        clear_path(options, t)

def make_copy(options, s, t):
    if not os.path.lexists(s):
        return
    make_directory(options, os.path.dirname(t))
    if options.debug:
        print >>sys.stderr, "cp -p %s %s" % (repr(s), repr(t))
    shutil.copy(s, t)

def make_directory(options, d):
    if os.path.lexists(d) and not os.path.isdir(d):
        error(options, "Can't overwrite existing file with dir: %s" % d)
    if os.path.isdir(d):
        return
    if options.debug:
        print >>sys.stderr, "mkdir -p %s" % repr(d)
    os.makedirs(d)

def make_symlink(options, s, t):
    if not os.path.lexists(s):
        return
    make_directory(options, os.path.dirname(t))
    if options.debug:
        print >>sys.stderr, "ln -s %s %s" % (repr(s), repr(t))
    os.symlink(s, t)

def merge_tree(options, s, t, copy=False):
    if not os.path.lexists(s):
        return
    if os.path.isdir(s):
        if os.path.lexists(t) and not os.path.isdir(t):
            error(options, "Can't overwrite non-dir with dir: %s" % repr(t))
        else:
            # Make sure directory is there
            make_directory(options, t)
            for f in dircache.listdir(s):
                if f == '.svn':
                    continue
                merge_tree(options, os.path.join(s, f), os.path.join(t, f),
                           copy)
    else:
        if os.path.lexists(t) and not os.path.islink(t) and not copy:
            error(options, "Can't overwrite existing with file: %s" % repr(t))
        else:
            if os.path.islink(t) or (copy and os.path.isfile(t)):
                # Replace existing with new.
                out_t = t
                if out_t.startswith(options.prefix):
                    out_t = out_t[len(options.prefix)+1:]
                print >>sys.stderr, "... replacing %s" % out_t
                os.unlink(t)
            if copy:
                make_copy(options, s, t)
            else:
                make_symlink(options, s, t)

## Installation and Unpacking ##########################################

def do_install_module_part(options, m, p):
    module_path = os.path.join(options.prefix, 'modules', m, p)
    part_path = os.path.join(options.prefix, p)
    copy = False
    if p == 'htdocs':
        part_path = options.htdocs_prefix
        copy = options.copy_htdocs
    elif p == 'doc':
        copy = options.copy_doc
    if m == 'CORE' or m == 'SKIN' or p == 'lib/python':
        merge_tree(options, module_path, part_path, copy)
    elif copy:
        merge_tree(options, module_path, os.path.join(part_path, m), copy)
    else:
        make_symlink(options, module_path, os.path.join(part_path, m))

def find_raved_py():
    # A little work to find the location of raved.py.  We'll look two places.
    # First, look in sys.exec_prefix.
    try:
        import rave
        rave_base = os.path.dirname(
            os.path.dirname(os.path.dirname(os.path.dirname(rave.__file__))))
        raved_py = os.path.join(rave_base, "bin", "raved.py")
        if os.path.exists(raved_py):
            return raved_py
    except:
        pass
    # Not there?  Assume it's in the python exec prefix
    raved_py = os.path.join(sys.exec_prefix, "bin", "raved.py")
    return raved_py

def do_expand_templates(options, part):
    # This is the target directory.  Go there and expand things.
    template_dir = os.path.join(options.prefix, part)
    template_vars = { 'prefix': options.prefix,
                      'htdocs_prefix': options.htdocs_prefix,
                      'python_interpreter': sys.executable,
                      'raved_py': find_raved_py(),
                    }
    print >>sys.stderr, "... expanding templates in", part
    skipped = False
    for n in dircache.listdir(template_dir):
        p = os.path.join(template_dir, n)
        if os.path.isfile(p) and p.endswith(".template"):
            target_name = p[:-9]
            if os.path.exists(target_name):
                print >>sys.stderr, "... skipping", target_name
                continue
            template_file = None
            output_file = None
            try:
                try:
                    template_file = open(p, "r")
                    template = template_file.read()
                    output_file = open(target_name, "w")
                    output_file.write(template % template_vars)
                except:
                    os.unlink(target_name)
                    raise
            finally:
                if template_file: template_file.close()
                if output_file: output_file.close()


def find_python_files(options, py_dir):
    py_files = []
    for n in dircache.listdir(py_dir):
        p = os.path.join(py_dir, n)
        if os.path.isdir(p):
            py_files.extend(find_python_files(options, p))
        elif os.path.isfile(p) and n.endswith(".py"):
            py_files.append(p)
    return py_files

def do_compile_python(options, part):
    py_dir = os.path.join(options.prefix, part)
    py_files = find_python_files(options, py_dir)
    print >>sys.stderr, "... compiling in", part
    distutils.util.byte_compile(py_files, 0, True, None, None, options.debug)
    distutils.util.byte_compile(py_files, 2, True, None, None, options.debug)

def do_compile_make(options, part):
    for mod in dircache.listdir(os.path.join(options.prefix, 'modules')):
        if mod == '.svn':
            continue
        part_path = os.path.join(options.prefix, 'modules', mod, part)
        if os.path.isfile(os.path.join(part_path, 'Makefile')):
            print >>sys.stderr, "... compiling in", \
                os.path.join("modules", mod, part)
            if options.debug:
                print >>sys.stderr, "cd %s; make" % part_path
                rc = os.system("cd %s; make" % part_path)
            else:
                rc = os.system("cd %s; make > /dev/null 2>&1" % part_path)
            if rc != 0:
                error(options, "Make in %s exited with rc %d" % rc)

def do_install(options):
    common_parts = ["lib/python", "doc"]
    analysis_parts = ["analyses"]
    web_parts = ["htdocs", "templates"]
    install_scripts = ["checkenv.py", "install.py"]
    parts = []
    compile_python_parts = []
    compile_make_parts = []
    expand_template_parts = []
    if options.do_install_common:
        if not options.prefix:
            error(options, "--prefix is required when installing")
        # Copy over modules
        merge_tree(options, os.path.join(options.source_prefix, "modules"),
                   os.path.join(options.prefix, "modules"), copy=True)
        # Copy over etc
        merge_tree(options, os.path.join(options.source_prefix, "etc"),
                   os.path.join(options.prefix, "etc"), copy=True)
        expand_template_parts.append("etc")
        for s in install_scripts:
            make_copy(options, os.path.join(options.source_prefix, s),
                      os.path.join(options.prefix, s))
        parts.extend(common_parts)
        compile_python_parts.append("lib/python")
    if options.do_install_ana:
        if not options.prefix:
            error(options, "--prefix is required when installing")
        parts.extend(analysis_parts)
        compile_python_parts.append("analyses")
        compile_make_parts.append("libexec")
    if options.do_install_web:
        if not options.prefix:
            error(options, "--prefix is required when installing")
        parts.extend(web_parts)
    if not parts:
        return
    print >>sys.stderr, "Clearing out previous installed files"
    clear_parts(options, parts)
    for mod in sorted(dircache.listdir(os.path.join(options.source_prefix,
                                                    "modules"))):
        if mod == '.svn':
            continue
        print >>sys.stderr, "Installing %s" % mod
        for p in parts:
            do_install_module_part(options, mod, p)
    if compile_python_parts:
        print >>sys.stderr, "Compiling Python modules"
        for p in compile_python_parts:
            do_compile_python(options, p)
    if compile_make_parts:
        print >>sys.stderr, "Compiling other modules"
        for p in compile_make_parts:
            do_compile_make(options, p)
    if expand_template_parts:
        print >>sys.stderr, "Expanding templates"
        for p in expand_template_parts:
            do_expand_templates(options, p)

def do_unpack(options):
    # Don't copy over modules or etc--the assumption here is that we are
    # unpacking an installation in source_prefix
    if not (options.do_unpack or options.do_clear_unpack):
        return
    # NOTE: doc is *not* unpacked
    unpack_parts = ["analyses", "lib/python", "htdocs", "templates"]
    options.prefix = options.source_prefix
    options.htdocs_prefix = os.path.join(options.prefix, "htdocs")
    options.copy_htdocs = False
    print >>sys.stderr, "Clearing out previous unpacked files"
    clear_parts(options, unpack_parts)
    if options.do_clear_unpack:
        return
    for mod in sorted(dircache.listdir(os.path.join(options.source_prefix,
                                                    "modules"))):
        if mod == '.svn':
            continue
        print >>sys.stderr, "Unpacking %s" % mod
        for p in unpack_parts:
            do_install_module_part(options, mod, p)
    for p in ['lib/python', 'analyses']:
        do_compile_python(options, p)
    for p in ['libexec']:
        do_compile_make(options, p)

## Testing #############################################################

def run_test(options, f):
    sys.stderr.write("... % -50s" % f.__doc__)
    sys.stdout.flush()
    stderr_r_fd, stderr_w_fd = os.pipe()
    pid = os.fork()
    if pid == 0:
        options.debug = True
        try:
            os.dup2(stderr_w_fd, sys.stderr.fileno())
        except:
            pass
        f()
        sys.exit(0)
    else:
        os.close(stderr_w_fd)
        test_out = open("test.log", "a")
        test_out.write("-" * 72)
        test_out.write("\nTesting: ")
        test_out.write(f.__doc__)
        test_out.write("\n")
        try:
            child_stderr_str = ''
            buff = None
            while buff != '':
                buff = os.read(stderr_r_fd, 4096)
                child_stderr_str += buff
            (done_pid, status) = os.waitpid(pid, 0)
            os.close(stderr_r_fd)
            if status == 0:
                test_out.write("pass\n")
                sys.stderr.write("pass\n")
                if child_stderr_str:
                    test_out.write(child_stderr_str)
                    test_out.write("\n")
                return True
            else:
                test_out.write("  FAIL\n")
                sys.stderr.write("  FAIL\n")
                if child_stderr_str:
                    test_out.write(child_stderr_str)
                    test_out.write("\n")
                return False
        finally:
            test_out.close()

def test_rave_installed():
    "RAVE installed"
    import rave.plugins

def test_raved_found():
    "raved.py found"
    raved_py = find_raved_py()
    if not os.path.isfile(raved_py):
        raise Exception("couldn't find raved.py")

def test_pil_installed():
    "PIL installed"
    from PIL import Image, ImageDraw, ImageFont, ImageColor

def test_pil_png_support():
    "PIL PNG support"
    from PIL import Image
    im = Image.open("modules/network_map/etc/hilbert-overlay.png")
    im.load()

def test_pil_ttf_support():
    "PIL TrueType font support"
    from PIL import ImageFont
    font = ImageFont.truetype("etc/fonts/Vera.ttf", 12)
    font.getsize("phil")

def test_mpl_installed():
    "matplotlib installed"
    import matplotlib.ticker
    import matplotlib.figure
    import matplotlib.axes
    import matplotlib.dates
    import matplotlib.numerix
    import matplotlib.font_manager
    import matplotlib.artist
    import matplotlib.text
    import matplotlib.lines
    import matplotlib.patches
    import matplotlib.font_manager 

def test_mpl_backend():
    "matplotlib AGG or Cairo support"
    try:
        from matplotlib.backends.backend_agg import FigureCanvasAgg
    except:
        from matploglib.backends.backend_cairo import FigureCanvasCairo

def test_mpl_font():
    "matplotlib font support"
    from matplotlib import font_manager
    p = font_manager.FontProperties(fname="etc/fonts/Vera.ttf")
    p.get_family()

def test_numpy_installed():
    "Numpy installed"
    import numpy

def test_scipy_installed():
    "Scipy installed"
    import scipy

def test_numpy_scipy_work():
    "Numpy and Scipy work"
    from numpy import matrix, arange
    from scipy.linalg import inv, det, eig
    r = arange(10)
    A = matrix([[1, 1, 1], [4, 4, 3], [7, 8, 5]])
    b = matrix([1,2,1]).transpose()
    det(A)
    inv(A)*b
    eig(A)

def test_psycopg2_installed():
    "psycopg2 installed"
    import psycopg2

common_tests = [
    test_rave_installed,
    ]
analysis_tests = [
    test_raved_found,
    test_pil_installed,
    test_pil_png_support,
    test_pil_ttf_support,
    test_mpl_installed,
    test_mpl_backend,
    test_mpl_font,
    test_numpy_installed,
    test_scipy_installed,
    test_numpy_scipy_work,
    test_psycopg2_installed,
    ]
web_tests = []

def do_checks(options, checks):
    test_count = 0
    pass_count = 0
    for t in checks:
        test_count += 1
        if run_test(options, t):
            pass_count += 1
    return (pass_count, test_count)

def do_check(options):
    if not (options.do_check_common or options.do_check_ana or
            options.do_check_web):
        return True
    print >>sys.stderr, "Running automated tests..."
    if os.path.isfile("test.log") and not os.path.islink("test.log"):
        os.unlink("test.log")
    test_count = 0
    pass_count = 0
    if options.do_check_common:
        (local_passes, local_tests) = do_checks(options, common_tests)
        test_count += local_tests
        pass_count += local_passes
    if options.do_check_ana:
        (local_passes, local_tests) = do_checks(options, analysis_tests)
        test_count += local_tests
        pass_count += local_passes
    if options.do_check_web:
        (local_passes, local_tests) = do_checks(options, web_tests)
        test_count += local_tests
        pass_count += local_passes
    print "%d/%d tests passed" % (pass_count, test_count)
    if pass_count < test_count:
        return False
    return True

## Main Program Code ###################################################

def parse_options():
    op = OptionParser(usage="""%prog [options] commands ...

Valid Commands:
  check                 Run automated tests for required libraries
  check-analysis        Run tests only for analysis-required libraries
  check-web             Run tests only for portal-required libraries
  install               Install the software
  install-analysis      Install only the analysis software
  install-web           Install only the portal software
  link                  Create source-tree symlinks for development testing
  clean                 Clean up source-tree symlinks""")
    op.set_defaults(
        do_check=False, do_install=False, do_unpack=False,
        do_clear_unpack=False,
        do_check_common=False, do_check_ana=False, do_check_web=False,
        do_install_common=False, do_install_ana=False, do_install_web=False,
        prefix=None, htdocs_prefix=None, root="/", make_rootdir=True,
        debug=False, force_install=False, copy_htdocs=True, copy_doc=False,
        source_prefix=os.path.dirname(os.path.abspath(__file__)))
    op.add_option("--analysis", action="store_true",
                  dest="do_install_ana", help=SUPPRESS_HELP)
    op.add_option("--web", dest="do_install_web",
                  action="store_true", help=SUPPRESS_HELP)
    op.add_option("--only-check-environment", action="store_true",
                  dest="do_check", help=SUPPRESS_HELP)
    op.add_option("--dont-check-environment", action="store_true",
                   dest="dont_check", help=SUPPRESS_HELP)
    op.add_option("--prefix", metavar="PATH",
                  help="Where to install RAVE components")
    op.add_option( "--portal-prefix", dest="prefix", help=SUPPRESS_HELP)
    op.add_option("--htdocs-prefix", metavar="PATH",
                  help="Where to install web pages, default $prefix/htdocs")
    op.add_option("--web-prefix", dest="htdocs_prefix", help=SUPPRESS_HELP)
    op.add_option("--root", default="/", metavar="PATH",
                  help="Alternative root directory for installation")
    op.add_option("--make-rootdir", action="store_true", help=SUPPRESS_HELP)
    op.add_option("-d", "--debug", action="store_true",
                  help="Enable debugging output")
    op.add_option("--force-install", action="store_true",
                  help="Install even if environment problems are discovered")
    options, args = op.parse_args()
    if not args:
        # No commands given
        op.print_help()
        sys.exit(-1)
    if options.prefix is not None and options.htdocs_prefix is None:
        options.htdocs_prefix = os.path.join(options.prefix, "htdocs")
    for command in args:
        if command == 'check':
            options.do_check = True
        elif command == 'check-analysis':
            options.do_check_ana = True
        elif command == 'check-web':
            options.do_check_web = True
        elif command == 'install':
            options.do_install = True
        elif command == 'install-analysis':
            options.do_install_ana = True
        elif command == 'install-web':
            options.do_install_web = True
        elif command == 'clean':
            options.do_clear_unpack = True
        elif command == 'link':
            options.do_unpack = True
        else:
            op.print_help()
            sys.exit(-1)
    if options.do_install:
        options.do_check = True
        options.do_install_ana = True
        options.do_install_common = True
        options.do_install_web = True
    if options.do_install_ana:
        options.do_check_ana = True
        options.do_install_common = True
    if options.do_install_web:
        options.do_check_web = True
        options.do_install_common = True
    if options.do_install_common:
        options.do_check_common = True
    if options.do_check:
        options.do_check_ana = True
        options.do_check_common = True
        options.do_check_web = True
    if options.dont_check:
        options.do_check = False
        options.do_check_ana = False
        options.do_check_common = False
        options.do_check_web = False
    if (options.do_install_ana or options.do_install_common or
        options.do_install_web):
        if not options.prefix:
            error(options, "--prefix is required when installing")
    return options

def main():
    class Holder: pass
    options = Holder()
    options.debug = True
    try:
        options = parse_options()
        if not do_check(options):
            if not options.force_install:
                print "ABORTING"
                sys.exit(-1)
        do_install(options)
        # Make sure unpacking is done last--it screws with the prefixes.
        do_unpack(options)
    except SystemExit:
        pass
    except:
        t, v, tb = sys.exc_info()
        if options.debug:
            traceback.print_exception(t, v, tb, file=sys.stderr)
        else:
            print >>sys.stderr, "ERROR: (%s) %s" % (str(t), str(v))
        sys.exit(1)

if __name__ == '__main__':
    main()
