from ConfigParser import SafeConfigParser
import os.path
import sys
import dircache

portal_path = os.path.abspath(os.path.join(os.path.dirname(__file__),
                                           '../../..'))

path = [ '/etc', '/usr/local/etc', portal_path ]
config_file_name = 'portal.conf'

## Defaults ############################################################

def init_defaults():
    config = SafeConfigParser()

    config.add_section('files')
    config.set('files', 'portal_path',    portal_path)
    config.set('files', 'module_path',    '%(portal_path)s/modules')
    config.set('files', 'analyses_path',  '%(portal_path)s/analyses')
    config.set('files', 'htdocs_path',    '%(portal_path)s/htdocs')
    config.set('files', 'templates_path', '%(portal_path)s/templates')
    config.set('files', 'font_path',      '%(portal_path)s/etc/fonts')

    config.add_section('databases')

    config.set('databases', 'portdb',           'postgresql:///portdb')

    config.add_section('rave')

    config.set('rave',      'base_url',          '/rave/')

    config.add_section('rendering')

    config.set('rendering', 'bgcolor',             '#ffffff')
    config.set('rendering', 'sparklines.maxcolor', '#00aa00')
    config.set('rendering', 'sparklines.mincolor', '#aa0000')
    config.set('rendering', 'sparklines.width',    '175')
    config.set('rendering', 'sparklines.height',   '20')
    config.set('rendering', 'chart.width',         '635')
    config.set('rendering', 'chart.height',        '400')
    config.set('rendering', 'icon_background',     'dark')

    config.add_section('site')
    config.set('site',      'site_name',           'SITE_NAME')
    config.set('site',      'url_root',            '.')
    config.set('site',      'disabled_modules',    '')

    # <group_name>: <sensor>,<sensor>,...
    config.add_section('sensor_groups')

    # <display_group>: <item_name>[:<item_display_name>],...
    config.add_section('sensor_display_groups')

    # <group_name>: <user>,<user>,<user>,...
    config.add_section('user_groups')

    # <user_or_group_name>: <sensor_or_group_name>,<sensor_or_group_name>,...
    config.add_section('user_group_sensors')

    # <rave-url>: allow:<group_name>,restrict-all-sensors,...
    #   allow:<group_name> means only that group is allowed to view at all
    #   restrict-all-sensors means "all" becomes a list of all allowed sensors
    config.add_section('restricted_urls')

    return config

config = init_defaults()

## Dynamic Updates #####################################################

last_mtime = {}

def get_mtime(path):
    try:
        return os.path.getmtime(path)
    except:
        return 0

def update_config(force=False):
    # Scan config files to see if any of them has changed.  If so, reload
    # everything.  Otherwise, do nothing.
    config_files = [os.path.join(d, config_file_name) for d in reversed(path)]
    reload = False
    if force:
        reload = True
    else:
        for f in config_files:
            if last_mtime.get(f,0) < get_mtime(f):
                reload = True
                break
    if reload:
        local_config = init_defaults()
        used_files = local_config.read(config_files)
        for f in config_files:
            last_mtime[f] = get_mtime(f)
        global config
        config = local_config

# Now read the configuration file(s)
update_config()

## Accessors ###########################################################

def file(path_name, file_name=None):
    update_config()
    if file_name == None:
        return config.get('files', path_name)
    else:
        return os.path.join(config.get('files', path_name), file_name)

def template(file_name):
    return file("templates_path", file_name)

def font(font_name):
    return file("font_path", font_name + ".ttf")

def module_file(module_name, file_name):
    result = file('module_path', os.path.join(module_name, file_name))
    return result

def module_template(module_name, file_name):
    return module_file(module_name, os.path.join("templates", file_name))

def database(dbname):
    update_config()
    return config.get('databases', dbname)

def rendering_option(opt_name):
    update_config()
    return config.get('rendering', opt_name)

def rave_option(opt_name):
    update_config()
    return config.get('rave', opt_name)

def site_option(opt_name):
    update_config()
    return config.get('site', opt_name)

def url(relative_url):
    return site_option("url_root") + "/" + relative_url

def module_url(module_name, relative_url):
    return site_option("url_root") + "/" + module_name + "/" + relative_url

def sensor_group(group_name):
    update_config()
    try:
        return [x.strip()
                for x in config.get('sensor_groups', group_name).split(',')]
    except:
        # Actually a sensor name.  Probably.
        return [group_name]

def sensor_option_groups(user=None):
    update_config()
    result = []
    group_display_names = [x.strip() for x in
                           config.get('sensor_display_groups', '*').split(',')]
    for group_display_name in group_display_names:
        group_info = config.get('sensor_display_groups', group_display_name)
        group_result = []
        for display_item in [x.strip() for x in group_info.split(',')]:
            if ':' in display_item:
                (opt, display_opt) = [x.strip()
                                      for x in display_item.split(':')]
            else:
                opt = display_item
                display_opt = display_item
            if user is not None:
                if opt in allowed_sensors(user):
                    group_result.append((display_opt, opt))
            else:
                group_result.append((display_opt, opt))
        if group_result:
            result.append((group_display_name, group_result))
    return result

def user_group(group_name):
    update_config()
    return [x.strip()
            for x in config.get('user_groups', group_name).split(',')]

def user_groups(user_name):
    update_config()
    result = [user_name]
    for (group_name, user_names) in config.items('user_groups'):
        if user_name in [x.strip() for x in user_names.split(',')]:
            result.append(group_name)
    return result

def allowed_sensors(user_or_group_name):
    update_config()
    groups = user_groups(user_or_group_name)
    sensor_groups = set([])
    for group_name in groups:
        try:
            group_sensors = config.get('user_group_sensors', group_name)
            sensor_groups = sensor_groups.union(
                [x.strip() for x in group_sensors.split(',')])
        except:
            pass
    result = set(['all'] + list(sensor_groups))
    for sensor_group_name in sensor_groups:
        try:
            result = result.union(config.sensor_group(sensor_group_name))
        except:
            pass
    return list(result)

def url_restrictions(rave_url):
    update_config()
    result = []
    try:
        url_restrictions = config.get('restricted_urls', rave_url)
        for restriction in [x.strip() for x in url_restrictions.split(',')]:
            result.append(restriction)
    except:
        pass
    return result

## Module Configuration ################################################

module_configs = {}
panel_configs = []

def read_module_config(p):
    config = SafeConfigParser()
    config.add_section('module')
    config.read(p)
    return config

def update_module_configs(force=False):
    update_config(force)
    reload = False
    module_path = config.get('files', 'module_path')
    disabled_modules = config.get('site', 'disabled_modules', '').split(',')
    disabled_modules = [x.strip() for x in disabled_modules]
    if force:
        reload = True
    else:
        if last_mtime.get(module_path,0) < get_mtime(module_path):
            reload = True
        else:
            for mod in dircache.listdir(module_path):
                if mod in disabled_modules:
                    continue
                conf_path = os.path.join(module_path, mod, "module.conf")
                if os.path.isfile(conf_path):
                    if last_mtime.get(conf_path,0) < get_mtime(conf_path):
                        reload = True
                        break
    if reload:
        # Reload all of the module configs.
        global module_configs
        global panel_configs
        local_configs = {}
        for mod in dircache.listdir(module_path):
            if mod in disabled_modules:
                continue
            conf_path = os.path.join(module_path, mod, "module.conf")
            if os.path.isfile(conf_path):
                local_configs[mod] = read_module_config(conf_path)
        local_panel_configs = {}
        for mod in local_configs:
            mod_conf = local_configs[mod]
            for section in mod_conf.sections():
                if section == 'module':
                    continue
                new_conf = {'order': '50',
                            'allowed_groups': None,
                            'title': section,
                            'href': 'index.psp',
                            'href_advanced': None,
                            'icon_light': None,
                            'icon_dark': None,
                            'template_front': None,
                            'template_summary': None,
                            'template_controls': None}
                for k in new_conf:
                    try:
                        new_conf[k] = mod_conf.get(section, k)
                    except:
                        pass
                new_conf['id'] = section
                new_conf['module'] = mod
                local_panel_configs[section] = new_conf
        module_configs = local_configs
        def compare_panel_configs(a, b):
            return cmp(int(local_panel_configs[a]['order']),
                       int(local_panel_configs[b]['order']))
        panel_configs = [local_panel_configs[x]
                         for x in sorted(local_panel_configs,
                                         compare_panel_configs)]

def module_list():
    update_module_configs()
    return sorted(module_configs.keys())

def get_module_configs():
    update_module_configs()
    return module_configs

def get_panel_configs():
    update_module_configs()
    return panel_configs

########################################################################

__all__ = ['file', 'template', 'font', 'module_file', 'module_template',
           'database', 'rendering_option', 'rave_option', 'site_option',
           'url', 'module_url', 'sensor_option_groups', 'sensor_group',
           'allowed_sensors', 'url_restrictions', 'user_groups',
           'module_list', 'get_module_configs', 'get_panel_configs']
