#!/usr/bin/env python

from PIL import Image, ImageDraw, ImageFont, ImageColor
from subprocess import Popen, PIPE
from tempfile import TemporaryFile
import sys, math

from optparse import OptionParser

default_font = '/Library/Fonts/BigCaslon.dfont'

parser = OptionParser(usage="usage: %prog [ options ] --output-file=FILE")
parser.add_option("--file", dest="file", default=None,
                  help="Read input data from FILE (default stdin)",
                  metavar="FILE")
parser.add_option("--output-file", dest="output_file", default=None,
                  help="Write output to FILE (required)",
                  metavar="FILE")
parser.add_option("--hibertize-program", dest="hilbertize_program",
                  default="./hilbertize", metavar="PATH",
                  help="Run program at PATH to convert to positions on the Hilbert curve (default './hilbertize')")
#parser.add_option("--width", dest="width", default=1024, type='int',
#                  help="Set width of output image to WIDTH (default 1024)",
#                  metavar="WIDTH")
#parser.add_option("--height", dest="height", default=None, type='int',
#                  help="Set height of output image to HEIGHT (default WIDTH)",
#                  metavar="HEIGHT")
parser.add_option("--address-bits", dest="addr_bits", metavar="N", type='int',
                  help="Process input addresses to N bits (default 24)",
                  default=24)

parser.add_option("--background-color", dest="background_color",
                  metavar="COLOR",
                  help="Set color for non-activity (default 'white')",
                  default="white")
parser.add_option("--label-color", dest="label_color", metavar="COLOR",
                  help="Set color for text labels (default 'red')",
                  default="red")
parser.add_option("--label-font", dest="label_font", metavar="FILE",
                  help="Set font file for labels (default '%s')" % default_font,
                  default=default_font)
parser.add_option("--min-color", dest="min_color", metavar='COLOR',
                  help="Set color for minimum value (default 'gray')",
                  default='gray')
parser.add_option("--max-color", dest="max_color", metavar='COLOR',
                  help="Set color for maximum value (default 'black')",
                  default='black')
parser.add_option("--min-value", dest="min_value", type='int', metavar='N',
                  help="Set minimum value (derived by default)",
                  default=None)
parser.add_option("--max-value", dest="max_value", type='int', metavar='N',
                  help="Set maximum value (derived by default)",
                  default=None)

(options, args) = parser.parse_args()

if args or (options.output_file == None):
    parser.print_help()
    sys.exit(-1)

in_file = sys.stdin
if options.file:
    in_file = file(options.file, 'r')
out_filename = options.output_file

hilbertize_program = options.hilbertize_program

#width = options.width
#height = options.height
#if height == None:
#    height = width

height_multiplier = 1
address_bits = options.addr_bits
address_nonbits = 32 - address_bits
flip_xy = True
if address_bits % 2 == 1:
    address_bits = address_bits + 1
    height_multiplier = 2
    flip_xy = False

data_width = 2**(address_bits/2)
data_height = 2**(address_bits/2)/height_multiplier

background_color = ImageColor.getrgb(options.background_color)
label_color = ImageColor.getrgb(options.label_color)
label_font = options.label_font
min_color = ImageColor.getrgb(options.min_color)
max_color = ImageColor.getrgb(options.max_color)

min_value = options.min_value
max_value = options.max_value

# First, send input file to hilbertize and find min/max values
temp_file = TemporaryFile()
hilbertize = Popen([hilbertize_program, `address_bits/2`],
                   stdin=PIPE, stdout=temp_file)
computed_min_value = 2**31
computed_max_value = 0
for line in in_file:
    address, value = line.split()
    if '.' in address:
        octets = [int(i) for i in address.split('.')]
        address = str((octets[0]<<24) + (octets[1]<<16) +
                      (octets[2]<<8) + (octets[3]))
    a = int(address)
    v = int(value)
    if v < computed_min_value:
        computed_min_value = v
    if v > computed_max_value:
        computed_max_value = v
    if address_nonbits > 0:
        a = a >> address_nonbits
    hilbertize.stdin.write("%d %d\n" % (a, v))
hilbertize.stdin.close()
hilbertize.wait()
temp_file.seek(0)
if min_value == None:
    min_value = computed_min_value
if max_value == None:
    max_value = computed_max_value

im = Image.new('RGB', (data_width, data_height), background_color)

im.load()
putpixel = im.im.putpixel

if max_value > 0: max_value = math.log(max_value)
if min_value > 0: min_value = math.log(min_value)
if max_value == min_value:
    min_value = max_value - 1
value_scale = max_value - min_value
r_scale = float(max_color[0] - min_color[0])
g_scale = float(max_color[1] - min_color[1])
b_scale = float(max_color[2] - min_color[2])

for line in temp_file:
    x, y, v = (int(i) for i in line.split())
    if v > 0: v = math.log(v)
    r = int(min_color[0] + (r_scale * (v - min_value)) / value_scale)
    g = int(min_color[1] + (g_scale * (v - min_value)) / value_scale)
    b = int(min_color[2] + (b_scale * (v - min_value)) / value_scale)
    if flip_xy:
        putpixel((y, x), (r,g,b))
    else:
        putpixel((x, y), (r,g,b))
temp_file.close()

temp_file = TemporaryFile()
hilbertize = Popen(['./hilbertize', `address_bits/2`],
                   stdin=PIPE, stdout=temp_file)

for i in range(256):
    i_pos = i << (address_bits - 8)
    i_pos = i_pos + (1 << (address_bits - 9))
    print >>hilbertize.stdin, "%d %d" % (i_pos, i)
hilbertize.stdin.close()
hilbertize.wait()
temp_file.seek(0)

if flip_xy:
    font_size = 60 * data_width / 4096
else:
    font_size = 60 * data_height / 4096
x_off = 110 * data_width / 4096
y_off = 128 * data_height * height_multiplier/ 4096
font = ImageFont.truetype(label_font, font_size)
draw = ImageDraw.Draw(im)
for line in temp_file:
    (x, y, l) = line.split()
    x = int(x)
    y = int(y)
    if flip_xy:
        draw.text((y-x_off,x-y_off), l, fill=label_color, font=font)
    else:
        draw.text((x-x_off,y-y_off), l, fill=label_color, font=font)

im.save(out_filename)

