from rave.plugins.dataset import Dataset
from rave.plugins.db import query
from rave.plugins.decorators import *
from rave.plugins.render_mpl import *
from rave.plugins.render_html import *
from rave.plugins.times import *
from datetime import timedelta

from portal.util import *
import portal.config

from urllib import urlencode

set_font_filename(portal.config.font('Vera'))

one_second = timedelta(seconds=1)

width = 680
height = 500

def gen_all_proto_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select protocol, sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by protocol
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['protocol']

def gen_all_proto(sdate, edate, dir, sensors, countries, n):
    protocols = gen_all_proto_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               cast(protocol as text) as protocol,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and protocol = any ( %(protocols)s )
          group by rb_time, protocol
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as protocol,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and protocol <> all ( %(protocols)s )
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'protocols': protocols})
    protocol_map = get_protocol_map()
    return Dataset(bin_time=data['rb_time'],
                   protocol=(str(x) + "\n" + protocol_map.get(str(x), "")
                             for x in data['protocol']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_proto_single(sdate, edate, dir, sensors, countries, protocol):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and protocol = %(protocol)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'protocol': protocol})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_sensor(sdate, edate, dir, sensors, countries):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sensor,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by rb_time, sensor
        order by rb_time, sensor
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return Dataset(bin_time=data['rb_time'],
                   sensor=data['sensor'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_sensor_single(sdate, edate, dir, sensors, countries, sensor):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and sensor = %(sensor)s
        group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'sensor': sensor})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_region(sdate, edate, dir, sensors, countries):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               coalesce((select region_short_name from cc_regions
                         where region_cc = country), 'Other') as region,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by rb_time, region
        order by rb_time, region
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return Dataset(bin_time=data['rb_time'],
                   region=data['region'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_subregion(sdate, edate, dir, sensors, countries, region):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               coalesce((select region_short_name from cc_subregions
                         where region_cc = country), 'Other') as subregion,
               sum(bytes) as bytes
          from all_protocol
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and country = any ( select region_cc from cc_regions
                                  where region_short_name = %(region)s )
        group by rb_time, subregion
        order by rb_time, subregion
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'region': region})
    return Dataset(bin_time=data['rb_time'],
                   subregion=data['subregion'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_country(sdate, edate, dir, sensors, countries, subregion):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
      select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
              trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
             country,
             sum(bytes) as bytes
        from all_protocol
        where bin_time >= %(sdate)s and bin_time < %(edate)s
          and direction = %(direction)s
          and sensor = any ( %(sensors)s )
          and country = any ( %(countries)s )
          and country = any ( select region_cc from cc_subregions
                                where region_short_name = %(subregion)s )
      group by rb_time, country
      order by rb_time, country
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'subregion': subregion})
    return Dataset(bin_time=data['rb_time'],
                   country=data['country'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_all_country_single(sdate, edate, dir, sensors, country):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
      select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
              trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
             sum(bytes) as bytes
        from all_protocol
        where bin_time >= %(sdate)s and bin_time < %(edate)s
          and direction = %(direction)s
          and sensor = any ( %(sensors)s )
          and country = %(country)s
      group by rb_time
      order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'country': country})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_tcp_sport_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select port, sum(bytes) as bytes
          from tcp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by port
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['port']

def gen_tcp_sport(sdate, edate, dir, sensors, countries, n):
    ports = gen_tcp_sport_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               cast(port as text) as port,
               sum(bytes) as bytes
          from tcp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = any ( %(ports)s )
          group by rb_time, port
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as port,
               sum(bytes) as bytes
          from tcp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port <> all ( %(ports)s )
          group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'ports': ports})
    service_map = get_service_map()
    return Dataset(bin_time=data['rb_time'],
                   port=(str(x) + "\n" + service_map.get("tcp/" + str(x), "")
                         for x in data['port']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_tcp_sport_single(sdate, edate, dir, sensors, countries, port):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from tcp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = %(port)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'port': port})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_tcp_dport_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select port, sum(bytes) as bytes
          from tcp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by port
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['port']

def gen_tcp_dport(sdate, edate, dir, sensors, countries, n):
    ports = gen_tcp_dport_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               cast(port as text) as port,
               sum(bytes) as bytes
          from tcp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = any ( %(ports)s )
          group by rb_time, port
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as port,
               sum(bytes) as bytes
          from tcp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port <> all ( %(ports)s )
          group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'ports': ports})
    service_map = get_service_map()
    return Dataset(bin_time=data['rb_time'],
                   port=(str(x) + "\n" + service_map.get("tcp/" + str(x), "")
                         for x in data['port']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_tcp_dport_single(sdate, edate, dir, sensors, countries, port):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from tcp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = %(port)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'port': port})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_udp_sport_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select port, sum(bytes) as bytes
          from udp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by port
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['port']

def gen_udp_sport(sdate, edate, dir, sensors, countries, n):
    ports = gen_udp_sport_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               cast(port as text) as port,
               sum(bytes) as bytes
          from udp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = any ( %(ports)s )
          group by rb_time, port
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as port,
               sum(bytes) as bytes
          from udp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port <> all ( %(ports)s )
          group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'ports': ports})
    service_map = get_service_map()
    return Dataset(bin_time=data['rb_time'],
                   port=(str(x) + "\n" + service_map.get("udp/" + str(x), "")
                         for x in data['port']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_udp_sport_single(sdate, edate, dir, sensors, countries, port):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from udp_sport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = %(port)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'port': port})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_udp_dport_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select port, sum(bytes) as bytes
          from udp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by port
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['port']

def gen_udp_dport(sdate, edate, dir, sensors, countries, n):
    ports = gen_udp_dport_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               cast(port as text) as port,
               sum(bytes) as bytes
          from udp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = any ( %(ports)s )
          group by rb_time, port
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as port,
               sum(bytes) as bytes
          from udp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port <> all ( %(ports)s )
          group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'ports': ports})
    service_map = get_service_map()
    return Dataset(bin_time=data['rb_time'],
                   port=(str(x) + "\n" + service_map.get("udp/" + str(x), "")
                         for x in data['port']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_udp_dport_single(sdate, edate, dir, sensors, countries, port):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from udp_dport
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and port = %(port)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'port': port})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_icmp_type_code_top(sdate, edate, dir, sensors, countries, n):
    data = query(portal.config.database('portdb'), """
        select (icmp_type || ',' || icmp_code) as type_code,
               sum(bytes) as bytes
          from icmp_type_code
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
        group by type_code
        order by bytes desc
        limit %(n)s
    """, {'sdate': sdate, 'edate': edate, 'n': n,
          'direction': dir, 'sensors': sensors, 'countries': countries})
    return data['type_code']

def gen_icmp_type_code(sdate, edate, dir, sensors, countries, n):
    tycodes = gen_icmp_type_code_top(sdate, edate, dir, sensors, countries, n)
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               (icmp_type || ',' || icmp_code) as type_code,
               sum(bytes) as bytes
          from icmp_type_code
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and (icmp_type || ',' || icmp_code) = any ( %(tycodes)s )
          group by rb_time, type_code
        union all
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               'other' as type_code,
               sum(bytes) as bytes
          from icmp_type_code
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and (icmp_type || ',' || icmp_code) <> all ( %(tycodes)s )
          group by rb_time
        order by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'tycodes': tycodes})
    service_map = get_service_map()
    return Dataset(bin_time=data['rb_time'],
                   type_code=(x + "\n" + service_map.get("icmp/" + x, "")
                              for x in data['type_code']),
                   bps=(float(x)/bin_size for x in data['bytes']))

def gen_icmp_type_code_single(sdate, edate, dir, sensors, countries,
                              icmp_type_code):
    bin_size = get_bin_size(sdate, edate)
    data = query(portal.config.database('portdb'), """
        select (timestamp 'epoch' + interval '1 second' * %(bin_size)s *
                trunc(extract(epoch from bin_time) / %(bin_size)s)) as rb_time,
               sum(bytes) as bytes
          from icmp_type_code
          where bin_time >= %(sdate)s and bin_time < %(edate)s
            and direction = %(direction)s
            and sensor = any ( %(sensors)s )
            and country = any ( %(countries)s )
            and (icmp_type || ',' || icmp_code) = %(icmp_type_code)s
          group by rb_time
    """, {'sdate': sdate, 'edate': edate, 'bin_size': bin_size,
          'direction': dir, 'sensors': sensors, 'countries': countries,
          'icmp_type_code': icmp_type_code})
    return Dataset(bin_time=data['rb_time'],
                   bps=(float(x)/bin_size for x in data['bytes']))


def get_advanced_exprs(dirs, sensors, countries, protocols,
                       tcp_ports, udp_ports, icmp_type_codes, breakdown,
                       port_direction):
    exprs = {}
    if port_direction == 'source':
        exprs['table_expr'] = 'all_sources'
    else:
        exprs['table_expr'] = 'all_dests'
    if breakdown == "dir":
        exprs['label_expr'] = "direction"
    elif breakdown == "sensor":
        exprs['label_expr'] = "sensor"
    elif breakdown == "country":
        exprs['label_expr'] = "country"
    elif breakdown == "port":
        exprs['label_expr'] = "label"
    else: # breakdown == "proto", or any unrecognized
        exprs['label_expr'] = "cast(protocol as text)"
    if dirs:
        exprs['direction_expr'] = """
            and direction = any ( %(dirs)s )
        """ 
    else:
        exprs['direction_expr'] = ""
    if sensors:
        exprs['sensor_expr'] = """
            and sensor = any ( %(sensors)s )
        """
    else:
        exprs['sensor_expr'] = ""
    if countries:
        exprs['country_expr'] = """
            and country = any ( %(countries)s )
        """
    else:
        exprs['country_expr'] = ""
    if protocols:
        exprs['protocol_expr'] = """
            and protocol = any ( %(protocols)s )
        """
    else:
        exprs['protocol_expr'] = ""
    if tcp_ports or udp_ports or icmp_type_codes:
        null_part = []
        non_null_part = []
        if tcp_ports:
            null_part.append("tcp_port is null")
            non_null_part.append("(tcp_port = any ( %(tcp_ports)s ) is true)")
        if udp_ports:
            null_part.append("udp_port is null")
            non_null_part.append("(udp_port = any ( %(udp_ports)s ) is true)")
        if icmp_type_codes:
            null_part.append("icmp_type_code is null")
            non_null_part.append("(icmp_type_code = any ( %(icmp_type_codes)s ) is true)")
        exprs['port_expr'] = """
            and ((%(null_part)s) or %(non_null_part)s)
        """ % {'null_part': ' and '.join(null_part),
               'non_null_part': ' or '.join(non_null_part)}
    else:
        exprs['port_expr'] = ""
    return exprs

def gen_advanced_top(sdate, edate, dirs, sensors, countries, protocols,
                     tcp_ports, udp_ports, icmp_type_codes, breakdown,
                     port_direction, n):
    exprs = get_advanced_exprs(dirs, sensors, countries, protocols,
                               tcp_ports, udp_ports,
                               icmp_type_codes, breakdown, port_direction)
    sql = """
        select label, sum(bytes) as bytes from
         (select (%(label_expr)s) as label, bytes
            from %(table_expr)s
            where bin_time >= %%(sdate)s and bin_time < %%(edate)s
              %(direction_expr)s
              %(sensor_expr)s
              %(country_expr)s
              %(protocol_expr)s
              %(port_expr)s) as y
          group by label
        order by bytes desc
        limit %%(n)s
    """ % exprs
    vals = {'sdate': sdate, 'edate': edate, 'dirs': dirs,
            'sensors': sensors, 'countries': countries,
            'protocols': protocols, 'tcp_ports': tcp_ports,
            'udp_ports': udp_ports,
            'icmp_type_codes': icmp_type_codes, 'n': n}
    data = query(portal.config.database('portdb'), sql, vals)
    return data['label']

def gen_advanced(sdate, edate, dirs, sensors, countries, protocols,
                 tcp_ports, udp_ports, icmp_type_codes, breakdown,
                 port_direction, n):
    labels = gen_advanced_top(sdate, edate, dirs, sensors, countries,
                              protocols, tcp_ports, udp_ports, icmp_type_codes,
                              breakdown, port_direction, n+1)
    if len(labels) > n:
        labels = labels[:n-1]
        other_label = ['other']
    else:
        other_label = []
    bin_size = get_bin_size(sdate, edate)
    exprs = get_advanced_exprs(dirs, sensors, countries, protocols,
                               tcp_ports, udp_ports,
                               icmp_type_codes, breakdown, port_direction)
    sql = """
        select rb_time, label, sum(bytes) as bytes from
         (select (timestamp 'epoch' + interval '1 second' *
                  %%(bin_size)s * trunc(extract(epoch from bin_time) /
                  %%(bin_size)s)) as rb_time,
                 (%(label_expr)s) as label,
                  bytes
            from %(table_expr)s
            where bin_time >= %%(sdate)s and bin_time < %%(edate)s
              %(direction_expr)s
              %(sensor_expr)s
              %(country_expr)s
              %(protocol_expr)s
              %(port_expr)s) as y
          where label = any ( %%(labels)s )
          group by y.rb_time, y.label
        union all
        select rb_time, 'other' as label, sum(bytes) as bytes from
         (select (timestamp 'epoch' + interval '1 second' *
                  %%(bin_size)s * trunc(extract(epoch from bin_time) /
                  %%(bin_size)s)) as rb_time,
                 (%(label_expr)s) as label,
                  bytes
            from %(table_expr)s
            where bin_time >= %%(sdate)s and bin_time < %%(edate)s
              %(direction_expr)s
              %(sensor_expr)s
              %(country_expr)s
              %(protocol_expr)s
              %(port_expr)s) as y
            where label <> all ( %%(labels)s )
          group by y.rb_time
    """ % exprs
    vals = {'sdate': sdate, 'edate': edate, 'dirs': dirs,
            'sensors': sensors, 'countries': countries,
            'protocols': protocols, 'tcp_ports': tcp_ports,
            'udp_ports': udp_ports, 'icmp_type_codes': icmp_type_codes,
            'labels': labels, 'bin_size': bin_size}
    data = query(portal.config.database('portdb'), sql, vals)
    return (data['label'],
            data['rb_time'],
            [float(x)/bin_size for x in data['bytes']],
            labels + other_label)

########################################################################


@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_all_proto(out_file, sdate, dur, dir, sensors, countries,
                  width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    protos = gen_all_proto_top(sdate, edate, dir, sensors, countries, 11)
    protocol_map = get_protocol_map()
    labels = [str(x) + "\n" + protocol_map.get(str(x), "")
              for x in protos]
    data = gen_all_proto(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other']
    render_timeseries_ind(out_file, data['protocol'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_all_proto_drilldown(out_file, sdate, dur, dir, sensors, countries,
                            width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_proto(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        protos = gen_all_proto_top(sdate, edate, dir, sensors, countries, 11)
        include_other = False
        if len(protos) > 10:
            include_other = True
            protos = protos[:9]
        proto_map = get_protocol_map()
        labels = []
        for proto in protos:
            protocol = proto_map.get(str(proto), None)
            if protocol is None:
                labels.append("%d" % proto)
            else:
                labels.append("%d (%s)" % (proto, protocol))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-proto',
                           '_show': 'Show',
                           'protocol': p})
                for p in protos]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height,
                                  labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         protocol=int, width=int, height=int)
@mime_type('image/png')
def vis_all_proto_single(out_file, sdate, dur, dir, sensors, countries,
                         protocol, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_proto_single(sdate, edate, dir, sensors,
                                countries, protocol)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_all_sensor(out_file, sdate, dur, dir, sensors, countries,
                   width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_sensor(sdate, edate, dir, sensors, countries)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries_ind(out_file, data['sensor'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_all_sensor_drilldown(out_file, sdate, dur, dir, sensors, countries,
                             width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_sensor(sdate, edate, dir, sensors, countries)
    if len(data) > 0:
        labels = list(set(data['sensor']))
        labels.sort()
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-sensor',
                           '_show': 'Show',
                           'sensor': l})
                for l in labels]
        render_html_chart_overlay(out_file, width, height,
                                  labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         sensor=str, width=int, height=int)
@mime_type('image/png')
def vis_all_sensor_single(out_file, sdate, dur, dir, sensors, countries,
                          sensor, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_sensor_single(sdate, edate, dir, sensors,
                                 countries, sensor)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")


@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_all_region(out_file, sdate, dur, dir, sensors, countries,
                   width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_region(sdate, edate, dir, sensors, countries)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries_ind(out_file, data['region'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=region_short_names)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_all_region_drilldown(out_file, sdate, dur, dir, sensors, countries,
                             width, height):
    # Check first if vis_all_regions would produce data
    orig_sensors = sensors
    sensors = get_sensors(sensors)
    orig_countries = countries
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_region(sdate, edate, dir, sensors, countries)
    urls = ["?%s" % urlencode({
                'sdate': iso_date(sdate), 'dur': dur, 'dir': dir,
                'sensors': orig_sensors, 'countries': orig_countries,
                'display': 'drilldown-region', '_show': 'Show',
                'region': r})
            for r in region_short_names]
    if len(data) > 0:
        render_html_chart_overlay(out_file, width, height,
                                  [region_long_name_map[x]
                                   for x in region_short_names], urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         region=str, width=int, height=int)
@mime_type('image/png')
def vis_all_subregion(out_file, sdate, dur, dir, sensors, countries, region,
                      width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_subregion(sdate, edate, dir, sensors, countries, region)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries_ind(out_file, data['subregion'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=subregion_short_names[region])

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         region=str, width=int, height=int)
@mime_type('text/html')
def vis_all_subregion_drilldown(out_file, sdate, dur, dir, sensors, countries,
                                region, width, height):
    # Check first if vis_all_subregion would produce data
    orig_sensors = sensors
    sensors = get_sensors(sensors)
    orig_countries = countries
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_subregion(sdate, edate, dir, sensors, countries, region)
    urls = ["?%s" %
            urlencode({'sdate': iso_date(sdate),
                       'dur': dur, 'dir': dir,
                       'sensors': orig_sensors,
                       'countries': orig_countries,
                       'display': 'drilldown-subregion',
                       '_show': 'Show',
                       'region': region,
                       'subregion': r})
            for r in subregion_short_names[region]]
    if len(data) > 0:
        render_html_chart_overlay(out_file, width, height,
                                  [subregion_long_name_map[x]
                                   for x in subregion_short_names[region]],
                                  urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         region=str, subregion=str, width=int, height=int)
@mime_type('image/png')
def vis_all_country(out_file, sdate, dur, dir, sensors, countries,
                    region, subregion, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_country(sdate, edate, dir, sensors, countries, subregion)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries_ind(out_file, data['country'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         region=str, subregion=str, width=int, height=int)
@mime_type('text/html')
def vis_all_country_drilldown(out_file, sdate, dur, dir, sensors, countries,
                              region, subregion, width, height):
    # Check first if vis_all_country would produce data
    orig_sensors = sensors
    sensors = get_sensors(sensors)
    orig_countries = countries
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_all_country(sdate, edate, dir, sensors, countries, subregion)
    if len(data) > 0:
        shown_countries = list(set(data['country']))
        def sort_key(l):
            if isinstance(l, (str, unicode)):
                l = l.lower()
                if l == 'other' or l == 'unknown': return ()
            return l
        shown_countries.sort(key=sort_key)
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-country',
                           '_show': 'Show',
                           'region': region,
                           'subregion': subregion,
                           'country': c})
                for c in shown_countries]
        render_html_chart_overlay(out_file, width, height,
                                  [country_long_name_map[x]
                                   for x in shown_countries],
                                  urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         country=str, width=int, height=int)
@mime_type('image/png')
def vis_all_country_single(out_file, sdate, dur, dir, sensors, countries,
                           country, width, height):
    sensors = get_sensors(sensors)
    edate = add_duration(sdate, dur)
    data = gen_all_country_single(sdate, edate, dir, sensors, country)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_tcp_sport(out_file, sdate, dur, dir, sensors, countries, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    ports = gen_tcp_sport_top(sdate, edate, dir, sensors, countries, 11)
    service_map = get_service_map()
    labels = [str(x) + "\n" + service_map.get("tcp/" + str(x), "")
              for x in ports]
    data = gen_tcp_sport(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other\n']
    render_timeseries_ind(out_file, data['port'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_tcp_sport_drilldown(out_file, sdate, dur, dir, sensors, countries,
                            width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_tcp_sport(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        ports = gen_tcp_sport_top(sdate, edate, dir, sensors, countries, 11)
        include_other = False
        if len(ports) > 10:
            include_other = True
            ports = ports[:9]
        service_map = get_service_map()
        labels = []
        for port in ports:
            service = service_map.get('tcp/%d' % port, None)
            if service is None:
                labels.append("tcp/%d" % port)
            else:
                labels.append("tcp/%d (%s)" % (port, service))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-tcp-sport',
                           '_show': 'Show',
                           'port': p})
                for p in ports]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height, labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         port=int, width=int, height=int)
@mime_type('image/png')
def vis_tcp_sport_single(out_file, sdate, dur, dir, sensors, countries,
                         port, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_tcp_sport_single(sdate, edate, dir, sensors,
                                countries, port)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_tcp_dport(out_file, sdate, dur, dir, sensors, countries, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    ports = gen_tcp_dport_top(sdate, edate, dir, sensors, countries, 11)
    service_map = get_service_map()
    labels = [str(x) + "\n" + service_map.get("tcp/" + str(x), "")
              for x in ports]
    data = gen_tcp_dport(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other\n']
    render_timeseries_ind(out_file, data['port'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_tcp_dport_drilldown(out_file, sdate, dur, dir, sensors, countries,
                            width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_tcp_dport(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        ports = gen_tcp_dport_top(sdate, edate, dir, sensors, countries, 11)
        include_other = False
        if len(ports) > 10:
            include_other = True
            ports = ports[:9]
        service_map = get_service_map()
        labels = []
        for port in ports:
            service = service_map.get('tcp/%d' % port, None)
            if service is None:
                labels.append("tcp/%d" % port)
            else:
                labels.append("tcp/%d (%s)" % (port, service))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-tcp-dport',
                           '_show': 'Show',
                           'port': p})
                for p in ports]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height, labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         port=int, width=int, height=int)
@mime_type('image/png')
def vis_tcp_dport_single(out_file, sdate, dur, dir, sensors, countries,
                         port, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_tcp_dport_single(sdate, edate, dir, sensors,
                                countries, port)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_udp_sport(out_file, sdate, dur, dir, sensors, countries, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    ports = gen_udp_sport_top(sdate, edate, dir, sensors, countries, 11)
    service_map = get_service_map()
    labels = [str(x) + "\n" + service_map.get("udp/" + str(x), "")
              for x in ports]
    data = gen_udp_sport(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other\n']
    render_timeseries_ind(out_file, data['port'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_udp_sport_drilldown(out_file, sdate, dur, dir, sensors, countries,
                            width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_udp_sport(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        ports = gen_udp_sport_top(sdate, edate, dir, sensors, countries, 11)
        include_other = False
        if len(ports) > 10:
            include_other = True
            ports = ports[:9]
        service_map = get_service_map()
        labels = []
        for port in ports:
            service = service_map.get('udp/%d' % port, None)
            if service is None:
                labels.append("udp/%d" % port)
            else:
                labels.append("udp/%d (%s)" % (port, service))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-udp-sport',
                           '_show': 'Show',
                           'port': p})
                for p in ports]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height, labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         port=int, width=int, height=int)
@mime_type('image/png')
def vis_udp_sport_single(out_file, sdate, dur, dir, sensors, countries,
                         port, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_udp_sport_single(sdate, edate, dir, sensors,
                                countries, port)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_udp_dport(out_file, sdate, dur, dir, sensors, countries, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    ports = gen_udp_dport_top(sdate, edate, dir, sensors, countries, 11)
    service_map = get_service_map()
    labels = [str(x) + "\n" + service_map.get("udp/" + str(x), "")
              for x in ports]
    data = gen_udp_dport(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other\n']
    render_timeseries_ind(out_file, data['port'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_udp_dport_drilldown(out_file, sdate, dur, dir, sensors, countries,
                            width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_udp_dport(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        ports = gen_udp_dport_top(sdate, edate, dir, sensors, countries, 11)
        include_other = False
        if len(ports) > 10:
            include_other = True
            ports = ports[:9]
        service_map = get_service_map()
        labels = []
        for port in ports:
            service = service_map.get('udp/%d' % port, None)
            if service is None:
                labels.append("udp/%d" % port)
            else:
                labels.append("udp/%d (%s)" % (port, service))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-udp-dport',
                           '_show': 'Show',
                           'port': p})
                for p in ports]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height, labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         port=int, width=int, height=int)
@mime_type('image/png')
def vis_udp_dport_single(out_file, sdate, dur, dir, sensors, countries,
                         port, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_udp_dport_single(sdate, edate, dir, sensors,
                                countries, port)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('image/png')
def vis_icmp_type_code(out_file, sdate, dur, dir, sensors, countries,
                       width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    tycodes = gen_icmp_type_code_top(sdate, edate, dir, sensors, countries, 11)
    service_map = get_service_map()
    labels = [str(x) + "\n" + service_map.get("icmp/" + str(x), "")
              for x in tycodes]
    data = gen_icmp_type_code(sdate, edate, dir, sensors, countries, 10)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 10:
        labels = labels[:9] + ['other\n']
    render_timeseries_ind(out_file, data['type_code'], data['bin_time'],
                          data['bps'], width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s",
                          labels=labels)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         width=int, height=int)
@mime_type('text/html')
def vis_icmp_type_code_drilldown(out_file, sdate, dur, dir, sensors, countries,
                                 width, height):
    orig_sensors = sensors
    orig_countries = countries
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_icmp_type_code(sdate, edate, dir, sensors, countries, 10)
    if len(data) > 0:
        tycodes = gen_icmp_type_code_top(sdate, edate, dir, sensors,
                                         countries, 11)
        include_other = False
        if len(tycodes) > 10:
            include_other = True
            tycodes = tycodes[:9]
        service_map = get_service_map()
        labels = []
        for tycode in tycodes:
            service = service_map.get('icmp/%s' % tycode, None)
            if service is None:
                labels.append("icmp/%s" % tycode)
            else:
                labels.append("icmp/%s (%s)" % (tycode, service))
        urls = ["?%s" %
                urlencode({'sdate': iso_date(sdate),
                           'dur': dur, 'dir': dir,
                           'sensors': orig_sensors,
                           'countries': orig_countries,
                           'display': 'drilldown-icmp-type-code',
                           '_show': 'Show',
                           'type_code': t})
                for t in tycodes]
        if include_other:
            urls.append(None)
            labels.append("other")
        render_html_chart_overlay(out_file, width, height, labels, urls)
    else:
        f = open(out_file, 'w')
        f.close()

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         type_code=str, width=int, height=int)
@mime_type('image/png')
def vis_icmp_type_code_single(out_file, sdate, dur, dir, sensors, countries,
                              type_code, width, height):
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_icmp_type_code_single(sdate, edate, dir, sensors,
                                     countries, type_code)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    render_timeseries(out_file, data['bin_time'], data['bps'], width, height,
                      sdate, edate, by=bin_size, average_by=avg_size,
                      y_units="B/s")

########################################################################

def format_port(protocol, port, proto_map=None):
    if proto_map is None:
        proto_map = get_protocol_map()
    protocol = proto_map.get(str(protocol), str(protocol))
    if port is not None:
        return "%s/%s" % (protocol, port)
    return protocol
        
def gen_top_n(sdate, edate, dir, sensors, countries, sort_by, protocol, n=20):
    if sort_by not in ['bytes', 'packets', 'flows']:
        sort_by = 'bytes'
    if protocol not in ['all', 'tcp', 'udp', 'icmp']:
        protocol = 'all'
    if protocol == 'all':
        protocol_clause = ''
    else:
        protocol_clause = 'and protocol = %(protocol)s'
        protocol = {'tcp': 6, 'udp': 17, 'icmp': 1}[protocol]
    data = query(portal.config.database('portdb'), """
      select label,
             sum(flows) as flows,
             sum(flows) /
              (select sum(flows) from all_dests
                 where bin_time >= %%(sdate)s
                   and bin_time < %%(edate)s
                   and direction = %%(direction)s
                   and sensor = any ( %%(sensors)s )
                   and country = any ( %%(countries)s )
                   %(protocol_clause)s ) as pct_flows,
             sum(packets) as packets,
             sum(packets) /
              (select sum(packets) from all_dests
                 where bin_time >= %%(sdate)s
                   and bin_time < %%(edate)s
                   and direction = %%(direction)s
                   and sensor = any ( %%(sensors)s )
                   and country = any ( %%(countries)s )
                   %(protocol_clause)s ) as pct_packets,
             sum(bytes) as bytes,
             sum(bytes) /
              (select sum(bytes) from all_dests
                 where bin_time >= %%(sdate)s
                   and bin_time < %%(edate)s
                   and direction = %%(direction)s
                   and sensor = any ( %%(sensors)s )
                   and country = any ( %%(countries)s )
                   %(protocol_clause)s ) as pct_bytes
        from all_dests
          where bin_time >= %%(sdate)s
            and bin_time < %%(edate)s
            and direction = %%(direction)s
            and sensor = any ( %%(sensors)s )
            and country = any ( %%(countries)s )
            %(protocol_clause)s
      group by label
      order by %(sort_by)s desc
      limit %%(n)s
    """ % {'sort_by': sort_by, 'protocol_clause': protocol_clause},
        {'sdate': sdate, 'edate': edate, 'direction': dir,
         'sensors': sensors, 'countries': countries,
         'n': n, 'protocol': protocol})
    proto_map = get_protocol_map()
    data = Dataset(proto_port=data['label'],
                   flows=data['flows'],
                   pct_flows=data['pct_flows'],
                   packets=data['packets'],
                   pct_packets=data['pct_packets'],
                   bytes=data['bytes'],
                   pct_bytes=data['pct_bytes'])
    return data

def commaize(n):
    l = list(str(int(n)))
    l.reverse()
    result = []
    while len(l) > 3:
        result.extend(l[:3])
        l = l[3:]
        result.append(',')
    result.extend(l)
    result.reverse()
    return ''.join(result)

@op_file
@typemap(sdate=datetime_obj, dur=str, dir=str, sensors=str, countries=str,
         sort_by=str, protocol=str)
@mime_type('text/html')
def vis_top_n_proto_port_table(out_file, sdate, dur, dir, sensors, countries,
                               sort_by='bytes', protocol='all'):
    if sort_by not in ['bytes', 'packets', 'flows']:
        sort_by = 'bytes'
    if protocol not in ['all', 'tcp', 'udp', 'icmp']:
        protocol = 'all'
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    edate = add_duration(sdate, dur)
    data = gen_top_n(sdate, edate, dir, sensors, countries, sort_by,
                     protocol, n=20)
    service_map = get_service_map()
    services = []
    for x in data['proto_port']:
        if x in service_map:
            services.append(service_map[x])
        else:
            services.append('')

    render_html_table(out_file,
        [{'title': 'Rank', 'align': 'right', 'data': range(1, len(data)+1)},
         {'title': 'Proto/Port', 'align': 'left', 'data': data['proto_port']},
         {'title': 'Service', 'align': 'left', 'data': services},
         {'title': 'Flows', 'align': 'right',
          'data': [commaize(x) for x in data['flows']]},
         {'title': '% Flows', 'align': 'right',
          'data': ['%0.2f%%' % (x*100) for x in data['pct_flows']]},
         {'title': 'Packets', 'align': 'right',
          'data': [commaize(x) for x in data['packets']]},
         {'title': '% Packets', 'align': 'right',
          'data': ['%0.2f%%' % (x*100) for x in data['pct_packets']]},
         {'title': 'Bytes', 'align': 'right',
          'data': [commaize(x) for x in data['bytes']]},
         {'title': '% Bytes', 'align': 'right',
          'data': ['%0.2f%%' % (x*100) for x in data['pct_bytes']]}])

@op_file
@typemap(sdate=datetime_obj, dur=str, breakdown=str, width=int, height=int,
         dirs=str, sensors=str, countries=str, protocols=str,
         tcp_ports=str, udp_ports=str, icmp_type_codes=str,
         port_direction=str, n=int)
def vis_advanced(out_file_name, sdate, dur, breakdown, width, height,
                 dirs='all', sensors='all', countries='all', protocols='all',
                 tcp_ports='all', udp_ports='all', icmp_type_codes='all',
                 port_direction='destination', n=10):
    edate = add_duration(sdate, dur)
    dirs = get_directions(dirs)
    sensors = get_sensors(sensors)
    countries = get_countries(countries)
    protocols = get_protocols(protocols)
    tcp_ports = get_tcp_ports(tcp_ports)
    udp_ports = get_udp_ports(udp_ports)
    icmp_type_codes = get_icmp_type_codes(icmp_type_codes)
    labels = gen_advanced_top(sdate, edate, dirs, sensors, countries,
                              protocols, tcp_ports, udp_ports,
                              icmp_type_codes, breakdown, port_direction, n+1)
    (label_data, time_data,
     value_data, labels) = gen_advanced(sdate, edate, dirs, sensors,
                                        countries, protocols, tcp_ports,
                                        udp_ports, icmp_type_codes, breakdown,
                                        port_direction, n)
    bin_size = get_bin_size(sdate, edate)
    avg_size = get_average_size(sdate, edate)
    if len(labels) > 1:
        render_timeseries_ind(out_file_name, label_data, time_data,
                              value_data, width, height, sdate, edate,
                              by=bin_size, average_by=avg_size,
                              y_units="B/s", labels=labels)
    else:
        render_timeseries(out_file_name, time_data, value_data, width, height,
                          sdate, edate, by=bin_size, average_by=avg_size,
                          y_units="B/s")

########################################################################

__export__ = { 'portdb/all-proto': vis_all_proto,
               'portdb/all-proto-drilldown': vis_all_proto_drilldown,
               'portdb/drilldown-proto': vis_all_proto_single,
               'portdb/all-sensor': vis_all_sensor,
               'portdb/all-sensor-drilldown': vis_all_sensor_drilldown,
               'portdb/drilldown-sensor': vis_all_sensor_single,
               'portdb/all-region': vis_all_region,
               'portdb/all-region-drilldown': vis_all_region_drilldown,
               'portdb/drilldown-region': vis_all_subregion,
               'portdb/drilldown-region-drilldown': vis_all_subregion_drilldown,
               'portdb/drilldown-subregion': vis_all_country,
               'portdb/drilldown-subregion-drilldown': vis_all_country_drilldown,
               'portdb/drilldown-country': vis_all_country_single,
               'portdb/tcp-sport': vis_tcp_sport,
               'portdb/tcp-sport-drilldown': vis_tcp_sport_drilldown,
               'portdb/drilldown-tcp-sport': vis_tcp_sport_single,
               'portdb/tcp-dport': vis_tcp_dport,
               'portdb/tcp-dport-drilldown': vis_tcp_dport_drilldown,
               'portdb/drilldown-tcp-dport': vis_tcp_dport_single,
               'portdb/udp-sport': vis_udp_sport,
               'portdb/udp-sport-drilldown': vis_udp_sport_drilldown,
               'portdb/drilldown-udp-sport': vis_udp_sport_single,
               'portdb/udp-dport': vis_udp_dport,
               'portdb/udp-dport-drilldown': vis_udp_dport_drilldown,
               'portdb/drilldown-udp-dport': vis_udp_dport_single,
               'portdb/icmp-type-code': vis_icmp_type_code,
               'portdb/icmp-type-code-drilldown': vis_icmp_type_code_drilldown,
               'portdb/drilldown-icmp-type-code': vis_icmp_type_code_single,
               'other/top-n-proto-port-table': vis_top_n_proto_port_table,
               'portdb/advanced': vis_advanced }
