begin work;

create table all_protocol (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    varchar(2) not null,
    protocol   smallint not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on all_protocol to public;

comment on table all_protocol is
    'All flows, broken down by protocol';

-- Each of these tables is used as a partitioned table, partitioned by
-- direction and hour.  The partitions are created in the following fashion:
--
-- 1) A table is created using the same physical layout:
--         create table all_protocol_in_YYYYMMDDtHH
--           ( like all_protocol including defaults including constraints );
-- 2) The direction column is set to default to the direction the table
--    contains:
--         alter table all_protocol_in_YYYYMMDDtHH
--           alter column direction set default 'in';
-- 3) The data is loaded into the partition:
--         copy all_protocol_in_YYYYMMDDtHH
--           ( sensor, country, protocol, bin_time, bytes, packets, flows )
--         from 'input_file.txt' with delimiter '|';
-- 4) Indices are created on the table:
--         create index all_protocol_in_YYYYMMDDtHH_sensor on
--           all_protocol_in_YYYYMMDDtHH ( sensor );
--         create index all_protocol_in_YYYYMMDDtHH_country on
--           all_protocol_in_YYYYMMDDtHH ( country );
--         create index all_protocol_in_YYYYMMDDtHH_protocol on
--           all_protocol_in_YYYYMMDDtHH ( protocol );
-- 5) Constraints are added on the table (NB: pseudo-code for the hour
--    after the hour the table contains):
--         alter table all_protocol_in_YYYYMMDDtHH
--           add constraint all_protocol_in_YYYYMMDDtHH_date_check
--             check ( bin_time >= 'YYYY-MM-DD HH:00:00' and
--                     bin_time < 'YYYY-MM-DD HH+01:00:00' );
--         alter table all_protocol_in_YYYYMMDDtHH
--           add constraint all_protocol_in_YYYYMMDDtHH_direction_check
--             check ( direction = 'in' );
-- 6) The table is analyzed for the query planner:
--         analyze all_protocol_in_YYYYMMDDtHH
-- 7) Finally, the table is made to inherit from the base table, which
--    manes its contents are included on queries for the base table:
--         alter table all_protocol_in_YYYYMMDDtHH
--           inherit all_protocol;
--
-- Now, this table will be queried as well when the table all_protocol
-- is queried.  In order for queries to be handled most efficiently,
-- the "constraint_exclusion" parameter on the database should be set
-- to "on", either database-wide or per-session using:
--         set constraint_exclusion = on;
-- This will cause PostgreSQL to automatically ignore the partitions
-- which do not contain data pertinent to the query.    

create table icmp_type_code (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    char(2) not null,
    icmp_type  smallint not null,
    icmp_code  smallint not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on icmp_type_code to public;

comment on table icmp_type_code is
    'ICMP flows, broken down by type and code';

create table tcp_sport (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    varchar(2) not null,
    port       integer not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on tcp_sport to public;

comment on table tcp_sport is
    'TCP flows, broken down by source port';

create table tcp_dport (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    varchar(2) not null,
    port       integer not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on tcp_dport to public;

comment on table tcp_dport is
    'TCP flows, broken down by destination port';

create table udp_sport (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    varchar(2) not null,
    port       integer not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on udp_sport to public;

comment on table udp_sport is
    'UDP flows, broken down by source port';

create table udp_dport (
    bin_time   timestamp not null,
    direction  varchar(8) not null,
    sensor     text not null,
    country    varchar(2) not null,
    port       integer not null,
    bytes      bigint not null,
    packets    bigint not null,
    flows      bigint not null
);

grant select on udp_dport to public;

comment on table udp_dport is
    'UDP flows, broken down by destination port';

-- Create a new empty schema.  The tables acting as partitions of the
-- above tables will be placed within.

create schema portdb_partitions;
grant usage on schema portdb_partitions to public;

comment on schema portdb_partitions is
    'Namespace for partitions of the port database tables';

------------------------------------------------------------------------

-- World region definitions

create table cc_subregions (
  region_short_name text not null,
  region_cc         char(2) primary key
);

grant select on cc_subregions to public;

comment on table cc_subregions is
    'Mapping from subregion names to included country codes';

create table cc_regions (
  region_short_name text not null,
  region_cc         char(2) primary key
);

grant select on cc_regions to public;

comment on table cc_regions is
    'Mapping from region names to included country codes';

insert into cc_subregions values
        ('Other', '--'),
        ('Other', 'a1'),
        ('Other', 'a2'),
        ('E Afr', 'bi'),
        ('E Afr', 'km'),
        ('E Afr', 'dj'),
        ('E Afr', 'er'),
        ('E Afr', 'et'),
        ('E Afr', 'ke'),
        ('E Afr', 'mg'),
        ('E Afr', 'mw'),
        ('E Afr', 'mu'),
        ('E Afr', 'yt'),
        ('E Afr', 'mz'),
        ('E Afr', 're'),
        ('E Afr', 'rw'),
        ('E Afr', 'sc'),
        ('E Afr', 'so'),
        ('E Afr', 'ug'),
        ('E Afr', 'tz'),
        ('E Afr', 'zm'),
        ('E Afr', 'zw'),
        ('M Afr', 'ao'),
        ('M Afr', 'cm'),
        ('M Afr', 'cf'),
        ('M Afr', 'td'),
        ('M Afr', 'cg'),
        ('M Afr', 'cd'),
        ('M Afr', 'gq'),
        ('M Afr', 'ga'),
        ('M Afr', 'st'),
        ('N Afr', 'dz'),
        ('N Afr', 'eg'),
        ('N Afr', 'ly'),
        ('N Afr', 'ma'),
        ('N Afr', 'sd'),
        ('N Afr', 'tn'),
        ('N Afr', 'eh'),
        ('S Afr', 'bw'),
        ('S Afr', 'ls'),
        ('S Afr', 'na'),
        ('S Afr', 'za'),
        ('S Afr', 'sz'),
        ('W Afr', 'bj'),
        ('W Afr', 'bf'),
        ('W Afr', 'cv'),
        ('W Afr', 'ci'),
        ('W Afr', 'gm'),
        ('W Afr', 'gh'),
        ('W Afr', 'gn'),
        ('W Afr', 'gw'),
        ('W Afr', 'lr'),
        ('W Afr', 'ml'),
        ('W Afr', 'mr'),
        ('W Afr', 'ne'),
        ('W Afr', 'ng'),
        ('W Afr', 'sh'),
        ('W Afr', 'sn'),
        ('W Afr', 'sl'),
        ('W Afr', 'tg'),
        ('Carib', 'ai'),
        ('Carib', 'ag'),
        ('Carib', 'aw'),
        ('Carib', 'bs'),
        ('Carib', 'bb'),
        ('Carib', 'vg'),
        ('Carib', 'ky'),
        ('Carib', 'cu'),
        ('Carib', 'dm'),
        ('Carib', 'do'),
        ('Carib', 'gd'),
        ('Carib', 'gp'),
        ('Carib', 'ht'),
        ('Carib', 'jm'),
        ('Carib', 'mq'),
        ('Carib', 'ms'),
        ('Carib', 'an'),
        ('Carib', 'pr'),
        ('Carib', 'kn'),
        ('Carib', 'lc'),
        ('Carib', 'vc'),
        ('Carib', 'tt'),
        ('Carib', 'tc'),
        ('Carib', 'vi'),
        ('C Am', 'bz'),
        ('C Am', 'cr'),
        ('C Am', 'sv'),
        ('C Am', 'gt'),
        ('C Am', 'hn'),
        ('C Am', 'mx'),
        ('C Am', 'ni'),
        ('C Am', 'pa'),
        ('S Am', 'ar'),
        ('S Am', 'bo'),
        ('S Am', 'br'),
        ('S Am', 'cl'),
        ('S Am', 'co'),
        ('S Am', 'ec'),
        ('S Am', 'fk'),
        ('S Am', 'gf'),
        ('S Am', 'gy'),
        ('S Am', 'py'),
        ('S Am', 'pe'),
        ('S Am', 'sr'),
        ('S Am', 'uy'),
        ('S Am', 've'),
        ('N Am', 'bm'),
        ('N Am', 'ca'),
        ('N Am', 'gl'),
        ('N Am', 'pm'),
        ('N Am', 'us'),
        ('C Asia', 'kz'),
        ('C Asia', 'kg'),
        ('C Asia', 'tj'),
        ('C Asia', 'tm'),
        ('C Asia', 'uz'),
        ('E Asia', 'cn'),
        ('E Asia', 'hk'),
        ('E Asia', 'mo'),
        ('E Asia', 'kp'),
        ('E Asia', 'jp'),
        ('E Asia', 'mn'),
        ('E Asia', 'kr'),
        ('E Asia', 'tw'),
        ('S Asia', 'af'),
        ('S Asia', 'bd'),
        ('S Asia', 'bt'),
        ('S Asia', 'in'),
        ('S Asia', 'ir'),
        ('S Asia', 'mv'),
        ('S Asia', 'np'),
        ('S Asia', 'pk'),
        ('S Asia', 'lk'),
        ('SE Asia', 'bn'),
        ('SE Asia', 'kh'),
        ('SE Asia', 'id'),
        ('SE Asia', 'la'),
        ('SE Asia', 'my'),
        ('SE Asia', 'mm'),
        ('SE Asia', 'ph'),
        ('SE Asia', 'sg'),
        ('SE Asia', 'th'),
        ('SE Asia', 'tl'),
        ('SE Asia', 'vn'),
        ('W Asia', 'am'),
        ('W Asia', 'az'),
        ('W Asia', 'bh'),
        ('W Asia', 'cy'),
        ('W Asia', 'ge'),
        ('W Asia', 'iq'),
        ('W Asia', 'il'),
        ('W Asia', 'jo'),
        ('W Asia', 'kw'),
        ('W Asia', 'lb'),
        ('W Asia', 'ps'),
        ('W Asia', 'om'),
        ('W Asia', 'qa'),
        ('W Asia', 'sa'),
        ('W Asia', 'sy'),
        ('W Asia', 'tr'),
        ('W Asia', 'ae'),
        ('W Asia', 'ye'),
        ('E Eur', 'by'),
        ('E Eur', 'bg'),
        ('E Eur', 'cz'),
        ('E Eur', 'hu'),
        ('E Eur', 'pl'),
        ('E Eur', 'md'),
        ('E Eur', 'ro'),
        ('E Eur', 'ru'),
        ('E Eur', 'sk'),
        ('E Eur', 'ua'),
        ('E Eur', 'yu'),
        ('N Eur', 'ax'),
        ('N Eur', 'dk'),
        ('N Eur', 'ee'),
        ('N Eur', 'fo'),
        ('N Eur', 'fi'),
        ('N Eur', 'gg'),
        ('N Eur', 'is'),
        ('N Eur', 'ie'),
        ('N Eur', 'im'),
        ('N Eur', 'je'),
        ('N Eur', 'lv'),
        ('N Eur', 'lt'),
        ('N Eur', 'no'),
        ('N Eur', 'sj'),
        ('N Eur', 'se'),
        ('N Eur', 'gb'),
        ('S Eur', 'al'),
        ('S Eur', 'ad'),
        ('S Eur', 'ba'),
        ('S Eur', 'hr'),
        ('S Eur', 'gi'),
        ('S Eur', 'gr'),
        ('S Eur', 'va'),
        ('S Eur', 'it'),
        ('S Eur', 'mt'),
        ('S Eur', 'me'),
        ('S Eur', 'pt'),
        ('S Eur', 'sm'),
        ('S Eur', 'rs'),
        ('S Eur', 'si'),
        ('S Eur', 'es'),
        ('S Eur', 'mk'),
        ('W Eur', 'at'),
        ('W Eur', 'be'),
        ('W Eur', 'fr'),
        ('W Eur', 'de'),
        ('W Eur', 'li'),
        ('W Eur', 'lu'),
        ('W Eur', 'mc'),
        ('W Eur', 'nl'),
        ('W Eur', 'ch'),
        ('W Eur', 'eu'),
        ('Aus/NZ', 'au'),
        ('Aus/NZ', 'nz'),
        ('Aus/NZ', 'nf'),
        ('Mela', 'fj'),
        ('Mela', 'nc'),
        ('Mela', 'pg'),
        ('Mela', 'sb'),
        ('Mela', 'vu'),
        ('Micro', 'gu'),
        ('Micro', 'ki'),
        ('Micro', 'mh'),
        ('Micro', 'fm'),
        ('Micro', 'nr'),
        ('Micro', 'mp'),
        ('Micro', 'pw'),
        ('Poly', 'as'),
        ('Poly', 'ck'),
        ('Poly', 'pf'),
        ('Poly', 'nu'),
        ('Poly', 'pn'),
        ('Poly', 'ws'),
        ('Poly', 'tk'),
        ('Poly', 'to'),
        ('Poly', 'tv'),
        ('Poly', 'wf');

insert into cc_regions
    select 'Africa' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('E Afr', 'M Afr', 'N Afr', 'S Afr',
                                    'W Afr');

insert into cc_regions
    select 'Lat Am' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('Carib', 'C Am', 'S Am');

insert into cc_regions
    select 'N Am' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('N Am');

insert into cc_regions
    select 'Asia' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('C Asia', 'E Asia', 'S Asia', 'SE Asia',
                                    'W Asia');

insert into cc_regions
    select 'Oceania' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('Aus/NZ', 'Mela', 'Micro', 'Poly');

insert into cc_regions
    select 'Europe' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('E Eur', 'N Eur', 'S Eur', 'W Eur');

insert into cc_regions
    select 'Other' as region_short_name, region_cc
      from cc_subregions c
      where c.region_short_name in ('Other');

------------------------------------------------------------------------

-- Protocol assigned numbers

create table protocol_names (
    protocol_num  smallint not null primary key,
    protocol_name text     not null unique
);

grant select on protocol_names to public;

comment on table protocol_names is
    'IANA-assigned names for protocol numbers, used by views';

insert into protocol_names values
    (0, 'hopopt'),
    (1, 'icmp'),
    (2, 'igmp'),
    (3, 'ggp'),
    (4, 'ip'),
    (5, 'st'),
    (6, 'tcp'),
    (7, 'cbt'),
    (8, 'egp'),
    (9, 'igp'),
    (10, 'bbn-rcc-mon'),
    (11, 'nvp-ii'),
    (12, 'pup'),
    (13, 'argus'),
    (14, 'emcon'),
    (15, 'xnet'),
    (16, 'chaos'),
    (17, 'udp'),
    (18, 'mux'),
    (19, 'dcn-meas'),
    (20, 'hmp'),
    (21, 'prm'),
    (22, 'xns-idp'),
    (23, 'trunk-1'),
    (24, 'trunk-2'),
    (25, 'leaf-1'),
    (26, 'lead-2'),
    (27, 'rdp'),
    (28, 'irtp'),
    (29, 'iso-tp4'),
    (30, 'netblt'),
    (31, 'mfe-nsp'),
    (32, 'merit-inp'),
    (33, 'dccp'),
    (34, '3pc'),
    (35, 'idpr'),
    (36, 'xtp'),
    (37, 'ddp'),
    (38, 'idpr-cmtp'),
    (39, 'tp++'),
    (40, 'il'),
    (41, 'ipv6'),
    (42, 'sdrp'),
    (43, 'ipv6-route'),
    (44, 'ipv6-frag'),
    (45, 'idrp'),
    (46, 'rsvp'),
    (47, 'gre'),
    (48, 'dsr'),
    (49, 'bna'),
    (50, 'esp'),
    (51, 'ah'),
    (52, 'i-nlsp'),
    (53, 'swipe'),
    (54, 'narp'),
    (55, 'mobile'),
    (56, 'tlsp'),
    (57, 'skip'),
    (58, 'ipv6-icmp'),
    (59, 'ipv6-nonxt'),
    (60, 'ipv6-opts'),
    (62, 'cftp'),
    (64, 'sat-expak'),
    (65, 'kryptolan'),
    (66, 'rvd'),
    (67, 'ippc'),
    (69, 'sat-mon'),
    (70, 'visa'),
    (71, 'ipcv'),
    (72, 'cpnx'),
    (73, 'cphb'),
    (74, 'wsn'),
    (75, 'pvp'),
    (76, 'br-sat-mon'),
    (77, 'sun-nd'),
    (78, 'wb-mon'),
    (79, 'wb-expak'),
    (80, 'iso-ip'),
    (81, 'vmtp'),
    (82, 'secure-vmtp'),
    (83, 'vines'),
    (84, 'ttp'),
    (85, 'nsfnet-igp'),
    (86, 'dgp'),
    (87, 'tcf'),
    (88, 'eigrp'),
    (89, 'ospfigp'),
    (90, 'sprite-rpc'),
    (91, 'larp'),
    (92, 'mtp'),
    (93, 'ax.25'),
    (94, 'ipip'),
    (95, 'micp'),
    (96, 'scc-sp'),
    (97, 'etherip'),
    (98, 'encap'),
    (100, 'gmtp'),
    (101, 'ifmp'),
    (102, 'pnni'),
    (103, 'pim'),
    (104, 'aris'),
    (105, 'scps'),
    (106, 'qnx'),
    (107, 'a/n'),
    (108, 'ipcomp'),
    (109, 'snp'),
    (110, 'compaq-peer'),
    (111, 'ipx-in-ip'),
    (112, 'vrrp'),
    (113, 'pgm'),
    (115, 'l2tp'),
    (116, 'ddx'),
    (117, 'iatp'),
    (118, 'stp'),
    (119, 'srp'),
    (120, 'uti'),
    (121, 'smp'),
    (122, 'sm'),
    (123, 'ptp'),
    (124, 'isis'),
    (125, 'fire'),
    (126, 'crtp'),
    (127, 'crudp'),
    (128, 'sscopmce'),
    (129, 'iplt'),
    (130, 'sps'),
    (131, 'pipe'),
    (132, 'sctp'),
    (133, 'fc'),
    (134, 'rsvp-e2e-ignore'),
    (135, 'mobility-header'),
    (136, 'udplite'),
    (137, 'mpls-in-ip');    

------------------------------------------------------------------------

-- Views for convenience on complicated multi-protocol lookups

create view all_sources as
  select bin_time, direction, sensor, country, protocol,
         coalesce((select protocol_name from protocol_names
                     where protocol_num = protocol),
                  cast(protocol as text)) as label,
         cast(null as integer) as tcp_port,
         cast(null as integer) as udp_port,
         cast(null as smallint) as icmp_type,
         cast(null as smallint) as icmp_code,
         flows, packets, bytes
    from all_protocol
    where protocol not in (1, 6, 17)
  union all
  select bin_time, direction, sensor, country, 6 as protocol,
         ('tcp/' || port) as label,
         port as tcp_port,
         null as udp_port,
         null as icmp_type,
         null as icmp_code,
         flows, packets, bytes
    from tcp_sport
  union all
  select bin_time, direction, sensor, country, 17 as protocol,
         ('udp/' || port) as label,
         null as tcp_port,
         port as udp_port,
         null as icmp_type,
         null as icmp_code,
         flows, packets, bytes
    from udp_sport
  union all
  select bin_time, direction, sensor, country, 1 as protocol,
         ('icmp/' || icmp_type || ',' || icmp_code) as label,
         null as tcp_port,
         null as udp_port,
         icmp_type,
         icmp_code,
         flows, packets, bytes
    from icmp_type_code;

comment on view all_sources is
    'All flows broken down by (tcp/sport, udp/sport, icmp/tc, protocol)';

create view all_dests as
  select bin_time, direction, sensor, country, protocol,
         (select protocol_name from protocol_names
             where protocol_num = protocol) as label,
         cast(null as integer) as tcp_port,
         cast(null as integer) as udp_port,
         cast(null as text) as icmp_type_code,
         cast(null as smallint) as icmp_type,
         cast(null as smallint) as icmp_code,
         flows, packets, bytes
    from all_protocol
    where protocol not in (1, 6, 17)
  union all
  select bin_time, direction, sensor, country, 6 as protocol,
         ('tcp/' || port) as label,
         port as tcp_port,
         null as udp_port,
         null as icmp_type_code,
         null as icmp_type,
         null as icmp_code,
         flows, packets, bytes
    from tcp_dport
  union all
  select bin_time, direction, sensor, country, 17 as protocol,
         ('udp/' || port) as label,
         null as tcp_port,
         port as udp_port,
         null as icmp_type_code,
         null as icmp_type,
         null as icmp_code,
         flows, packets, bytes
    from udp_dport
  union all
  select bin_time, direction, sensor, country, 1 as protocol,
         ('icmp/' || icmp_type || ',' || icmp_code) as label,
         null as tcp_port,
         null as udp_port,
         (icmp_type || '-' || icmp_code) as icmp_type_code,
         icmp_type,
         icmp_code,
         flows, packets, bytes
    from icmp_type_code;

comment on view all_dests is
    'All flows broken down by (tcp/dport, udp/dport, icmp/tc, protocol)';

------------------------------------------------------------------------

commit work;
