# TODO:
# * Supply expiration strategies

# sensors is:
# * A comma-separated list of sensor names, or 'all'
#   ('all' == all sensors')




import sys
import copy

import urllib
from datetime import datetime, timedelta
import subprocess

from rave.plugins.decorators import *
from rave.plugins.render_mpl import *
from rave.plugins.render_html import *
from rave.plugins.render_pil import (render_timeseries_primitive,
                                     render_histogram_primitive,
                                     MetricFormatter,)
from rave.plugins.times import *

from silk import *

from portal.util import *
import portal.config

set_font_filename(portal.config.font('Vera'))

load_scheme=1
max_flow_duration=350

# For silk: Cache summarized data forever
# Don't do this yet: Have to handle recent data cases.
#set_summary_merged_cache_dur(None)
#set_summary_cache_dur(None)

## Utilities ###########################################################

# Used by both darknet and susp-hosts to look for traffic between two
# sets of hosts.

def gen_flows_from_ipsets(sdate, edate, sensors, sipset, dipset):
#   Subtract one second from end date
    edate = datetime_obj(edate) - timedelta(seconds=1)
    flows_by_sensor = []
    for s in sensors:
        data = rwfilter(type="all", sensors=s,
                     start_date=silk_hour(sdate),
                     end_date=silk_hour(edate),
                     sipset=sipset, dipset=dipset,
                     rwfilter_output=RWF_PASS)
        data._proc_get_parts()
        flows_by_sensor.append(data)
    return rwcat(*flows_by_sensor)


# Spreads a pairwise histogram (bin numbers in one series, bin values in the
# other series) into a single list with one value for every bin.

def normalize_histogram(bins, values, max_bins):
    sparse_values = [0] * max_bins
    for i in xrange(len(bins)):
        if bins[i] < max_bins:
            sparse_values[int(bins[i])] = values[i]
    return range(len(sparse_values)), sparse_values


# Takes an irregular time-series and fills out a regular time-series

def regularize(d, date_col, count_col, sdate, edate, bin_size=60):
#   Assume d is a dataset containing a sorted, possibly incomplete
#   time-series of binned counts, and fill absent bins in the time
#   series with zeroes. sdate, edate and bin_size should generate
#   bins equivalent to those in d.
    sdate = datetime_obj(sdate)
    edate   = datetime_obj(edate)
    bin_size = timedelta(seconds=bin_size)

    d = d.sort_col(date_col)

    dates = []
    values = []
    curr = copy.copy(sdate)
    for row in d:
        while curr < row[date_col]:
            dates.append(curr)
            values.append(0)
            curr = curr + bin_size
        if curr == datetime_obj(row[date_col]):
            dates.append(curr)
            values.append(row[count_col])
            curr = curr + bin_size
        else:
            raise RuntimeError("Data %s doesn't match time series parameters "
                               "(%s)" % (`row[date_col]`, `curr`))
    while curr <= edate:
        dates.append(curr)
        values.append(0)
        curr = curr + bin_size

    return Dataset(**{date_col: dates, count_col: values})

# Add commas to a number when converting to string representation
def commaize(n):
    l = list(str(int(n)))
    l.reverse()
    result = []
    while len(l) > 3:
        result.extend(l[:3])
        l = l[3:]
        result.append(',')
    result.extend(l)
    result.reverse()
    return ''.join(result)

  
## Data Generation #####################################################


# Suspicious hosts

@op
@version('20070618_01')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_susp_hosts_timeseries_sensor(sdate, edate, sensor, ip):
    bin_size = get_bin_size(sdate, edate)
    flows = gen_flows_from_ipsets(
        sdate, edate, [sensor], 
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/blacklist.set'))
    flows = rwfilter(flows, saddress=ip)
    data = rwcount(flows, load_scheme=load_scheme, bin_size=bin_size,
                   skip_zeroes=True)
    if len(data) == 0:
        return []
    else:
        return zip(data['time'], data['packets'])

def gen_susp_hosts_timeseries(sdate, edate, sensors, ip):
    bin_size = get_bin_size(sdate, edate)
    data = {}
#   Aggregate each sensor's numbers into data
    for s in sensors:
        new_data = _gen_susp_hosts_timeseries_sensor(sdate, edate, s, ip)
        for k,v in new_data:
            data[k] = data.get(k, 0) + v
#   Convert data into a Dataset, for regularize
    times = []
    packets = []
    for k, v in data.items():
        times.append(k)
        packets.append(v)
    data = Dataset(time=times, packets=packets)
    return regularize(data, 'time', 'packets', sdate, edate, bin_size)
    

@op
@version('20070618_01')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_susp_hosts_histogram_sensor(sdate, edate, sensor, ip):
    flows = gen_flows_from_ipsets(
        sdate, edate, [sensor],
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/blacklist.set'))
    flows = rwfilter(flows, saddress=ip)
    data = rwtotal(flows, duration=True, skip_zeroes=True)
    if len(data) == 0:
        data = Dataset(dur=[], flows=[])
    return normalize_histogram(data['dur'], data['flows'],
                               max_flow_duration)


def gen_susp_hosts_histogram(sdate, edate, sensors, ip):
    bins = values = None
    for sensor in sensors:
        s_bins, s_values = _gen_susp_hosts_histogram_sensor(sdate, edate, sensor, ip)
        if bins is None:
            bins = s_bins
        if values is None:
            values = s_values
        else:
            for i in xrange(len(values)):
                values[i] += s_values[i]
    return bins, values
        
 
def _gen_susp_hosts_list_sensor(sdate, edate, sensor):
    flows = gen_flows_from_ipsets(
        sdate, edate, sensor,
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/blacklist.set'))
    return rwbag(flows, rwbag_output=RWB_SIP_PACKETS)
    

def gen_susp_hosts_list(sdate, edate, sensors, threshold):
    bags = []
    for s in sensors:
        bags.append(_gen_susp_hosts_list_sensor(sdate, edate, [s]))
    host_counts = rwbagtool(rwbagtool_output=RWB_ADD, *bags)
    data = rwbagcat(host_counts, mincount=threshold)
    return data.sort_col('count', key=(lambda i: int(i)), reverse=True)

# Suspicious ports


@op
@version('20070618_01')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_susp_ports_timeseries_sensor(sdate, edate, sensor, ip):
    bin_size = get_bin_size(sdate, edate)
    flows = gen_susp_ports_flows(sdate, edate, [sensor]) 
    flows = rwfilter(flows, saddress=ip)
    data = rwcount(flows, load_scheme=load_scheme, bin_size=bin_size,
                   skip_zeroes=True)
    if len(data) == 0:
        return []
    else:
        return zip(data['time'], data['packets'])


def gen_susp_ports_timeseries(sdate, edate, sensors, ip):
    bin_size = get_bin_size(sdate, edate)
    data = {}
#   Aggregate each sensor's numbers into data
    for s in sensors:
        new_data = _gen_susp_ports_timeseries_sensor(sdate, edate, s, ip)
        for k,v in new_data:
            data[k] = data.get(k, 0) + v
#   Convert data into a Dataset, for regularize
    times = []
    packets = []
    for k, v in data.items():
        times.append(k)
        packets.append(v)
    data = Dataset(time=times, packets=packets)
    return regularize(data, 'time', 'packets', sdate, edate, bin_size)
    

@op
@version('20070618_01')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_susp_ports_histogram_sensor(sdate, edate, sensor, ip):
    flows = gen_susp_ports_flows(sdate, edate, [sensor]) 
    flows = rwfilter(flows, saddress=ip)
    data = rwtotal(flows, duration=True, skip_zeroes=True)
    if len(data) == 0:
        data = Dataset(dur=[], flows=[])
    return normalize_histogram(data['dur'], data['flows'],
                               max_flow_duration)


def gen_susp_ports_histogram(sdate, edate, sensors, ip):
    bins   = None
    values = None
    for sensor in sensors:
        s_bins, s_values = _gen_susp_ports_histogram_sensor(sdate, edate, sensor, ip)
        if bins is None:
            bins = s_bins
        if values is None:
            values = s_values
        else:
            for i in xrange(len(values)):
                values[i] += s_values[i]
    return bins, values
        


def _gen_susp_ports_list_sensor(sdate, edate, sensor):
    flows = gen_susp_ports_flows(sdate, edate, [sensor])
    return rwbag(flows, rwbag_output=RWB_SIP_PACKETS)
    

def gen_susp_ports_list(sdate, edate, sensors, threshold):
    bags = []
    for s in sensors:
        bags.append(_gen_susp_ports_list_sensor(sdate, edate, [s]))
    host_counts = rwbagtool(rwbagtool_output=RWB_ADD, *bags)
    data = rwbagcat(host_counts, mincount=threshold)
    return data.sort_col('count', key=(lambda i: int(i)), reverse=True)


# Darknet

@op
@version('20070618_02')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_darknet_timeseries_sensor(sdate, edate, sensor, ip):
    bin_size = get_bin_size(sdate, edate)
    flows = gen_flows_from_ipsets(
        sdate, edate, [sensor], 
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/darkspace.set'))
    flows = rwfilter(flows, saddress=ip)
    data = rwcount(flows, load_scheme=load_scheme, bin_size=bin_size,
                   skip_zeroes=True)
    if len(data) == 0:
        return []
    else:
        return zip(data['time'], data['packets'])

def gen_darknet_timeseries(sdate, edate, sensors, ip):
    bin_size = get_bin_size(sdate, edate)
    data = {}
#   Aggregate each sensor's numbers into data
    for s in sensors:
        new_data = _gen_darknet_timeseries_sensor(sdate, edate, s, ip)
        for k,v in new_data:
            data[k] = data.get(k, 0) + v
#   Convert data into a Dataset, for regularize
    times = []
    packets = []
    for k, v in data.items():
        times.append(k)
        packets.append(v)
    data = Dataset(time=times, packets=packets)
    return regularize(data, 'time', 'packets', sdate, edate, bin_size)


@op
@version('20070618_01')
@expires(duration_strategy("30d")) # TODO: real strategy
def _gen_darknet_histogram_sensor(sdate, edate, sensor, ip):
    flows = gen_flows_from_ipsets(
        sdate, edate, [sensor],
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/darkspace.set'))
    flows = rwfilter(flows, saddress=ip)
    data = rwtotal(flows, duration=True, skip_zeroes=True)
    if len(data) == 0:
        data = Dataset(dur=[], flows=[])
    return normalize_histogram(data['dur'], data['flows'],
                               max_flow_duration)


def gen_darknet_histogram(sdate, edate, sensors, ip):
    bins = values = None
    for sensor in sensors:
        s_bins, s_values = _gen_darknet_histogram_sensor(sdate, edate, sensor, ip)
        if bins is None:
            bins = s_bins
        if values is None:
            values = s_values
        else:
            for i in xrange(len(values)):
                values[i] += s_values[i]
    return bins, values


def _gen_darknet_list_sensor(sdate, edate, sensor):
    flows = gen_flows_from_ipsets(
        sdate, edate, [sensor],
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/darkspace.set'))
    return rwbag(flows, rwbag_output=RWB_SIP_PACKETS)
    

def gen_darknet_list(sdate, edate, sensors, threshold):
    bags = []
    for s in sensors:
        bags.append(_gen_darknet_list_sensor(sdate, edate, [s]))
    host_counts = rwbagtool(rwbagtool_output=RWB_ADD, *bags)
    data = rwbagcat(host_counts, mincount=threshold)
    return data.sort_col('count', key=(lambda i: int(i)), reverse=True)


## Renderers ###########################################################


def render_tx_timeseries(out_file_name, flows, sdate, edate,
                         width, height, bgcolor):
    bin_size = get_bin_size(sdate, edate)
    average_size = get_average_size(sdate, edate)
    data = rwuniq(flows, fields='stime', packets=True, bin_time=60)
    if len(data) == 0:
        data = Dataset(stime=[], packets=[])
    data = Dataset(stime=data['stime'],
                   pps=(float(x)/bin_size for x in data['packets']))
    
    render_timeseries(out_file_name, data['stime'], data['pps'],
                      width, height, sdate, edate,
                      by=bin_size, average_by=average_size,
                      ylab='traffic (pkt/s)',
                      bgcolor=bgcolor)

def render_watchlist_html(out_file_name, sdate, dur, sensors, data,
                          timeseries_url, histogram_url,
                          stylesheet=None):
    spark_width = portal.config.rendering_option('sparklines.width')
    spark_height = portal.config.rendering_option('sparklines.height')
    def graphic_html(ip, url, width):
        template = ("<span class='chart-tiny' "
                    "style='width: %(width)spx; height: %(height)spx'>"
                    "<img src='%(base_url)s/%(url)s?%(url_params)s' "
                    "style='visibility: hidden; "
                    "width: %(width)spx; height: %(height)spx' "
                    "onload='this.style.visibility = \"visible\";' "
                    "alt='' />"
                    "</span>")
        return (template %
                {'base_url': portal.config.rave_option('base_url'),
                 'url': url,
                 'url_params': urllib.urlencode({
                        'sdate': sdate, 'dur': dur,
                        'sensors': sensors, 'ip': ip,
                        'width': width, 'height': spark_height}),
                 'width': width,
                 'height': spark_height})
    render_html_table(out_file_name, [
            {'title': 'Address', 'align': 'left', 'data': data['key']},
            {'title': 'Packets', 'data': [commaize(x) for x in data['count']]},
            {'title': 'Traffic (Packets/Second)',
             'data' : [graphic_html(ip, timeseries_url, int(spark_width) + 80)
                       for ip in data['key']]}, 
            {'title': 'Connection duration (seconds)',
             'data' : [graphic_html(ip, histogram_url, spark_width)
                       for ip in data['key']]}, 
            ], stylesheet)


## Darknet #############################################################


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str, width=int, height=int)
def vis_darknet_timeseries(out_file_name, sdate, dur, sensors, width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    
    flows = gen_flows_from_ipsets(
        sdate, edate, sensors,
        portal.config.module_file('watchlists', 'etc/internal.set'),
        portal.config.module_file('watchlists', 'etc/darkspace.set'))
    render_tx_timeseries(out_file_name, flows, sdate, edate, width, height,
                         bgcolor=portal.config.rendering_option('bgcolor'))


@op_file
@mime_type('text/html')
@typemap(sdate=datetime_obj, dur=str, sensors=str, threshold=int,
         stylesheet=one_of(None, str))
def vis_darknet_table(out_file_name, sdate, dur, sensors,
                      threshold=1, stylesheet=None):
    edate = add_duration(sdate, dur)
    sensor_spec = sensors
    sensors = get_sensors(sensors)
    data = gen_darknet_list(sdate, edate, sensors, threshold)
    data = data[:20]
#   Warning: Don't try to recreate the sensor spec from the sensor list!
#   You may accidentally introduce sensors into the visualization that
#   the user may not be allowed to see. BEWARE! BEWARE!
    render_watchlist_html(
        out_file_name, sdate, dur, sensor_spec, data,
        "vis/watchlists/darknet-timeseries-spark",
        "vis/watchlists/darknet-histogram-spark",
        stylesheet)


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str, ip=str,
         width=int, height=int)
def vis_darknet_timeseries_spark(out_file_name, sdate, dur,sensors, ip,
                                 width, height):
    edate = add_duration(sdate, dur)
    sensor_spec = sensors
    sensors = get_sensors(sensors)
    data = gen_darknet_timeseries(sdate, edate, sensors, ip)

    def get_formatter(color):
        return MetricFormatter(portal.config.font("Vera"), 
                               10, color, "%(num).1f", "%(short)s")

    bin_size = get_bin_size(sdate, edate)
    pps=[float(x)/bin_size for x in data['packets']]
    render_timeseries_primitive(
        out_file_name, pps, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'),
        max_color=portal.config.rendering_option('sparklines.maxcolor'),
        min_color=portal.config.rendering_option('sparklines.mincolor'),
        max_fmt=get_formatter(
            portal.config.rendering_option('sparklines.maxcolor')),
        min_fmt=get_formatter(
            portal.config.rendering_option('sparklines.mincolor')),)







@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str,
         ip=str, width=int, height=int)
def vis_darknet_histogram_spark(out_file_name, sdate, dur, sensors,
                                ip, width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    bins, values = gen_darknet_histogram(sdate, edate, sensors, ip)
    render_histogram_primitive(
        out_file_name, values, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'))




## Suspicious Ports ####################################################


susp_ports = [ 7, 9, 11, 13, 17, 19, 23, 67, 68, (109, 111), 135, (137, 139),
               143, 162, 177, 220, 411, 412, 445, (512, 515), 631, 1099, 1214,
               1433, 1434, 1521, 1801, 2049, 3283, 3306, 3389, 3527, 4662,
               4899, 5432, 5737, 5800, 5900, 5950, (6000, 6007), 6257,
               (6346, 6348), 6699, (6881, 6885), (9100, 9103), 10080, 13720,
               13721, 13724, 13782, 13783 ]


def gen_susp_ports_flows(sdate, edate, sensors):
#   Subtract one second from end date
    edate = datetime_obj(edate) - timedelta(seconds=1)
    flows_by_sensor = []
    internal_set = portal.config.module_file('watchlists', 'etc/internal.set')
    for s in sensors:
        flows_by_sensor.append(
            rwfilter(type="all", sensors=s,
                    start_date=silk_hour(sdate),
                    end_date=silk_hour(edate),
                    sipset=internal_set,
                    not_dipset=internal_set,
                    dport=susp_ports,
                    not_daddress="192.168.x.x",
                    rwfilter_output=RWF_PASS)
        )
    return rwcat(*flows_by_sensor)



@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str, width=int, height=int)
def vis_susp_ports_timeseries(out_file_name, sdate, dur,
                              sensors, width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    flows = gen_susp_ports_flows(sdate, edate, sensors)
    render_tx_timeseries(out_file_name, flows, sdate, edate, width, height,
                         bgcolor=portal.config.rendering_option('bgcolor'))



@op_file
@mime_type('text/html')
@typemap(sdate=datetime_obj, dur=str, sensors=str, threshold=int,
         stylesheet=one_of(None, str))
def vis_susp_ports_table(out_file_name, sdate, dur, sensors=str, threshold=1,
                         stylesheet=None):
    edate = add_duration(sdate, dur)
    sensor_spec = sensors
    sensors = get_sensors(sensors)
    data = gen_susp_ports_list(sdate, edate, sensors, threshold)
    data = data[:20]
#   Warning: Don't try to recreate the sensor spec from the sensor list!
#   You may accidentally introduce sensors into the visualization that
#   the user may not be allowed to see. BEWARE! BEWARE!
    render_watchlist_html(
        out_file_name, sdate, dur, sensor_spec, data,
        "vis/watchlists/susp-ports-timeseries-spark",
        "vis/watchlists/susp-ports-histogram-spark",
        stylesheet)


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str,
         ip=str, width=int, height=int)
def vis_susp_ports_timeseries_spark(out_file_name, sdate, dur, sensors, ip,
                                    width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    data = gen_susp_ports_timeseries(sdate, edate, sensors, ip)

    def get_formatter(color):
        return MetricFormatter(portal.config.font("Vera"), 
                               10, color, "%(num).1f", "%(short)s")

    bin_size = get_bin_size(sdate, edate)
    pps=[float(x)/bin_size for x in data['packets']]
    render_timeseries_primitive(
        out_file_name, pps, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'),
        max_color=portal.config.rendering_option('sparklines.maxcolor'),
        min_color=portal.config.rendering_option('sparklines.mincolor'),
        max_fmt=get_formatter(
            portal.config.rendering_option('sparklines.maxcolor')),
        min_fmt=get_formatter(
            portal.config.rendering_option('sparklines.mincolor')),)


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str,
         ip=str, width=int, height=int)
def vis_susp_ports_histogram_spark(out_file_name, sdate, dur, sensors, ip,
                                   width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    bins, values = gen_susp_ports_histogram(sdate, edate, sensors, ip)
    render_histogram_primitive(
        out_file_name, values, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'))


## Suspicious Hosts (Corruption) #######################################


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str, width=int, height=int)
def vis_susp_hosts_timeseries(out_file_name, sdate, dur,
                              sensors, width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    internal_set = portal.config.module_file('watchlists', 'etc/internal.set')
    blacklist_set = portal.config.module_file('watchlists', 'etc/blacklist.set')
    flows = gen_flows_from_ipsets(sdate, edate, sensors,
                                  internal_set, blacklist_set)
    render_tx_timeseries(out_file_name, flows, sdate, edate, width, height,
                         bgcolor=portal.config.rendering_option('bgcolor'))


@op_file
@mime_type('text/html')
@typemap(sdate=datetime_obj, dur=str, sensors=str, threshold=int,
         stylesheet=one_of(None, str))
def vis_susp_hosts_table(out_file_name, sdate, dur, sensors, threshold=1,
                         stylesheet=None):
    edate = add_duration(sdate, dur)
    sensor_spec = sensors
    sensors = get_sensors(sensors)
    data = gen_susp_hosts_list(sdate, edate, sensors, threshold)
    data = data[:20]
#   Warning: Don't try to recreate the sensor spec from the sensor list!
#   You may accidentally introduce sensors into the visualization that
#   the user may not be allowed to see. BEWARE! BEWARE!
    render_watchlist_html(
        out_file_name, sdate, dur, sensor_spec, data,
        "vis/watchlists/susp-hosts-timeseries-spark",
        "vis/watchlists/susp-hosts-histogram-spark",
        stylesheet)


@op_file
@mime_type('image/png')
@typemap(sdate=datetime_obj, dur=str, sensors=str, ip=str, 
         width=int, height=int)
def vis_susp_hosts_timeseries_spark(out_file_name, sdate, dur, sensors, ip,
                                    width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    data = gen_susp_hosts_timeseries(sdate, edate, sensors, ip)

    def get_formatter(color):
        return MetricFormatter(portal.config.font("Vera"), 
                               10, color, "%(num).1f", "%(short)s")

    bin_size = get_bin_size(sdate, edate)
    pps=[float(x)/bin_size for x in data['packets']]
    render_timeseries_primitive(
        out_file_name, pps, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'),
        max_color=portal.config.rendering_option('sparklines.maxcolor'),
        min_color=portal.config.rendering_option('sparklines.mincolor'),
        max_fmt=get_formatter(
            portal.config.rendering_option('sparklines.maxcolor')),
        min_fmt=get_formatter(
            portal.config.rendering_option('sparklines.mincolor')),)


@op_file
@mime_type('image/png')
@expires(nocache)
@typemap(sdate=datetime_obj, dur=str, sensors=str,
         ip=str, width=int, height=int)
def vis_susp_hosts_histogram_spark(out_file_name, sdate, dur, sensors, ip,
                                   width, height):
    edate = add_duration(sdate, dur)
    sensors = get_sensors(sensors)
    bins, values = gen_susp_hosts_histogram(sdate, edate, sensors, ip)
    render_histogram_primitive(
        out_file_name, values, width, height,
        bgcolor=portal.config.rendering_option('bgcolor'))


########################################################################


__export__ = {
    'watchlists/darknet-timeseries': vis_darknet_timeseries,
    'watchlists/darknet-table': vis_darknet_table,
    'watchlists/darknet-timeseries-spark': vis_darknet_timeseries_spark,
    'watchlists/darknet-histogram-spark': vis_darknet_histogram_spark,

    'watchlists/susp-ports-timeseries': vis_susp_ports_timeseries,
    'watchlists/susp-ports-table': vis_susp_ports_table,
    'watchlists/susp-ports-timeseries-spark': vis_susp_ports_timeseries_spark,
    'watchlists/susp-ports-histogram-spark': vis_susp_ports_histogram_spark,
    
    'watchlists/susp-hosts-timeseries': vis_susp_hosts_timeseries,
    'watchlists/susp-hosts-table': vis_susp_hosts_table,
    'watchlists/susp-hosts-timeseries-spark': vis_susp_hosts_timeseries_spark,
    'watchlists/susp-hosts-histogram-spark': vis_susp_hosts_histogram_spark,
}
