import time
import codecs
from types import NoneType
from httplib import HTTPConnection
try:
    from httplib import HTTPSConnection
    https_supported = True
except:
    pass

from urllib import urlencode
from urlparse import urlsplit
import xml.dom.minidom as minidom
from rave.util import json
import socket

from rave.util import json

__VERSION__ = '1.1'

# Namespace URI for RAVE elements
ns_rave = 'http://www.cert.org/schema/rave'

# Codec for decoding JSON data
utf8 = codecs.getwriter("utf8")

class RaveCommunicationError(Exception):
    "Problem talking to RAVE"
    def __init__(self, code, msg=None):
        Exception.__init__(self)
        self.code = code
        if msg:
            self.msg = msg
        else:
            self.msg = "General failure"
    def __str__(self):
        return "RaveCommunicationError(%d, %s)" % (self.code, self.msg)

class RaveTimeoutError(Exception):
    "Timeout while getting visualization"
    pass

class Ticket(object):
    def __init__(self, uri, content_type, stream=None):
        self.uri = str(uri)
        self.content_type = str(content_type)
        self.stream = stream
    def get_uri(self):
        return self.uri
    def get_content_type(self):
        return self.content_type
    def get_stream(self):
        if self.stream:
            result = self.stream
            self.stream = None
            return result
        else:
            return self.fetch_stream()
    def fetch_stream(self):
        return fetch_stream(self.uri)

def OLDparse_response(xml):
    "Extract the visualization URI and the content-type from RAVE XML."
    doc = minidom.parseString(xml)
    uri_els = doc.getElementsByTagNameNS(ns_rave, "uri")
    content_type_els = doc.getElementsByTagNameNS(ns_rave, "content-type")
    return Ticket(uri_els[0].childNodes[0].data,
                  content_type_els[0].childNodes[0].data)


def parse_response(data):
    "Extract the visualization URI and the content-type from RAVE JSON."
    obj = json.parse(data)
    return Ticket(obj['uri'], obj['content-type'])


def parse_url(url):
    (scheme, network_loc, path, query, frag_id) = urlsplit(url)
    if scheme == 'http':
        default_port = 80
    elif scheme == 'https' and https_supported:
        default_port = 443
    else:
        raise RaveCommunicationError(500, 'Unsupported scheme in URL: %s'
                                     % scheme)
    if network_loc.find(':') == -1:
        host = network_loc
        port = default_port
    else:
        (host, port) = network_loc.split(':')
        port = int(port)
    return (scheme, host, port, path, query)

def compose_params(params):
    return urlencode(params)

def get_connection(scheme, host, port):
    if scheme == 'http':
        connection = HTTPConnection
    elif scheme == 'https' and https_supported:
        connection = HTTPSConnection
    else:
        raise RaveCommunicationError('Unsupported scheme: %s' % scheme)
    return connection(host, port)

def fetch_stream(stream_url, timeout=600, delay=10):
    (scheme, host, port, path, query) = parse_url(stream_url)
    request_url = path
    if query:
        request_url = request_url + '?' + query
    start_time = time.time()
    conn = None
    while True:
        # DO NOT CLOSE!
        conn = get_connection(scheme, host, port)
        conn.request('GET', request_url)
        res = conn.getresponse()
        if res.status == 200:
            return res
        elif res.status <> 404:
            raise RuntimeError("Error %d from server: %s" %
                               (res.status, res.reason))
        cur_time = time.time()
        if timeout and (cur_time - start_time) >= timeout:
            raise RaveTimeoutError()
        else:
            time.sleep(delay)

def fetch_ticket(service_url, query_name, params=None, wait=None):
    if params and not isinstance(params, str):
        params = compose_params(params)
    (scheme, host, port, path, query) = parse_url(service_url)
    if not path.endswith('/'):
        path = path + '/'
    request_url = path + query_name
    request_wait = 'forever'
    if params:
        request_url = request_url + "?" + params
    if wait:
        request_wait = str(wait)
    conn = None
    try:
        try:
            conn = get_connection(scheme, host, port)
            conn.request('GET', request_url,
                         headers={'X-Rave-Wait': request_wait})
            res = conn.getresponse()
        except socket.error, e:
            raise RaveCommunicationError(500, 'Socket error connecting: %s'
                                         % e[1])
    finally:
        if conn: conn.close()
    if res.status != 200:
        raise RaveCommunicationError(res.status,
                                     'HTTP Error from RAVE: %d %s' %
                                     (res.status, res.reason))
    try:
        return parse_response(res.read())
    except Exception, e:
        raise RaveCommunicationError(500, 'Problem parsing response data: %s'
                                     % str(e))
                                     
