#
# Exceptions
#

import traceback
import sys

class RaveException(Exception):
    """Base class for RAVE exceptions."""
    pass

class RaveNestedException(RaveException):
    """An exception wrapping another exception."""
    def __init__(self, msg, exc_info):
        self.exc_info =  exc_info
        RaveException.__init__(self, msg)

    def nested_tb_list(self):
        try:
            return traceback.format_exception(*self.exc_info)
        except:
            return ["[Couldn't print traceback information]"]

    def nested_tb(self):
        try:
            return ''.join(traceback.format_exception(*self.exc_info))
        except:
            return "[Couldn't print traceback information]"


class ConfigurationError(RaveException):
    """Problem with RAVE configuration."""
    pass

class IllegalTypemap(ConfigurationError):
    """The typemap is improperly specified."""
    pass
class TypemapTypeError(IllegalTypemap):
    """Couldn't reconcile typemap with function signature."""
    def __init__(self, msg):
        msg = ("Couldn't reconcile typemap with "
               "function signature: %s" % msg)
        ConfigurationError.__init__(self, msg)

class IncompleteCacheError(ConfigurationError):
    """RAVE cache hasn't been completely created."""
    def __init__(self, root, msg=None):
        if msg is None:
            msg = "Cache in '%s' is corrupt, " \
                  "incomplete or an outdated version" % root
        ConfigurationError.__init__(self, msg)
        self.root_dir = root


class ExportError(RaveException):
    "Problem with __export__ in a plugin module."
    pass

class NoSuchOperation(RaveException):
    "An Operation was requested that does not exist."
    msgformat = "No such operation '%s'"
    def __init__(self, ns):
        RaveException.__init__(self, self.msgformat % ns)

class NoSuchNamespace(NoSuchOperation):
    "Someone requested a non-existant namespace."
    msgformat = "No such namespace '%s'"

class NoSuchModule(NoSuchOperation):
    "A module was requested that does not exist"
    msgformat = "No such module '%s'"

class NamespaceLoadError(RaveNestedException):
    "A module couldn't be loaded."
    def __init__(self, modname, exc_info):
        errmsg = "Couldn't load namespace '%s': (%s: %s)" % (
            modname, exc_info[0], exc_info[1])
        RaveNestedException.__init__(self, errmsg, exc_info)

class ParameterError(RaveException):
    "There was a problem with the parameters passed into a function."
    pass

class TypeMismatch(ParameterError):
    "Attempt to coerce parameter into uncoercable type."
    pass

class DecoratorError(RaveException):
    "Problem in one of the RAVE decorators."
    pass

class CacheError(RaveException):
    "Problem in the RAVE cache."
    pass

class ModuleRemovedError(RaveException):
    """
    A module previously loaded by a Library has been removed in the filesystem. The
    enclosed message is the a tuple of the name of the (now-missing) module and its
    (now-nonexistant) path.
    """
    pass
