import logging
import re

class RegexFilter(logging.Filter):
    """
    logging Filter that rejects messages based on a regular expression.
    """
    def __init__(self, re_str, name=''):
        logging.Filter.__init__(self, name)
        self.re = re.compile(re_str)

    def filter(self, record):
        m = self.re.search(record.getMessage())
        if m == None:
            return True
        else:
            return False

class log_factory(object):
    """
    Easily retrieve module-specific Logger, or an extension thereof.

    To use this class, create a module-scope instance at the top of
    the module:

        import rave.logging as rlog
        log = rlog.log_factory("a.b.c")

    Then use it in your code thusly:

        log().debug("debug msg to a.b.c")
        log("d").debug("debug msg to a.b.c.d")
    """
    def __init__(self, basename=None, filters=[]):
        super(log_factory, self).__init__()
        self.basename = basename
        self.filters = filters
        for f in self.filters:
            self.log.addFilter(f)

    def __call__(self, ext=None):
        if ext == None:
            return self
        else:
            return log_factory(self.basename + "." + ext)

    def __getattr__(self, attr):
        log = logging.getLogger(self.basename)
        return getattr(log, attr)


def deprecated(log, feature, ver, msg=""):
    """
    Warn to the log that a feature has been deprecated.
    Parameters:
    log
        logger
            instance to log with
        feature
            string name of feature being deprecated
        ver
            version in which feature was deprecated
        msg
            (optional) additional information about deprecation
    """
    import traceback as tb
    frame = tb.extract_stack()[1]
    log.warn(
          "Deprecation warning: '%s' deprecated in version %s: %s"
        , feature, ver, msg
    )
    log.warn(
          "Deprecated feature apparently used at %s line %d (function '%s') " 
        , frame[0], frame[1], frame[2]
    )
