"""
The RAVE Request object, and functions used to obtain and release it.
"""
import threading

# Thread-local, module-scope storage
class ThreadContext(threading.local):
    def __init__(self):
        self.requests = []
    def export_state(self):
        return (self.requests[0], self.requests[-1])
    def import_state(self, state):
        self.requests = list(state)
    def current_request(self):
        return self.requests[-1]
    def allocate(self, *args, **kwargs):
        if len(self.requests) == 0:
            kwargs['global_req'] = None
        else:
            kwargs['global_req'] = self.requests[0]
        req = Request(*args, **kwargs)
        self.requests.append(req)
        return req
    def free(self):
        self.requests.pop()
            
storage = ThreadContext()

class Request(object):
    """
    A state container for invocations of RAVE operations.
    """
    def __init__(self, op, options, args, kwargs, cache_key, global_req=None):
        """Called internally by RAVE to create Request objects.
        Users should not have to create Requests themselves."""
        self._op = op
        self._options = options
        self._args = args
        self._kwargs = kwargs
        self._cache_key = cache_key
        self._global_req = global_req
        self._job_scheduled = False
        self._shared_state = {}
    def args(self):
        "Positional arguments used to invoke this function."
        return self._args
    def kwargs(self):
        "Positional arguments used to invoke this function."
        return self._kwargs
    def job_scheduled(self, scheduled=None):
        if scheduled is not None:
            self._job_scheduled = scheduled
        return self._job_scheduled
    def cache_key(self):
        return self._cache_key
    def operation(self):
        "The Operation currently being run."
        return self._op
    def cache(self):
        """The cache into which data from the current operation will be 
        written.""" 
        return self.operation().cache()
    def normalize(self, *args, **kwargs):
        """
        Normalize arguments into the form the cache expects. Operations that 
        must manipulate the cache directly must do this so the cache can 
        compare entries effectively. This is an advanced technique; most 
        operations should never use this method.
        """
        return self.operation().normalize(*args, **kwargs)
    def option(self, optname, check_global=False):
        """
        Get meta-information passed to the operation as special
        arguments.  This facility is used internally by RAVE to
        direct how and whether to cache the results of operations;
        certain options are exposed and documented in the _RAVE
        Administrator's Guide_. They are read-only, and presented
        only for informational purposes.  Options have a local scope
        (the current operation being run) and a global scope (the
        topmost parent operation of the operation being run).

        Options are read-only. If you need to share state with other
        functions (like your expiration strategy) use the get() and
        set() methods.

        Parameters:
        optname: str
            The name of the option to look for
        check_global: boolean
            Whether to look for the option in global scope 
            By default, look only in the local object.
        """
        try:
            return self._options[optname]
        except KeyError:
            if check_global and self._global_req:
                return self._global_req.option(optname)
            else:
                return None
    def get(self, name):
        """
        Get an arbitrary property associated with this request. Use
        these properties to share information between the operation
        functions like the expiration strategy which also have
        access to the Request object.

        Parameters:
        name: <any>
            The name of the shared value
        Returns: rc
            rc: <any> | None
                The stored value, or None if there is no value by
                that name.
        """
        return self._shared_state[name]
    def set(self, name, value):
        """
        Set a property associated with this request. See get() for
        a slightly fuller discussion of this facility.

        Parameters:
        name: <any>
            The name of the shared value
        value: <any>
            The value of the shared value
        """
        self._shared_state[name] = value
        
        

__all__ = ['Request']
