import types
import inspect

def immute(v):
    """
    Convert a mutable object into an immutable one for hashing purposes,
    if necessary.
    """
    if type(v) == types.ListType or type(v) == types.TupleType:
        v = immute_list(v)
    elif type(v) == types.DictType:
        v = immute_dict(v)
    return v

def immute_dict(d):
    """
    Convert a dictionary into an immutable structure.
    """
    d_tuples = []
    for k,v in sorted(d.items()):
        v = immute(v)
        d_tuples.append((k,v))
    d_tuples = tuple(d_tuples)
    return d_tuples

def immute_list(l):
    """
    Convert a list into an immutable structure.
    """
    v_items = []
    for v in l:
        v = immute(v)
        v_items.append(v)
    v_items = tuple(v_items)
    return v_items

class LazyProxy(object):
    """A proxy object that doesn't instantiate the proxied object until
    it is first accessed. Handy for deferring the execution of code in
    objects with global scope until after other initialization has taken
    place."""
    def __init__(self, cls, *args, **kwargs):
        self.proxied_cls = cls
        self.proxied_args = (args, kwargs)
        self.proxied_instance = None
    def __getattribute__(self, name):
        def get(name):
            return object.__getattribute__(self, name)
        def set(name, val):
            object.__setattr__(self, name, val)
        if name == "lp_constructed":
            return get('proxied_instance') != None
        if not get('proxied_instance'):
            cls = get('proxied_cls')
            args = get('proxied_args')
            set('proxied_instance', cls(*args[0], **args[1]))
        return getattr(get('proxied_instance'), name)

def deprecated(msg, log=None, err=None):
    caller = inspect.currentframe(1)
    callerinfo = inspect.getframeinfo(caller)
    caller_parent = inspect.currentframe(2)
    parentinfo = inspect.getframeinfo(caller_parent)
    location = "Deprecation warning in line %s, %s (called from line %d, %s)" % (
        callerinfo[1], callerinfo[0], parentinfo[1], parentinfo[0]
    )
    if log is None:
        print location
        print msg
    else:
        log.warning(location)
        log.warning(msg)
    if err != None:
        raise err

def get_units(bytes):
    "Convert count to a larger unit type."
    kb = 1024.0
    mb = 1024.0 * kb
    gb = 1024.0 * mb
    tb = 1024.0 * gb

    if bytes >= tb:
        return bytes/tb, 'T'
    if bytes >= gb:
        return bytes/gb, 'G'
    if bytes >= mb:
        return bytes/mb, 'M'
    if bytes >= kb:
        return bytes/kb, 'k'
    return bytes, 'b'


def format_bytes(bytes):
    return "%.2f%s" % get_units(bytes)
