"""Disk And Execution MONitor (Daemon)

Default daemon behaviors (they can be modified):
   1.) Ignore SIGHUP signals.
   2.) Default current working directory to the "/" directory.
   3.) Set the current file creation mode mask to 0.
   4.) Close all open files (0 to [SC_OPEN_MAX or 256]).
   5.) Redirect standard I/O streams to "/dev/null".

Failed fork() calls will return a tuple: (errno, strerror).  This behavior
can be modified to meet your program's needs.

Resources:
   Advanced Programming in the Unix Environment: W. Richard Stevens
   Unix Network Programming (Volume 1): W. Richard Stevens
   http://www.erlenstar.demon.co.uk/unix/faq_2.html#SEC16
"""

#
# Code shamelessly ripped (and modified) from:
# <http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/278731>
#

import os               # Miscellaneous OS interfaces.
import sys              # System-specific parameters and functions.
import signal           # Set handlers for asynchronous events.

def daemonize(wd="/", stdout=None, stderr=None):
    """Detach a process from the controlling terminal and run it in the
    background as a daemon.

    Parameters:
    stdout: str
        name of file to which to write standard output of forked process
    stderr: str
        name of file to which to write standard error of forked process

    """

    if stdout is None:
        stdout = "/dev/null"
    if stderr is None:
        stderr = "/dev/null"

    pid = os.fork()
    if (pid == 0): # Child
        os.setsid()
    #   Ignore SIGHUP so the upcoming child process doesn't barf when
    #   we, the parent, exit mere milliseconds from starting him.
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        pid = os.fork()
        if (pid == 0): # Grandchild
        #   Set the working directory to wd so we don't stand in the way
        #   of unmounting any filesystems
            os.chdir(wd)
        #   Give daemon process unlimited control on permissions of
        #   files it creates.
            os.umask(0)
        else: # (Still) Child
            os._exit(0)
    else: # Parent
        os._exit(0)
        

    # Still executing? Must be the now-daemonized grandchild process....

    # Close all open files.  Try the system configuration
    # variable, SC_OPEN_MAX, but default if it isn't there.
    try:
        maxfd = os.sysconf("SC_OPEN_MAX")
    except (AttributeError, ValueError):
        # SC_OPEN_MAX didn't exist. Make something up.
        maxfd = 256       # default maximum

    for fd in range(0, maxfd):
        try:
            os.close(fd)
        except OSError:
        #   Ignore
            pass



    # Now that all the file descriptors are closed, any new ones
    # we open will get the lowest fd numbers available -- 0,1, and
    # 2, which correspond to STDIN, STDOUT and STDERR.


    # Redirect the standard file descriptors to specified file, or
    # /dev/null. (This is programming by coincidence -- we just
    # closed these fds, so we should get them when in this order
    # when we open them. This _is not guaranteed_ if you have
    # other threads running in parallel with daemonize().
    os.open("/dev/null", os.O_RDONLY)       # standard input  (0)
    os.open(stdout, os.O_RDWR | os.O_CREAT) # standard output (1)
    os.open(stderr, os.O_RDWR | os.O_CREAT) # standard error  (2)

    # Redirect Python STD* streams
    sys.stdin  = os.fdopen(0, 'rb')
    sys.stdout = os.fdopen(1, 'wb', 1)  # line-buffer STDOUT
    sys.stderr = os.fdopen(2, 'wb', 0)  # unbuffer STDERR
