"""
A simple wrapper over python's getopt to make argument parsing slightly
clearer, more informative and more convenient.
"""
import getopt
import sys

from ConfigParser import ConfigParser

class OptionError(Exception):
    pass

class Option(object):
#   Number of characters before the end of a tabstop (in help string,
#   between long/short option and opt documentation)
    tabstop = 33

    def __init__(self, shortopt, longopt, opt_valdesc, type, doc):
        assert(longopt != None)
        self.shortopt = shortopt
        self.longopt = longopt
        self.opt_valdesc = opt_valdesc
        self.type = type
        self.doc = doc

    def as_shortopt(self):
        if not self.shortopt:
            return ''
        else:
            if self.opt_valdesc:
                return "%s:" % self.shortopt
            else:
                return self.shortopt

    def as_longopt(self):
        if self.opt_valdesc:
            return "%s=" % self.longopt
        else:
            return self.longopt

    def __str__(self):
        if self.shortopt and self.longopt:
            opts = "-%s, --%s" % (self.shortopt, self.longopt)
        elif self.shortopt:
            opts = "-%s" % self.shortopt
        elif self.longopt:
            opts = "--%s" % self.longopt

        if self.opt_valdesc:
            if self.longopt:
                val = "=%s" % self.opt_valdesc
            else:
                val = " %s" % self.opt_valdesc
        else:
            val = ''

        doc = self.doc or ''
        kvpair = "%s%s" % (opts, val)
        if len(kvpair) < self.tabstop:
            kvpair = kvpair + ''.join(
                ' ' for x in range(self.tabstop - len(kvpair))
            )
        return "%s%s%s" % (kvpair, ' ', doc)

class OptionList(object):
    @staticmethod
    def from_options(opts):
        olist = OptionList()
        for o in opts:
            olist.add(o)
     #  Add help option
        olist.add(Option('?', 'help', None, None, "Print help and exit"))
        return olist

    def __init__(self):
        self._options = {}
        self.short2long = {}

    def add(self, opt):
        self._options[opt.longopt] = opt
        if opt.shortopt:
            self.short2long[opt.shortopt] = opt.longopt

    def options(self):
        return self._options.values()


    def convert_option_name(self, rawname):
        if rawname.startswith('--'):
            if rawname.endswith('='):
                return rawname[2:-1]
            else:
                return rawname[2:]
        else:
            try:
                return self.short2long[rawname[1]]
            except KeyError:
                raise OptionError("Unknown option '%s'" % rawname)


    def convert_value(self, name, v):
        try:
            opt = self._options[name]
        except:
            raise OptionError("Unknown option '%s'" % name)

        if v:
            if opt.type:
                return opt.type(v)
            else:
                return v
        else:
            return True
        

    def parse(self, args):
        "Parse a sys.argv-like command line."
        shortopts = []
        longopts  = []
        for opt in self.options():
            shortopts.append(opt.as_shortopt())
            longopts.append(opt.as_longopt())
        shortopts = ''.join(shortopts)
        (parsed, unparsed) = getopt.getopt(args, shortopts, longopts)
        params = {}
        for k,v in parsed:
            if k.startswith('--'):
                if k.endswith('='):
                    name = k[2:-1]
                else:
                    name = k[2:]
            else:
                try:
                    name = self.short2long[k[1]]
                except KeyError:
                    raise OptionError("Unknown option '%s'" % k)
            try:
                opt = self._options[name]
            except:
                raise OptionError("Unknown option '%s'" % name)

            if v:
                if opt.type:
                    params[name] = opt.type(v)
                else:
                    params[name] = v 
            else:
                params[name] = True

        return params


    def parse_cp(self, cp, section):
        params = {}
        for name in cp.options(section):
            val = cp.get(section, name)
            param[name] = self.convert_value(name, val)
        return params
            

    def help_str(self):
        return "Options:\n%s" % ''.join(
            "%s\n" % opt for opt in sorted(self.options()))

