import sys, string, time, os
from rave.plugins.decorators import Expert
from threading import Condition, Thread
from subprocess import Popen, PIPE
from rave.plugins.shell import parse_rwuniq, parse_rwtotal, parse_rwcount, \
    parse_generic, parse_rwcut
import rave.cache.core
from rave.plugins.dataset import Dataset
from silk.util import run_process, run_in_thread
from datetime import datetime
import silk

### Expiry #############################################################

raw_merged_cache_dur = 600
raw_cache_dur = 600
summary_merged_cache_dur = 3600
summary_cache_dur = 3600

def set_raw_merged_cache_dur(x):
    global raw_merged_cache_dur
    raw_merged_cache_dur = x

def set_raw_cache_dur(x):
    global raw_cache_dur
    raw_cache_dur = x

def set_summary_merged_cache_dur(x):
    global summary_merged_cache_dur
    summary_merged_cache_dur = x

def set_summary_cache_dur(x):
    global summary_cache_dur
    summary_cache_dur = x

def raw_merged_expire_time():
    if raw_merged_cache_dur is not None:
        return time.time() + raw_merged_cache_dur
    return rave.cache.core.NEVER

def raw_expire_time():
    if raw_cache_dur is not None:
        return time.time() + raw_cache_dur
    return rave.cache.core.NEVER

def summary_merged_expire_time():
    if summary_merged_cache_dur is not None:
        return time.time() + summary_merged_cache_dur
    return rave.cache.core.NEVER

def summary_expire_time():
    if summary_cache_dur is not None:
        return time.time() + summary_cache_dur
    return rave.cache.core.NEVER

### Cache Simplification ###############################################

def cache_factory(label, version, mime_type="application/octet-stream"):
    def get_or_reserve(inputs, *args, **kwargs):
        safe_inputs = []
        for i in inputs:
            if issubclass(type(i), proc):
                safe_inputs.append(i._proc_label())
            else:
                safe_inputs.append(repr(i))
        r = Expert(label, version, (safe_inputs, args, kwargs), mime_type)
        return r
    return get_or_reserve

### Core Processor Class ###############################################

class proc(object):
    """
    """
    ST_READY = 0
    ST_WORK = 1
    ST_RESULT_VALUE = 2
    ST_RESULT_EXCEPTION = 3
    __slots__ = '_proc_name', '_proc_tag', '_proc_inputs', '_proc_kwargs', \
        '_proc_status', '_proc_result', '_proc_cond', '_proc_parts', \
        '_proc_cmdargs'
    def __init__(self, name, inputs, kwargs, tag=None, _cond=None):
        if _cond == None: _cond = Condition()
        object.__init__(self)
        self._proc_name = name
        self._proc_inputs = list(inputs)
        self._proc_kwargs = dict(kwargs)
        self._proc_status = proc.ST_READY
        self._proc_result = None
        self._proc_tag = tag
        self._proc_cond = _cond
        self._proc_parts = None
        self._proc_cmdargs = []
    def _proc_make_input_parts(self):
        self._proc_parts = [proc_part(self, i)
                            for i in self._proc_expanded_inputs()]
    def _proc_progress(self):
        if not self._proc_parts:
            if self._proc_status <= proc.ST_WORK:
                return 0.0
            else:
                return 1.0
        else:
            progress = 0.0
            for p in self._proc_parts:
                progress += p._proc_progress()
            if self._proc_status <= proc.ST_WORK:
                return (progress / len(self._proc_parts)) * 0.9
            else:
                return (progress / len(self._proc_parts))
    def _proc_eval(self):
        # Default behavior
        if self._proc_parts is None:
            raise TypeError("proc subclass '%s' did not implement _proc_eval" % type(self).__name__)
        # If we have parts, use eval part instead
        return self._proc_merge(self._proc_parts)
    def _proc_eval_part(self, input):
        raise TypeError("proc subclass did not implement _proc_eval_part")
    def _proc_merge(self, values):
        raise TypeError("proc subclass did not implement _proc_merge")
    def _proc_get_parts(self):
        self._proc_cond.acquire()
        try:
            for p in self._proc_parts:
                p._proc_get()
        finally:
            self._proc_cond.release()
    def _proc_get(self):
        self._proc_cond.acquire()
        try:
            # If we're ready, evaluate here.
            if self._proc_status == proc.ST_READY:
                self._proc_status = proc.ST_WORK
                try:
                    self._proc_result = self._proc_eval()
                    self._proc_status = proc.ST_RESULT_VALUE
                except:
                    self._proc_result = sys.exc_info()
                    self._proc_status = proc.ST_RESULT_EXCEPTION
                self._proc_cond.notifyAll()
            # If somebody else is working on it, wait for them.
            while self._proc_status == proc.ST_WORK:
                # We wake up periodically to be sure signals work
                self._proc_cond.wait(2)
            # Now the result is definitely available, do it
            if self._proc_status == proc.ST_RESULT_VALUE:
                return self._proc_result
            else: # self._proc_status == proc.ST_RESULT_EXCEPTION
                raise self._proc_result[0], self._proc_result[1], \
                    self._proc_result[2]
        finally:
            self._proc_cond.release()
    def _proc_expanded_inputs(self):
        expanded_inputs = []
        for i in self._proc_inputs:
            if issubclass(type(i), proc):
                expanded_inputs += i._proc_expanded_parts()
            else:
                expanded_inputs.append(i)
        return expanded_inputs
    def _proc_expanded_parts(self):
        """
        Returns the process expanded into all of its individual
        sub-parts.  The default behavior is to return self.        
        """
        if self._proc_parts:
            return self._proc_parts
        else:
            return [self]
    def _proc_label(self):
        args = []
        for i in self._proc_inputs:
            if issubclass(type(i), proc):
                args.append("%s" % i._proc_label())
            else:
                args.append("%s" % repr(i))
        kwarg_keys = self._proc_kwargs.keys()
        kwarg_keys.sort()
        for k in kwarg_keys:
            v = self._proc_kwargs[k]
            if issubclass(type(v), proc):
                args.append("%s=%s" % (k, v._proc_label()))
            else:
                args.append("%s=%s" % (k, repr(v)))
        if self._proc_tag:
            return "%s<%s>(%s)" % (self._proc_name, self._proc_tag,
                                   ', '.join(args))
        else:
            return "%s(%s)" % (self._proc_name, ', '.join(args))
    def _proc_repr(self):
        label = self._proc_label()
        status = self._proc_status
        if status == proc.ST_WORK:
            return label + ' [%0.0f%% complete]' % (self._proc_progress()
                                                    * 100.0)
        elif status == proc.ST_RESULT_VALUE:
            return label + ' => return %s' % repr(self._proc_result)
        elif status == proc.ST_RESULT_EXCEPTION:
            return label + ' => raise %s' % repr(self._proc_result[1])
        else:
            return label + ' [waiting, %0.0f%% complete]' % \
                (self._proc_progress() * 100.0)
    def _proc_file_inputs(self):
        result = set()
        for i in self._proc_inputs:
            if issubclass(type(i), proc):
                result.update(i._proc_file_inputs())
            else:
                result.add(i)
        return result
    ####################################################################
    ### __repr__ is special cased to output the current status if
    ### we're currently computing--otherwise, it passes on to the
    ### underlying result
    def __repr__(self):
        if self._proc_status <= proc.ST_WORK:
            return self._proc_repr()
        else:
            return repr(self._proc_value)
    ### All other methods pass through purely to the result, forcing
    ### the underlying promise.
    def __abs__(self):               return abs(self._proc_value)
    def __add__(self, o):            return self._proc_value + o
    def __and__(self, o):            return self._proc_value & o
    def __call__(self, *args, **kwargs):
        return self._proc_value(*args, **kwargs)
    def __cmp__(self, o):            return cmp(self._proc_value, o)
    def __coerce__(self, o):         return coerce(self._proc_value, o)
    def __complex__(self):           return complex(self._proc_value)
    def __contains__(self, x):       return x in self._proc_value
    def __delattr__(self, name):
        if name == '_proc_value':
            self._proc_value = None
            self._proc_status = proc.ST_READY
        elif name.startswitch('_proc_'):
            object.__delattr__(self, name)
        else:
            delattr(self._proc_value, name)
    def __delitem__(self, i):        del self._proc_value[i]
    def __delslice(self, i, j):      del self._proc_value[i:j]
    def __div__(self, o):            return self._proc_value / o
    def __divmod__(self, o):         return divmod(self._proc_value, o)
    def __eq__(self, o):             return self._proc_value == o
    def __float__(self):             return float(self._proc_value)
    def __floordiv__(self, o):       return self._proc_value // o
    def __ge__(self, o):             return self._proc_value >= o
    def __getattribute__(self, name):
        if name == '_proc_value':
            if self._proc_status <= proc.ST_WORK:
                return self._proc_get()
            elif self._proc_status == proc.ST_RESULT_VALUE:
                return self._proc_result
            else:
                raise self._proc_result[0], self._proc_result[1], \
                    self._proc_result[2]
        elif name.startswith('_proc_'):
            return object.__getattribute__(self, name)
        else:
            return getattr(self._proc_get(), name)
    def __getitem__(self, i):        return self._proc_value[i]
    def __getslice__(self, i, j):    return self._proc_value[i:j]
    def __gt__(self, o):             return self._proc_value > o
    def __hash__(self):              return hash(self._proc_value)
    def __hex__(self):               return hex(self._proc_value)
    def __iadd__(self, o):
        self._proc_value += o
        return self
    def __iand__(self, o):
        self._proc_value &= o
        return self
    def __idiv__(self, o):
        self._proc_value /= o
        return self
    def __ifloordiv__(self, o):
        self._proc_value //= o
        return self
    def __imod__(self, o):
        self._proc_value %= o
        return self
    def __ilshift__(self, o):
        self._proc_value <<= o
        return self
    def __imul__(self, o):
        self._proc_value *= o
        return self
    def __int__(self):               return int(self._proc_value)
    def __invert__(self):            return ~self._proc_value
    def __ior__(self, o):
        self._proc_value |= o
        return self
    def __ipow__(self, o):
        self._proc_value **= o
        return self
    def __irshift__(self, o):
        self._proc_value >>= o
        return self
    def __isub__(self, o):
        self._proc_value -= o
        return self
    def __iter__(self):              return iter(self._proc_value)
    def __itruediv__(self, o):
        self._proc_value /= o
        return self
    def __ixor__(self, o):
        self._proc_value ^= o
        return self
    def __le__(self, o):             return self._proc_value <= o
    def __len__(self):               return len(self._proc_value)
    def __long__(self):              return long(self._proc_value)
    def __lshift__(self, o):         return self._proc_value << o
    def __lt__(self, o):             return self._proc_value < o
    def __mod__(self, o):            return self._proc_value % o
    def __mul__(self, o):            return self._proc_value * o
    def __ne__(self, o):             return self._proc_value != o
    def __neg__(self):               return -self._proc_value
    def __nonzero__(self):           return bool(self._proc_value)
    def __oct__(self):               return oct(self._proc_value)
    def __or__(self, o):             return self._proc_value | o
    def __pos__(self):               return +self._proc_value
    def __pow__(self, *args):        return pow(self._proc_value, *args)
    def __radd__(self, o):           return o + self._proc_value
    def __rand__(self, o):           return o & self._proc_value
    def __rdiv__(self, o):           return o / self._proc_value
    def __rdivmod__(self, o):        return divmod(o, self._proc_value)
    def __rfloordiv__(self, o):      return o // self._proc_value
    def __rlshift__(self, o):        return o << self._proc_value
    def __rmod__(self, o):           return o % self._proc_value
    def __rmul__(self, o):           return o * self._proc_value
    def __ror__(self, o):            return o | self._proc_value
    def __rpow__(self, o):           return pow(o, self._proc_value)
    def __rrshift__(self, o):        return o >> self._proc_value
    def __rshift__(self, o):         return self._proc_value >> o
    def __rsub__(self, o):           return o - self._proc_value
    def __rtruediv__(self, o):       return o / self._proc_value
    def __rxor__(self, o):           return o ^ self._proc_value
    def __setattr__(self, name, val):
        if name.startswith('_proc_'):
            object.__setattr__(self, name, val)
        else:
            setattr(self._proc_value, name, val)
    def __setitem__(self, i, x):     self._proc_value[i] = v
    def __setslice__(self, i, j, x): self._proc_value[i:j] = x
    def __str__(self):               return str(self._proc_value)
    def __sub__(self, o):            return self._proc_value - o
    def __truediv__(self, o):        return self._proc_value / o
    def __xor__(self, o):            return self._proc_value ^ o

class proc_part(proc):
    __slots__ = '_proc_parent'
    def __init__(self, parent, input, kwargs=None, tag=None):
        proc.__init__(self, parent._proc_name + '#', [input],
                      (kwargs or parent._proc_kwargs),
                      tag=(tag or parent._proc_tag),
                      _cond=parent._proc_cond)
        self._proc_parent = parent
    def _proc_eval(self):
        return self._proc_parent._proc_eval_part(self._proc_inputs[0])

# Utility functions for procs

# Notice that issubclass(type(p), proc) is used instead of
# isinstance(p, proc).  If isinstance is used, the proc will be
# forced, and the type compared will be that of the result, not the
# proc itself.

# Return the text label for a proc.  This respresents all parameters
# of the proc, including all inputs and their parameters
# (recursively).
def proc_label(p):
    if not issubclass(type(p), proc):
        raise TypeError("%s object is not a silk.proc" % type(p))
    return p._proc_label()

# Return the current progress of the proc, as a floating point value
# between 0.0 (no progress) and 1.0 (completed).
def proc_progress(p):
    if not issubclass(type(p), proc):
        raise TypeError("%s object is not a silk.proc" % type(p))
    return p._proc_progress()

# Wait for the proc to finish, calling the provided callback to
# provide status updates.  The timeout determines the maximum
# resolution of updates, and may be a floating point number for better
# than 1s resolution, if the OS supports this.

# Note: This will not return the actual value of the proc (you should
# use the proc directly as that.)  But, it will raise any exceptions
# from the proc.

def proc_wait(p, callback, timeout=2):
    # If something's already done, return immediately
    if p._proc_status > proc.ST_WORK:
        p.proc_get()
        return
    # Otherwise, make sure that some other thread is already running
    # things before we get there.
    if p._proc_status == proc.ST_READY:
        run_in_thread(p._proc_get)
    while p._proc_status <= proc.ST_WORK:
        last_progress = None
        # Wait the timeout
        time.sleep(timeout)
        # Return right now if we're done--no more callback
        if p._proc_status > proc.ST_WORK:
            break
        # Otherwise, call back with progress, if changed
        new_progress = p._proc_progress()
        if new_progress != last_progress:
            last_progress = new_progress
            callback(p, last_progress)
    p.proc_get()
    return

# Return a set of the filenames of static file inputs used in this
# proc, including any filenames used as inputs in proc inputs,
# recursively.
def proc_file_inputs(p):
    return p._proc_file_inputs()

### Tool Support #######################################################

tr_underscore_to_hyphen = string.maketrans('_', '-')

def hyphenate(x): return x.translate(tr_underscore_to_hyphen)

def arg_str(cmdargs, kwargs, k):
    if k in kwargs:
        v = str(kwargs[k])
        cmdargs.append('--%s=%s' % (hyphenate(k), v))
        del kwargs[k]

def arg_str_choice(cmdargs, kwargs, k, option_list):
    if k in kwargs:
        v = str(kwargs[k])
        if v not in option_list:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are: %s") %
                                 (`v`, `k`,
                                  ', '.join(`o` for o in option_list)))
        cmdargs.append('--%s=%s' % (hyphenate(k), v))
        del kwargs[k]

def arg_str_list(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        if isinstance(v, (str, unicode)):
            v = v.split(',')
        if not isinstance(v, (list, tuple)) or \
                filter((lambda x: not isinstance(x, (str, unicode))), v):
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are strings delimited by " +
                                  "commas, or sequences of strings.") %
                                 (`kwargs[k]`, `k`))
        cmdargs.append('--%s=%s' % (hyphenate(k), ','.join(str(x) for x in v)))
        del kwargs[k]

def arg_str_many(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        if not isinstance(v, (list, tuple)) or \
                filter((lambda x: not isinstance(x, (str, unicode))), v):
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are sequences of strings.") %
                                 (`kwargs[k]`, `k`))
        for s in v:
            cmdargs.append('--%s=%s' % (hyphenate(k), s))
        del kwargs[k]

def arg_float(cmdargs, kwargs, k):
    if k in kwargs:
        try:
            v = float(kwargs[k])
        except ValueError:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Only numbers are allowed.") %
                                 (`kwargs[k]`, `k`))
        cmdargs.append('--%s=%f' % (hyphenate(k), v))
        del kwargs[k]

def arg_int(cmdargs, kwargs, k):
    if k in kwargs:
        try:
            v = int(kwargs[k])
        except ValueError:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Only integers are allowed.") %
                                 (`kwargs[k]`, `k`))
        cmdargs.append('--%s=%d' % (hyphenate(k), v))
        del kwargs[k]

def arg_int_list(cmdargs, kwargs, k):
    if k in kwargs:
        try:
            values = []
            v = kwargs[k]
            if isinstance(v, (int, long, str, unicode)):
                v = [v]
            for x in v:
                if isinstance(x, tuple):
                    values.append("%d-%d" % (int(x[0]), int(x[1])))
                elif isinstance(x, (str, unicode)):
                    values.append(x)
                else:
                    values.append("%d" % int(x))
        except ValueError:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Only sequences of integers are allowed.") %
                                 (`kwargs[k]`, `k`))
        cmdargs.append('--%s=%s' % (hyphenate(k), ','.join(values)))
        del kwargs[k]

def arg_bool(cmdargs, kwargs, k):
    if k in kwargs:
        if kwargs[k] == True:
            cmdargs.append('--%s=1' % hyphenate(k))
            del kwargs[k]
        elif kwargs[k] == False:
            cmdargs.append('--%s=0' % hyphenate(k))
            del kwargs[k]
        else:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are: True, False") %
                                 (`kwargs[k]`, `k`))

def arg_flag(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        if v == True:
            cmdargs.append('--%s' % hyphenate(k))
            del kwargs[k]
        elif v == False:
            del kwargs[k]
        else:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are: True, False") %
                                 (`kwargs[k]`, `k`))

def arg_antiflag(cmdargs, kwargs, k):
    if k in kwargs:
        if k == True:
            del kwargs[k]
        elif k == False:
            cmdargs.append('--no-%s' % hyphenate(k))
            del kwargs[k]
        else:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are: True, False") %
                                 (`kwargs[k]`, `k`))
    
def arg_flag_antiflag(cmdargs, kwargs, k):
    if k in kwargs:
        if k == True:
            cmdargs.append('--%s' % hyphenate(k))
            del kwargs[k]
        elif k == False:
            cmdargs.append('--no-%s' % hyphenate(k))
            del kwargs[k]
        else:
            raise silk.SilkError(("Value %s not allowed for arg %s.  " +
                                  "Valid options are: True, False") %
                                 (`kwargs[k]`, `k`))

def arg_illegal(cmdargs, kwargs, k):
    if k in kwargs:
        raise silk.SilkError("Argument %s is not allowed." % `k`)

def arg_flag_value(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        if v is True:
            cmdargs.append('--%s' % hyphenate(k))
            del kwargs[k]
        elif v is False:
            del kwargs[k]
        else:
            cmdargs.append('--%s=%s' % (hyphenate(k), str(v)))
            del kwargs[k]

def arg_remainder(cmdargs, kwargs):
    switches = []
    for k in kwargs.keys():
        if silk.DEBUG_WARN_UNKNOWN:
            silk.debug("WARNING: Unknown argument %s with value %s" %
                       (`k`, `kwargs[k]`))
        switch = '--' + k.translate(tr_underscore_to_hyphen)
        if kwargs[k] is True:
            switches.append(switch)
        elif kwargs[k] is not False:
            switches.append("%s=%s" % (switch, str(kwargs[k])))
        del kwargs[k]
    cmdargs += switches

def arg_hour(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        del kwargs[k]
        v = silk_hour(v)
        cmdargs.append('--%s=%s' % (hyphenate(k), v))

def arg_datetime(cmdargs, kwargs, k):
    if k in kwargs:
        v = kwargs[k]
        del kwargs[k]
        v = silk_datetime(v)
        cmdargs.append('--%s=%s' % (hyphenate(k), v))

def arg_datetime_range(cmdargs, kwargs, k):
    if k in kwargs:
        s = None
        e = None
        try:
            (s, e) = kwargs[k]
        except:
            s = kwargs[k]
        del kwargs[k]
        s = silk_datetime(s)
        if e is None:
            cmdargs.append('--%s=%s' % (hyphenate(k), s))
        else:
            e = silk_datetime(e)
            cmdargs.append('--%s=%s-%s' % (hyphenate(k), s, e))

arg_wild_ip = arg_str # XXX

### Tool: rwcat ########################################################

rwcat_cache = cache_factory('rwcat', 20070413, 'application/x-silk-flow')
class proc_rwcat(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwcat', inputs, kwargs)
        args = self._proc_cmdargs
        arg_illegal    (args, kwargs, 'compression_method')
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'output_path')
        arg_illegal    (args, kwargs, 'print_filenames')
        arg_str        (args, kwargs, 'site_config_file')
        arg_illegal    (args, kwargs, 'version')
        arg_illegal    (args, kwargs, 'xargs')
        arg_remainder  (args, kwargs)
        self._proc_parts = self._proc_expanded_inputs()
    def _proc_merge(self, values):
        ent = rwcat_cache(self._proc_parts)
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'wb')
                run_process('rwcat', values + self._proc_cmdargs, stdout=f)
                f.close()
                ent.commit(raw_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwcat(*inputs, **kwargs):
    try:
        return proc_rwcat(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwsort #######################################################

rwsort_cache = cache_factory('rwsort', 20070413, 'application/x-silk-flow')

class proc_rwsort(proc):
    __slos__ = ()
    def __init__(self, input, kwargs):
        proc.__init__(self, 'rwsort', [input], kwargs)
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_str       (args, kwargs, 'site_config_file')
        arg_str_list  (args, kwargs, 'fields')
        arg_illegal   (args, kwargs, 'output_path')
        arg_illegal   (args, kwargs, 'input_pipe')
        arg_str_many  (args, kwargs, 'dynamic_library')
        arg_str       (args, kwargs, 'temp_directory')
        arg_illegal   (args, kwargs, 'compression_method')
        arg_str       (args, kwargs, 'pmap_file')
        arg_remainder (args, kwargs)
        run_in_thread(self._proc_get)
    def _proc_eval(self):
        ent = rwsort_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'w')
                run_process('rwsort', self._proc_inputs + self._proc_cmdargs,
                            stdout=f)
                f.close()
                ent.commit(raw_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwsort(input, **kwargs):
    try:
        return proc_rwsort(input, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwtotal #####################################################

rwtotal_cache = cache_factory('rwtotal', 20070405)
rwtotal_sum_cols = [ 'bytes', 'packets', 'flows' ]
rwtotal_standard_args = [ '--delimited=|' ]

class proc_rwtotal(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwtotal', inputs, kwargs)
        args = self._proc_cmdargs
        arg_flag       (args, kwargs, 'sip_first_8')
        arg_flag       (args, kwargs, 'sip_first_16')
        arg_flag       (args, kwargs, 'sip_first_24')
        arg_flag       (args, kwargs, 'sip_last_8')
        arg_flag       (args, kwargs, 'sip_last_16')
        arg_flag       (args, kwargs, 'dip_first_8')
        arg_flag       (args, kwargs, 'dip_first_16')
        arg_flag       (args, kwargs, 'dip_first_24')
        arg_flag       (args, kwargs, 'dip_last_8')
        arg_flag       (args, kwargs, 'dip_last_16')
        arg_flag       (args, kwargs, 'sport')
        arg_flag       (args, kwargs, 'dport')
        arg_flag       (args, kwargs, 'proto')
        arg_flag       (args, kwargs, 'packets')
        arg_flag       (args, kwargs, 'bytes')
        arg_flag       (args, kwargs, 'duration')
        arg_flag       (args, kwargs, 'icmp_code')
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'version')
        arg_str        (args, kwargs, 'site_config_file')
        arg_flag       (args, kwargs, 'skip_zeroes')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_illegal    (args, kwargs, 'no_columns')
        arg_illegal    (args, kwargs, 'column_separator')
        arg_illegal    (args, kwargs, 'delimited')
        arg_illegal    (args, kwargs, 'print_filenames')
        arg_illegal    (args, kwargs, 'pager')
        arg_illegal    (args, kwargs, 'copy_input')
        arg_remainder  (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwtotal_cache([input], sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwtotal', [input] + rwtotal_standard_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_rwtotal(f)
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwtotal_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                d = values[0]
                key_cols = []
                sum_cols = []
                for c in d.columns:
                    if c in rwtotal_sum_cols:
                        sum_cols.append(c)
                    else:
                        key_cols.append(c)
                for v in values[1:]:
                    d = v + d
                    d = d.sum_uniq(key_cols, sum_cols)
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwtotal(*inputs, **kwargs):
    try:
        return proc_rwtotal(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwcount #####################################################

rwcount_cache = cache_factory('rwcount', 20070405)
rwcount_sum_cols = [ 'bytes', 'packets', 'flows' ]
rwcount_min_cols = [ 'mintime' ]
rwcount_max_cols = [ 'maxtime' ]
rwcount_standard_args = [ '--epoch-slots', '--delimited=|' ]

class proc_rwcount(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwcount', inputs, kwargs)
        args = self._proc_cmdargs
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'version')
        arg_int        (args, kwargs, 'bin_size')
        arg_str_choice (args, kwargs, 'load_scheme', ['0', '1', '2', '3', '4'])
        arg_flag       (args, kwargs, 'skip_zeroes')
        arg_flag       (args, kwargs, 'epoch_slots')
        arg_flag       (args, kwargs, 'bin_slots')
        arg_int        (args, kwargs, 'start_epoch')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_illegal    (args, kwargs, 'no_columns')
        arg_illegal    (args, kwargs, 'column_separator')
        arg_illegal    (args, kwargs, 'delimited')
        arg_illegal    (args, kwargs, 'print_filenames')
        arg_illegal    (args, kwargs, 'pager')
        arg_illegal    (args, kwargs, 'legacy_timestamps')
        arg_remainder  (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwcount_cache([input], sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwcount', [input] + rwcount_standard_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_rwcount(f)
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwcount_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    d = values[0]
                else:
                    d = values[0]
                    key_cols = []
                    sum_cols = []
                    min_cols = []
                    max_cols = []
                    for c in d.columns:
                        if c in rwcount_sum_cols:
                            sum_cols.append(c)
                        elif c in rwcount_min_cols:
                            min_cols.append(c)
                        elif c in rwcount_max_cols:
                            max_cols.append(c)
                        else:
                            key_cols.append(c)
                    for v in values[1:]:
                        d = v + d
                        d = d.merge_uniq(key_cols, sum_cols, min_cols, max_cols)
                # Remove the degenerate epoch value
                epoch = datetime.utcfromtimestamp(0)
                d = Dataset(r for r in d if r['time'] <> epoch)
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwcount(*inputs, **kwargs):
    try:
        return proc_rwcount(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwaddrcount #################################################

rwaddrcount_cache = cache_factory('rwaddrcount', 20070405)
rwaddrcount_sum_cols = [ 'bytes', 'packets', 'flows' ]
rwaddrcount_standard_args = [ '--epoch-slots', '--delimited=|' ]

class proc_rwaddrcount(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwaddrcount', inputs, kwargs)
        args = self._proc_cmdargs
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'version')
        arg_flag       (args, kwargs, 'print_recs')
        arg_flag       (args, kwargs, 'print_stat')
        arg_flag       (args, kwargs, 'print_ips')
        arg_flag       (args, kwargs, 'use_dest')
        arg_int        (args, kwargs, 'byte_min')
        arg_int        (args, kwargs, 'packet_min')
        arg_int        (args, kwargs, 'rec_min')
        arg_int        (args, kwargs, 'byte_max')
        arg_int        (args, kwargs, 'packet_max')
        arg_int        (args, kwargs, 'rec_max')
        arg_str        (args, kwargs, 'set_file')
        arg_illegal    (args, kwargs, 'integer_ips')
        arg_illegal    (args, kwargs, 'zero_pad_ips')
        arg_str        (args, kwargs, 'sort_ips')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_illegal    (args, kwargs, 'no_columns')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_illegal    (args, kwargs, 'pager')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_remainder  (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwcount_cache([input], sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwaddrcount', [input] + rwaddrcount_standard_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_rwaddrcount(f)
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwaddrcount_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                d = values[0]
                key_cols = []
                sum_cols = []
                for c in d.columns:
                    if c in rwaddrcount_sum_cols:
                        sum_cols.append(c)
                    else:
                        key_cols.append(c)
                for v in values[1:]:
                    d = v + d
                    d = d.sum_uniq(key_cols, sum_cols)
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwaddrcount(*inputs, **kwargs):
    try:
        return proc_rwaddrcount(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwset #######################################################

RWS_SIP = "sip"
RWS_DIP = "dip"
RWS_NHIP = "nhip"

rwset_sip_cache = cache_factory('rwset<sip>', 20070405)
rwset_dip_cache = cache_factory('rwset<dip>', 20070405)
rwset_nhip_cache = cache_factory('rwset<nhip>', 20070405)

class proc_rwset(proc):
    __slots__ = ['_proc_output_type']
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwset', inputs, kwargs)
        if 'rwset_output' in kwargs:
            self._proc_output_type = kwargs['rwset_output']
            del kwargs['rwset_output']
        else:
            self._proc_output_type = RWS_SIP
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_str       (args, kwargs, 'site_config_file')
        arg_illegal   (args, kwargs, 'sip_file')
        arg_illegal   (args, kwargs, 'dip_file')
        arg_illegal   (args, kwargs, 'nhip_file')
        arg_illegal   (args, kwargs, 'print_filenames')
        arg_illegal   (args, kwargs, 'copy_input')
        arg_illegal   (args, kwargs, 'compression_method')
        arg_illegal   (args, kwargs, 'saddress')
        arg_illegal   (args, kwargs, 'daddress')
        arg_remainder (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        if self._proc_output_type == RWS_SIP:
            ent = rwset_sip_cache([input], sorted(self._proc_cmdargs))
            output_args = ['--sip-file=stdout']
        elif self._proc_output_type == RWS_DIP:
            ent = rwset_dip_cache([input], sorted(self._proc_cmdargs))
            output_args = ['--dip-file=stdout']
        elif self._proc_output_type == RWS_NHIP:
            ent = rwset_nhip_cache([input], sorted(self._proc_cmdargs))
            output_args = ['--nhip-file=stdout']
        else:
            raise ValueError("Unrecognized rwset_output option")
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'w')
                run_process('rwset', [input] + output_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                ent.commit(summary_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        if self._proc_output_type == RWS_SIP:
            ent = rwset_sip_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        elif self._proc_output_type == RWS_DIP:
            ent = rwset_dip_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        elif self._proc_output_type == RWS_NHIP:
            ent = rwset_nhip_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        else:
            raise ValueError("Unrecognized rwset_output option")
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                f = open(ent.data_file(), 'w')
                run_process('rwsettool', ['--union'] + values, stdout=f)
                f.close()
                ent.commit(summary_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwset(*inputs, **kwargs):
    try:
        if 'rwset_output' in kwargs:
            if kwargs['rwset_output'] not in (RWS_SIP, RWS_DIP, RWS_NHIP):
                raise ValueError("Unrecognized rwset_output option")
        return proc_rwset(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwsetcat ####################################################

rwsetcat_cache = cache_factory('rwsetcat', 20070405)
rwsetcat_standard_args = ['--delimited=|']

class proc_rwsetcat(proc):
    __slots__ = ()
    def __init__(self, input, kwargs):
        proc.__init__(self, 'rwsetcat', [input], kwargs)
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_illegal   (args, kwargs, 'count_ips')
        arg_illegal   (args, kwargs, 'print_ips')
        arg_illegal   (args, kwargs, 'network_structure')
        arg_illegal   (args, kwargs, 'print_statistics')
        arg_illegal   (args, kwargs, 'cidr_blocks')
        arg_flag      (args, kwargs, 'integer_ips')
        arg_illegal   (args, kwargs, 'zero_pad_ips')
        arg_illegal   (args, kwargs, 'no_columns')
        arg_illegal   (args, kwargs, 'column_separator')
        arg_illegal   (args, kwargs, 'delimited')
        arg_illegal   (args, kwargs, 'pager')
        arg_remainder (args, kwargs)
        run_in_thread(self._proc_get)
    def _proc_eval(self):
        ent = rwsetcat_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwsetcat', self._proc_inputs + 
                            rwsetcat_standard_args + self._proc_cmdargs,
                            stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_generic(f, ['ip'])
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwsetcat(input, **kwargs):
    try:
        return proc_rwsetcat(input, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwsettool ###################################################

rwsettool_union_cache = cache_factory('rwsettool<union>', 20070405)
rwsettool_intersect_cache = cache_factory('rwsettool<intersect>', 20070405)
rwsettool_difference_cache = cache_factory('rwsettool<difference>', 20070405)
rwsettool_sample_cache = cache_factory('rwsettool<sample>', 20070405)

RWS_UNION = "union"
RWS_INTERSECT = "intersect"
RWS_DIFFERENCE = "difference"
RWS_SAMPLE = "sample"

class proc_rwsettool(proc):
    __slots__ = ['_proc_output_type']
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwsettool', inputs, kwargs)
        if 'rwsettool_output' in kwargs:
            self._proc_output_type = kwargs['rwsettool_output']
            del kwargs['rwsettool_output']
        else:
            self._proc_output_type = RWS_UNION
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_illegal   (args, kwargs, 'union')
        arg_illegal   (args, kwargs, 'intersect')
        arg_illegal   (args, kwargs, 'difference')
        arg_illegal   (args, kwargs, 'sample')
        arg_int       (args, kwargs, 'size')
        arg_float     (args, kwargs, 'ratio')
        arg_illegal   (args, kwargs, 'output_path')
        arg_illegal   (args, kwargs, 'compression_method')
        arg_remainder (args, kwargs)
        run_in_thread(self._proc_get)
    def _proc_eval(self):
        if self._proc_output_type == RWS_UNION:
            ent = rwsettool_union_cache(self._proc_inputs,
                                        sorted(self._proc_cmdargs))
            output_args = ['--union']
        elif self._proc_output_type == RWS_INTERSECT:
            ent = rwsettool_intersect_cache(self._proc_inputs,
                                            sorted(self._proc_cmdargs))
            output_args = ['--intersect']
        elif self._proc_output_type == RWS_DIFFERENCE:
            ent = rwsettool_difference_cache(self._proc_inputs,
                                             sorted(self._proc_cmdargs))
            output_args = ['--difference']
        elif self._proc_output_type == RWS_SAMPLE:
            ent = rwsettool_sample_cache(self._proc_inputs,
                                         sorted(self._proc_cmdargs))
            output_args = ['--sample']
        else:
            raise ValueError("Unrecognized rwsettool_output option")
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'w')
                run_process('rwsettool', output_args + self._proc_inputs + 
                            self._proc_cmdargs, stdout=f)
                f.close()
                ent.commit(summary_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwsettool(*inputs, **kwargs):
    try:
        if 'rwsettool_output_type' in kwargs:
            if kwargs['rwsettool_output_type'] not in \
                    (RWS_UNION, RWS_INTERSECT, RWS_DIFFERENCE, RWS_SAMPLE):
                raise ValueError("Unrecognized rwsettool_output option")
        return proc_rwsettool(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwsetbuild ##################################################

### Tool: rwbag #######################################################

RWB_SIP_FLOWS = "sip-flows"
RWB_SIP_PACKETS = "sip-packets"
RWB_SIP_BYTES = "sip-bytes"
RWB_DIP_FLOWS = "dip-flows"
RWB_DIP_PACKETS = "dip-packets"
RWB_DIP_BYTES = "dip-bytes"
RWB_NHIP_FLOWS = "nhip-flows"
RWB_NHIP_PACKETS = "nhip-packets"
RWB_NHIP_BYTES = "nhip-bytes"
RWB_SPORT_FLOWS = "sport-flows"
RWB_SPORT_PACKETS = "sport-packets"
RWB_SPORT_BYTES = "sport-bytes"
RWB_DPORT_FLOWS = "dport-flows"
RWB_DPORT_PACKETS = "dport-packets"
RWB_DPORT_BYTES = "dport-bytes"
RWB_PROTO_FLOWS = "proto-flows"
RWB_PROTO_PACKETS = "proto-packets"
RWB_PROTO_BYTES = "proto-bytes"
RWB_SENSOR_FLOWS = "sensor-flows"
RWB_SENSOR_PACKETS = "sensor-packets"
RWB_SENSOR_BYTES = "sensor-bytes"
RWB_INPUT_FLOWS = "input-flows"
RWB_INPUT_PACKETS = "input-packets"
RWB_INPUT_BYTES = "input-bytes"
RWB_OUTPUT_FLOWS = "output-flows"
RWB_OUTPUT_PACKETS = "output-packets"
RWB_OUTPUT_BYTES = "output-bytes"

rwbag_cache = cache_factory('rwbag', 20070405)

class proc_rwbag(proc):
    __slots__ = ['_proc_output_type']
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwbag', inputs, kwargs)
        if 'rwbag_output' in kwargs:
            self._proc_output_type = kwargs['rwbag_output']
            del kwargs['rwbag_output']
        else:
            self._proc_output_type = RWB_SIP_FLOWS
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_str       (args, kwargs, 'site_config_file')
        arg_illegal   (args, kwargs, 'compression_method')
        arg_illegal   (args, kwargs, 'sip_flows')
        arg_illegal   (args, kwargs, 'sip_packets')
        arg_illegal   (args, kwargs, 'sip_bytes')
        arg_illegal   (args, kwargs, 'dip_flows')
        arg_illegal   (args, kwargs, 'dip_packets')
        arg_illegal   (args, kwargs, 'dip_bytes')
        arg_illegal   (args, kwargs, 'nhip_flows')
        arg_illegal   (args, kwargs, 'nhip_packets')
        arg_illegal   (args, kwargs, 'nhip_bytes')
        arg_illegal   (args, kwargs, 'sport_flows')
        arg_illegal   (args, kwargs, 'sport_packets')
        arg_illegal   (args, kwargs, 'sport_bytes')
        arg_illegal   (args, kwargs, 'dport_flows')
        arg_illegal   (args, kwargs, 'dport_packets')
        arg_illegal   (args, kwargs, 'dport_bytes')
        arg_illegal   (args, kwargs, 'proto_flows')
        arg_illegal   (args, kwargs, 'proto_packets')
        arg_illegal   (args, kwargs, 'proto_bytes')
        arg_illegal   (args, kwargs, 'sensor_flows')
        arg_illegal   (args, kwargs, 'sensor_packets')
        arg_illegal   (args, kwargs, 'sensor_bytes')
        arg_illegal   (args, kwargs, 'input_flows')
        arg_illegal   (args, kwargs, 'input_packets')
        arg_illegal   (args, kwargs, 'input_bytes')
        arg_illegal   (args, kwargs, 'output_flows')
        arg_illegal   (args, kwargs, 'output_packets')
        arg_illegal   (args, kwargs, 'output_bytes')
        arg_illegal   (args, kwargs, 'print_filenames')
        arg_illegal   (args, kwargs, 'copy_input')
        arg_illegal   (args, kwargs, 'legacy-help')
        arg_remainder (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwbag_cache([self._proc_output_type] + [input],
                          sorted(self._proc_cmdargs))
        output_args = ['--' + self._proc_output_type + '=stdout']
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'w')
                run_process('rwbag', [input] + output_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                ent.commit(summary_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwbag_cache([self._proc_output_type] + self._proc_inputs,
                          sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                f = open(ent.data_file(), 'w')
                run_process('rwbagtool', ['--add'] + values, stdout=f)
                f.close()
                ent.commit(summary_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwbag(*inputs, **kwargs):
    try:
        if 'rwbag_output' in kwargs:
            if kwargs['rwbag_output'] not in \
                    (RWB_SIP_FLOWS, RWB_SIP_PACKETS, RWB_SIP_BYTES,
                     RWB_DIP_FLOWS, RWB_DIP_PACKETS, RWB_DIP_BYTES,
                     RWB_NHIP_FLOWS, RWB_NHIP_PACKETS, RWB_NHIP_BYTES,
                     RWB_SPORT_FLOWS, RWB_SPORT_PACKETS, RWB_SPORT_BYTES,
                     RWB_DPORT_FLOWS, RWB_DPORT_PACKETS, RWB_DPORT_BYTES,
                     RWB_PROTO_FLOWS, RWB_PROTO_PACKETS, RWB_PROTO_BYTES,
                     RWB_SENSOR_FLOWS, RWB_SENSOR_PACKETS, RWB_SENSOR_BYTES,
                     RWB_INPUT_FLOWS, RWB_INPUT_PACKETS, RWB_INPUT_BYTES,
                     RWB_OUTPUT_FLOWS, RWB_OUTPUT_PACKETS, RWB_OUTPUT_BYTES):
                raise ValueError("Unrecognized rwbag_output option")
        return proc_rwbag(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwbagcat ####################################################

rwbagcat_cache = cache_factory('rwbagcat', 20070405)
rwbagcat_standard_args = ['--delimited=|']

class proc_rwbagcat(proc):
    __slots__ = ()
    def __init__(self, input, kwargs):
        proc.__init__(self, 'rwbagcat', [input], kwargs)
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_illegal   (args, kwargs, 'network_structure')
        arg_illegal   (args, kwargs, 'bin_ips')
        arg_illegal   (args, kwargs, 'stats')
        arg_illegal   (args, kwargs, 'tree_stats')
        arg_int       (args, kwargs, 'mincount')
        arg_int       (args, kwargs, 'maxcount')
        arg_str       (args, kwargs, 'minkey')
        arg_str       (args, kwargs, 'maxkey')
        arg_flag      (args, kwargs, 'zero_counts')
        arg_illegal   (args, kwargs, 'output')
        arg_flag      (args, kwargs, 'integer_keys')
        arg_illegal   (args, kwargs, 'zero_page_ips')
        arg_illegal   (args, kwargs, 'no_columns')
        arg_illegal   (args, kwargs, 'column_separator')
        arg_illegal   (args, kwargs, 'delimited')
        arg_illegal   (args, kwargs, 'pager')
        arg_remainder (args, kwargs)
        run_in_thread(self._proc_get)
    def _proc_eval(self):
        ent = rwbagcat_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwbagcat', self._proc_inputs + 
                            rwbagcat_standard_args + self._proc_cmdargs,
                            stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_generic(f, ['key', 'count', '_'])
                d = Dataset(('key', 'count'), _data=(d['key'],
                                                     [int(x) for x
                                                      in d['count']]))
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwbagcat(input, **kwargs):
    try:
        return proc_rwbagcat(input, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwbagtool ###################################################

rwbagtool_add_cache = cache_factory('rwbagtool<add>', 20070405)
rwbagtool_divide_cache = cache_factory('rwbagtool<divide>', 20070405)
rwbagtool_subtract_cache = cache_factory('rwbagtool<subtract>', 20070405)
rwbagtool_coverset_cache = cache_factory('rwbagtool<coverset>', 20070405)
rwbagtool_invert_cache = cache_factory('rwbagtool<invert>', 20070405)

RWB_ADD = "add"
RWB_DIVIDE = "divide"
RWB_SUBTRACT = "subtract"
RWB_COVERSET = "coverset"
RWB_INVERT = "invert"

class proc_rwbagtool(proc):
    __slots__ = ['_proc_output_type']
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwbagtool', inputs, kwargs)
        if 'rwbagtool_output' in kwargs:
            self._proc_output_type = kwargs['rwbagtool_output']
            del kwargs['rwbagtool_output']
        else:
            self._proc_output_type = RWB_ADD
        args = self._proc_cmdargs
        arg_illegal   (args, kwargs, 'help')
        arg_illegal   (args, kwargs, 'version')
        arg_illegal   (args, kwargs, 'add')
        arg_illegal   (args, kwargs, 'intersect')
        arg_illegal   (args, kwargs, 'complement_intersect')
        arg_illegal   (args, kwargs, 'divide')
        arg_illegal   (args, kwargs, 'subtract')
        arg_illegal   (args, kwargs, 'coverset')
        arg_illegal   (args, kwargs, 'invert')
        arg_int       (args, kwargs, 'mincount')
        arg_int       (args, kwargs, 'maxcount')
        arg_str       (args, kwargs, 'minkey')
        arg_str       (args, kwargs, 'maxkey')
        arg_illegal   (args, kwargs, 'output')
        arg_illegal   (args, kwargs, 'compression_method')
        arg_remainder (args, kwargs)
        run_in_thread(self._proc_get)
    def _proc_eval(self):
        if self._proc_output_type == RWB_ADD:
            ent = rwbagtool_add_cache(self._proc_inputs,
                                      sorted(self._proc_cmdargs))
            output_args = ['--add']
        elif self._proc_output_type == RWB_DIVIDE:
            ent = rwsettool_divide_cache(self._proc_inputs,
                                         sorted(self._proc_cmdargs))
            output_args = ['--divide']
        elif self._proc_output_type == RWB_SUBTRACT:
            ent = rwsettool_subtract_cache(self._proc_inputs,
                                           sorted(self._proc_cmdargs))
            output_args = ['--subtract']
        elif self._proc_output_type == RWB_COVERSET:
            ent = rwbagtool_coverset_cache(self._proc_inputs,
                                           sorted(self._proc_cmdargs))
            output_args = ['--coverset']
        elif self._proc_output_type == RWB_INVERT:
            ent = rwbagtool_invert_cache(self._proc_inputs,
                                         sorted(self._proc_cmdargs))
            output_args = ['--invert']
        else:
            raise ValueError("Unrecognized rwsettool_output option")
        try:
            if not ent.is_cached():
                f = open(ent.data_file(), 'w')
                run_process('rwbagtool', output_args + self._proc_inputs + 
                            self._proc_cmdargs, stdout=f)
                f.close()
                ent.commit(summary_merged_expire_time())
            return ent.data_file()
        except:
            ent.rollback()
            raise

def rwbagtool(*inputs, **kwargs):
    try:
        if 'rwbagtool_output_type' in kwargs:
            if kwargs['rwbagtool_output_type'] not in \
                    (RWB_ADD, RWB_DIVIDE, RWB_SUBTRACT,
                     RWB_COVERSET, RWB_INVERT):
                raise ValueError("Unrecognized rwbagtool_output option")
        return proc_rwbagtool(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwbagbuild ##################################################

### Tool: nafalize #####################################################
### Tool: nafscii #####################################################

### Tool: rwuniq #######################################################

rwuniq_cache = cache_factory('rwuniq', 20070405)
rwuniq_sum_cols = [ 'bytes', 'packets', 'flows' ]
rwuniq_standard_args = [ '--epoch-time', '--delimited=|' ]

class proc_rwuniq(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwuniq', inputs, kwargs)
        args = self._proc_cmdargs
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'version')
        arg_str        (args, kwargs, 'site_config_file')
        arg_str_list   (args, kwargs, 'fields')
        arg_flag       (args, kwargs, 'presorted_input')
        arg_flag       (args, kwargs, 'all_counts')
        arg_flag_value (args, kwargs, 'bytes')
        arg_flag_value (args, kwargs, 'packets')
        arg_flag_value (args, kwargs, 'flows')
        arg_flag       (args, kwargs, 'stime')
        arg_flag       (args, kwargs, 'etime')
        arg_flag_value (args, kwargs, 'sip_distinct')
        arg_flag_value (args, kwargs, 'dip_distinct')
        arg_str_many   (args, kwargs, 'dynamic_library')
        arg_flag_value (args, kwargs, 'bin_time')
        arg_illegal    (args, kwargs, 'epoch_time')
        arg_flag       (args, kwargs, 'integer_ips')
        arg_flag       (args, kwargs, 'zero_pad_ips')
        arg_flag       (args, kwargs, 'integer_sensors')
        arg_illegal    (args, kwargs, 'no_titles')
        arg_illegal    (args, kwargs, 'no_columns')
        arg_illegal    (args, kwargs, 'column_separator')
        arg_illegal    (args, kwargs, 'delimited')
        arg_illegal    (args, kwargs, 'copy_input')
        arg_illegal    (args, kwargs, 'print_filenames')
        arg_illegal    (args, kwargs, 'output_path')
        arg_illegal    (args, kwargs, 'pager')
        arg_illegal    (args, kwargs, 'legacy_timestamps')
        arg_str        (args, kwargs, 'pmap_file')
        arg_remainder  (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwuniq_cache([input], sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwuniq', [input] + rwuniq_standard_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_rwuniq(f)
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwuniq_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                d = values[0]
                key_cols = []
                sum_cols = []
                for c in d.columns:
                    if c in rwuniq_sum_cols:
                        sum_cols.append(c)
                    else:
                        key_cols.append(c)
                for v in values[1:]:
                    d = v + d
                    d = d.sum_uniq(key_cols, sum_cols)
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwuniq(*inputs, **kwargs):
    try:
        return proc_rwuniq(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwcut ########################################################

rwcut_cache = cache_factory('rwcut', 20070405)
rwcut_standard_args = [ '--epoch-time', '--delimited=|' ]

class proc_rwcut(proc):
    __slots__ = ()
    def __init__(self, inputs, kwargs):
        proc.__init__(self, 'rwcut', inputs, kwargs)
        args = self._proc_cmdargs
        arg_illegal    (args, kwargs, 'help')
        arg_illegal    (args, kwargs, 'version')
        arg_str        (args, kwargs, 'site_config_file')
        arg_str_list   (args, kwargs, 'fields')
        arg_flag       (args, kwargs, 'integer_ips')
        arg_flag       (args, kwargs, 'zero_pad_ips')
        arg_int        (args, kwargs, 'num_recs')
        arg_int        (args, kwargs, 'start_rec_num')
        arg_int        (args, kwargs, 'end_rec_num')
        arg_illegal    (args, kwargs, 'no_title')
        arg_illegal    (args, kwargs, 'no_columns')
        arg_illegal    (args, kwargs, 'column_separator')
        arg_illegal    (args, kwargs, 'delimited')
        arg_illegal    (args, kwargs, 'epoch_time')
        arg_flag       (args, kwargs, 'icmp_type_and_code')
        arg_str_many   (args, kwargs, 'dynamic_library')
        arg_flag       (args, kwargs, 'integer_sensors')
        arg_illegal    (args, kwargs, 'dry_run')
        arg_illegal    (args, kwargs, 'print_filenames')
        arg_illegal    (args, kwargs, 'copy_input')
        arg_illegal    (args, kwargs, 'output_path')
        arg_illegal    (args, kwargs, 'pager')
        arg_illegal    (args, kwargs, 'legacy_timestamps')
        arg_str        (args, kwargs, 'pmap_file')
        arg_remainder  (args, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        ent = rwcut_cache([input], sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                # First write the data to the temp file
                f = open(ent.data_file(), 'w')
                run_process('rwcut', [input] + rwcut_standard_args +
                            self._proc_cmdargs, stdout=f)
                f.close()
                f = open(ent.data_file(), 'r')
                d = parse_rwcut(f)
                f.close()
                os.remove(ent.data_file())
                ent.commit(summary_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise
    def _proc_merge(self, values):
        ent = rwcut_cache(self._proc_inputs, sorted(self._proc_cmdargs))
        try:
            if not ent.is_cached():
                if len(values) == 1:
                    return values[0]
                d = values[0]
                for v in values[1:]:
                    d = v + d
                ent.commit(summary_merged_expire_time(), d)
            return ent.value()
        except:
            ent.rollback()
            raise

def rwcut(*inputs, **kwargs):
    try:
        return proc_rwcut(inputs, kwargs)
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

### Tool: rwfilter #####################################################

RWF_PASS = 'pass'
RWF_PASS_FAIL = 'pass+fail'
RWF_PASS_ALL = 'pass+all'
RWF_PASS_FAIL_ALL = 'pass+fail+all'
RWF_FAIL = 'fail'
RWF_FAIL_ALL = 'fail+all'
RWF_ALL = 'all'

import re

re_silk_datetime = re.compile(r"""
          (?P<year>\d\d\d\d)
    /     (?P<month>\d\d)
    /     (?P<day>\d\d)
(?: [ :T] (?P<hour>\d\d)
(?: :     (?P<minute>\d\d)
(?: :     (?P<second>\d\d)
(?: \.    (?P<fracsecond>\d+) )? )? )? )?
""", re.VERBOSE)

from datetime import datetime, timedelta

rwfilter_pass_cache = cache_factory('rwfilter<pass>', 20070405)
rwfilter_fail_cache = cache_factory('rwfilter<fail>', 20070405)
rwfilter_all_cache = cache_factory('rwfilter<all>', 20070405)

def arg_rwfilter(args, kwargs):
    arg_illegal        (args, kwargs, 'help')
    arg_illegal        (args, kwargs, 'version')
    arg_illegal        (args, kwargs, 'compression_method')
    arg_illegal        (args, kwargs, 'dry_run')
    arg_illegal        (args, kwargs, 'input_pipe')
    arg_illegal        (args, kwargs, 'xargs')
    arg_int            (args, kwargs, 'max_pass_records')
    arg_illegal        (args, kwargs, 'print_filenames')
    arg_str_many       (args, kwargs, 'dynamic_library')
    arg_illegal        (args, kwargs, 'pass_destination')
    arg_illegal        (args, kwargs, 'fail_destination')
    arg_illegal        (args, kwargs, 'all_destination')
    arg_illegal        (args, kwargs, 'print_statistics')
    arg_illegal        (args, kwargs, 'print_volume_statistics')
    arg_str            (args, kwargs, 'class')
    arg_str_list       (args, kwargs, 'type')
    arg_str_list       (args, kwargs, 'sensors')
    arg_illegal        (args, kwargs, 'start_date')
    arg_illegal        (args, kwargs, 'end_date')
    arg_illegal        (args, kwargs, 'print_missing_files')
    arg_str            (args, kwargs, 'data_rootdir')
    arg_str            (args, kwargs, 'site_config_file')
    arg_datetime_range (args, kwargs, 'stime')
    arg_datetime_range (args, kwargs, 'etime')
    arg_datetime_range (args, kwargs, 'active_time')
    arg_int_list       (args, kwargs, 'duration')
    arg_int_list       (args, kwargs, 'sport')
    arg_int_list       (args, kwargs, 'dport')
    arg_int_list       (args, kwargs, 'aport')
    arg_int_list       (args, kwargs, 'protocol')
    arg_int_list       (args, kwargs, 'icmp_type')
    arg_int_list       (args, kwargs, 'icmp_code')
    arg_int_list       (args, kwargs, 'bytes')
    arg_int_list       (args, kwargs, 'packets')
    arg_int_list       (args, kwargs, 'bytes_per_packet')
    arg_wild_ip        (args, kwargs, 'saddress')
    arg_wild_ip        (args, kwargs, 'daddress')
    arg_wild_ip        (args, kwargs, 'any_address')
    arg_wild_ip        (args, kwargs, 'next_hop_id')
    arg_wild_ip        (args, kwargs, 'not_saddress')
    arg_wild_ip        (args, kwargs, 'not_daddress')
    arg_wild_ip        (args, kwargs, 'not_any_address')
    arg_wild_ip        (args, kwargs, 'not_next_hop_id')
    arg_wild_ip        (args, kwargs, 'sipset')
    arg_wild_ip        (args, kwargs, 'dipset')
    arg_wild_ip        (args, kwargs, 'anyset')
    arg_wild_ip        (args, kwargs, 'nhipset')
    arg_wild_ip        (args, kwargs, 'not_sipset')
    arg_wild_ip        (args, kwargs, 'not_dipset')
    arg_wild_ip        (args, kwargs, 'not_anyset')
    arg_wild_ip        (args, kwargs, 'not_nhipset')
    arg_int_list       (args, kwargs, 'input_index')
    arg_int_list       (args, kwargs, 'output_index')
    arg_str            (args, kwargs, 'tcp_flags')
    arg_bool           (args, kwargs, 'fin_flag')
    arg_bool           (args, kwargs, 'syn_flag')
    arg_bool           (args, kwargs, 'rst_flag')
    arg_bool           (args, kwargs, 'psh_flag')
    arg_bool           (args, kwargs, 'ack_flag')
    arg_bool           (args, kwargs, 'urg_flag')
    arg_bool           (args, kwargs, 'ece_flag')
    arg_bool           (args, kwargs, 'cwr_flag')
    arg_str            (args, kwargs, 'flags-all')
    arg_str_list       (args, kwargs, 'scc')
    arg_str_list       (args, kwargs, 'dcc')
    arg_str_list       (args, kwargs, 'sval')
    arg_str_list       (args, kwargs, 'dval')
    arg_remainder(args, kwargs)

class proc_rwfilter_hours(proc):
    __slots__ = '_proc_output_type', '_proc_start_date', '_proc_end_date', \
        '_proc_part_kwargs'
    def __init__(self, kwargs):
        # No inputs, or we wouldn't be here
        if 'rwfilter_output' in kwargs:
            self._proc_output_type = kwargs['rwfilter_output']
            del kwargs['rwfilter_output']
        else:
            self._proc_output_type = RWF_PASS
        # Now we know who we are
        proc.__init__(self, 'rwfilter', [], kwargs, tag=self._proc_output_type)
        if 'start_date' in kwargs:
            start_date = kwargs['start_date']
            del kwargs['start_date']
        else:
            raise silk.SilkError("'start_date' argument required " +
                                 "when no inputs are given")
        if 'end_date' in kwargs:
            end_date = kwargs['end_date']
            del kwargs['end_date']
        else:
            end_date = start_date
        part_kwargs = dict(kwargs)
        # Try parsing the dates
        m = re_silk_datetime.match(start_date)
        if not m or len(start_date) <> len(m.group()):
            raise silk.SilkError(("Value %s could not be parsed for "+
                                  "'start_date'") % `start_date`)
        d = m.groupdict(None)
        start_datetime = datetime(int(d['year']), int(d['month']),
                                  int(d['day']), int(d['hour'] or '0'))
        m = re_silk_datetime.match(end_date)
        if not m or len(end_date) <> len(m.group()):
            raise silk.SilkError(("Value %s could not be parsed for "+
                                  "'end_date'") % `end_date`)
        d = m.groupdict(None)
        end_datetime = datetime(int(d['year']), int(d['month']),
                                int(d['day']), int(d['hour'] or '23'),
                                59, 59, 999999)
        self._proc_start_date = start_datetime
        self._proc_end_date = end_datetime
        # Get the non-date arguments
        arg_rwfilter(self._proc_cmdargs, kwargs)
        # Set up the parts
        one_hour = timedelta(hours=1)
        self._proc_parts = []
        t = self._proc_start_date
        while t <= self._proc_end_date:
            self._proc_parts.append(proc_part(self, t.strftime('%Y/%m/%d:%H'),
                                              kwargs=part_kwargs))
            t += one_hour
        # Start the engines
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        generating = False
        pass_txn = None
        fail_txn = None
        all_txn = None
        kwargs = self._proc_kwargs
        cmdargs = list(self._proc_cmdargs)
        try:
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn = rwfilter_pass_cache([input],
                                               sorted(self._proc_cmdargs))
                if not pass_txn.is_cached():
                    cmdargs.append('--pass-destination=%s' %
                                   pass_txn.data_file())
                    generating = True
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn = rwfilter_fail_cache([input],
                                               sorted(self._proc_cmdargs))
                if not fail_txn.is_cached():
                    cmdargs.append('--fail-destination=%s' %
                                   fail_txn.data_file())
                    generating = True
            if self._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                all_txn = rwfilter_all_cache([input],
                                             sorted(self._proc_cmdargs))
                if not all_txn.is_cached():
                    cmdargs.append('--all-destination=%s' %
                                   all_txn.data_file())
                    generating = True
            if generating:
                run_process('rwfilter', ['--start-date=%s' % input] + cmdargs)
            result = {}
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn.commit(raw_expire_time())
                result['pass'] = pass_txn.data_file()
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn.commit(raw_expire_time())
                result['fail'] = fail_txn.data_file()
            if self._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                all_txn.commit(raw_expire_time())
                result['all'] = all_txn.data_file()
            return result
        except:
            if pass_txn is not None: pass_txn.rollback()
            if fail_txn is not None: fail_txn.rollback()
            if all_txn is not None: all_txn.rollback()
            raise
    def _proc_merge(self, values):
        if len(values) == 1:
            return values[0]
        result = {}
        pass_txn = None
        fail_txn = None
        all_txn = None    
        try:
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn = rwfilter_pass_cache(
                    [],
                    self._proc_start_date.strftime('%Y/%m/%d:%H'),
                    self._proc_end_date.strftime('%Y/%m/%d:%H'),
                    sorted(self._proc_cmdargs))
                if not pass_txn.is_cached():
                    f = open(pass_txn.data_file(), 'wb')
                    run_process('rwcat', [v['pass'] for v in values], stdout=f)
                    f.close()
                    pass_txn.commit(raw_merged_expire_time())
                result['pass'] = pass_txn.data_file()
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn = rwfilter_fail_cache(
                    [],
                    self._proc_start_date.strftime('%Y/%m/%d:%H'),
                    self._proc_end_date.strftime('%Y/%m/%d:%H'),
                    sorted(self._proc_cmdargs))
                if not fail_txn.is_cached():
                    f = open(fail_txn.data_file(), 'wb')
                    run_process('rwcat', [v['fail'] for v in values], stdout=f)
                    f.close()
                    fail_txn.commit(raw_merged_expire_time())
                result['fail'] = fail_txn.data_file()
            if self._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                all_txn = rwfilter_all_cache(
                    [],
                    self._proc_start_date.strftime('%Y/%m/%d:%H'),
                    self._proc_end_date.strftime('%Y/%m/%d:%H'),
                    sorted(self._proc_cmdargs))
                if not all_txn.is_cached():
                    f = open(all_txn.data_file(), 'wb')
                    run_process('rwcat', [v['all'] for v in values], stdout=f)
                    f.close()
                    all_txn.commit(raw_merged_expire_time())
                result['all'] = all_txn.data_file()
            return result
        except:
            if pass_txn is not None: pass_txn.rollback()
            if fail_txn is not None: fail_txn.rollback()
            if all_txn is not None: all_txn.rollback()
            raise

class proc_rwfilter(proc):
    __slots__ = '_proc_output_type'
    def __init__(self, inputs, kwargs):
        if 'rwfilter_output' in kwargs:
            self._proc_output_type = kwargs['rwfilter_output']
            del kwargs['rwfilter_output']
        else:
            self._proc_output_type = RWF_PASS
        proc.__init__(self, 'rwfilter', inputs, kwargs,
                      tag=self._proc_output_type)
        arg_rwfilter   (self._proc_cmdargs, kwargs)
        arg_remainder  (self._proc_cmdargs, kwargs)
        self._proc_make_input_parts()
        run_in_thread(self._proc_get_parts)
    def _proc_eval_part(self, input):
        generating = False
        pass_txn = None
        fail_txn = None
        cmdargs = list(self._proc_cmdargs)
        try:
            kwargs = dict(self._proc_kwargs)
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn = rwfilter_pass_cache([input],
                                               sorted(self._proc_cmdargs))
                if not pass_txn.is_cached():
                    cmdargs.append('--pass-destination=%s' %
                                   pass_txn.data_file())
                    generating = True
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn = rwfilter_fail_cache([input],
                                               sorted(self._proc_cmdargs))
                if not fail_txn.is_cached():
                    cmdargs.append('--fail-destination=%s' %
                                   fail_txn.data_file())
                    generating = True
            # ALL is handled by re-using the input
            if generating:
                run_process('rwfilter', [str(input)] + cmdargs)
            result = {}
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn.commit(raw_expire_time())
                result['pass'] = pass_txn.data_file()
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn.commit(raw_expire_time())
                result['fail'] = fail_txn.data_file()
            if self._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                result['all'] = str(input)
            return result
        except:
            if pass_txn is not None: pass_txn.rollback()
            if fail_txn is not None: fail_txn.rollback()
            raise
    def _proc_merge(self, values):
        if len(values) == 1:
            return values[0]
        result = {}
        pass_txn = None
        fail_txn = None
        all_txn = None    
        try:
            if self._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                          RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
                pass_txn = rwfilter_pass_cache(self._proc_inputs, 
                                               sorted(self._proc_cmdargs))
                if not pass_txn.is_cached():
                    f = open(pass_txn.data_file(), 'wb')
                    run_process('rwcat', [v['pass'] for v in values], stdout=f)
                    f.close()
                    pass_txn.commit(raw_merged_expire_time())
                result['pass'] = pass_txn.data_file()
            if self._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                fail_txn = rwfilter_fail_cache(self._proc_inputs,
                                               sorted(self._proc_cmdargs))
                if not fail_txn.is_cached():
                    f = open(fail_txn.data_file(), 'wb')
                    run_process('rwcat', [v['fail'] for v in values], stdout=f)
                    f.close()
                    fail_txn.commit(raw_merged_expire_time())
                result['fail'] = fail_txn.data_file()
            if self._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                          RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
                all_txn = rwfilter_fail_cache(self._proc_inputs,
                                              sorted(self._proc_cmdargs))
                if not all_txn.is_cached():
                    f = open(all_txn.data_file(), 'wb')
                    run_process('rwcat', [v['all'] for v in values], stdout=f)
                    f.close()
                    all_txn.commit(raw_merged_expire_time())
                result['all'] = all_txn.data_file()
            return result
        except:
            if pass_txn is not None: pass_txn.rollback()
            if fail_txn is not None: fail_txn.rollback()
            if all_txn is not None: all_txn.rollback()
            raise

class proc_rwfilter_part(proc):
    __slots__ = '_proc_parent'
    def __init__(self, parent, tag):
        proc.__init__(self, parent._proc_name, parent._proc_inputs,
                      parent._proc_kwargs, tag, _cond=parent._proc_cond)
        self._proc_parent = parent
        self._proc_parts = []
        for p in parent._proc_parts:
            self._proc_parts.append(proc_rwfilter_part_part(p, tag))
        run_in_thread(self._proc_get_parts)
    def _proc_eval(self):
        return self._proc_parent[self._proc_tag]

class proc_rwfilter_part_part(proc):
    __slots__ = '_proc_parent_part'
    def __init__(self, parent_part, tag):
        proc.__init__(self, parent_part._proc_name, parent_part._proc_inputs,
                      parent_part._proc_kwargs, tag,
                      _cond=parent_part._proc_cond)
        self._proc_parent_part = parent_part
    def _proc_eval(self):
        return self._proc_parent_part[self._proc_tag]

def rwfilter(*inputs, **kwargs):
    try:
        if inputs:
            # There is input, just do it
            x = proc_rwfilter(inputs, kwargs)
        else:
            # No input, use fglob support
            x = proc_rwfilter_hours(kwargs)
        result = ()
        if x._proc_output_type in (RWF_PASS, RWF_PASS_FAIL,
                                   RWF_PASS_ALL, RWF_PASS_FAIL_ALL):
            result += (proc_rwfilter_part(x, 'pass'),)
        if x._proc_output_type in (RWF_FAIL, RWF_PASS_FAIL,
                                   RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
            result += (proc_rwfilter_part(x, 'fail'),)
        if x._proc_output_type in (RWF_ALL, RWF_PASS_ALL,
                                   RWF_FAIL_ALL, RWF_PASS_FAIL_ALL):
            result += (proc_rwfilter_part(x, 'all'),)
        if len(result) == 1:
            return result[0]
        return result
    except:
        if silk.DEBUG_ORIGINAL_EXCEPTIONS: raise
        error = sys.exc_info()[1]
        raise error

__all__ = ('rwcat', 'rwfilter', 'rwsort',
           'RWF_PASS', 'RWF_FAIL', 'RWF_ALL', 'RWF_PASS_FAIL', 'RWF_PASS_ALL',
           'RWF_FAIL_ALL', 'RWF_PASS_FAIL_ALL',
           'rwuniq', 'rwtotal', 'rwcount', 'rwcut',
           'rwset', 'rwsetcat', 'rwsettool',
           'RWS_SIP', 'RWS_DIP', 'RWS_NHIP',
           'RWS_UNION', 'RWS_INTERSECT', 'RWS_DIFFERENCE', 'RWS_SAMPLE',
           'rwbag', 'rwbagcat', 'rwbagtool',
           'RWB_SIP_FLOWS', 'RWB_SIP_PACKETS', 'RWB_SIP_BYTES',
           'RWB_DIP_FLOWS', 'RWB_DIP_PACKETS', 'RWB_DIP_BYTES',
           'RWB_NHIP_FLOWS', 'RWB_NHIP_PACKETS', 'RWB_NHIP_BYTES',
           'RWB_SPORT_FLOWS', 'RWB_SPORT_PACKETS', 'RWB_SPORT_BYTES',
           'RWB_DPORT_FLOWS', 'RWB_DPORT_PACKETS', 'RWB_DPORT_BYTES',
           'RWB_PROTO_FLOWS', 'RWB_PROTO_PACKETS', 'RWB_PROTO_BYTES',
           'RWB_SENSOR_FLOWS', 'RWB_SENSOR_PACKETS', 'RWB_SENSOR_BYTES',
           'RWB_INPUT_FLOWS', 'RWB_INPUT_PACKETS', 'RWB_INPUT_BYTES',
           'RWB_OUTPUT_FLOWS', 'RWB_OUTPUT_PACKETS', 'RWB_OUTPUT_BYTES',
           'RWB_ADD', 'RWB_DIVIDE', 'RWB_SUBTRACT', 'RWB_COVERSET',
           'RWB_INVERT',
           'set_raw_merged_cache_dur', 'set_raw_cache_dur',
           'set_summary_merged_cache_dur', 'set_summary_cache_dur',
           'proc_label', 'proc_progress', 'proc_wait', 'proc_file_inputs'
           )
