/*
** Copyright (C) 2003-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: flowcap.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/libflowsource.h>
#include <silk/skheader.h>
#include <silk/skthread.h>      /* MUTEX_LOCK,MUTEX_UNLOCK */
#include <silk/sktimer.h>
#include <sys/un.h>
#ifdef SK_HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#endif
#include "flowcap.h"


/* TYPEDEFS AND DEFINES */

/* Whether a probe's priority is high */
#define FC_PROBE_IS_HIGH_PRIORITY(probe)        \
    (skpcProbeGetPriority(probe) > 50)

/*
 *  Specify the maximum size (in terms of RECORDS) of the buffer used
 *  to hold records that have been read from the flow-source but not
 *  yet processed.  This value is the number of records as read from
 *  the wire (e.g., PDUs for a NetFlow v5 probe) per PROBE.  The
 *  maximum memory per NetFlow v5 probe will be BUF_REC_COUNT * 1464.
 *  The maximum memory per IPFIX or NetFlow v9 probe will be
 *  BUF_REC_COUNT * 52 (or BUF_REC_COUNT * 88 for IPv6-enabled SiLK).
 *  If records are processed as quickly as they are read, the normal
 *  memory use per probe will be CIRCBUF_CHUNK_MAX_SIZE bytes.
 */
#define BUF_REC_COUNT  32768

/* We need big socket buffers.  Let's try 8MB */
#define BIG_SOCK_BUF  (1024 * 1024 * 8)

/* values used to set flags used by closeFile() */
#define FC_REOPEN     (1 << 0)
#define FC_TIMED_OUT  (1 << 1)



typedef struct flowcap_reader_st {
    /* the source of data that this reader captures */
    union {
        pduSource_t   pdu;
#if SK_ENABLE_IPFIX
        ipfixSource_t ipfix;
#endif
    }                   source;

    /* probe that this reader is capturing */
    const skpc_probe_t *probe;

    /* the skStream that is used for writing */
    skstream_t         *ios;

    /* base name of the file; a pointer into 'path' */
    char               *name;

    /* complete path to file */
    char                path[PATH_MAX];

    /* close timer */
    skTimer_t           timer;

    /* reader lock */
    pthread_mutex_t     mutex;

    /* reader thread */
    pthread_t           reader_thread;

    /* time when the file was opened */
    time_t              start_time;

    /* number of records written to current file */
    uint32_t            records;

    /* whether this file is due to be closed. */
    unsigned            close       : 1;

    /* whether this file is due to be closed due to getting too large. */
    unsigned            overfull    : 1;

    /* whether this file is in the process of being closed---protect
     * against size limit and time limit firing simultaneously. */
    unsigned            closing     : 1;

    /* is the source object valid */
    unsigned            valid_source: 1;

    /* is this thread running? */
    unsigned            running     : 1;
} flowcap_reader_t;


/* EXPORTED VARIABLES */

/* Where to write files */
const char *destination_dir = NULL;

/* Compression method for output files */
sk_compmethod_t comp_method;

/* To ensure records are sent along in a timely manner, the files are
 * closed when a timer fires or once they get to a certain size.
 * These variables define those values. */
uint32_t write_timeout = 60;
uint32_t max_file_size = 0;

/* Timer base (0 if none) from which we calculate timeouts */
sktime_t clock_time = 0;

/* Amount of disk space to allow for a new file when determining
 * whether there is disk space available.  This will be max_file_size
 * plus some overhead should the compressed data be larger than the
 * raw data. */
uint64_t alloc_file_size = 0;

/* The version of flowcap files to produce */
uint8_t flowcap_version = FC_VERSION_DEFAULT;


#ifdef SK_HAVE_STATVFS
/* leave at least this much free space on the disk; specified by
 * --freespace-minimum.  Gets set to DEFAULT_FREESPACE_MINIMUM */
int64_t freespace_minimum = -1;

/* take no more that this amount of the disk; as a percentage.
 * specified by --space-maximum-percent */
double space_maximum_percent = DEFAULT_SPACE_MAXIMUM_PERCENT;
#endif /* SK_HAVE_STATVFS */


/* LOCAL VARIABLES */

/* Reader shut down flag (0 == stop) */
static volatile uint8_t reading;

/* Indicator of whether flowcap is in the process of shutting down */
static volatile int shuttingDown = 0;

/* Main thread id. */
static pthread_t main_thread;

/* The array of readers, and the array's size. */
static flowcap_reader_t *fc_readers;
static size_t num_fc_readers;


/* LOCAL FUNCTION PROTOTYPES */

static void appTeardown(void);
static skTimerRepeat_t timerHandler(
    void               *vreader);
static void freeReaders(void);
static void openFile(
    flowcap_reader_t   *reader);
static void openFileBase(
    flowcap_reader_t   *reader);
static void closeFile(
    flowcap_reader_t   *reader,
    int                 flags);
static void closeFileBase(
    flowcap_reader_t   *reader,
    int                 flags);
static uint32_t writeHeader(
    flowcap_reader_t   *reader,
    int                 fd,
    const char         *pathname);
static int  startReaders(void);
static void stopReaders(void);
static void *readerMain(
    void               *vreader);

#if SK_HAVE_STATVFS
static void checkDiskSpace(void);
#else
#  define checkDiskSpace()
#endif


/* FUNCTION DEFINITIONS */

/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
    static uint8_t teardownFlag = 0;

    MUTEX_LOCK(&mutex);

    if (teardownFlag) {
        MUTEX_UNLOCK(&mutex);
        return;
    }
    teardownFlag = 1;
    MUTEX_UNLOCK(&mutex);

    NOTICEMSG("Shutting down...");
    shuttingDown = 1;

    stopReaders();
    freeReaders();

    skpcTeardown();
    skdaemonTeardown();
    skAppUnregister();
}


/*
 *  repeat = timerHandler(reader);
 *
 *    The timer fired for 'reader'.  Close the file and restart the
 *    timer.
 */
static skTimerRepeat_t timerHandler(void *vreader)
{
    register flowcap_reader_t *reader = (flowcap_reader_t*)vreader;

    if (shuttingDown) {
        return SK_TIMER_END;
    }

    /* Set the close flag first. */
    reader->close = 1;

    /* Timer handler stuff */
    INFOMSG("Timer fired for '%s'", reader->name);

    /* Close the file, and open a new one. */
    closeFile(reader, (FC_REOPEN | FC_TIMED_OUT));

    return SK_TIMER_REPEAT;
}


/*
 *  createReaders(probe_vec);
 *
 *    Creates all the flowcap reader structures: one for each probe.
 *    Does not open the files.
 */
int createReaders(const sk_vector_t *probe_vec)
{
    skpc_probe_t *probe;
    flowcap_reader_t *reader;
    size_t i;

    num_fc_readers = skVectorGetCount(probe_vec);

    fc_readers = calloc(num_fc_readers, sizeof(flowcap_reader_t));
    if (fc_readers == NULL) {
        skAppPrintErr("Cannot create readers: out of memory");
        return -1;
    }

    for (i = 0, reader = fc_readers; i < num_fc_readers; ++i, ++reader) {
        skVectorGetValue(&probe, probe_vec, i);

        /* Fill in the probe */
        reader->probe = probe;
    }

    return 0;
}


/*
 *  freeReaders(void);
 *
 *    Close all the files, destroy any remaining flow-sources, and
 *    destroy the reader array.
 */
static void freeReaders(void)
{
    flowcap_reader_t *reader;
    size_t i;

    INFOMSG("Closing all files...");

    /* close all files */
    for (i = 0, reader = fc_readers; i < num_fc_readers; ++i, ++reader) {
        reader->close = 1;
        closeFile(reader, 0);

        if (reader->valid_source) {
            reader->valid_source = 0;

            DEBUGMSG("Destroying source for probe %s",
                     skpcProbeGetName(reader->probe));
            switch (skpcProbeGetType(reader->probe)) {
              case PROBE_ENUM_NETFLOW_V5:
                pduSourceDestroy(reader->source.pdu);
                break;
              default:
                assert(0);
            }
        }
    }

    INFOMSG("Closed all files.");

    free(fc_readers);
    fc_readers = NULL;
}


/*
 *  openFile(reader);
 *
 *    Get the lock for 'reader' and call openFileBase().
 */
static void openFile(
    flowcap_reader_t   *reader)
{
    MUTEX_LOCK(&reader->mutex);

    openFileBase(reader);

    MUTEX_UNLOCK(&reader->mutex);
}


/*
 *  openFileBase(reader);
 *
 *    Open a disk file to store the flows that are being read from the
 *    probe associated with 'reader'.
 *
 *    This function assumes it has the lock for 'reader'.
 *
 *    This function creates two files: a placeholder file and a
 *    temporary file that has the same name as the placeholder folder
 *    except it is prefixed with a dot.  The leading dot tells
 *    rwsender's directory poller to ignore the file.  We write the
 *    data into the temporary file.  In the closeFileBase() function,
 *    we move the temporary file over the placeholder file.
 *
 *    A timer is created for the 'reader' unless one already exists.
 *
 *    This function writes the SiLK header to the temporary file.
 *
 *    This function calls checkDiskSpace(), which may terminate the
 *    program.
 */
static void openFileBase(
    flowcap_reader_t   *reader)
{
    char dotpath[PATH_MAX];
    char ts[FC_TIMESTAMP_MAX + 1]; /* timestamp */
    struct timeval tv;
    struct tm ut;
    int fd;
    int rv;

    INFOMSG("Opening new file...");

    /* make sure there is space available or exit */
    checkDiskSpace();

    /* Create a timestamp */
    gettimeofday(&tv, NULL);
    gmtime_r(&tv.tv_sec, &ut);
    strftime(ts, sizeof(ts), "%Y%m%d%H%M%S", &ut);

    /* Create a pathname from the directory, timestamp, and probe */
    if ((size_t)snprintf(reader->path, sizeof(reader->path), "%s/%s_%s.XXXXXX",
                         destination_dir, ts, skpcProbeGetName(reader->probe))
        >= sizeof(reader->path))
    {
        CRITMSG("Pathname exceeded maximum filename size.");
        exit(EXIT_FAILURE);
    }

    /* Open the file; making sure its name is unique */
    fd = mkstemp(reader->path);
    if (fd == -1) {
        CRITMSG("Unable to create file '%s': %s",
                reader->path, strerror(errno));
        exit(EXIT_FAILURE);
    }

    DEBUGMSG("Opened placeholder file '%s'", reader->path);

    /* Set the permissions */
    fchmod(fd, 0644);

    rv = close(fd);
    fd = -1;
    if (-1 == rv) {
        CRITMSG("Unable to close file '%s': %s",
                reader->path, strerror(errno));
        exit(EXIT_FAILURE);
    }

    /* Get the basename of the file */
    reader->name = strrchr(reader->path, '/');
    ++reader->name;

    /* Create the name of the dotfile */
    if ((size_t)snprintf(dotpath, sizeof(dotpath), "%s/.%s",
                         destination_dir, reader->name)
        >= sizeof(dotpath))
    {
        CRITMSG("Dot pathname exceeded buffer size.");
        exit(EXIT_FAILURE);
    }

    /* Open the dot file */
    fd = open(dotpath, O_WRONLY | O_CREAT | O_EXCL, 0644);
    if (fd == -1) {
        CRITMSG("Unable to create file '%s': %s",
                dotpath, strerror(errno));
        unlink(reader->path);
        exit(EXIT_FAILURE);
    }

    DEBUGMSG("Opened working file '%s'", dotpath);

    /* create an skstream_t from the 'fd' and write the header */
    if (writeHeader(reader, fd, dotpath)) {
        CRITMSG("Fatal error writing to file '%s'", dotpath);
        exit(EXIT_FAILURE);
    }

    /* Set up default values */
    reader->start_time = tv.tv_sec;
    reader->records    = 0;
    reader->closing    = 0;
    reader->close      = 0;
    reader->overfull   = 0;

    /* set the timer to write_timeout */
    if (NULL == reader->timer) {
        if (clock_time) {
            skTimerCreateAtTime(&reader->timer, write_timeout, clock_time,
                                &timerHandler, (void*)reader);
        } else {
            skTimerCreate(&reader->timer, write_timeout,
                          &timerHandler, (void*)reader);
        }
    }

    INFOMSG("Opened new file '%s'", reader->name);
}


/*
 *  closeFile(reader, flags);
 *
 *    Close the current disk file associated with 'reader'.
 *
 *    When 'flags' contains FC_REOPEN, close the file and then call
 *    openFileBase() to open a new file.
 *
 *    This function must protect against attempts by the size limit
 *    and the time limit to close the file simultaneously.  If
 *    FC_REOPEN is set and if 'reader' is already in the state of
 *    being closed, simply return.
 *
 *    Otherwise, get the lock for 'reader' and call closeFileBase() to
 *    close the disk file associated with 'reader'.
 */
static void closeFile(
    flowcap_reader_t   *reader,
    int                 flags)
{
    static pthread_mutex_t close_lock = PTHREAD_MUTEX_INITIALIZER;
    uint8_t quit = 0;

    /* Ah, the perils of threads.  reader->closing keeps us from
     * double-closing a reader.  reader->close makes sure we don't honor
     * a request to close a reader that has been closed and reopened
     * since the request. */
    MUTEX_LOCK(&close_lock);

    if (reader->closing || !reader->close) {
        quit = 1;
    } else {
        reader->closing = 1;
    }
    MUTEX_UNLOCK(&close_lock);

    if (quit && (flags & FC_REOPEN)) {
        DEBUGMSG("Avoiding duplicate call to closeFile.");
        return;
    }

    MUTEX_LOCK(&reader->mutex);

    closeFileBase(reader, flags);
    if (flags & FC_REOPEN) {
        openFileBase(reader);
    }

    MUTEX_UNLOCK(&reader->mutex);
}


/*
 *  closeFileBase(reader, flags);
 *
 *    Close the disk file associated with the 'reader'.
 *
 *    This function assumes it has the lock for 'reader'.
 *
 *    The function closes the temporary dot file.  If the dot file
 *    contains no records, the dot file and placeholder file are
 *    removed.  If the dot file contains records, the dot file is
 *    moved on top of the placeholder file.
 *
 *    If 'reader' has a timer associated with it, the timer is
 *    destroyed unless this function has been called because the timer
 *    fired---that is, if 'flags' contains FC_TIMED_OUT.
 */
static void closeFileBase(
    flowcap_reader_t   *reader,
    int                 flags)
{
    const sk_file_header_t *hdr;
    char dotpath[PATH_MAX];
    int64_t uncompress_size;
    int64_t size;
    double change;
    time_t end_time;
    int rv;

    if (NULL == reader->name) {
        /* Do not close an unopened file.  An unopened file can occur
         * during start up when there are multiple sources and a
         * source (other than the final source) fails to start. */
        if (reader->timer && !(flags & FC_TIMED_OUT)) {
            DEBUGMSG("Destroying timer");
            skTimerDestroy(reader->timer);
            reader->timer = NULL;
        }
        return;
    }

    INFOMSG("Closing file '%s'...", reader->name);

    /* Make certain the timer for this file doesn't fire.  If the file
     * timed out, however, keep the timer, which will just restart.
     * The assumption is that the time to create a new file after this
     * point is less than the timer fire time. */
    if (reader->timer && !(flags & FC_TIMED_OUT)) {
        DEBUGMSG("Destroying timer");
        skTimerDestroy(reader->timer);
        reader->timer = NULL;
    }

    /* get path to the dot file. */
    sprintf(dotpath, "%s/.%s",
            destination_dir, reader->name);

    /* if no records were written, close and remove the file */
    if (reader->records == 0) {
        end_time = time(NULL);
        rv = skStreamClose(reader->ios);
        if (rv) {
            skStreamPrintLastErr(reader->ios, rv, &ERRMSG);
            CRITMSG("Fatal error closing '%s'", dotpath);
            exit(EXIT_FAILURE);
        }
        skStreamDestroy(&reader->ios);
        unlink(dotpath);
        unlink(reader->path);
        INFOMSG(("Removed empty file '%s': %" PRId64 " seconds"),
                reader->name, (int64_t)(end_time - reader->start_time));
        return;
    }

    /* flush the file so we can get its final size */
    rv = skStreamFlush(reader->ios);
    if (rv) {
        skStreamPrintLastErr(reader->ios, rv, &ERRMSG);
        CRITMSG("Fatal error flushing file '%s'", reader->path);
        exit(EXIT_FAILURE);
    }
    end_time = time(NULL);

    /* how many uncompressed bytes were processed? */
    hdr = skStreamGetSilkHeader(reader->ios);
    uncompress_size = (skHeaderGetLength(hdr)
                       + reader->records * skHeaderGetRecordLength(hdr));

    /* how many bytes were written to disk? */
    size = (int64_t)skStreamTell(reader->ios);

    /* what's the compression ratio? */
    if (uncompress_size == 0) {
        change = 0.0;
    } else {
        change = (100.0 * (double)(uncompress_size - size)
                  / (double)uncompress_size);
    }

    INFOMSG(("'%s': Closing file '%s': %" PRId64 " seconds,"
             " %" PRIu32 " records, %" PRId64 " bytes, %4.1f%% compression"),
            skpcProbeGetName(reader->probe),
            reader->name,
            (int64_t)(end_time - reader->start_time),
            reader->records,
            size,
            change);

    switch (skpcProbeGetType(reader->probe)) {
      case PROBE_ENUM_NETFLOW_V5:
        assert(reader->valid_source);
        pduSourceLogStatsAndClear(reader->source.pdu,
                                  skpcProbeGetName(reader->probe));
        break;
      default:
        break;
    }

    /* close the file and destroy the handle */
    rv = skStreamClose(reader->ios);
    if (rv) {
        skStreamPrintLastErr(reader->ios, rv, &ERRMSG);
        CRITMSG("Fatal error closing '%s'", dotpath);
        exit(EXIT_FAILURE);
    }
    skStreamDestroy(&(reader->ios));

    /* move the dot-file over the placeholder file. */
    rv = rename(dotpath, reader->path);
    if (rv != 0) {
        CRITMSG("Failed to replace '%s' with '%s': %s",
                reader->path, dotpath, strerror(errno));
        exit(EXIT_FAILURE);
    }

    INFOMSG("Finished closing '%s'", reader->name);
}


/*
 *  status = writeHeader(reader, fd, pathname);
 *
 *    Create an skstream_t at 'pathname', bind it to the file
 *    descriptor 'fd', write the SiLK header to the stream, and store
 *    the stream on 'reader'.
 */
static uint32_t writeHeader(
    flowcap_reader_t   *reader,
    int                 fd,
    const char         *pathname)
{
    skstream_t *rwios = NULL;
    sk_file_header_t *hdr;
    int rv;

    rv = skStreamCreate(&rwios, SK_IO_WRITE, SK_CONTENT_SILK_FLOW);
    if (rv) {
        return -1;
    }
    rv = skStreamBind(rwios, pathname);
    if (rv) {
        skStreamDestroy(&rwios);
        return -1;
    }

    hdr = skStreamGetSilkHeader(rwios);

    /* choose file format based on probe type.  There is no need to
     * use an IPv6 format if the probe is incapable of producing IPv6
     * data */
    switch (skpcProbeGetType(reader->probe)) {
      case PROBE_ENUM_NETFLOW_V5:
        /* IPv6 not supported */
        rv = skHeaderSetFileFormat(hdr, FT_FLOWCAP);
        if (rv == SKSTREAM_OK) {
            rv = skHeaderSetRecordVersion(hdr, flowcap_version);
        }
        break;
      default:
#if SK_ENABLE_IPV6
        rv = skHeaderSetFileFormat(hdr, FT_RWIPV6ROUTING);
#else
        rv = skHeaderSetFileFormat(hdr, FT_FLOWCAP);
        if (rv == SKSTREAM_OK) {
            rv = skHeaderSetRecordVersion(hdr, flowcap_version);
        }
#endif /* SK_ENABLE_IPV6 */
        break;
    }

    if (rv == SKSTREAM_OK) {
        rv = skHeaderSetByteOrder(hdr, SILK_ENDIAN_BIG);
    }
    if (rv == SKSTREAM_OK) {
        rv = skHeaderSetCompressionMethod(hdr, comp_method);
    }
    if (rv == SKSTREAM_OK) {
        rv = skHeaderAddProbename(hdr, skpcProbeGetName(reader->probe));
    }
    if (rv == SKSTREAM_OK) {
        rv = skStreamFDOpen(rwios, fd);
    }
    if (rv == SKSTREAM_OK) {
        rv = skStreamWriteSilkHeader(rwios);
    }

    if (rv) {
        skStreamPrintLastErr(rwios, rv, &ERRMSG);
        skStreamDestroy(&rwios);
        return -1;
    }
    reader->ios = rwios;
    return 0;
}


#ifdef SK_HAVE_STATVFS
/*
 *  checkDiskSpace();
 *
 *    Verify that we haven't reached the limits of the file system
 *    usage specified by the command line parameters.  If we're out of
 *    space, we exit the program.
 */
static void checkDiskSpace(void)
{
    struct statvfs vfs;
    int64_t free_space, total, newfree;
    int rv;
    double percent_used;

    rv = statvfs(destination_dir, &vfs);
    if (rv != 0) {
        CRITMSG("Could not statvfs '%s'", destination_dir);
        exit(EXIT_FAILURE);
    }

    /* free bytes is fundamental block size multiplied by the
     * available (non-privileged) blocks. */
    free_space = ((int64_t)vfs.f_frsize * (int64_t)vfs.f_bavail);
    /* to compute the total (non-privileged) blocks, subtract the
     * available blocks from the free (privileged) blocks to get
     * the count of privileged-only blocks, subtract that from the
     * total blocks, and multiply the result by the block size. */
    total = ((int64_t)vfs.f_frsize
             * ((int64_t)vfs.f_blocks
                - ((int64_t)vfs.f_bfree - (int64_t)vfs.f_bavail)));

    newfree = free_space - alloc_file_size * num_fc_readers;
    percent_used = ((double)(total - newfree) /
                    ((double)total / 100.0));

    if (newfree < freespace_minimum) {
        CRITMSG(("Free disk space limit overrun: "
                 "free=%" PRId64 " < min=%" PRId64 " (used %.4f%%)"),
                newfree, freespace_minimum, percent_used);
        /* TODO: Create a wait routine instead of exiting? */
        exit(EXIT_FAILURE);
    }
    if (percent_used > space_maximum_percent) {
        CRITMSG(("Free disk space limit overrun: "
                 "used=%.4f%% > max=%.4f%% (free %" PRId64 " bytes)"),
                percent_used, space_maximum_percent, newfree);
        /* TODO: Create a wait routine instead of exiting? */
        exit(EXIT_FAILURE);
    }

    DEBUGMSG(("Free space available is %" PRId64 " bytes (%.4f%%)"),
             newfree, percent_used);
}
#endif /* SK_HAVE_STATVFS */


/*
 *  readerMain(reader);
 *
 *    Thread entry point for each reader_thread.
 *
 *    Reads data from the probe/flow-source associated with the
 *    'reader' and writes that data to to disk file associated with
 *    'reader'.
 *
 *    If the file reaches the maximum size, the file is closed and a
 *    new file is opened.
 *
 *    This function runs until the flow-source is stopped, at which
 *    point it returns.
 *
 *    If there is an error writing to the disk or if the file system
 *    usage reaches the limits set by the command line parameters, the
 *    function terminates the program.
 */
static void *readerMain(void *vreader)
{
    flowcap_reader_t *reader = (flowcap_reader_t*)vreader;
    sigset_t sigs;
    skpc_probetype_t probe_type;
    rwRec rec;
    int rv;

    assert(reader);

    /* ignore all signals */
    sigfillset(&sigs);
    pthread_sigmask(SIG_SETMASK, &sigs, NULL);

    NOTICEMSG("Reader for probe %s started.",
              skpcProbeGetName(reader->probe));

    probe_type = skpcProbeGetType(reader->probe);

    /* Infloop */
    while (reading) {

        /* Get the next record */
        switch (probe_type) {
          case PROBE_ENUM_NETFLOW_V5:
            rv = pduSourceGetGeneric(reader->source.pdu, &rec);
            break;
#if SK_ENABLE_IPFIX
          case PROBE_ENUM_IPFIX:
          case PROBE_ENUM_NETFLOW_V9:
            rv = ipfixSourceGetGeneric(reader->source.ipfix, &rec);
            break;
#endif
          default:
            CRITMSG("Invalid probe id '%d'", probe_type);
            skAbortBadCase(probe_type);
        }

        if (rv == -1) {
            break;
        }

        MUTEX_LOCK(&reader->mutex);

        /* Write the record to the file */
        rv = skStreamWriteRecord(reader->ios, &rec);
        if (rv) {
            skStreamPrintLastErr(reader->ios, rv, &ERRMSG);
            CRITMSG("Fatal error writing record.");
            exit(EXIT_FAILURE);
        }
        reader->records++;

        /* Check to see if we have reached the size limit */
        if (skStreamGetUpperBound(reader->ios) >= max_file_size) {
            reader->close = 1;
            reader->overfull = 1;
        }

        MUTEX_UNLOCK(&reader->mutex);

        /* Close and acquire new file if necessary */
        if (reader->overfull) {
            /* Close and ship the file and open a new one in its place */
            closeFile(reader, FC_REOPEN);
        }
    }

    NOTICEMSG("Reader for probe %s ended.",
              skpcProbeGetName(reader->probe));

    /* End thread */
    return NULL;
}


/*
 *  status = startReaders();
 *
 *    Create the flow-source object associated with the probe that is
 *    stored on each 'reader' object.  Have the flow-sources begin to
 *    collect network traffic, and create a thread for each 'reader'
 *    to read the flows.
 *
 *    Return 0 on success, non-zero otherwise.
 */
static int startReaders(void)
{
    flowcap_reader_t *reader;
    size_t i;
    skpc_probetype_t probe_type;
    pduSourcePool_t pdupool;

    DEBUGMSG("Creating PDU source pool");
    pdupool = pduSourcePoolCreate();
    if (pdupool == NULL) {
        return 1;
    }

#if SK_ENABLE_IPFIX
    DEBUGMSG("Setting up IPFIX");
    if (ipfixSourcesSetup()) {
        return 1;
    }
#endif  /* SK_ENABLE_IPFIX */

    INFOMSG("Starting all readers...");
    reading = 1;

    for (i = 0, reader = fc_readers; i < num_fc_readers; ++i, ++reader) {
        /* Initialize mutex */
        pthread_mutex_init(&reader->mutex, NULL);

        /* Create the first file */
        openFile(reader);

        probe_type = skpcProbeGetType(reader->probe);
        DEBUGMSG("Starting %s source for probe %s",
                 skpcProbetypeEnumtoName(probe_type),
                 skpcProbeGetName(reader->probe));

        switch (probe_type) {
          case PROBE_ENUM_NETFLOW_V5:
            reader->source.pdu =
                pduSourceCreateFromProbeDef(pdupool,
                                            reader->probe,
                                            BUF_REC_COUNT,
                                            BIG_SOCK_BUF/num_fc_readers);
            if (reader->source.pdu == NULL) {
                WARNINGMSG("Failed to start %s source for probe %s",
                           skpcProbetypeEnumtoName(probe_type),
                           skpcProbeGetName(reader->probe));
                return 1;
            }
            break;
#if SK_ENABLE_IPFIX
          case PROBE_ENUM_IPFIX:
          case PROBE_ENUM_NETFLOW_V9:
            reader->source.ipfix =
                ipfixSourceCreateFromProbeDef(reader->probe,
                                              BUF_REC_COUNT);
            if (reader->source.ipfix == NULL) {
                WARNINGMSG("Failed to start %s source for probe %s",
                           skpcProbetypeEnumtoName(probe_type),
                           skpcProbeGetName(reader->probe));
                return 1;
            }
            break;
#endif  /* SK_ENABLE_IPFIX */
          default:
            CRITMSG("Unsupported probe type %d",
                    (int)skpcProbeGetType(reader->probe));
            skAbortBadCase(probe_type);
        } /* switch () */

        reader->valid_source = 1;

        DEBUGMSG("Creating thread for probe %s",
                 skpcProbeGetName(reader->probe));
        if (pthread_create(&reader->reader_thread, NULL,
                           readerMain, (void*)reader))
        {
            return 1;
        }
        reader->running = 1;
    }

    INFOMSG("Started all readers.");

    DEBUGMSG("Destroying pools");
    pduSourcePoolDestroy(pdupool);

    return 0;
}


/*
 *  stopReaders();
 *
 *    Stop all the flow-sources.
 *
 *    For flow-sources that have separate Stop() and Destroy()
 *    functions, call the Stop() function; otherwise, call the
 *    Destroy() function.
 *
 *    Wait for each reader_thread to terminate.
 */
static void stopReaders(void)
{
    flowcap_reader_t *reader;
    size_t i;

    if (!reading) {
        return;
    }

    INFOMSG("Stopping all readers...");
    reading = 0;

    for (i = 0, reader = fc_readers; i < num_fc_readers; ++i, ++reader) {
        /* Stop the flow-source */
        if (reader->valid_source) {
            DEBUGMSG("Stopping %s source for probe %s",
                     skpcProbetypeEnumtoName(skpcProbeGetType(reader->probe)),
                     skpcProbeGetName(reader->probe));
            switch (skpcProbeGetType(reader->probe)) {
              case PROBE_ENUM_NETFLOW_V5:
                pduSourceStop(reader->source.pdu);
                break;
#if SK_ENABLE_IPFIX
              case PROBE_ENUM_IPFIX:
              case PROBE_ENUM_NETFLOW_V9:
                ipfixSourceDestroy(reader->source.ipfix);
                reader->valid_source = 0;
                break;
#endif
              default:
                assert(0);
            }
        }

        /* Wait for the thread to end */
        if (reader->running) {
            DEBUGMSG("Waiting for probe %s",
                     skpcProbeGetName(reader->probe));
            pthread_join(reader->reader_thread, NULL);
            reader->running = 0;
        }

        /* Don't destroy until after the files are closed, since we
         * want to get the final stats from the flow-source. */
    }

    INFOMSG("Stopped all readers.");
}


/* Program entry point. */
int main(int argc, char **argv)
{
    appSetup(argc, argv);               /* never returns on failure */

    sklogOpen();

    /* start the logger and become a daemon */
#ifdef DEBUG
    skdaemonDontFork();
#endif
    if (skdaemonize(&shuttingDown, &appTeardown) == -1) {
        exit(EXIT_FAILURE);
    }

    /* Store the main thread ID */
    main_thread = pthread_self();

    /* Start the reader threads */
    if (startReaders()) {
        CRITMSG("Failed to start all readers. Exiting.");
        exit(EXIT_FAILURE);
    }

    /* We now run forever, excepting signals */
    while (!shuttingDown) {
        pause();
    }

    appTeardown();

    exit(EXIT_SUCCESS);
    return 0;                   /* NOTREACHED */
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
