/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _SKPREFIXMAP_H
#define _SKPREFIXMAP_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_SKPREFIXMAP_H, "$SiLK: skprefixmap.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skstream.h>
#include <silk/skplugin.h>


/*
**  prefixmap.h
**
**  John McClary Prevost
**  December 1st, 2004
**
**    This is a tree data structure for prefixes of IP addresses
**    (including full addresses, if necessary).  It is suitable for
**    representing a categorization of the IPv4 address space--for
**    example, mapping IP addresses to country codes.
**
**    Strictly speaking, it maps IP address to 31-bit unsigned values.
**    These values could be used to represent anything desired.
*/

#define SKPREFIXMAP_NOT_FOUND       (0xFFFFFFFF)

#define SKPREFIXMAP_MAX_VALUE       (~0x80000000u)

typedef enum skPrefixMapErr_en {
    SKPREFIXMAP_OK = 0,
    SKPREFIXMAP_ERR_ARGS = 1,
    SKPREFIXMAP_ERR_MEMORY = 2,
    SKPREFIXMAP_ERR_IO = 3,
    SKPREFIXMAP_ERR_DUPLICATE = 4,
    SKPREFIXMAP_ERR_NOTEMPTY = 5
} skPrefixMapErr_t;

/* For addresses: key is 32-bit value representing IP address in
 * network byte-order.
 *
 * For protocol/ports: key is a 32-bit value with the (8-bit) protocol
 * number occupying the high 16-bits and the (16-bit) port number (if
 * the protocol is TCP or UDP) occupying the low 16-bits.  The high
 * 8 bits are always zero, since the protocol is an 8-bit value.
 */

typedef enum skPrefixMapContent_en {
    SKPREFIXMAP_CONT_ADDR = 0,
    SKPREFIXMAP_CONT_PROTO_PORT = 1
} skPrefixMapContent_t;


typedef struct skPrefixMap_st skPrefixMap_t;


typedef struct skPrefixMapIterator_st {
    const skPrefixMap_t    *map;
    uint32_t                start;
    uint32_t                end;
} skPrefixMapIterator_t;



skPrefixMapErr_t skPrefixMapAddRange(
    skPrefixMap_t  *map,
    uint32_t        low_val,
    uint32_t        high_val,
    uint32_t        dict_val);
/*
 *    Add a new key->value mapping to the prefix map 'map', specifying
 *    that all keys from 'low_key' to 'high_key' should be mapped to
 *    'dict_val'.  'dict_val' should not be greater than
 *    SKPREFIXMAP_MAX_VALUE.
 */


skPrefixMapErr_t skPrefixMapCreate(skPrefixMap_t **map);
/*
 *    Create a new prefix map at the memory pointed at by 'map'.
 */


void skPrefixMapDelete(skPrefixMap_t *map);
/*
 *   Frees the resources used by the prefixmap pointed to by map, then
 *   frees map itself.  After calling, map is an invalid pointer.
 */


int skPrefixMapDictionaryGetEntry(
    const skPrefixMap_t    *map,
    uint32_t                dict_val,
    char                   *out_buf,
    size_t                  bufsize);
/*
 *    Fills the character array 'out_buf' with the dictionary entry
 *    (label) for the given integer 'dict_val' and returns the string
 *    length of the label.  If the label's length is larger than
 *    'bufsize', the buffer was too small to hold the label and only
 *    the first bufsize-1 characters were written to 'out_buf'.  A
 *    prefixmap will give a result for any input value; if 'dict_val'
 *    is not in the dictionary, 'out_buf' is filled with a string
 *    representation of the value.
 */


uint32_t skPrefixMapDictionaryGetMaxWordSize(const skPrefixMap_t *map);
/*
 *    Returns the length in characters of the longest word in the
 *    dictionary.
 */


uint32_t skPrefixMapDictionaryGetWordCount(const skPrefixMap_t *map);
/*
 *    Returns the number of words in the dictionary.  Returns 0 if the
 *    prefix map does not contain a dictionary.
 */


skPrefixMapErr_t skPrefixMapDictionaryInsert(
    skPrefixMap_t  *map,
    uint32_t        id,
    const char     *word);
/*
 *    Add a new 'word' to the dictionary on the prefix map 'map' at
 *    the location specified by 'id'.  The entry at 'id' must be empty
 *    or greater than the current maximum dictionary ID.  If either of
 *    these conditions is not met, SKPREFIXMAP_ERR_DUPLICATE is
 *    returned.
 */


uint32_t skPrefixMapDictionaryLookup(
    const skPrefixMap_t    *map,
    const char             *word);
/*
 *    Return the value for a given 'word' in the dictionary associated
 *    with 'map'., Return SKPREFIXMAP_NOT_FOUND if 'word' is not in
 *    the dictionary.
 *
 *    See also skPrefixMapDictionarySearch().
 */


skPrefixMapErr_t skPrefixMapDictionarySearch(
    skPrefixMap_t  *map,
    const char     *word,
    uint32_t       *out_dict_val);
/*
 *    See if 'word' exists in the dictionary on the prefix map 'map'.
 *    If so, place the ID of its entry in the memory pointed at by
 *    'out_dict_val' and return SKPREFIXMAP_OK.
 *
 *    If 'word' does not exist, append it to the dictionary, fill
 *    'out_dict_val' with the newly created ID, and return
 *    SKPREFIXMAP_OK.
 *
 *    The function may return SKPREFIXMAP_ERR_ALLOC if it is unable to
 *    grow the dictionary.
 *
 *    See also skPrefixMapDictionaryLookup(), which will not modify
 *    the prefix map 'map'.
 */


uint32_t skPrefixMapGet(const skPrefixMap_t *map, uint32_t key);
/*
 *    Returns the mapped value for the given 32 bit key.  A prefixmap
 *    will give a result for any input value.  This function is
 *    guaranteed to terminate.  A valid result is never greater than
 *    SKPREFIXMAP_MAX_VALUE.  Returns SKPREFIXMAP_NOT_FOUND if the key
 *    is not found in the tree.
 *
 *    See also skPrefixMapGetString().
 */


const char *skPrefixMapGetContentName(int content_id);
/*
 *    Given the content type 'content_id'---which should be a value
 *    from skPrefixMapContent_t---return a textual representation of
 *    it.
 */


const char *skPrefixMapGetMapName(const skPrefixMap_t *map);
/*
 *    Returns the map-name of the prefix map.  NULL if no map-name is
 *    defined.
 */


skPrefixMapContent_t skPrefixMapGetContentType(const skPrefixMap_t *map);
/*
 *    Return the content type of the prefix map.  One of:
 *
 *    SKPREFIXMAP_CONTENTS_ADDR - IP Addresses
 *    SKPREFIXMAP_CONTENTS_PROTO_PORT - Protocol and Port
 */


int skPrefixMapGetString(
    const skPrefixMap_t    *map,
    uint32_t                key,
    char                   *out_buf,
    size_t                  bufsize);
/*
 *    Fills the character array 'out_buf' with the mapped value
 *    (label) for the given 32 bit 'key' and returns the string length
 *    of the label.  If the label's length is larger than 'bufsize',
 *    the buffer was too small to hold the label and only the first
 *    bufsize-1 characters were written to 'out_buf'.  A prefixmap
 *    will give a result for any input value.
 */


int skPrefixMapIteratorBind(
    skPrefixMapIterator_t  *iter,
    const skPrefixMap_t    *map);
/*
 *    Binds the prefix map iterator 'iter' to iterate over all the
 *    entries in the prefix map 'map'.  Returns 0 on success, non-zero
 *    otherwise.
 */


int skPrefixMapIteratorCreate(
    skPrefixMapIterator_t **out_iter,
    const skPrefixMap_t    *map);
/*
 *    Creates a new iterator at the address pointed to by 'out_iter'
 *    and binds it to iterate over all the entries in the prefix map
 *    'map'.  Returns 0 on success, non-zero otherwise.
 */


void skPrefixMapIteratorDestroy(skPrefixMapIterator_t **out_iter);
/*
 *    Destroys the iterator pointed to by 'out_iter'.
 */


skIteratorStatus_t skPrefixMapIteratorNext(
    skPrefixMapIterator_t  *iter,
    uint32_t               *out_key_start,
    uint32_t               *out_key_end,
    uint32_t               *out_dict_val);
/*
 *    If there are more entries in the prefix map, this function fills
 *    the locations pointed to by 'out_key_start', 'out_key_end', and
 *    'out_dict_val' with the value of the starting and ending values
 *    of the range and the value, respectively, and returns
 *    SK_ITERATOR_OK.  Otherwise, the output values are not touched
 *    and SK_ITERATOR_NO_MORE_ENTRIES is returned.
 */


void skPrefixMapIteratorReset(skPrefixMapIterator_t *iter);
/*
 *    Resets the iterator 'iter' to begin looping through the entries
 *    in the prefix map again.
 */


skPrefixMapErr_t skPrefixMapLoad(skPrefixMap_t **map, const char *path);
/*
 *    Opens a stream to the file at 'path' and calls skPrefixMapRead()
 *    to read the prefixmap from the stream.
 */


skPrefixMapErr_t skPrefixMapRead(skPrefixMap_t **map, skstream_t *in);
/*
 *    Allocates a new prefixmap and assigns a pointer to it into the
 *    provided space, 'map'.  Then reads a prefixmap from the stream
 *    'in'.  Before calling, map should not be NULL, *map may have any
 *    value, and 'in' should open to a valid stream.
 *    skPrefixMapRead() returns successfully memory has been
 *    allocated, and *map will contain a new valid skPrefixMap_t
 *    pointer.  If skPrefixMapRead() returns an error, *map will not
 *    contain a valid pointer, and any allocated memory will have been
 *    freed.
 */


skPrefixMapErr_t skPrefixMapSave(
    skPrefixMap_t  *map,
    const char     *pathname);
/*
 *    Open a file at the location specified by 'pathname', and then
 *    call skPrefixMapWrite() to write the prefix map 'map' to that
 *    location.  Finally, close the stream.
 */


skPrefixMapErr_t skPrefixMapSetContentType(
    skPrefixMap_t          *map,
    skPrefixMapContent_t    content_type);
/*
 *    Set the content type of the prefix map 'map' to 'content_type'.
 */


skPrefixMapErr_t skPrefixMapSetDefaultVal(
    skPrefixMap_t  *map,
    uint32_t        dict_val);
/*
 *    Set the default value to use for the prefix map 'map' to
 *    'dict_val'.  The map must be empty and must have had no default
 *    set previously; if either of these conditions is not met,
 *    SKPREFIXMAP_ERR_NOTEMPTY is returned.
 */


skPrefixMapErr_t skPrefixMapSetMapName(skPrefixMap_t *map, const char *name);
/*
 *    Set the mapname of the prefix map 'map' to 'name'.  Overwrites
 *    the current name.  If 'name' is NULL, the current name is
 *    cleared.  Makes a copy of 'name'.  Returns SKPREFIXMAP_OK on
 *    success.  If there is an allocation error the original mapname
 *    is unchanged and SKPREFIXMAP_ERR_MEMORY is returned.
 */


const char *skPrefixMapStrerror(int error_code);
/*
 *    Given the 'error_code'---a skPrefixMapErr_t---return a textual
 *    representation of it.
 */


skPrefixMapErr_t skPrefixMapWrite(
    skPrefixMap_t  *map,
    skstream_t     *stream);
/*
 *    Write the binary prefix map 'map' to the stream 'stream'.
 */


skplugin_err_t skPrefixMapAddFields(
    uint16_t major_version,
    uint16_t minor_version,
    void     *UNUSED(data));
/*
 *    Provide support in the calling SiLK application for loading a
 *    prefix map and using the switches and fields that are created as
 *    a result of loading the prefix map.
 */


skplugin_err_t skAddressTypesAddFields(
    uint16_t major_version,
    uint16_t minor_version,
    void     UNUSED(*pi_data));
/*
 *    Add support for the --stype, --dtype switches in rwfilter, and
 *    the stype and dtype fields in rwcut, rwgroup, rwsort, rwuniq,
 *    and rwstats.
 */


#endif /* _SKPREFIXMAP_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
