/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _SKSTRINGMAP_H
#define _SKSTRINGMAP_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_STRINGMAP_H, "$SiLK: skstringmap.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skdllist.h>
#include <silk/skvector.h>

/*
**  skstringmap.h
**
**  An ordered list of name->id mappings to support named columns.
**  name is a C-string; id is a uint32_t.
**
**  name matching is case-insensitive.
**
**  The basic usage is to create a sk_stringmap_t and to add
**  sk_stringmap_entry_t items to it, each of which is a name/value
**  pair.
**
**  Then, once processing begins, call skStringMapMatch() with the
**  user's string, and it will return either a valid result set (an
**  sk_vector_t of sk_stringmap_entry_t*), or a parse error.
**
**  Sample Usage (does not check error cases)
**
**    sk_stringmap_t *name_id_map;
**    sk_stringmap_entry_t mappings[] = {
**        { "foo", 1, "Description of foo", NULL },
**        { "bar", 2, "Description of bar", NULL },
**    };
**    char *user_input = "foo,baz";
**    sk_vector_t *match_vec;
**    sk_stringmap_status_t rv;
**    sk_stringmap_id_t found_id;
**
**    // Create map and add entries
**    skStringMapCreate( &name_id_map );
**    skStringMapAddEntries( name_id_map, 2, mappings );
**
**    // Create vector for results of search
**    skVectorNew( &match_vec );
**
**    // Map user's input and process each entry
**    rv = skStringMapMatch( name_id_map, user_input, match_vec, NULL );
**    if( rv == SKSTRINGMAP_OK ) {
**        size_t count = skVectorGetCount( match_vec );
**        size_t i;
**        sk_stringmap_entry_t *entry;
**        for( i = 0; i < count; ++i ) {
**            skVectorGetValue( &entry, match_vec, i );
**            found_id = entry->id;
**            // DO STUFF WITH THE IDS MATCHED
**        }
**    }
**
**    // Clean up
**    skVectorDestroy( match_vec );
**    skStringMapDestroy( name_id_map );
*/


/* function result status */
typedef enum {
    SKSTRINGMAP_OK = 0,
    /* Command was successful */

    SKSTRINGMAP_ERR_INPUT = -127,
    /* Indicates bad input, e.g. NULL pointer */

    SKSTRINGMAP_ERR_MEM,
    /* A memory allocation call failed */

    SKSTRINGMAP_ERR_LIST,
    /* Some unexpected error occured in the linked list */

    /* The following values can be returned while adding a new
     * key/value pair to the map. */

    SKSTRINGMAP_ERR_DUPLICATE_ENTRY,
    /* The new key was found to be a duplicate of a key already in the
     * map. */

    SKSTRINGMAP_ERR_ZERO_LENGTH_ENTRY,
    /* The new key was found to be the empty string. */

    SKSTRINGMAP_ERR_NUMERIC_START_ENTRY,
    /* The key was found to start with a number but to contain
     * non-numeric characters. */

    SKSTRINGMAP_ERR_ALPHANUM_START_ENTRY,
    /* The key was found to start with a non-alphanumeric
     * character. */

    SKSTRINGMAP_ERR_PARSER,
    /* The parser encountered an unexpected error unrelated to the
     * user's input */

    /* The following values can be returned while parsing user input
     * and finding entries in the map. */

    SKSTRINGMAP_PARSE_NO_MATCH,
    /* The user's input is not an exact match nor a prefix match for
     * any key. */

    SKSTRINGMAP_PARSE_AMBIGUOUS,
    /* The user's input matches matches no key exactly and is a prefix
     * match for multiple keys. */

    SKSTRINGMAP_PARSE_UNPARSABLE
    /* The user's input is not parsable or contains an invalid range
     * e.g., 3-2 */

} sk_stringmap_status_t;


typedef enum {
    SKSTRINGMAP_DUPES_KEEP = 0,
    SKSTRINGMAP_DUPES_REMOVE_SILENT,
    SKSTRINGMAP_DUPES_REMOVE_WARN,
    SKSTRINGMAP_DUPES_ERROR
} sk_stringmap_dupes_t;


typedef sk_dllist_t sk_stringmap_t;
typedef uint32_t sk_stringmap_id_t;

typedef struct sk_stringmap_entry_st {
    /* string name key */
    const char             *name;
    /* unsigned integer id value */
    sk_stringmap_id_t       id;
    /* optional description of this entry */
    void                   *description;
    /* data pointer maintained by the caller */
    void                   *userdata;
} sk_stringmap_entry_t;


#define SK_STRINGMAP_SENTINEL   {NULL, 0, NULL, NULL}


sk_stringmap_status_t skStringMapCreate(
    sk_stringmap_t        **out_str_map);
/*
 *    Create a new string-map and puts the result into 'out_str_map'.
 *    Return SKSTRINGMAP_OK on success, or SKSTRINGMAP_ERR_MEM if
 *    allocation fails.
 *
 */


sk_stringmap_status_t skStringMapDestroy(
    sk_stringmap_t         *map);
/*
 *    Destroy the string-map 'str_map'.
 *
 */


#define skStringMapIsEmpty(str_map) skDLListIsEmpty(str_map)
/*
 *    Return 1 if the string-map 'str_map' is empty; 0 if it contains
 *    entries.
 */


sk_stringmap_status_t skStringMapAddEntries(
    sk_stringmap_t             *str_map,
    int                         entryc,
    const sk_stringmap_entry_t *entryv);
/*
 *    Add multiple entries from the 'entryv' array to the StringMap
 *    'str_map'.
 *
 *    When 'entryc' is positive or zero, it is taken as the count of
 *    entries in 'entryv'.  When 'entryc' is negative, 'entryv' is
 *    considered to be a NULL-terminated array of entries; i.e., all
 *    entries in 'entryv' are added one of the form
 *    SKSTRINGMAP_SENTINEL is reached.
 *
 *    Returns SKSTRINGMAP_OK if all entries are successfully added.
 *    If an error occurs, some entries may have been added to the
 *    StringMap.  Returns SKSTRINGMAP_ERR_INPUT if 'str_map' or
 *    'entryv' is NULL, or if 'entryc' is positive and a "name" in
 *    entryv is NULL.  Returns SKSTRINGMAP_ERR_MEM if allocation
 *    fails.  May also return any of SKSTRINGMAP_ZERO_LENGTH_ENTRY,
 *    SKSTRINGMAP_ERR_NUMERIC_START_ENTRY,
 *    SKSTRINGMAP_ERR_ALPHANUM_START_ENTRY,
 *    SKSTRINGMAP_DUPLICATE_ENTRY.
 */


sk_stringmap_status_t skStringMapRemoveByName(
    sk_stringmap_t         *str_map,
    const char             *name);
/*
 *    Remove the single entry from the StringMap 'str_map' whose name
 *    is 'name'.  Return SKSTRINGMAP_ERR_INPUT if 'str_map' or 'name'
 *    is NULL, else return SKSTRINGMAP_OK.
 */


sk_stringmap_status_t skStringMapRemoveByID(
    sk_stringmap_t         *str_map,
    sk_stringmap_id_t       id);
/*
 *    Remove all entries from the StringMap 'str_map' whose ID is
 *    'id'.  Return SKSTRINGMAP_ERR_INPUT if 'str_map' is NULL, else
 *    return SKSTRINGMAP_OK.
 */


sk_stringmap_status_t skStringMapRemoveEntries(
    sk_stringmap_t             *str_map,
    int                         entryc,
    const sk_stringmap_entry_t *entryv);
/*
 *    Remove multiple entries from the StringMap 'str_map' by calling
 *    skStringMapRemoveByName() for all entries in 'entryv'.
 *
 *    When 'entryc' is positive or zero, it is taken as the count of
 *    entries in 'entryv'.  When 'entryc' is negative, 'entryv' is
 *    considered to be a NULL-terminated array of entries; i.e., all
 *    entries in 'entryv' are removed one of the form
 *    SKSTRINGMAP_SENTINEL is reached.
 */


sk_stringmap_status_t skStringMapMatch(
    const sk_stringmap_t   *str_map,
    const char             *user_string,
    sk_vector_t            *out_vec,
    char                  **bad_token);
/*
 *    Take the user entered string, 'user_string' containing tokens
 *    separated by commas(,) and hyphens(-), parse it to get a list of
 *    keys, search the StringMap 'str_map' for those keys, and fill
 *    the vector 'out_vec' with pointers to the sk_stringmap_entry_t's
 *    that matched the keys.
 *
 *    When non-NULL, 'bad_token' should contain a pointer to a
 *    C-string to hold the token that was being parsed when a parse
 *    error occurred.  It is the caller's responsibility to free()
 *    this string.
 *
 *    Pointers to the matched entries will be appened to 'out_vec',
 *    whose elements should be sizeof(sk_stringmap_entry_t*).  The
 *    caller should not modify nor free the elements in the vector.
 */


sk_stringmap_status_t skStringMapParse(
    const sk_stringmap_t   *field_map,
    const char             *user_input,
    sk_stringmap_dupes_t    handle_dupes,
    sk_vector_t            *out_vec,
    char                  **errmsg);


sk_stringmap_status_t skStringMapGetByName(
    const sk_stringmap_t   *str_map,
    const char             *user_string,
    sk_stringmap_entry_t  **out_entry);
/*
 *    Take the user entered string 'user_string' containing a SINGLE
 *    key and search the StringMap 'str_map' to find the
 *    sk_stringmap_entry_t that has that string as a key, setting
 *    'out_entry' to point to that key and returning SKSTRINGMAP_OK.
 *    The caller should not modify nor free the returned entry.
 *
 *    Return SKSTRINGMAP_PARSE_AMBIGUOUS if 'user_string' matches
 *    multiple entries, SKSTRINGMAP_PARSE_NO_MATCH if it matches no
 *    entries.
 */


sk_stringmap_status_t skStringMapGetByID(
    const sk_stringmap_t   *str_map,
    sk_stringmap_id_t       id,
    sk_vector_t            *out_vec);
/*
 *    Given an 'id', append to the vector 'out_vec' all entries in the
 *    StringMap 'str_map' that have keys mapping to that id.
 *
 *    'out_vec' should be a vector whose elements are
 *    sizeof(sk_stringmap_entry_t*).  The caller should not modify nor
 *    free the elements in the vector.
 */


const char *skStringMapGetFirstName(
    const sk_stringmap_t   *str_map,
    sk_stringmap_id_t       id);
/*
 *    Find the first entry in the StringMap 'str_map' whose ID is 'id'
 *    and return the name associated with the entry.  Return NULL if
 *    'id' is not in the StringMap.  The caller should not modify nor
 *    free the returned value.
 */


sk_stringmap_status_t skStringMapPrintMap(
    const sk_stringmap_t   *str_map,
    FILE                   *outstream);
/*
 *    Print the map keys and values to the given file stream.  The map
 *    is printed in a human readable format of the form
 *
 *        { "key1" : value1, "key2" : value2, ... }
 *
 *  Arguments
 *
 *    FILE *outstream - the output stream to which to print
 *
 *    sk_stringmap_t *str_map - pointer to the StringMap to print
 */


void skStringMapPrintUsage(
    const sk_stringmap_t   *field_map,
    FILE                   *fh,
    const int               indent_len);


const char *skStringMapStrerror(int error_code);
/*
 *    Given the 'error_code' return a textual representation of it.
 */


#endif /* _SKSTRINGMAP_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
