/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Deque API
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skdeque.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "skdqprivate.h"


/* Creates a copy of a deque.  Operations on both deques will affect each
   other. */
skDeque_t skDequeCopy(volatile skDeque_t deque)
{
    int oldtype;
    int die = 0;

    if (deque == NULL || deque->data == NULL) {
        return NULL;
    }
    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);
    if (deque->data == NULL) {
        die = 1;
    } else {
        deque->ref++;
    }
    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    if (die) {
        return NULL;
    }

    return deque;
}


/* Destroy and free a deque.  (Not reponsible for freeing the elements
 * within the deque). */
skDQErr_t skDequeDestroy(skDeque_t deque)
{
    int oldtype;
    int destroy;

    if (deque == NULL) {
        return SKDQ_ERROR;
    }

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    if ((destroy = (--deque->ref == 0))) {

        /* Call destructor method */
        deque->methods.destroy(deque);

        /* Mark as destroyed */
        deque->data = NULL;

        /* Give all the condition waiting threads a chance to exit. */
        pthread_cond_broadcast(deque->cond);
    }

    pthread_cleanup_pop(1);

    if (destroy) {
        while (pthread_mutex_destroy(deque->mutex) == EBUSY);
        while (pthread_cond_destroy(deque->cond) == EBUSY) {
            pthread_cond_broadcast(deque->cond);
        }

        free(deque);
    }

    pthread_setcanceltype(oldtype, NULL);

    return SKDQ_SUCCESS;
}


/* Return the status of a deque. */
skDQErr_t skDequeStatus(skDeque_t deque)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.status(deque);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}

/* Return the size of a deque. */
uint32_t skDequeSize(skDeque_t deque)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.size(deque);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}


/* Pop an element from a deque.  */
static skDQErr_t sk_deque_pop(
    skDeque_t   deque,
    void      **item,
    uint8_t     block,
    uint8_t     front,
    uint32_t    seconds)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.pop(deque, item, block, front, seconds);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}



/*
 *  Pop an element from a deque.
 *
 *  skDequePop{Front,Back} will block until an item is available in the
 *  deque.  skDequePop{Front,Back}NB will return SKDQ_EMPTY if the
 *  deque is empty instead.
 */
skDQErr_t skDequePopFront(skDeque_t deque, void **item)
{
    return sk_deque_pop(deque, item, 1, 1, 0);
}
skDQErr_t skDequePopFrontNB(skDeque_t deque, void **item)
{
    return sk_deque_pop(deque, item, 0, 1, 0);
}
skDQErr_t skDequePopFrontTimed(skDeque_t deque, void **item, uint32_t seconds)
{
    return sk_deque_pop(deque, item, 1, 1, seconds);
}
skDQErr_t skDequePopBack(skDeque_t deque, void **item)
{
    return sk_deque_pop(deque, item, 1, 0, 0);
}
skDQErr_t skDequePopBackNB(skDeque_t deque, void **item)
{
    return sk_deque_pop(deque, item, 0, 0, 0);
}
skDQErr_t skDequePopBackTimed(skDeque_t deque, void **item, uint32_t seconds)
{
    return sk_deque_pop(deque, item, 1, 0, seconds);
}


/* Peek at an element from a deque. */
static skDQErr_t sk_deque_peek(skDeque_t deque, void **item, uint8_t front)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.peek(deque, item, front);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}


/* Return the first or last element of a deque without removing it, or
   SKDQ_EMPTY if the deque is empty.  */
skDQErr_t skDequeFront(skDeque_t deque, void **item)
{
    return sk_deque_peek(deque, item, 1);
}
skDQErr_t skDequeBack(skDeque_t deque, void **item)
{
    return sk_deque_peek(deque, item, 0);
}

/* Peek at an element from a deque. */
static skDQErr_t sk_deque_push(skDeque_t deque, void *item, uint8_t front)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.push(deque, item, front);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}

skDQErr_t skDequeUnblock(skDeque_t deque)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.block(deque, 0);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}


skDQErr_t skDequeBlock(skDeque_t deque)
{
    int oldtype;
    skDQErr_t retval;

    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &oldtype);
    pthread_cleanup_push((callbackfn_t)pthread_mutex_unlock,
                         (void *)deque->mutex);
    pthread_mutex_lock(deque->mutex);

    retval = deque->methods.block(deque, 1);

    pthread_cleanup_pop(1);
    pthread_setcanceltype(oldtype, NULL);

    return retval;
}

/* Push an item onto a deque. */
skDQErr_t skDequePushFront(skDeque_t deque, void *item)
{
    return sk_deque_push(deque, item, 1);
}
skDQErr_t skDequePushBack(skDeque_t deque, void *item)
{
    return sk_deque_push(deque, item, 0);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
