/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
** rwaugroutingio.c
**
**      routines to do io stuff with augrouting records.
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwaugroutingio.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "skstream_priv.h"


/* Version to use when SK_RECORD_VERSION_ANY is specified */
#define DEFAULT_RECORD_VERSION 4


/* ********************************************************************* */

/*
**  RWAUGROUTING VERSION 4
**
**    uint32_t      stime_bb1;       //  0- 3
**    // uint32_t     stime     :22  //        Start time:msec offset from hour
**    // uint32_t     bPPkt1    :10; //        Whole bytes-per-packet (hi 10)
**
**    uint32_t      bb2_elapsed;     //  4- 7
**    // uint32_t     bPPkt2    : 4; //        Whole bytes-per-packet (low 4)
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     elapsed   :22; //        Duration of flow in msec
**
**    uint32_t      pro_flg_pkts;    //  8-11
**    // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
**                                   //        is_tcp==1 &&
**                                   //          tcp_state==0:TCPflags/All pkts
**                                   //          tcp_state==1:TCPflags/1st pkt
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     padding   : 2; //
**    // uint32_t     pkts      :20; //        Count of packets
**
**    uint32_t     tcp_state;        // 12     TCP state machine info
**    uint32_t     rest_flags;       // 13     is_tcp==0: Flow's reported flags
**                                   //        is_tcp==1 &&
**                                   //          tcp_state==0:Empty
**                                   //          tcp_state==1:TCPflags/!1st pkt
**    uint32_t     application;      // 14-15  Type of traffic
**
**    uint16_t      sPort;           // 16-17  Source port
**    uint16_t      dPort;           // 18-19  Destination port
**
**    uint16_t      input;           // 20-21  Router incoming SNMP interface
**    uint16_t      output;          // 22-23  Router outgoing SNMP interface
**
**    uint32_t      sIP;             // 24-27  Source IP
**    uint32_t      dIP;             // 28-31  Destination IP
**
**    uint32_t      nhIP;            // 32-35  Router Next Hop IP
**
**
**  36 bytes on disk.
*/

#define RECLEN_RWAUGROUTING_V4 36


/*
 *    Byte swap the RWAUGROUTING v4 record 'ar' in place.
 */
#define augroutingioRecordSwap_V4(ar)                           \
    {                                                           \
        SWAP_DATA32((ar) +  0);   /* stime_bb1 */               \
        SWAP_DATA32((ar) +  4);   /* bb2_elapsed */             \
        SWAP_DATA32((ar) +  8);   /* pro_flg_pkts */            \
        /* two single bytes (12)tcp_state, (13)rest_flags */    \
        SWAP_DATA16((ar) + 14);   /* application */             \
        SWAP_DATA16((ar) + 16);   /* sPort */                   \
        SWAP_DATA16((ar) + 18);   /* dPort */                   \
        SWAP_DATA16((ar) + 20);   /* input */                   \
        SWAP_DATA16((ar) + 22);   /* output */                  \
        SWAP_DATA32((ar) + 24);   /* sIP */                     \
        SWAP_DATA32((ar) + 28);   /* dIP */                     \
        SWAP_DATA32((ar) + 32);   /* nhIP */                    \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int augroutingioRecordUnpack_V4(
    skstream_t             *rwIOS,
    rwGenericRec_V5        *rwrec,
    uint8_t                *ar)
{
    /* swap if required */
    if (rwIOS->swapFlag) {
        augroutingioRecordSwap_V4(ar);
    }

    /* sTime, elapsed, pkts, bytes, proto, tcp-flags, state, application */
    rwpackUnpackFlagsTimesVolumes(rwrec, ar, rwIOS->hdr_starttime, 16, 0);

    /* sPort, dPort */
    rwRecMemSetSPort(rwrec, &ar[16]);
    rwRecMemSetDPort(rwrec, &ar[18]);

    /* input, output */
    rwRecMemSetInput(rwrec, &ar[20]);
    rwRecMemSetOutput(rwrec, &ar[22]);

    /* sIP, dIP, nhIP */
    rwRecMemSetSIPv4(rwrec, &ar[24]);
    rwRecMemSetDIPv4(rwrec, &ar[28]);
    rwRecMemSetNhIPv4(rwrec, &ar[32]);

    /* sensor, flow_type from file name/header */
    rwRecSetSensor(rwrec, rwIOS->hdr_sensor);
    rwRecSetFlowType(rwrec, rwIOS->hdr_flowtype);

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int augroutingioRecordPack_V4(
    skstream_t             *rwIOS,
    const rwGenericRec_V5  *rwrec,
    uint8_t                *ar)
{
    int rv = SKSTREAM_OK; /* return value */

    /* sTime, elapsed, pkts, bytes, proto, tcp-flags, state, application */
    rv = rwpackPackFlagsTimesVolumes(ar, rwrec, rwIOS->hdr_starttime, 16);
    if (rv) {
        return rv;
    }

    /* sPort, dPort */
    rwRecMemGetSPort(rwrec, &ar[16]);
    rwRecMemGetDPort(rwrec, &ar[18]);

    /* input, output */
    rwRecMemGetInput(rwrec, &ar[20]);
    rwRecMemGetOutput(rwrec, &ar[22]);

    /* sIP, dIP, nhIP */
    rwRecMemGetSIPv4(rwrec, &ar[24]);
    rwRecMemGetDIPv4(rwrec, &ar[28]);
    rwRecMemGetNhIPv4(rwrec, &ar[32]);

    /* swap if required */
    if (rwIOS->swapFlag) {
        augroutingioRecordSwap_V4(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
**  RWAUGROUTING VERSION 1
**  RWAUGROUTING VERSION 2
**  RWAUGROUTING VERSION 3
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      pkts_stime;      // 12-15
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     sTime     :12; //        Start time--offset from hour
**
**    uint32_t      bbe;             // 16-19
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     elapsed   :12; //        Duration of flow
**
**    uint32_t      msec_flags       // 20-23
**    // uint32_t     sTime_msec:10; //        Fractional sTime (millisec)
**    // uint32_t     elaps_msec:10; //        Fractional elapsed (millisec)
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     padding   : 2; //        padding/reserved
**    // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
**                                   //        is_tcp==1 &&
**                                   //          tcp_state==0:TCPflags/All pkts
**                                   //          tcp_state==1:TCPflags/1st pkt
**
**    uint16_t      application;     // 24-25  Type of traffic
**
**    uint8_t       tcp_state;       // 26     TCP state machine info
**    uint8_t       rest_flags;      // 27     is_tcp==0: Flow's reported flags
**                                   //        is_tcp==1 &&
**                                   //          tcp_state==0:Empty
**                                   //          tcp_state==1:TCPflags/!1st pkt
**
**    uint32_t      nhIP;            // 28-31  Router Next Hop IP
**
**    uint16_t      input;           // 32-33  Router incoming SNMP interface
**    uint16_t      output;          // 34-35  Router outgoing SNMP interface
**
**
**  36 bytes on disk.
*/

#define RECLEN_RWAUGROUTING_V1 36
#define RECLEN_RWAUGROUTING_V2 36
#define RECLEN_RWAUGROUTING_V3 36


/*
 *    Byte swap the RWAUGROUTING v1 record 'ar' in place.
 */
#define augroutingioRecordSwap_V1(ar)                           \
    {                                                           \
        SWAP_DATA32((ar) +  0);   /* sIP */                     \
        SWAP_DATA32((ar) +  4);   /* dIP */                     \
        SWAP_DATA16((ar) +  8);   /* sPort */                   \
        SWAP_DATA16((ar) + 10);   /* dPort */                   \
        SWAP_DATA32((ar) + 12);   /* pkts_stime */              \
        SWAP_DATA32((ar) + 16);   /* bbe */                     \
        SWAP_DATA32((ar) + 20);   /* msec_flags */              \
        SWAP_DATA16((ar) + 24);   /* application */             \
        /* Two single bytes: (26)tcp_state, (27)rest_flags */   \
        SWAP_DATA32((ar) + 28);   /* nhIP */                    \
        SWAP_DATA16((ar) + 32);   /* input */                   \
        SWAP_DATA16((ar) + 34);   /* output */                  \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int augroutingioRecordUnpack_V1(
    skstream_t             *rwIOS,
    rwGenericRec_V5        *rwrec,
    uint8_t                *ar)
{
    uint32_t msec_flags;
    uint8_t is_tcp, prot_flags;

    /* swap if required */
    if (rwIOS->swapFlag) {
        augroutingioRecordSwap_V1(ar);
    }

    /* sIP, dIP, sPort, dPort */
    rwRecMemSetSIPv4(rwrec, &ar[0]);
    rwRecMemSetDIPv4(rwrec, &ar[4]);
    rwRecMemSetSPort(rwrec, &ar[8]);
    rwRecMemSetDPort(rwrec, &ar[10]);

    /* msec times, proto or flags */
    memcpy(&msec_flags, &ar[20],  4);

    /* application */
    rwRecMemSetApplication(rwrec, &ar[24]);

    /* sTime, pkts, bytes, elapsed, proto, tcp-flags, bpp */
    rwpackUnpackTimeBytesPktsFlags(rwrec, rwIOS->hdr_starttime,
                                   (uint32_t*)&ar[12], (uint32_t*)&ar[16],
                                   &msec_flags);

    /* Augrouting TCP information */
    is_tcp = (uint8_t)GET_MASKED_BITS(msec_flags, 10, 1);
    prot_flags = (uint8_t)GET_MASKED_BITS(msec_flags, 0, 8);
    rwpackUnpackProtoFlags(rwrec, is_tcp, prot_flags, ar[26], ar[27]);

    /* next hop, input & output interfaces */
    rwRecMemSetNhIPv4(rwrec, &ar[28]);
    rwRecMemSetInput(rwrec, &ar[32]);
    rwRecMemSetOutput(rwrec, &ar[34]);

    /* sensor, flow_type from file name/header */
    rwRecSetSensor(rwrec, rwIOS->hdr_sensor);
    rwRecSetFlowType(rwrec, rwIOS->hdr_flowtype);

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int augroutingioRecordPack_V1(
    skstream_t             *rwIOS,
    const rwGenericRec_V5  *rwrec,
    uint8_t                *ar)
{
    int rv = SKSTREAM_OK; /* return value */
    uint32_t msec_flags;
    uint8_t is_tcp, prot_flags;

    /* sTime, pkts, bytes, elapsed, proto, tcp-flags, bpp */
    rv = rwpackPackTimeBytesPktsFlags((uint32_t*)&ar[12], (uint32_t*)&ar[16],
                                      &msec_flags,
                                      rwrec, rwIOS->hdr_starttime);
    if (rv) {
        return rv;
    }

    rwpackPackProtoFlags(&is_tcp, &prot_flags, &ar[26], &ar[27], rwrec);

    /* msec_flags: sTime_msec:10; elaps_msec:10; pflag:1;
     *             is_tcp:1; pad:2; prot_flags:8; */
    /* overwrite the least significant 11 bits */
    msec_flags = ((msec_flags & (MASKARRAY_21 << 11))
                  | (is_tcp ? (1 << 10) : 0)
                  | prot_flags);

    /* sIP, dIP, sPort, dPort */
    rwRecMemGetSIPv4(rwrec, &ar[0]);
    rwRecMemGetDIPv4(rwrec, &ar[4]);
    rwRecMemGetSPort(rwrec, &ar[8]);
    rwRecMemGetDPort(rwrec, &ar[10]);

    /* msec_flags */
    memcpy(&ar[20], &msec_flags, 4);

    /* application */
    rwRecMemGetApplication(rwrec, &ar[24]);

    /* next hop, input & output interfaces */
    rwRecMemGetNhIPv4(rwrec, &ar[28]);
    rwRecMemGetInput(rwrec, &ar[32]);
    rwRecMemGetOutput(rwrec, &ar[34]);

    /* swap if required */
    if (rwIOS->swapFlag) {
        augroutingioRecordSwap_V1(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
 *  Return length of record of specified version, or 0 if no such
 *  version exists.  See skstream_priv.h for details.
 */
uint16_t augroutingioGetRecLen(fileVersion_t vers)
{
    switch (vers) {
      case 1:
        return RECLEN_RWAUGROUTING_V1;
      case 2:
        return RECLEN_RWAUGROUTING_V2;
      case 3:
        return RECLEN_RWAUGROUTING_V3;
      case 4:
        return RECLEN_RWAUGROUTING_V4;
      default:
        return 0;
    }
}


/*
 *  status = augroutingioPrepare(&rwIOSPtr);
 *
 *    Sets the record version to the default if it is unspecified,
 *    checks that the record format supports the requested record
 *    version, sets the record length, and sets the pack and unpack
 *    functions for this record format and version.
 */
int augroutingioPrepare(skstream_t *rwIOS)
{
#define FILE_FORMAT "FT_RWAUGROUTING"
    sk_file_header_t *hdr = rwIOS->silk_hdr;
    int rv = SKSTREAM_OK; /* return value */

    assert(skHeaderGetFileFormat(hdr) == FT_RWAUGROUTING);

    /* Set version if none was selected by caller */
    if ((rwIOS->io_mode == SK_IO_WRITE)
        && (skHeaderGetRecordVersion(hdr) == SK_RECORD_VERSION_ANY))
    {
        skHeaderSetRecordVersion(hdr, DEFAULT_RECORD_VERSION);
    }

    /* version check; set values based on version */
    switch (skHeaderGetRecordVersion(hdr)) {
      case 4:
        rwIOS->rwUnpackFn = &augroutingioRecordUnpack_V4;
        rwIOS->rwPackFn   = &augroutingioRecordPack_V4;
        break;
      case 3:
      case 2:
      case 1:
        /* V1 and V2 differ only in the padding of the header */
        /* V2 and V3 differ only in that V3 supports compression on
         * read and write; V2 supports compression only on read */
        rwIOS->rwUnpackFn = &augroutingioRecordUnpack_V1;
        rwIOS->rwPackFn   = &augroutingioRecordPack_V1;
        break;
      case 0:
      default:
        rv = SKSTREAM_ERR_UNSUPPORT_VERSION;
        goto END;
    }

    rwIOS->recLen = augroutingioGetRecLen(skHeaderGetRecordVersion(hdr));

    /* verify lengths */
    if (rwIOS->recLen == 0) {
        skAppPrintErr("Record length not set for %s version %u",
                      FILE_FORMAT, (unsigned)skHeaderGetRecordVersion(hdr));
        skAbort();
    }
    if (rwIOS->recLen != skHeaderGetRecordLength(hdr)) {
        if (0 == skHeaderGetRecordLength(hdr)) {
            skHeaderSetRecordLength(hdr, rwIOS->recLen);
        } else {
            skAppPrintErr(("Record length mismatch for %s version %u\n"
                           "\tcode = %" PRIu16 " bytes;  header = %lu bytes"),
                          FILE_FORMAT, (unsigned)skHeaderGetRecordVersion(hdr),
                          rwIOS->recLen,
                          (unsigned long)skHeaderGetRecordLength(hdr));
            skAbort();
        }
    }

  END:
    return rv;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
