/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  skbitmap.c
**
**    Bitmap creatation and deletion.
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: skbitmap.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* Return number of 32bit words needed to hold a bitmap with
 * 'num_bits' elements */
#define BITMAP_GET_WORD_COUNT(num_bits)                         \
    (((num_bits) >> 5) + ((((num_bits) & 0x1F)==0) ? 0 : 1))


/* FUNCTION DEFINITIONS */

/*
 *    Return the number of trailing zeros in 'v'.
 */
static uint8_t bitmapCountTrailingZeros(uint32_t v)
{
    uint8_t c = 1;
    if ((v & 0xffff) == 0) {
        v >>= 16;
        c += 16;
    }
    if ((v & 0xff) == 0) {
        v >>= 8;
        c += 8;
    }
    if ((v & 0xf) == 0) {
        v >>= 4;
        c += 4;
    }
    if ((v & 0x3) == 0) {
        v >>= 2;
        c += 2;
    }
    return c - (uint8_t)(v & 0x1);
}


int skBitmapBind(
    sk_bitmap_t    *bitmap,
    uint32_t        num_bits,
    uint32_t       *bitarray,
    size_t          sizeof_bitarray)
{
    uint32_t word_count = BITMAP_GET_WORD_COUNT(num_bits);

    if ( !(bitmap && num_bits && bitarray && sizeof_bitarray)) {
        return -1;
    }

    if (sizeof_bitarray < (word_count * sizeof(uint32_t))) {
        return -1;
    }

    memset(bitarray, 0, sizeof_bitarray);
    bitmap->map = bitarray;
    bitmap->num_bits = num_bits;
    bitmap->count = 0;

    return 0;
}


int skBitmapCreate(sk_bitmap_t **bitmap_out, uint32_t num_bits)
{
    uint32_t word_count = BITMAP_GET_WORD_COUNT(num_bits);

    assert(bitmap_out);

    if (num_bits == 0) {
        return -1;
    }

    *bitmap_out = calloc(1, sizeof(sk_bitmap_t));
    if (*bitmap_out == NULL) {
        return -1;
    }

    (*bitmap_out)->map = calloc(word_count, sizeof(uint32_t));
    if (NULL == (*bitmap_out)->map) {
        free(*bitmap_out);
        return -1;
    }

    (*bitmap_out)->num_bits = num_bits;

    return 0;
}


void skBitmapDestroy(sk_bitmap_t **bitmap)
{
    if (!bitmap || !*bitmap) {
        return;
    }

    free((*bitmap)->map);
    (*bitmap)->map = NULL;
    free(*bitmap);
    *bitmap = NULL;
}


void skBitmapClearAllBits(sk_bitmap_t *bitmap)
{
    uint32_t word_count;

    assert(bitmap);

    word_count = BITMAP_GET_WORD_COUNT(bitmap->num_bits);

    memset(bitmap->map, 0, word_count * sizeof(uint32_t));
    bitmap->count = 0;
}


uint32_t skBitmapGetSizeF(const sk_bitmap_t *bitmap)
{
    assert(bitmap);

    return bitmap->num_bits;
}


uint32_t skBitmapGetHighCountF(const sk_bitmap_t *bitmap)
{
    assert(bitmap);

    return bitmap->count;
}


int skBitmapGetBitF(const sk_bitmap_t *bitmap, uint32_t pos)
{
    assert(bitmap);
    assert(pos < bitmap->num_bits);

    return (_BMAP_IS_SET(bitmap, pos) ? 1 : 0);
}


void skBitmapSetBitF(sk_bitmap_t *bitmap, uint32_t pos)
{
    assert(bitmap);
    assert(pos < bitmap->num_bits);

    if ( !_BMAP_IS_SET(bitmap, pos)) {
        (bitmap)->map[_BMAP_INDEX(pos)] |= _BMAP_OFFSET(pos);
        ++(bitmap)->count;
    }
}


void skBitmapClearBitF(sk_bitmap_t *bitmap, uint32_t pos)
{
    assert(bitmap);
    assert(pos < bitmap->num_bits);

    if (_BMAP_IS_SET(bitmap, pos)) {
        (bitmap)->map[_BMAP_INDEX(pos)] &= ~_BMAP_OFFSET(pos);
        --(bitmap)->count;
    }
}


void skBitmapComplement(sk_bitmap_t *bitmap)
{
    uint32_t i, limit;
    uint32_t bits;

    assert(bitmap);

    limit = BITMAP_GET_WORD_COUNT(bitmap->num_bits);
    bitmap->count = 0;

    /* if last word is not fully used, need to handle it specially so
     * we don't complement bits that are not in use */
    if ((bitmap->num_bits & 0x1F) != 0) {
        --limit;
        bitmap->map[limit] = GET_MASKED_BITS(~bitmap->map[limit], 0,
                                             (bitmap->num_bits & 0x1F));
        BITS_IN_WORD(&bits, bitmap->map[limit]);
        bitmap->count += bits;
    }

    for (i = 0; i < limit; i++) {
        bitmap->map[i] = ~bitmap->map[i];
        BITS_IN_WORD(&bits, bitmap->map[i]);
        bitmap->count += bits;
    }
}


int skBitmapIntersection(sk_bitmap_t *dest, const sk_bitmap_t *src)
{
    uint32_t i, limit;
    uint32_t bits;

    assert(dest);
    assert(src);

    if (dest->num_bits != src->num_bits) {
        return -1;
    }
    limit = BITMAP_GET_WORD_COUNT(src->num_bits);
    dest->count = 0;
    for (i = 0; i < limit; i++) {
        dest->map[i] &= src->map[i];
        BITS_IN_WORD(&bits, dest->map[i]);
        dest->count += bits;
    }

    return 0;
}


int skBitmapUnion(sk_bitmap_t *dest, const sk_bitmap_t *src)
{
    uint32_t i, limit;
    uint32_t bits;

    assert(dest);
    assert(src);

    if (dest->num_bits != src->num_bits) {
        return -1;
    }
    limit = BITMAP_GET_WORD_COUNT(src->num_bits);
    dest->count = 0;
    for (i = 0; i < limit; i++) {
        dest->map[i] |= src->map[i];
        BITS_IN_WORD(&bits, dest->map[i]);
        dest->count += bits;
    }

    return 0;
}


void skBitmapIteratorBind(const sk_bitmap_t *bitmap, sk_bitmap_iter_t *iter)
{
    assert(bitmap);
    assert(iter);

    memset(iter, 0, sizeof(sk_bitmap_iter_t));
    iter->bitmap = bitmap;
    skBitmapIteratorReset(iter);
}


int skBitmapIteratorNext(sk_bitmap_iter_t *iter, uint32_t *pos)
{
    uint32_t limit;
    uint32_t bits;

    assert(iter);
    assert(pos);

    if (iter->bits) {
        /* there are still bits in the current word */
        iter->pos += bitmapCountTrailingZeros(iter->bitmap->map[iter->map_idx]
                                              >> iter->pos);
        *pos = ((iter->map_idx << 5) | iter->pos);
        ++iter->pos;
        --iter->bits;
        return SK_ITERATOR_OK;
    }

    limit = BITMAP_GET_WORD_COUNT(iter->bitmap->num_bits);
    if (limit == iter->map_idx) {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }

    /* find next word with bits */
    for (++iter->map_idx; iter->map_idx < limit; ++iter->map_idx) {
        BITS_IN_WORD(&bits, iter->bitmap->map[iter->map_idx]);
        if (bits) {
            iter->pos
                = bitmapCountTrailingZeros(iter->bitmap->map[iter->map_idx]);
            *pos = ((iter->map_idx << 5) | iter->pos);
            ++iter->pos;
            iter->bits = (uint8_t)bits - 1;
            return SK_ITERATOR_OK;
        }
    }

    return SK_ITERATOR_NO_MORE_ENTRIES;
}


void skBitmapIteratorReset(sk_bitmap_iter_t *iter)
{
    uint32_t limit;
    uint32_t bits;

    assert(iter);

    limit = BITMAP_GET_WORD_COUNT(iter->bitmap->num_bits);
    for (iter->map_idx = 0; iter->map_idx < limit; ++iter->map_idx) {
        BITS_IN_WORD(&bits, iter->bitmap->map[iter->map_idx]);
        if (bits) {
            iter->bits = (uint8_t)bits;
            iter->pos = 0;
            return;
        }
    }
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
