/*
** Copyright (C) 2008-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  skipset.c
**
**    skipset.c provides a data structure for maintaining IP
**    addresses.  The implementation uses a Radix Tree (aka Patricia
**    Trie) to keep IP addresses and their prefixes.  The
**    implementation can support IPv4 or IPv6 addresses, though each
**    instance of an IPset can only hold one type of IP address.
**
**    skipset.c is a replacement for the skIPTree_t data structure
**    defined in iptree.c.
**
**    Mark Thomas
**    September 2008
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skipset.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skipset.h>
#include <silk/utils.h>
#include <silk/skvector.h>


/*
**  IMPLEMENTATION
**
**    The nodes in the radix tree are stored in a single array---this
**    array is allocated as a uint8_t[] to support IPv4 and IPv6
**    nodes.  When the array is full, the space is realloc()ed.
**    "Pointers" to other nodes in the tree are stored as offsets into
**    the array, though these offsets are in terms of nodes, not
**    bytes.
**
**    Each node has indexes to the left and right nodes (ie, a leaf is
**    identical to an internal node).  Each node has a complete IP
**    address; a 'prefix' value in the node says how many of the bits
**    in that IP address are valid.  If the 'prefix' does not
**    represent a complete address (either 32 or 128) and the node has
**    no children, the node prepresents a CIDR block.
**
**    The IPset object maintains a list of 'free' nodes in the array.
**    When a node is "deleted", it is moved into the free list.  These
**    nodes are linked together through the "left" pointer on each
**    node.  When a new node is requested, nodes are handed out from
**    the end of the array first; when there are no more entries at
**    the end of the array, the free list is used.  If the free list
**    is also empty, the array is reallocated.
**
**    The on-disk storage matches the in-core storage.  This allows us
**    to mmap() the data section of the file when reading a set---as
**    long as the set is in native byte order and the data section is
**    not compressed.
*/


/* LOCAL DEFINES AND TYPEDEFS */

/* Legacy version of the IPset file */
#define  IPSET_LEGACY_VERSION  2


#define skIPTreeNodeSet(addr, node)                                     \
    ((node)->addressBlock[(((addr) & 0xFFFF)>>5)] |= (1 << ((addr)&0x1F)))

#define skIPTreeNodeClear(addr, node)                                   \
    ((node)->addressBlock[(((addr) & 0xFFFF)>>5)] &= ~(1 << ((addr)&0x1F)))

#define IPTREE_NODE_ALLOC(high16)                                       \
    if (ipset->iptree->nodes[(high16)]) { /*no-op*/ } else {            \
        ipset->iptree->nodes[(high16)] = calloc(1, sizeof(skIPNode_t)); \
        if (NULL == ipset->iptree->nodes[(high16)]) {                   \
            return SKIP_ERR_ALLOC;                                      \
        }                                                               \
    }



/* THE IPSET Structure */
struct skipset_st {
    skIPTree_t *iptree;
};




/* LOCAL VARIABLE DEFINITIONS */



/* LOCAL FUNCTION PROTOTYPES */



/* FUNCTION DEFINITIONS */


/*
 *  status = ipsetCreate(ipset, iptree);
 *
 *    Create a new IPset at the address specified by 'ipset'.  If
 *    'iptree' is non-NULL, use it as the IPTree object that 'ipset'
 *    wraps.  Otherwise, allocate a new IPTree.  Return SKIPSET_OK on
 *    success, or SKIPSET_ERR_ALLOC for memory allocation failure.
 */
static int ipsetCreate(
    skipset_t         **ipset,
    skIPTree_t         *iptree)
{
    int rv = 0;

    *ipset = calloc(1, sizeof(skipset_t));
    if (!*ipset) {
        return SKIPSET_ERR_ALLOC;
    }

    if (iptree) {
        (*ipset)->iptree = iptree;
    } else {
        rv = skIPTreeCreate(&((*ipset)->iptree));
        if (rv) {
            free(*ipset);
            *ipset = NULL;
        }
    }
    return rv;
}


/* Return true if 'ipset' contains 'ipaddr' */
int skIPSetCheckAddress(
    const skipset_t    *ipset,
    const skipaddr_t   *ipaddr)
{
    uint32_t ipv4;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        /* attempt to convert to IPv4 */
        if (skipaddrGetAsV4(ipaddr, &ipv4)) {
            /* conversion failed; this IP is not in the IPSet */
            return 0;
        }
    } else
#endif
    {
        ipv4 = skipaddrGetV4(ipaddr);
    }

    return skIPTreeCheckAddress(ipset->iptree, ipv4);
}


int skIPSetCheckIPSet(
    const skipset_t    *ipset1,
    const skipset_t    *ipset2)
{
    return skIPTreeCheckIntersectIPTree(ipset1->iptree, ipset2->iptree);
}


/* Return true if 'ipset' contains any IPs represened by 'ipwild' */
int skIPSetCheckIPWildcard(
    const skipset_t        *ipset,
    const skIPWildcard_t   *ipwild)
{
    return skIPTreeCheckIntersectIPWildcard(ipset->iptree, ipwild);
}


/* Return true if the specified address on 'rwrec' is in the 'ipset' */
int skIPSetCheckRecord(
    const skipset_t        *ipset,
    const rwRec            *rwrec,
    int                     src_dst_nh)
{
    uint32_t ipv4;

#if SK_ENABLE_IPV6
    skipaddr_t ipaddr;

    if (rwRecIsIPv6(rwrec)) {
        /* attempt to convert to IPv4 */
        switch (src_dst_nh) {
          case 1:
            rwRecMemGetSIPv6(rwrec, &ipaddr);
            break;
          case 2:
            rwRecMemGetDIPv6(rwrec, &ipaddr);
            break;
          case 4:
            rwRecMemGetNhIPv6(rwrec, &ipaddr);
            break;
          default:
            skAbortBadCase(src_dst_nh);
        }
        if (skipaddrGetAsV4(&ipaddr, &ipv4)) {
            /* conversion failed; this IP is not in the IPSet */
            return 0;
        }
    } else
#endif
    {
        switch (src_dst_nh) {
          case 1:
            ipv4 = rwRecGetSIPv4(rwrec);
            break;
          case 2:
            ipv4 = rwRecGetDIPv4(rwrec);
            break;
          case 4:
            ipv4 = rwRecGetNhIPv4(rwrec);
            break;
          default:
            skAbortBadCase(src_dst_nh);
        }
    }

    return skIPTreeCheckAddress(ipset->iptree, ipv4);
}


int skIPSetClean(
    skipset_t          UNUSED(*ipset))
{
    return 0;
}


/* Return true if 'ipset' contains any IPs that cannot be represented
 * as an IPv4 address. */
int skIPSetContainsV6(
    const skipset_t    UNUSED(*ipset))
{
    return 0;
}


int skIPSetConvert(
    skipset_t          *ipset,
    int                 target_ip_version)
{
    if (ipset == NULL) {
        return SKIPSET_ERR_BADINPUT;
    }
    switch (target_ip_version) {
      case 4:
        return SKIPSET_OK;
      case 6:
        return SKIPSET_ERR_IPV6;
      default:
        return SKIPSET_ERR_BADINPUT;
    }
}


/* Return number of IPs in the IPset */
uint64_t skIPSetCountIPs(
    const skipset_t    *ipset,
    double             *count)
{
    uint64_t c = skIPTreeCountIPs(ipset->iptree);

    if (count) {
        *count = (double)c;
    }
    return c;
}


/* Create a new IPset */
int skIPSetCreate(
    skipset_t         **ipset,
    int                 support_ipv6)
{
    if (!ipset) {
        return SKIPSET_ERR_BADINPUT;
    }

    if (support_ipv6) {
        return SKIPSET_ERR_IPV6;
    }

    return ipsetCreate(ipset, NULL);
}


/* Destroy an IPset. */
void skIPSetDestroy(
    skipset_t         **ipset)
{
    if (!ipset) {
        return;
    }

    skIPTreeDelete(&((*ipset)->iptree));
    free(*ipset);
    *ipset = NULL;
}


/* Insert the IP address into the IPset. */
int skIPSetInsertAddress(
    skipset_t          *ipset,
    const skipaddr_t   *ipaddr,
    uint32_t            prefix)
{
    uint32_t ipv4;
    uint32_t ipv4_end;

    assert(ipset);

#if  SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        /* attempt to convert V6 ipaddr to V4 */
        if (skipaddrGetAsV4(ipaddr, &ipv4)) {
            /* cannot store V6 ipaddr in a V4 IPSet */
            return SKIPSET_ERR_IPV6;
        }
        if (prefix > 128) {
            return SKIPSET_ERR_PREFIX;
        }
        if ((128 == prefix) || (0 == prefix)) {
            prefix = 32;
            ipv4_end = ipv4;
        } else if (prefix <= 96) {
            return SKIPSET_ERR_IPV6;
        } else {
            prefix -= 96;
            /* max IP */
            ipv4_end = (UINT32_MAX >> prefix) | ipv4;
            /* min IP */
            ipv4 &= ~(UINT32_MAX >> prefix);
        }
    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        if (prefix > 32) {
            return SKIPSET_ERR_PREFIX;
        }
        if (0 == prefix) {
            prefix = 32;
            ipv4_end = ipv4 = skipaddrGetV4(ipaddr);
        } else if (32 == prefix) {
            ipv4_end = ipv4 = skipaddrGetV4(ipaddr);
        } else {
            /* max IP */
            ipv4_end = (UINT32_MAX >> prefix) | skipaddrGetV4(ipaddr);
            /* min IP */
            ipv4 = ipv4_end & ~(UINT32_MAX >> prefix);
        }
    }

    if (prefix <= 16) {
        ipv4 >>= 16;
        ipv4_end >>= 16;
        do {
            IPTREE_NODE_ALLOC(ipv4);
            memset(ipset->iptree->nodes[ipv4]->addressBlock,
                   0xFF, sizeof(skIPNode_t));
        } while (ipv4++ < ipv4_end);

    } else if (prefix <= 27) {
        ipv4 >>= 5;
        ipv4_end >>= 5;
        do {
            /* nodes are allocated on /16, already shifted by 5, need
             * to shift by 11 more to get the complete shift by 16.
             * Mask by 0x7ff (0xffff>>5) to get addressBlock */
            IPTREE_NODE_ALLOC(ipv4 >> 11);
            memset(&ipset->iptree->nodes[ipv4>>11]->addressBlock[ipv4 & 0x07FF],
                   0xFF, sizeof(uint32_t));
        } while (ipv4++ < ipv4_end);

    } else {
        do {
            IPTREE_NODE_ALLOC(ipv4 >> 16);
            skIPTreeNodeSet(ipv4, ipset->iptree->nodes[ipv4 >> 16]);
        } while (ipv4++ < ipv4_end);
    }
    return SKIPSET_OK;
}


/* Add each IP in the IPWildcard to the IPset. */
int skIPSetInsertIPWildcard(
    skipset_t              *ipset,
    const skIPWildcard_t   *ipwild)
{
    return skIPTreeAddIPWildcard(ipset->iptree, ipwild);
}


int skIPSetInsertRange(
    skipset_t          *ipset,
    const skipaddr_t   *ipaddr_start,
    const skipaddr_t   *ipaddr_end)
{
    uint32_t ipv4;
    uint32_t ipv4_end;
    int rv;

    rv = skipaddrCompare(ipaddr_start, ipaddr_end);
    if (rv > 0) {
        return SKIPSET_ERR_BADINPUT;
    }

    assert(ipset);

#if  SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr_start)) {
        /* attempt to convert V6 ipaddr to V4 */
        if (skipaddrGetAsV4(ipaddr_start, &ipv4)) {
            /* cannot store V6 ipaddr in a V4 IPSet */
            return SKIPSET_ERR_IPV6;
        }
    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        ipv4 = skipaddrGetV4(ipaddr_start);
    }

#if  SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr_end)) {
        /* attempt to convert V6 ipaddr to V4 */
        if (skipaddrGetAsV4(ipaddr_end, &ipv4_end)) {
            /* cannot store V6 ipaddr in a V4 IPSet */
            return SKIPSET_ERR_IPV6;
        }
    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        ipv4_end = skipaddrGetV4(ipaddr_end);
    }

    /* Make this more efficient */
    do {
        IPTREE_NODE_ALLOC(ipv4 >> 16);
        skIPTreeNodeSet(ipv4, ipset->iptree->nodes[ipv4 >> 16]);
    } while (ipv4++ < ipv4_end);

    return SKIPSET_OK;
}


/* Make 'result_ipset' hold the intersection of itself with 'ipset' */
int skIPSetIntersect(
    skipset_t          *result_ipset,
    const skipset_t    *ipset)
{
    skIPTreeIntersect(result_ipset->iptree, ipset->iptree);
    return SKIPSET_OK;
}


/* Return true if 'ipset' can hold IPv6 addresses */
int skIPSetIsV6(
    const skipset_t     UNUSED(*ipset))
{
    return 0;
}


/* Bind the iterator to an IPset */
int skIPSetIteratorBind(
    skipset_iterator_t *iter,
    const skipset_t    *ipset,
    uint32_t            cidr_blocks,
    sk_ipv6policy_t     policy)
{
    if (!ipset || !iter) {
        return SKIPSET_ERR_BADINPUT;
    }
    if (cidr_blocks > 1) {
        return SKIPSET_ERR_BADINPUT;
    }

    memset(iter, 0, sizeof(skipset_iterator_t));
    iter->ipv6_policy = policy;
    iter->cidr_blocks = cidr_blocks;
    if (cidr_blocks) {
        return skIPTreeCIDRBlockIteratorBind(&iter->iter.cidr, ipset->iptree);
    }
    return skIPTreeIteratorBind(&iter->iter.ip, ipset->iptree);
}


/* Return the next IP or CIDR block associated with the IPset */
int skIPSetIteratorNext(
    skipset_iterator_t *iter,
    skipaddr_t         *ipaddr,
    uint32_t           *prefix)
{
    skIPTreeCIDRBlock_t cidr;
    int rv;

    if (iter->cidr_blocks) {
        rv = skIPTreeCIDRBlockIteratorNext(&cidr, &iter->iter.cidr);
    } else {
        rv = skIPTreeIteratorNext(&cidr.addr, &iter->iter.ip);
        cidr.mask = 32;
    }
    if (rv == SK_ITERATOR_OK) {
#if !SK_ENABLE_IPV6
        skipaddrSetV4(ipaddr, &cidr.addr);
        *prefix = cidr.mask;
#else
        switch (iter->ipv6_policy) {
          case SK_IPV6POLICY_ONLY:
            /* since we do not yet support IPv6 addresses in IPsets,
             * there is nothing to return */
            return SK_ITERATOR_NO_MORE_ENTRIES;

          case SK_IPV6POLICY_FORCE:
            skipaddrSetV6FromUint32(ipaddr, &cidr.addr);
            *prefix = 96 + cidr.mask;
            break;

          case SK_IPV6POLICY_MIX:
          case SK_IPV6POLICY_ASV4:
          case SK_IPV6POLICY_IGNORE:
            skipaddrSetV4(ipaddr, &cidr.addr);
            *prefix = cidr.mask;
            break;
        }
#endif  /* SK_ENABLE_IPV6 */
    }
    return rv;
}


/* Reset iterator so we can visit the IPs again */
void skIPSetIteratorReset(
    skipset_iterator_t *iter)
{
    if (iter->cidr_blocks) {
        skIPTreeCIDRBlockIteratorReset(&iter->iter.cidr);
    } else {
        skIPTreeIteratorReset(&iter->iter.ip);
    }
}


/* Read IPSet from filename---a wrapper around skIPSetRead(). */
int skIPSetLoad(
    skipset_t         **ipset,
    const char         *filename)
{
    skstream_t *stream = NULL;
    int rv;

    if (filename == NULL || ipset == NULL) {
        return SKIPSET_ERR_BADINPUT;
    }

    if ((rv = skStreamCreate(&stream, SK_IO_READ, SK_CONTENT_SILK))
        || (rv = skStreamBind(stream, filename))
        || (rv = skStreamOpen(stream)))
    {
        /* skStreamPrintLastErr(stream, rv, &skAppPrintErr); */
        rv = SKIPSET_ERR_OPEN;
        goto END;
    }

    rv = skIPSetRead(ipset, stream);
    if (rv) {
        goto END;
    }

  END:
    skStreamDestroy(&stream);
    return rv;
}


/* For each occupied block of size 'prefix', set a single IP */
int skIPSetMask(
    skipset_t          *ipset,
    uint32_t            prefix)
{
    if (!ipset) {
        return SKIPSET_ERR_BADINPUT;
    }

    /* verify prefix value is valid */
    if (prefix == 0 || prefix >= 32) {
        return SKIPSET_ERR_PREFIX;
    }

    skIPTreeMask(ipset->iptree, prefix);
    return SKIPSET_OK;
}


/* Print the IPs in ipset to 'stream'. */
void skIPSetPrint(
    const skipset_t    *ipset,
    skstream_t         *stream,
    skipaddr_flags_t    ip_format,
    int                 as_cidr)
{
    skIPTreePrint(ipset->iptree, stream, ip_format, as_cidr);
}


int skIPSetRead(
    skipset_t         **ipset,
    skstream_t         *stream)
{
    skIPTree_t *iptree = NULL;
    int rv;

    if (stream == NULL || ipset == NULL) {
        return SKIPSET_ERR_BADINPUT;
    }

    rv = skIPTreeRead(&iptree, stream);
    if (rv != SKIP_OK) {
        return rv;
    }

    rv = ipsetCreate(ipset, iptree);
    if (rv) {
        skIPTreeDelete(&iptree);
        return rv;
    }

    return SKIPSET_OK;
}


int skIPSetRemoveAddress(
    skipset_t          *ipset,
    const skipaddr_t   *ipaddr,
    uint32_t            prefix)
{
    uint32_t ipv4;
    uint32_t ipv4_end;

    assert(ipset);

#if  SK_ENABLE_IPV6
    if (skipaddrIsV6(ipaddr)) {
        /* attempt to convert V6 ipaddr to V4 */
        if (skipaddrGetAsV4(ipaddr, &ipv4)) {
            /* cannot store V6 ipaddr in a V4 IPSet */
            return SKIPSET_ERR_IPV6;
        }

        if (prefix > 128) {
            return SKIPSET_ERR_PREFIX;
        }
        if ((128 == prefix) || (0 == prefix)) {
            ipv4_end = ipv4;
        } else if (prefix <= 96) {
            return SKIPSET_ERR_IPV6;
        } else {
            prefix -= 96;
            /* max IP */
            ipv4_end = (UINT32_MAX >> prefix) | ipv4;
            /* min IP */
            ipv4 &= ~(UINT32_MAX >> prefix);
        }
    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        if (prefix > 32) {
            return SKIPSET_ERR_PREFIX;
        }
        if ((32 == prefix) || (0 == prefix)) {
            ipv4_end = ipv4 = skipaddrGetV4(ipaddr);
        } else {
            /* max IP */
            ipv4_end = (UINT32_MAX >> prefix) | skipaddrGetV4(ipaddr);

            /* min IP */
            ipv4 = ipv4_end & ~(UINT32_MAX >> prefix);
        }
    }

    /* Make this more efficient */
    do {
        if (ipset->iptree->nodes[ipv4 >> 16]) {
            skIPTreeNodeClear(ipv4, ipset->iptree->nodes[ipv4 >> 16]);
        }
    } while (ipv4++ < ipv4_end);

    return SKIPSET_OK;
}


int skIPSetRemoveAll(
    skipset_t          *ipset)
{
    return skIPTreeRemoveAll(ipset->iptree);
}


/* Write 'ipset' to 'filename'--a wrapper around skIPSetWrite(). */
int skIPSetSave(
    const skipset_t    *ipset,
    const char         *filename)
{
    skstream_t *stream = NULL;
    int rv;

    if (filename == NULL || ipset == NULL) {
        return SKIPSET_ERR_BADINPUT;
    }

    if ((rv = skStreamCreate(&stream, SK_IO_WRITE, SK_CONTENT_SILK))
        || (rv = skStreamBind(stream, filename))
        || (rv = skStreamOpen(stream)))
    {
        /* skStreamPrintLastErr(stream, rv, &skAppPrintErr); */
        rv = SKIPSET_ERR_OPEN;
        goto END;
    }

    rv = skIPSetWrite(ipset, stream);

  END:
    skStreamDestroy(&stream);
    return rv;
}


/* Convert 'error_code' to a string. */
const char *skIPSetStrerror(int error_code)
{
    static char errbuf[128];

    switch ((skipset_return_t)error_code) {
      case SKIPSET_OK:
        return "Success";
      case SKIPSET_ERR_EMPTY:
        return "IPset is empty";
      case SKIPSET_ERR_PREFIX:
        return "Invalid prefix";
      case SKIPSET_ERR_NOTFOUND:
        return "Value not found in IPset";
      case SKIPSET_ERR_ALLOC:
        return "Unable to allocate memory";
      case SKIPSET_ERR_BADINPUT:
        return "Empty input value";
      case SKIPSET_ERR_FILEIO:
        return "Error in read/write";
      case SKIPSET_ERR_FILETYPE:
        return "Input is not an IPset";
      case SKIPSET_ERR_OPEN:
        return "Error opening file";
      case SKIPSET_ERR_IPV6:
        return "IPsets do not support IPv6 addresses";
      case SKIPSET_ERR_FILEVERSION:
        return "Unsupported file version";
    }

    snprintf(errbuf, sizeof(errbuf),
             "Unrecognized IPset error code %d", error_code);
    return errbuf;
}


/* Turn off IPs of 'result_ipset' that are on in 'ipset'. */
int skIPSetSubtract(
    skipset_t          *result_ipset,
    const skipset_t    *ipset)
{
    skIPTreeSubtract(result_ipset->iptree, ipset->iptree);
    return SKIPSET_OK;
}


/* Turn on IPs of 'result_ipset' that are on in 'ipset'. */
int skIPSetUnion(
    skipset_t          *result_ipset,
    const skipset_t    *ipset)
{
    return skIPTreeUnion(result_ipset->iptree, ipset->iptree);
}


/* Invoke the 'callback' function on all IPs in the 'ipset' */
int skIPSetWalk(
    const skipset_t    *ipset,
    uint32_t            cidr_blocks,
    sk_ipv6policy_t     policy,
    skipset_walk_fn_t   callback,
    void               *data)
{
    skipaddr_t ipaddr;
    int rv;

    if (cidr_blocks > 1) {
        return SKIPSET_ERR_BADINPUT;
    }
    if (SK_IPV6POLICY_ONLY == policy) {
        return SKIPSET_OK;
    }

    if (cidr_blocks) {
        skIPTreeCIDRBlockIterator_t cidr_iter;
        skIPTreeCIDRBlock_t cidr;

        rv = skIPTreeCIDRBlockIteratorBind(&cidr_iter, ipset->iptree);
        if (rv) {
            return rv;
        }

#if SK_ENABLE_IPV6
        if (SK_IPV6POLICY_FORCE == policy) {
            while ((rv = skIPTreeCIDRBlockIteratorNext(&cidr, &cidr_iter))
                   == SK_ITERATOR_OK)
            {
                skipaddrSetV6FromUint32(&ipaddr, &cidr.addr);
                rv = (*callback)(&ipaddr, cidr.mask, data);
                if (rv) {
                    return rv;
                }
            }
        } else
#endif  /* SK_ENABLE_IPV6 */
        {
            while ((rv = skIPTreeCIDRBlockIteratorNext(&cidr, &cidr_iter))
                   == SK_ITERATOR_OK)
            {
                skipaddrSetV4(&ipaddr, &cidr.addr);
                rv = (*callback)(&ipaddr, cidr.mask, data);
                if (rv) {
                    return rv;
                }
            }
        }

    } else {
        skIPTreeIterator_t ip_iter;
        uint32_t ip;

        rv = skIPTreeIteratorBind(&ip_iter, ipset->iptree);
        if (rv) {
            return rv;
        }

#if SK_ENABLE_IPV6
        if (SK_IPV6POLICY_FORCE == policy) {
            while ((rv = skIPTreeIteratorNext(&ip, &ip_iter))
                   == SK_ITERATOR_OK)
            {
                skipaddrSetV6FromUint32(&ipaddr, &ip);
                rv = (*callback)(&ipaddr, 96, data);
                if (rv) {
                    return rv;
                }
            }
        } else
#endif  /* SK_ENABLE_IPV6 */
        {
            while ((rv = skIPTreeIteratorNext(&ip, &ip_iter))
                   == SK_ITERATOR_OK)
            {
                skipaddrSetV4(&ipaddr, &ip);
                rv = (*callback)(&ipaddr, 32, data);
                if (rv) {
                    return rv;
                }
            }
        }
    }

    return SKIPSET_OK;
}


int skIPSetWrite(
    const skipset_t    *ipset,
    skstream_t         *stream)
{
    return skIPTreeWrite(ipset->iptree, stream);
}


/* Write 'ipset' to 'stream' in the old IPTree format. */
int skIPSetWriteIPTree(
    const skipset_t    *ipset,
    skstream_t         *stream)
{
    return skIPTreeWrite(ipset->iptree, stream);
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
