/*
** Copyright (C) 2005-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skstringmap.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skstringmap.h>

/* DEFINES AND TYPEDEFS */

/*
 * The width of a print buffer needed to represent an integer.  A
 * 64-bit integer fits in 20 digits, plus an additional digit for the
 * NUL.  I added one more on top of that, just to be safe, and in
 * case we want to parse signed numbers in the future.
 */
#define SKSTRING_PRINT_DIGITS 22


/* LOCAL FUNCTION DECLARATIONS */

static sk_stringmap_status_t stringMapCheckValidName(
    sk_stringmap_t         *str_map,
    const char             *name);

static void stringMapFreeEntry(
    sk_stringmap_entry_t   *map_entry);

static sk_stringmap_status_t stringMapParseList(
    const char             *user_string,
    sk_dllist_t           **token_list,
    char                  **bad_token);

static sk_stringmap_status_t stringMapExpandRanges(
    sk_dllist_t            *token_list,
    char                  **bad_token,
    char                    range_delimiter);

static sk_stringmap_status_t stringMapFindEntry(
    const sk_stringmap_t   *str_map,
    const char             *token,
    sk_stringmap_entry_t  **map_entry);

static sk_stringmap_status_t stringMapTokenize(
    const char             *user_string,
    const char              delimiter,
    sk_dllist_t            *result_list);


/* FUNCTION DEFINITIONS */

/* Create a new string map */
sk_stringmap_status_t skStringMapCreate(
    sk_stringmap_t        **out_str_map)
{
    *out_str_map = skDLListCreate((sk_dll_free_fn_t)&stringMapFreeEntry);
    if (*out_str_map == NULL) {
        return SKSTRINGMAP_ERR_MEM;
    }

    return SKSTRINGMAP_OK;
}


/* Destroy a string map */
sk_stringmap_status_t skStringMapDestroy(
    sk_stringmap_t         *str_map)
{
    skDLListDestroy(str_map);
    return SKSTRINGMAP_OK;
}


/* add multiple keys to a StringMap */
sk_stringmap_status_t skStringMapAddEntries(
    sk_stringmap_t             *str_map,
    int                         entryc,
    const sk_stringmap_entry_t *entryv)
{
    sk_stringmap_entry_t *map_entry = NULL;
    const sk_stringmap_entry_t *e;
    sk_stringmap_status_t rv;
    int i;
    int rv_list;

    /* check inputs */
    if (str_map == NULL || entryv == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    if (entryc < 0) {
        /* a null-terminated array; get the element count */
        entryc = 0;
        while (entryv[entryc].name) {
            ++entryc;
        }
    }

    for (i = 0, e = entryv; i < entryc; ++i, ++e) {
        if (NULL == e->name) {
            return SKSTRINGMAP_ERR_INPUT;
        }

        /* check to see if the name is valid */
        rv = stringMapCheckValidName(str_map, e->name);
        if (SKSTRINGMAP_OK != rv) {
            return rv;
        }

        /* allocate entry */
        map_entry = malloc(sizeof(sk_stringmap_entry_t));
        if (NULL == map_entry) {
            return SKSTRINGMAP_ERR_MEM;
        }

        /* copy the entry from the caller */
        map_entry->id = e->id;
        map_entry->userdata = e->userdata;
        map_entry->description = NULL;

        /* duplicate strings for our own use */
        map_entry->name = strdup(e->name);
        if (NULL == map_entry->name) {
            rv = SKSTRINGMAP_ERR_MEM;
            goto ERROR;
        }
        if (e->description) {
            map_entry->description = strdup(e->description);
            if (NULL == map_entry->description) {
                rv = SKSTRINGMAP_ERR_MEM;
                goto ERROR;
            }
        }

        /* add entry to end of list */
        rv_list = skDLListPushTail(str_map, (void *)map_entry);
        if (rv_list != 0) {
            rv = SKSTRINGMAP_ERR_MEM;
            goto ERROR;
        }
    }

    return SKSTRINGMAP_OK;

  ERROR:
    stringMapFreeEntry(map_entry);
    return rv;
}


/* remove a key from a StringMap */
sk_stringmap_status_t skStringMapRemoveByName(
    sk_stringmap_t     *str_map,
    const char         *name)
{
    int rv_list;
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;

    /* check inputs */
    if (str_map == NULL || name == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    skDLLAssignIter(&map_node, str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        if (strcasecmp(map_entry->name, name) == 0) {
            rv_list = skDLLIterDel(&map_node);
            if (rv_list != 0) {
                assert(0);
                return SKSTRINGMAP_ERR_LIST;
            }
            stringMapFreeEntry(map_entry);
        }
    }

    return SKSTRINGMAP_OK;
}


/* remove all entries have given ID from a StringMap */
sk_stringmap_status_t skStringMapRemoveByID(
    sk_stringmap_t     *str_map,
    sk_stringmap_id_t   id)
{
    int rv_list;
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;

    /* check inputs */
    if (str_map == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    skDLLAssignIter(&map_node, str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        if (id == map_entry->id) {
            rv_list = skDLLIterDel(&map_node);
            if (rv_list != 0) {
                assert(0);
                return SKSTRINGMAP_ERR_LIST;
            }
            stringMapFreeEntry(map_entry);
        }
    }

    return SKSTRINGMAP_OK;
}


/* remove multiple keys from a StringMap */
sk_stringmap_status_t skStringMapRemoveEntries(
    sk_stringmap_t             *str_map,
    int                         entryc,
    const sk_stringmap_entry_t *entryv)
{
    int i;
    sk_stringmap_status_t rv;

    /* check inputs */
    if (str_map == NULL || entryv == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    for (i = 0; (i < entryc && NULL != entryv[i].name); ++i) {
        rv = skStringMapRemoveByName(str_map, entryv[i].name);
        if (rv != SKSTRINGMAP_OK) {
            return rv;
        }
    }

    return SKSTRINGMAP_OK;
}


/*
 *  stringMapFindEntry(str_map, token_string, &found_match);
 *
 *    Search in 'str_map' for an entry that matches 'token_string'.
 *    If found, set found_match to that entry and return
 *    SKSTRINGMAP_OK.  If an ambiguous entry is found, return
 *    SKSTRINGMAP_PARSE_AMBIGUOUS and set found_match to one of the
 *    ambiguous entries; if no match, return
 *    SKSTRINGMAP_PARSE_NO_MATCH, and set *found_match to NULL.
 */
static sk_stringmap_status_t stringMapFindEntry(
    const sk_stringmap_t   *str_map,
    const char             *token_string,
    sk_stringmap_entry_t  **found_match)
{
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;
    size_t len;
    int unique = 1;

    assert(found_match);
    assert(str_map);
    assert(token_string);

    len = strlen(token_string);
    *found_match = NULL;

    /* Typecast away const.  We are still treating it as const
     * though. */
    skDLLAssignIter(&map_node, (sk_stringmap_t *)str_map);
    /* check the token against each entry in the map */
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {

        if (0 != strncasecmp(map_entry->name, token_string, len)) {
            /* no match, try next entry in the map */
            continue;
        }

        /* first 'len' chars match.  see if exact match by comparing
         * string lengths */
        if (len == strlen(map_entry->name)) {
            /* exact match; set found_match here and return */
            *found_match = map_entry;
            return SKSTRINGMAP_OK;
        }
        /* else not an exact match. */

        if (isdigit((int)*token_string)) {
            /* partial number match doesn't make sense; try again */
            continue;
        }
        /* else partial match. */

        /* If '*found_match' has not been set, set it to this location
         * as a potential match.  Else if '*found_match' is set,
         * compare the IDs on the current entry and *found_match and
         * if they are different, return 'ambiguous'. */
        if (*found_match == NULL) {
            *found_match = map_entry;
        } else if ((*found_match)->id != map_entry->id) {
            /* The token_string matches two entries with different
             * IDs; mark as possibly ambiguous.  Continue in case
             * there is an exact match. */
            unique = 0;
        }
        /* else token matches two entries that map to same ID, so
         * allow it and don't complain. */
    }

    if (!unique) {
        /* Multiple matches were found, mark as ambiguous */
        return SKSTRINGMAP_PARSE_AMBIGUOUS;
    }

    /* did we find a match? */
    if (*found_match == NULL) {
        return SKSTRINGMAP_PARSE_NO_MATCH;
    }

    return SKSTRINGMAP_OK;
}


/* match a single key against a StringMap, filling out_entry with a
 * pointer to the entry. */
sk_stringmap_status_t skStringMapGetByName(
    const sk_stringmap_t   *str_map,
    const char             *user_string,
    sk_stringmap_entry_t  **out_entry)
{
    sk_stringmap_status_t rv;
    sk_stringmap_entry_t *found_match;

    /* check inputs */
    if (out_entry == NULL || str_map == NULL || user_string == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    rv = stringMapFindEntry(str_map, user_string, &found_match);
    if (rv == SKSTRINGMAP_OK) {
        *out_entry = found_match;
    }

    return rv;
}


/* parse a user string for a list of keys, and match those keys
 * againts a StringMap */
sk_stringmap_status_t skStringMapMatch(
    const sk_stringmap_t   *str_map,
    const char             *user_string,
    sk_vector_t            *v_result,
    char                  **bad_token)
{
    sk_stringmap_status_t rv;

    /* for parsing from comma delimited string into list of tokens */
    sk_dllist_t   *token_list = NULL;
    sk_dll_iter_t  token;

    /* for parsing list of tokens into individual names/ids */
    char *token_string;

    /* matching against string map */
    sk_stringmap_entry_t *found_match;

    /* check inputs */
    if (str_map == NULL || v_result == NULL || user_string == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    if (skVectorGetElementSize(v_result) != sizeof(sk_stringmap_entry_t*)) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    /* convert the user's input into a linked list of tokens */
    rv = stringMapParseList(user_string, &token_list, bad_token);
    if (rv != SKSTRINGMAP_OK) {
        goto END;
    }

    /* loop over the tokens we got from the user's input */
    skDLLAssignIter(&token, token_list);
    while (skDLLIterForward(&token, (void **)&token_string) == 0) {
        rv = stringMapFindEntry(str_map, token_string, &found_match);
        switch (rv) {
          case SKSTRINGMAP_OK:
            /* found a single match; add it to vector */
            if (0 != skVectorAppendValue(v_result, &found_match)) {
                rv = SKSTRINGMAP_ERR_MEM;
                goto END;
            }
            break;

          case SKSTRINGMAP_PARSE_AMBIGUOUS:
          case SKSTRINGMAP_PARSE_NO_MATCH:
            if (bad_token != NULL) {
                /* let the user know where things failed. */
                *bad_token = strdup(token_string);
            }
            goto END;

          default:
            /* bad news */
            goto END;
        }
    }

    /* success */
    rv = SKSTRINGMAP_OK;

  END:
    if (token_list != NULL) {
        skDLListDestroy(token_list);
    }

    return rv;
}


sk_stringmap_status_t skStringMapParse(
    const sk_stringmap_t   *str_map,
    const char             *user_string,
    sk_stringmap_dupes_t    handle_dupes,
    sk_vector_t            *out_vec,
    char                  **errmsg)
{
    static char errbuf[1024];

    /* for parsing from comma delimited string into list of tokens */
    sk_dllist_t   *token_list = NULL;
    sk_dll_iter_t  token;

    /* for parsing list of tokens into individual names/ids */
    char *token_string;

    sk_stringmap_status_t rv;
    sk_stringmap_entry_t *found_match;
    sk_stringmap_entry_t **entry;
    size_t j;
    char *bad_token = NULL;

    /* check inputs */
    if (str_map == NULL || out_vec == NULL) {
        snprintf(errbuf, sizeof(errbuf),
                 "Programmer error: NULL passed to function");
        rv = SKSTRINGMAP_ERR_INPUT;
        goto END;
    }

    if (user_string == NULL || user_string[0] == '\0') {
        snprintf(errbuf, sizeof(errbuf), "Missing value");
        rv = SKSTRINGMAP_ERR_INPUT;
        goto END;
    }

    /* convert the user's input into a linked list of tokens */
    rv = stringMapParseList(user_string, &token_list, &bad_token);
    if (rv != SKSTRINGMAP_OK) {
        snprintf(errbuf, sizeof(errbuf), "Unable to parse the name '%s'",
                 bad_token);
        goto END;
    }

    /* loop over the tokens we got from the user's input */
    skDLLAssignIter(&token, token_list);
    while (skDLLIterForward(&token, (void **)&token_string) == 0) {
        found_match = NULL;
        rv = stringMapFindEntry(str_map, token_string, &found_match);
        switch (rv) {
          case SKSTRINGMAP_OK:
            /* found a single match. it may be a duplicate */
            if (SKSTRINGMAP_DUPES_KEEP == handle_dupes) {
                /* dupes are ok. */
                break;
            }
            /* else search for a duplicate */
            for (j = 0;
                 (found_match
                  && NULL != (entry = skVectorGetValuePointer(out_vec, j)));
                 ++j)
            {
                if ((*entry)->id != found_match->id) {
                    continue;
                }
                /* found a dupe */
                switch (handle_dupes) {
                  case SKSTRINGMAP_DUPES_ERROR:
                    snprintf(errbuf, sizeof(errbuf),
                             "Duplicate name '%s'", token_string);
                    rv = SKSTRINGMAP_ERR_DUPLICATE_ENTRY;
                    goto END;
                  case SKSTRINGMAP_DUPES_REMOVE_WARN:
                    snprintf(errbuf, sizeof(errbuf),
                             "Ignoring duplicate value '%s'", token_string);
                    /* FALLTHROUGH */
                  case SKSTRINGMAP_DUPES_REMOVE_SILENT:
                    /* force us out of the for-loop */
                    found_match = NULL;
                    break;
                  case SKSTRINGMAP_DUPES_KEEP:
                    /* shouldn't be in this loop at all */
                    skAbortBadCase(handle_dupes);
                }
            }
            break;

          case SKSTRINGMAP_PARSE_AMBIGUOUS:
            snprintf(errbuf, sizeof(errbuf),
                     "The field '%s' matches multiple keys", token_string);
            goto END;

          case SKSTRINGMAP_PARSE_NO_MATCH:
            snprintf(errbuf, sizeof(errbuf),
                     "No match was found for the field '%s'", token_string);
            goto END;

          default:
            snprintf(errbuf, sizeof(errbuf),
                     "Unexpected return value from field parser (%d)", rv);
            goto END;
        }

        if (found_match) {
            if (0 != skVectorAppendValue(out_vec, &found_match)) {
                snprintf(errbuf, sizeof(errbuf), "Out of memory");
                rv = SKSTRINGMAP_ERR_MEM;
                goto END;
            }
        }
    } /* outer while-loop over tokens  */

    /* success */
    rv = SKSTRINGMAP_OK;

  END:
    if (token_list != NULL) {
        skDLListDestroy(token_list);
    }
    if (bad_token) {
        free(bad_token);
    }
    if (rv != SKSTRINGMAP_OK) {
        if (errmsg) {
            *errmsg = errbuf;
        }
    }

    return rv;
}


/* add to a list the string names which map to a particular value */
sk_stringmap_status_t skStringMapGetByID(
    const sk_stringmap_t   *str_map,
    sk_stringmap_id_t       id,
    sk_vector_t            *out_vec)
{
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;

    /* check inputs */
    if (out_vec == NULL || str_map == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }
    if (skVectorGetElementSize(out_vec) != sizeof(sk_stringmap_entry_t*)) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    /* Typecast away const.  We are still treating it as const
     * though. */
    skDLLAssignIter(&map_node, (sk_stringmap_t *)str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        /* add name if the id matches */
        if (map_entry->id == id) {
            if (0 != skVectorAppendValue(out_vec, &map_entry)) {
                return SKSTRINGMAP_ERR_MEM;
            }
        }
    }

    return SKSTRINGMAP_OK;
}


const char *skStringMapGetFirstName(
    const sk_stringmap_t   *str_map,
    sk_stringmap_id_t       id)
{
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;

    /* Typecast away const.  We are still treating it as const
     * though. */
    skDLLAssignIter(&map_node, (sk_stringmap_t *)str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        /* add name if the id matches */
        if (map_entry->id == id) {
            return (const char *)(map_entry->name);
        }
    }

    return NULL;
}


/*
 * Helper functions
 */

/* print the StringMap to an output stream in human-readable form */
sk_stringmap_status_t skStringMapPrintMap(
    const sk_stringmap_t   *str_map,
    FILE                   *outstream)
{
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;
    int first_entry_skip_comma = 1;

    if (str_map == NULL || outstream == NULL) {
        return SKSTRINGMAP_ERR_INPUT;
    }

    fprintf(outstream, "{");
    /* Typecast away const.  We are still treating it as const
     * though. */
    skDLLAssignIter(&map_node, (sk_stringmap_t *)str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        if (!first_entry_skip_comma) {
            fprintf(outstream, ", ");
        } else {
            first_entry_skip_comma = 0;
        }

        fprintf(outstream, (" \"%s\" : %" PRIu32),
                map_entry->name, (uint32_t)map_entry->id);
    }
    fprintf(outstream, " }");

    return SKSTRINGMAP_OK;
}


void skStringMapPrintUsage(
    const sk_stringmap_t   *str_map,
    FILE                   *fh,
    const int               indent_len)
{
    const char column_sep = ';';
    const char alias_sep = ',';
    char line_buf[81];
    sk_stringmap_entry_t *entry;
    sk_stringmap_entry_t *old_entry;
    sk_dll_iter_t node;
    int len;
    int avail_len;
    int entry_len;
    int total_len;
    int last_field_end;

    assert(indent_len < (int)sizeof(line_buf));

    if (NULL == str_map) {
        fprintf(fh, "\t[Fields not available]\n");
        return;
    }

    fprintf(fh,
            "\t(Semicolon separates unique columns."
            " Comma separates column aliases.\n"
            "\t Names are case-insenstive and"
            " can be abbreviated to the shortest\n"
            "\t unique prefix.)\n");

    /* previous value from map */
    old_entry = NULL;

    /* set indentation */
    memset(line_buf, ' ', sizeof(line_buf));
    total_len = indent_len;
    avail_len = sizeof(line_buf) - indent_len - 1;
    last_field_end = 0;

    /* loop through all entries in the map */
    skDLLAssignIter(&node, (sk_stringmap_t*)str_map);
    while (skDLLIterForward(&node, (void **)&entry) == 0) {
        entry_len = strlen(entry->name);

        if (last_field_end == 0) {
            /* very first field */
            last_field_end = total_len - 1;
        } else if ((old_entry != NULL) && (old_entry->id == entry->id)) {
            /* 'entry' is an alias for 'old_entry' */
            len = snprintf(&(line_buf[total_len]), avail_len, "%c",
                           alias_sep);
            assert(len < avail_len);
            total_len += len;
            avail_len -= len;
            entry_len += len;
        } else {
            /* start of a new field */
            len = snprintf(&(line_buf[total_len]), avail_len, "%c ",
                           column_sep);
            assert(len < avail_len);
            total_len += len;
            avail_len -= len;
            entry_len += len;
            last_field_end = total_len - 1;
        }

        if (entry_len >= avail_len) {
            /* need to start a new line */
            int to_move;
            if (last_field_end <= indent_len) {
                skAppPrintErr("Too many aliases or switch names too long");
                skAbort();
            }
            line_buf[last_field_end] = '\0';
            fprintf(fh, "%s\n", line_buf);
            ++last_field_end;
            to_move = total_len - last_field_end;
            if (to_move > 0) {
                memmove(&(line_buf[indent_len]), &(line_buf[last_field_end]),
                        to_move);
            }
            avail_len = sizeof(line_buf) - indent_len - to_move - 1;
            total_len = indent_len + to_move;
            last_field_end = indent_len - 1;
        }

        old_entry = entry;
        len = snprintf(&(line_buf[total_len]), avail_len, "%s", entry->name);
        assert(len < avail_len);
        total_len += len;
        avail_len -= len;
    }

    /* close out last line */
    if (last_field_end > 0) {
        fprintf(fh, "%s%c\n", line_buf, column_sep);
    }
}


/* Return a textual representation of the specified error code. */
const char *skStringMapStrerror(int error_code)
{
    static char buf[256];

    switch ((sk_stringmap_status_t)error_code) {
      case SKSTRINGMAP_OK:
        return "Command was successful";

      case SKSTRINGMAP_ERR_INPUT:
        return "Bad input to function";

      case SKSTRINGMAP_ERR_MEM:
        return "Memory allocation failed";

      case SKSTRINGMAP_ERR_LIST:
        return "Unexpected error occured in the linked list";

      case SKSTRINGMAP_ERR_DUPLICATE_ENTRY:
        return "Name is already in use";

      case SKSTRINGMAP_ERR_ZERO_LENGTH_ENTRY:
        return "Name is the empty string";

      case SKSTRINGMAP_ERR_NUMERIC_START_ENTRY:
        return "Name cannot begin with digit";

      case SKSTRINGMAP_ERR_ALPHANUM_START_ENTRY:
        return "Name cannot begin with a non-alphanumeric";

      case SKSTRINGMAP_ERR_PARSER:
        return "Unexpected error during parsing";

      case SKSTRINGMAP_PARSE_NO_MATCH:
        return "Input does not match any names";

      case SKSTRINGMAP_PARSE_AMBIGUOUS:
        return "Input matches multiple names";

      case SKSTRINGMAP_PARSE_UNPARSABLE:
        return "Input not parsable";
    }

    snprintf(buf, sizeof(buf), "Unrecognized string map error code %d",
             error_code);
    return buf;
}



/*
 *  stringMapCheckValidName(name, map);
 *
 *    Parse a key to be inserted into a StringMap to determine if it is
 *    legal or not.  Assumes arguments are non-NULL.
 *
 *  Arguments
 *
 *    const char *name - the key to check
 *
 *    sk_stringmap_t *str_map - the StringMap against whose contents the
 *    string key should be validated
 *
 *  Return Values
 *
 *    if name is the empty string, returns
 *    SKSTRINGMAP_ZERO_LENGTH_ENTRY
 *
 *    if name starts with a number, but does not contain all numeric
 *    characters (e.g., "34yh"), returns
 *    SKSTRINGMAP_ERR_NUMERIC_START_ENTRY
 *
 *    if name does not start with a alpha-numeric, return
 *    SKSTRINGMAP_ERR_ALPHANUM_START_ENTRY
 *
 *    if name already exists in the StringMap, returns
 *    SKSTRINGMAP_DUPLICATE_ENTRY
 *
 */
static sk_stringmap_status_t stringMapCheckValidName(
    sk_stringmap_t         *str_map,
    const char             *name)
{
    sk_dll_iter_t map_node;
    sk_stringmap_entry_t *map_entry;
    size_t i;

    assert(name != NULL);
    assert(str_map != NULL);

    if (name[0] == '\0') {
        return SKSTRINGMAP_ERR_ZERO_LENGTH_ENTRY;
    }

    if (isdigit((int)name[0])) {
        /* if the first character is a digit, they ALL have to be digits */
        for (i = strlen(name) - 1; i > 0; --i) {
            if ( !isdigit((int)name[i])) {
                return SKSTRINGMAP_ERR_NUMERIC_START_ENTRY;
            }
        }
    } else if (!isalpha((int)name[0])) {
        return SKSTRINGMAP_ERR_ALPHANUM_START_ENTRY;
    }

    skDLLAssignIter(&map_node, str_map);
    while (skDLLIterForward(&map_node, (void **)&map_entry) == 0) {
        if (0 == strcasecmp(map_entry->name, name)) {
            return SKSTRINGMAP_ERR_DUPLICATE_ENTRY;
        }
    }

    return SKSTRINGMAP_OK;
}


/*
 *  stringMapFreeEntry(map_entry);
 *
 *    Internal helper function to free a single entry 'map_entry' from
 *    a StringMap.
 */
static void stringMapFreeEntry(
    sk_stringmap_entry_t   *map_entry)
{
    if (map_entry) {
        /* free char string in entry */
        if (NULL != map_entry->name) {
            /* we know we allocated this ourselves, so the cast to (char*)
             * is safe */
            free((char*)map_entry->name);
        }
        if (NULL != map_entry->description) {
            free((char*)map_entry->description);
        }
        /* free entry itself */
        free(map_entry);
    }
}


/*
 *  status = stringMapParseList(user_string, &token_list, bad_token);
 *
 *    Treat 'user_string' as a C-string containing a comma separated
 *    list of tokens, integers, and/or integer-ranges.  Convert
 *    'user_string' into a list of tokens, then do number-range
 *    expansion on each token.  Put the resulting tokens into the
 *    linked list pointed at by 'token_list', which the function will
 *    allocate.  If parsing of a integer range fails, 'bad_token' will
 *    contain a copy of string where parsing failed.
 */
static sk_stringmap_status_t stringMapParseList(
    const char             *user_string,
    sk_dllist_t           **token_list,
    char                  **bad_token)
{
    sk_stringmap_status_t rv;

    assert(token_list);
    assert(user_string);

    /* tokenize input using comma as a delimiter.  each entry in the
     * list will be a string name, a string id, or a string range of
     * ids. */

    /* create list of comma-delimited tokens */
    *token_list = skDLListCreate(free);
    if (*token_list == NULL) {
        rv = SKSTRINGMAP_ERR_MEM;
        goto END;
    }

    rv = stringMapTokenize(user_string, ',', *token_list);
    if (rv != SKSTRINGMAP_OK) {
        goto END;
    }

    /* parse tokens representing ranges into individual numbers */
    rv = stringMapExpandRanges(*token_list, bad_token, '-');
    if (rv != SKSTRINGMAP_OK) {
        goto END;
    }

  END:
    if (rv != SKSTRINGMAP_OK) {
        if (*token_list != NULL) {
            skDLListDestroy(*token_list);
            *token_list = NULL;
        }
    }

    return rv;
}


/*
 *  ok = stringMapExpandRanges(token_list, &bad_token, range_delim);
 *
 *    Expand any numeric ranges in the list of strings present in
 *    'token_list'.  'range_delim' is the character that denotes a
 *    numeric range.  If parsing of a range fails and 'bad_token' is
 *    non-NULL, a copy of the string where the error occurred will be
 *    copied into 'bad_token'.
 *
 *    That is, the function takes a linked list where each data
 *    pointer of the list points to a string.  The functin goes
 *    through the list and finds any entries which match an unsigned
 *    integer range (e.g., "2-5" or "7-30"), removes that entry, and
 *    replaces it with multiple string entries for each integer in the
 *    range (e.g., "2-5" becomes four nodes, "2" "3" "4" "5").
 */
static sk_stringmap_status_t stringMapExpandRanges(
    sk_dllist_t        *token_list,
    char              **bad_token,
    char                range_delimiter)
{
    sk_stringmap_status_t rv;
    int rv_list;

    sk_dll_iter_t map_node;
    char *string = NULL;
    char *hyphen;

    int rv_int;
    uint32_t start_number;
    uint32_t end_number;

    uint32_t i;

    int chars_printed;
    char print_buffer[SKSTRING_PRINT_DIGITS];
    char *final_buffer;

    skDLLAssignIter(&map_node, token_list);
    while (skDLLIterForward(&map_node, (void **)&string) == 0) {

        if (string == NULL) {
            return SKSTRINGMAP_ERR_PARSER;
        }

        /* Don't treat the delimiter specially unless this token is a
         * number or the token begins with the delimiter character. */
        if (!isdigit((int)string[0]) && string[0] != range_delimiter) {
            continue;
        }

        /*
         * find range delimiter in string.  if it exists, then
         * something looks like a range and we try to parse the
         * numbers on either side to get the starting and ending
         * points of the range.  if they don't parse, return an error.
         */

        hyphen = strchr(string, range_delimiter);
        if (hyphen != NULL) {

            /*
             * parse the start number (we should have cruft after the
             * range parsing, namely, the hyphen
             */
            rv_int = skStringParseUint32(&start_number, string, 0, 0);
            /* rv_int shouldn't be zero, because of the hyphen */
            if (rv_int <= 0) {
                rv = SKSTRINGMAP_PARSE_UNPARSABLE;
                goto END;
            }

            /*
             * this check handles the case where there is a number,
             * followed by some cruft, followed by the hyphen (e.g.,
             * "4jg-6" should be unparsable, not parsed as "4-6"
             */
            if (string + rv_int != hyphen) {
                rv = SKSTRINGMAP_PARSE_UNPARSABLE;
                goto END;
            }

            /* parse the end number */
            rv_int = skStringParseUint32(&end_number, hyphen + 1, 0, 0);
            if (rv_int < 0) {
                rv = SKSTRINGMAP_PARSE_UNPARSABLE;
                goto END;
            }

            /*
             * this check handles the case where there is cruft after
             * the second number. (e.g. "4-6g")
             */
            if (rv_int > 0) {
                rv = SKSTRINGMAP_PARSE_UNPARSABLE;
                goto END;
            }

            /* not a valid range */
            if (end_number < start_number) {
                rv = SKSTRINGMAP_PARSE_UNPARSABLE;
                goto END;
            }

            /*
             * add entries for each number in the range
             */
            for (i = start_number; i <= end_number; ++i) {
                chars_printed =
                    snprintf(print_buffer, SKSTRING_PRINT_DIGITS,
                             ("%" PRIu32), i);

                /* we should never run out of buffer space.  if we
                   do, the buffer should be enlarged. */
                assert(chars_printed < SKSTRING_PRINT_DIGITS);

                final_buffer = (char *) malloc(chars_printed + 1);
                if (final_buffer == NULL) {
                    rv = SKSTRINGMAP_ERR_MEM;
                    goto END;
                }
                memcpy(final_buffer, print_buffer, chars_printed + 1);


                rv_list = skDLLIterAddBefore(&map_node, (void *)final_buffer);
                if (rv_list != 0) {
                    rv = SKSTRINGMAP_ERR_MEM;
                    goto END;
                }
            }

            /*
             * remove item containing the range and continue
             */
            free(string);
            rv_list = skDLLIterDel(&map_node);
            if (rv_list != 0) {
                /* we should never reach this case */
                assert(0);
                return SKSTRINGMAP_ERR_LIST;
            }
        }
    }

    rv = SKSTRINGMAP_OK;

  END:
    if (rv != SKSTRINGMAP_OK) {
        if ((bad_token != NULL) && (string != NULL)) {
            *bad_token = strdup(string);
        }
    }

    return rv;
}


/*
 *  ok = stringMapTokenize(user_string, delimiter, token_list);
 *
 *    Take the string 'user_string' and tokenize it based on the
 *    specified 'delimiter'.  Each token string is duplicated and
 *    stored in the linked list 'token_list'.  The original string is
 *    not modified in any way.  Returns SKSTRINGMAP_OK on success, or
 *    SKSTRINGMAP_ERR_MEM if memory allocation fails.
 */
static sk_stringmap_status_t stringMapTokenize(
    const char         *user_string,
    const char          delimiter,
    sk_dllist_t        *token_list)
{
    sk_stringmap_status_t rv;
    int rv_list;
    char *token;
    const char *cp;
    const char *ep;
    size_t len;

    /* check inputs */
    assert(token_list != NULL);
    assert(user_string != NULL);

    /* loop through string, tokenizing as we go */
    cp = user_string;
    while (*cp) {
        ep = strchr(cp, (int)delimiter);
        if (ep == NULL) {
            /* at end of string; get its length and put 'ep' on the
             * final \0. */
            len = strlen(cp);
            ep = cp + len;
        } else if (ep == cp) {
            /* double delimiter; ignore */
            ++cp;
            continue;
        } else {
            /* found a token; get its length and move 'ep' to start of
             * next token. */
            len = ep - cp;
            ++ep;
        }

        /* copy the token; add 1 for the \0. */
        token = malloc((1 + len) * sizeof(char));
        if (token == NULL) {
            rv = SKSTRINGMAP_ERR_MEM;
            goto END;
        }
        strncpy(token, cp, len);
        token[len] = '\0';
        cp = ep;

        /* add to list */
        rv_list = skDLListPushTail(token_list, (void*)token);
        if (rv_list != 0) {
            rv = SKSTRINGMAP_ERR_MEM;
            goto END;
        }
    }

    rv = SKSTRINGMAP_OK;

  END:
    return rv;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
