/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  sktempfile.c
**
**    Functions to handle temp file creation and access.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: sktempfile.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skvector.h>
#include <silk/sktempfile.h>

#ifdef SKTEMPFILE_TRACE_LEVEL
#define TRACEMSG_LEVEL 1
#endif
#define TRACEMSG(x)  TRACEMSG_TO_TRACEMSGLVL(1, x)
#include <silk/sktracemsg.h>


/* EXPORTED VARIABLES */

/* placeholder for files that have been removed from the vector */
const char * const sktempfile_null = "NULL";


/* LOCAL VARIABLES */

/* template used to make temporary files. */
static char temp_file_template[PATH_MAX];

/* names of temporary files */
static sk_vector_t *tmp_names = NULL;


/* FUNCTION DEFINITIONS */


/* find the tmpdir to use, initialize template, create vector */
int skTempFileInitialize(
    const char     *user_temp_dir,
    sk_msg_fn_t     err_fn)
{
    const char *tmp_dir = NULL;
    int rv;

    assert(NULL == tmp_names);

    tmp_dir = skTempDir(user_temp_dir, err_fn);
    if (NULL == tmp_dir) {
        return -1;
    }

    rv = snprintf(temp_file_template, sizeof(temp_file_template),
                  "%s/%s_tmp.XXXXXXXX", tmp_dir, skAppName());
    if ((size_t)rv >= sizeof(temp_file_template) || rv < 0) {
        return -1;
    }

    /* initialize tmp_names */
    tmp_names = skVectorNew(sizeof(char*));
    if (NULL == tmp_names) {
        return -1;
    }

    TRACEMSG(("skTempFile: Temporary directory = '%s'", tmp_dir));

    return 0;
}


/* remove all temp files and destroy vector */
void skTempFileTeardown(void)
{
    int i;

    if (tmp_names) {
        for (i = skVectorGetCount(tmp_names)-1; i >= 0; --i) {
            skTempFileRemove(i);
        }
        skVectorDestroy(tmp_names);
        tmp_names = NULL;
        TRACEMSG(("skTempFile: Teardown complete."));
    }
}


/* create new file */
FILE *skTempFileCreate(
    int            *tmp_idx,
    char          **out_name)
{
    int saved_errno;
    FILE *fp = NULL;
    char *name;
    int fd;

    /* copy template name */
    name = strdup(temp_file_template);
    if (NULL == name) {
        return NULL;
    }

    /* open file */
    fd = mkstemp(name);
    if (fd == -1) {
        saved_errno = errno;
        free(name);
        errno = saved_errno;
        return NULL;
    }

    /* convert descriptor to pointer */
    fp = fdopen(fd, "w");
    if (fp == NULL) {
        saved_errno = errno;
        close(fd);
        free(name);
        errno = saved_errno;
        return NULL;
    }

    /* append to vector */
    if (skVectorAppendValue(tmp_names, &name)) {
        saved_errno = errno;
        close(fd);
        free(name);
        errno = saved_errno;
        return NULL;
    }
    *tmp_idx = skVectorGetCount(tmp_names) - 1;

    TRACEMSG(("skTempFile: Created temp %d => '%s'", *tmp_idx, name));

    if (out_name) {
        *out_name = name;
    }
    return fp;
}


/* get name of file */
const char *skTempFileGetName(
    int             tmp_idx)
{
    char **name;

    name = skVectorGetValuePointer(tmp_names, tmp_idx);
    if (name && *name) {
        return *name;
    }
    return sktempfile_null;
}


/* open existing file */
FILE *skTempFileOpen(
    int             tmp_idx)
{
    const char *name;

    name = skTempFileGetName(tmp_idx);
    TRACEMSG(("skTempFile: Open existing temp %d => '%s'", tmp_idx, name));

    if (name == sktempfile_null) {
        return NULL;
    }
    return fopen(name, "r");
}


/* remove file */
void skTempFileRemove(
    int             tmp_idx)
{
    const char *name;

    name = skTempFileGetName(tmp_idx);
    if (name == sktempfile_null) {
        TRACEMSG(("skTempFile: Remove temp %d => '%s'", tmp_idx, name));
        return;
    }

    TRACEMSG(("skTempFile: Remove temp %d => '%s' of size %" PRId64,
              tmp_idx, name, (int64_t)skFileSize(name)));
    if (-1 == unlink(name)) {
        if (skFileExists(name)) {
            TRACEMSG(("skTempFile: Cannot remove file '%s': %s",
                      name, strerror(errno)));
        }
    }

    free((char*)name);
    skVectorSetValue(tmp_names, tmp_idx, &sktempfile_null);
}


/*
 *  ok = skTempFileWriteBuffer(&tmp_idx, buffer, size, count);
 *
 *    Print 'count' records having size 'size' bytes from 'buffer'
 *    into a newly created temporary file.  Index the temp file by
 *    'tmp_idx'.  Returns 0 on success, or -1 if a temp file cannot be
 *    created, or the write or close calls fail.
 */
int skTempFileWriteBuffer(
    int            *tmp_idx,
    const void     *rec_buffer,
    uint32_t        rec_size,
    uint32_t        rec_count)
{
    int saved_errno = 0;
    FILE* temp_filep = NULL;
    char *name;
    int rv = -1; /* return value */

    temp_filep = skTempFileCreate(tmp_idx, &name);
    if (temp_filep == NULL) {
        saved_errno = errno;
        goto END;
    }

    TRACEMSG(("skTempFile: Writing %" PRIu32 " records to temp %d => '%s'...",
              rec_count, *tmp_idx, name));
    if (rec_count != fwrite(rec_buffer, rec_size, rec_count, temp_filep)) {
        /* error writing */
        saved_errno = errno;
        goto END;
    }

    /* Success so far */
    rv = 0;

  END:
    /* close the file if open */
    if (temp_filep) {
        if (rv != 0) {
            /* already an error, so just close the file */
            fclose(temp_filep);
        } else if (fclose(temp_filep) == EOF) {
            saved_errno = errno;
            rv = -1;
        }
        TRACEMSG(("%s", ((rv == 0) ? "done." : "ERROR!")));
    }
    errno = saved_errno;
    return rv;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
