/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  skipfix.c
**
**    SiLK Flow Record / IPFIX translation core
**
**    Brian Trammell
**    February 2007
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skipfix.c 4913fa12fec0 2012-06-25 18:19:12Z mthomas $");

#include <silk/skipfix.h>
#include <silk/utils.h>


/* Values for debugging/tracig.  Set SKIPFIX_DEBUG to the bitwise OR
 * of these values */
#define SKI_TRACE_REC_DECODE  1


#ifndef  SKIPFIX_DEBUG
#  define SKIPFIX_DEBUG 0
#endif


/* The IPFIX Private Enterprise Number for CERT */
#define IPFIX_CERT_PEN  6871

/*
 *  val = CLAMP_VAL(val, max);
 *
 *    If 'val' is greater then 'max', return 'max'.  Otherwise,
 *    return (max & val).
 */
#define CLAMP_VAL(val, max) \
    (((val) > (max)) ? (max) : ((max) & (val)))



/* Define the IPFIX information elements in IPFIX_CERT_PEN space for
 * SiLK */
static fbInfoElement_t ski_info_elements[] = {
    FB_IE_INIT("initialTCPFlags",              IPFIX_CERT_PEN, 14, 1,
               FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("unionTCPFlags",                IPFIX_CERT_PEN, 15, 1,
               FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("reverseFlowDeltaMilliseconds", IPFIX_CERT_PEN, 21, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkFlowType",                 IPFIX_CERT_PEN, 30, 1,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkFlowSensor",               IPFIX_CERT_PEN, 31, 2,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkTCPState",                 IPFIX_CERT_PEN, 32, 1,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkAppLabel",                 IPFIX_CERT_PEN, 33, 2,
               FB_IE_F_ENDIAN),
    FB_IE_NULL
};


/* Bytes of padding to add to ski_flow_spec to get a multiple of 64bits */
#define SKI_FLOW_SPEC_PADDING  6

/* This is an IPFIX encoding of a standard SiLK Flow record (rwRec).
 * Keep this in sync with the ski_rwrec_t below. Pad to 64bits */
static fbInfoElementSpec_t ski_flow_spec[] = {
    /* Millisecond start and end (epoch) (native time) */
    { "flowStartMilliseconds",              0, 0 },
    { "flowEndMilliseconds",                0, 0 },
    /* 4-tuple */
    { "sourceIPv6Address",                  0, 0 },
    { "destinationIPv6Address",             0, 0 },
    { "sourceIPv4Address",                  0, 0 },
    { "destinationIPv4Address",             0, 0 },
    { "sourceTransportPort",                0, 0 },
    { "destinationTransportPort",           0, 0 },
    /* Router interface information */
    { "ipNextHopIPv4Address",               0, 0 },
    { "ipNextHopIPv6Address",               0, 0 },
    { "ingressInterface",                   0, 0 },
    { "egressInterface",                    0, 0 },
    /* Counters (reduced length encoding for SiLK) */
    { "packetDeltaCount",                   0, 0 },
    { "octetDeltaCount",                    0, 0 },
    /* Protocol; sensor information */
    { "protocolIdentifier",                 0, 0 },
    { "silkFlowType",                       0, 0 },
    { "silkFlowSensor",                     0, 0 },
    /* Flags */
    { "tcpControlBits",                     0, 0 },
    { "initialTCPFlags",                    0, 0 },
    { "unionTCPFlags",                      0, 0 },
    { "silkTCPState",                       0, 0 },
    { "silkAppLabel",                       0, 0 },
    /* pad record to 64-bit boundary */
#if SKI_FLOW_SPEC_PADDING != 0
    { "paddingOctets",  SKI_FLOW_SPEC_PADDING, 0 },
#endif
    FB_IESPEC_NULL
};

/* Keep this in sync with the ski_flow_spec above.. Pad to 64bits */
typedef struct ski_rwrec_st {
    uint64_t        flowStartMilliseconds;          /*   0-  7 */
    uint64_t        flowEndMilliseconds;            /*   8- 15 */

    uint8_t         sourceIPv6Address[16];          /*  16- 31 */
    uint8_t         destinationIPv6Address[16];     /*  32- 47 */

    uint32_t        sourceIPv4Address;              /*  48- 51 */
    uint32_t        destinationIPv4Address;         /*  52- 55 */

    uint16_t        sourceTransportPort;            /*  56- 57 */
    uint16_t        destinationTransportPort;       /*  58- 59 */

    uint32_t        ipNextHopIPv4Address;           /*  60- 63 */
    uint8_t         ipNextHopIPv6Address[16];       /*  64- 79 */
    uint32_t        ingressInterface;               /*  80- 83 */
    uint32_t        egressInterface;                /*  84- 87 */

    uint64_t        packetDeltaCount;               /*  88- 95 */
    uint64_t        octetDeltaCount;                /*  96-103 */

    uint8_t         protocolIdentifier;             /* 104     */
    flowtypeID_t    silkFlowType;                   /* 105     */
    sensorID_t      silkFlowSensor;                 /* 106-107 */

    uint8_t         tcpControlBits;                 /* 108     */
    uint8_t         initialTCPFlags;                /* 109     */
    uint8_t         unionTCPFlags;                  /* 110     */
    uint8_t         silkTCPState;                   /* 111     */
    uint16_t        silkAppLabel;                   /* 112-113 */
#if SKI_FLOW_SPEC_PADDING != 0
    uint8_t         pad[SKI_FLOW_SPEC_PADDING];     /* 114-119 */
#endif
} ski_rwrec_t;



/* Bytes of padding to add to ski_extflow_spec to get a multiple of 64bits */
#define SKI_EXTFLOW_SPEC_PADDING  4

/* These are additional IPFIX fields (or different encodings of the
 * SiLK fields) that we may get from other flowmeters.  They will be
 * appended to the ski_flow_spec (above) to create the complete
 * ski_extrwrec_t. */
static fbInfoElementSpec_t ski_extflow_spec[] = {
    /* Total counter support */
    { "packetTotalCount",                   0, 0 },
    { "octetTotalCount",                    0, 0 },
    /* Reverse counter support */
    { "reversePacketDeltaCount",            0, 0 },
    { "reverseOctetDeltaCount",             0, 0 },
    { "reversePacketTotalCount",            0, 0 },
    { "reverseOctetTotalCount",             0, 0 },
    /* Microsecond start and end (RFC1305-style) (extended time) */
    { "flowStartMicroseconds",              0, 0 },
    { "flowEndMicroseconds",                0, 0 },
    /* Second start and end (extended time) */
    { "flowStartSeconds",                   0, 0 },
    { "flowEndSeconds",                     0, 0 },
    /* Flow durations (extended time) */
    { "flowDurationMicroseconds",           0, 0 },
    { "flowDurationMilliseconds",           0, 0 },
    /* Microsecond delta start and end (extended time) */
    { "flowStartDeltaMicroseconds",         0, 0 },
    { "flowEndDeltaMicroseconds",           0, 0 },
    /* Initial packet roundtrip */
    { "reverseFlowDeltaMilliseconds",       0, 0 },
    /* Reverse flags */
    { "reverseTcpControlBits",              0, 0 },
    { "reverseInitialTCPFlags",             0, 0 },
    { "reverseUnionTCPFlags",               0, 0 },
    /* End reason */
    { "flowEndReason",                      0, 0 },
    /* Vlan IDs */
    { "vlanId",                             0, 0 },
    { "postVlanId",                         0, 0 },
    /* pad record to 64-bit boundary. */
#if SKI_EXTFLOW_SPEC_PADDING != 0
    { "paddingOctets", SKI_EXTFLOW_SPEC_PADDING, 0 },
#endif
    FB_IESPEC_NULL
};

typedef struct ski_extrwrec_st {
    ski_rwrec_t     rw;                             /*   0-119 */

    uint64_t        packetTotalCount;               /* 120-127 */
    uint64_t        octetTotalCount;                /* 128-135 */

    uint64_t        reversePacketDeltaCount;        /* 136-143 */
    uint64_t        reverseOctetDeltaCount;         /* 144-151 */
    uint64_t        reversePacketTotalCount;        /* 152-159 */
    uint64_t        reverseOctetTotalCount;         /* 160-167 */

    /* Time can be represented in many different formats: */

    /* start time as NTP (RFC1305); may either have end Time in same
     * format or as an flowDurationMicroseconds value. */
    uint64_t        flowStartMicroseconds;          /* 168-175 */
    uint64_t        flowEndMicroseconds;            /* 176-183 */

    /* start time and end times as seconds since UNIX epoch. no
     * flowDuration field */
    uint32_t        flowStartSeconds;               /* 184-187 */
    uint32_t        flowEndSeconds;                 /* 188-191 */

    /* elapsed time as either microsec or millisec.  used when the
     * flowEnd time is not given. */
    uint32_t        flowDurationMicroseconds;       /* 192-195 */
    uint32_t        flowDurationMilliseconds;       /* 196-199 */

    /* start time as delta (negative microsec offsets) from the export
     * time; may either have end time in same format or a
     * flowDurationMicroseconds value */
    uint32_t        flowStartDeltaMicroseconds;     /* 200-203 */
    uint32_t        flowEndDeltaMicroseconds;       /* 204-207 */

    /* start time of reverse flow, as millisec offset from start time
     * of forward flow */
    uint32_t        reverseFlowDeltaMilliseconds;   /* 208-211 */

    /* Flags for the reverse flow: */
    uint8_t         reverseTcpControlBits;          /* 212     */
    uint8_t         reverseInitialTCPFlags;         /* 213     */
    uint8_t         reverseUnionTCPFlags;           /* 214     */

    uint8_t         flowEndReason;                  /* 215     */

    /* vlan IDs */
    uint16_t        vlanId;                         /* 216-217 */
    uint16_t        postVlanId;                     /* 218-219 */

    /* padding */
#if SKI_EXTFLOW_SPEC_PADDING != 0
    uint8_t         pad[SKI_EXTFLOW_SPEC_PADDING];  /* 220-223 */
#endif
} ski_extrwrec_t;

#define SKI_END_IDLE            1
#define SKI_END_ACTIVE          2
#define SKI_END_CLOSED          3
#define SKI_END_FORCED          4
#define SKI_END_RESOURCE        5
#define SKI_END_MASK            0x1f
#define SKI_END_ISCONT          0x80

/* SiLK will ignore flows with a flowEndReason of
 * SKI_END_YAF_INTERMEDIATE_FLOW */
#define SKI_END_YAF_INTERMEDIATE_FLOW 0x1F



/* These are individual elements whose presence we want to check for. */
static fbInfoElement_t tcp_init_flags_ie =
    FB_IE_INIT("initialTCPFlags", IPFIX_CERT_PEN,
               14, 1, FB_IE_F_ENDIAN);
static fbInfoElementSpec_t reverse_tcp_ctrl_ies =
    {"reverseTcpControlBits", 0, 0};
static fbInfoElementSpec_t reverse_tcp_init_ies =
    {"reverseInitialTCPFlags", 0, 0};
static fbInfoElementSpec_t post_vlan_ies =
    {"postVlanId", 0, 0};

static fbInfoModel_t *ski_model = NULL;


/* FUNCTION DEFINITIONS */

static fbInfoModel_t *skiInfoModel(void)
{
    if (!ski_model) {
        ski_model = fbInfoModelAlloc();
        fbInfoModelAddElementArray(ski_model, ski_info_elements);
    }

    return ski_model;
}


static void skiInfoModelFree(void)
{
    if (ski_model) {
        fbInfoModelFree(ski_model);
        ski_model = NULL;
    }
}


void skiCheckDataStructure(FILE *fh)
{
    unsigned long pos;

#define PRINT_TITLE(s_)                                         \
    fprintf(fh, "===> %s\n%5s|%5s|%5s|%5s|%5s|%s\n", #s_,       \
            "begin", "end", "size", "alerr", "hole", "member")

#define PRINT_OFFSET(pos_, s_, mem_)                                    \
    {                                                                   \
        s_ x;                                                           \
        unsigned long off_ = (unsigned long)offsetof(s_, mem_);         \
        unsigned long sz_  = (unsigned long)sizeof(x.mem_);             \
        unsigned long end_ = off_ + sz_ - 1;                            \
        int align_ = ((off_ % sz_) == 0);                               \
        int hole_ = (pos_ != off_);                                     \
        pos_ += sz_;                                                    \
        fprintf(fh, "%5lu|%5lu|%5lu|%5s|%5s|%s\n",                      \
                off_, end_, sz_, (align_ ? "" : "alerr"),               \
                (hole_ ? "hole" : ""), #mem_);                          \
    }

    pos = 0;
    PRINT_TITLE(ski_rwrec_t);
    PRINT_OFFSET(pos, ski_rwrec_t, flowStartMilliseconds);
    PRINT_OFFSET(pos, ski_rwrec_t, flowEndMilliseconds);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceTransportPort);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationTransportPort);
    PRINT_OFFSET(pos, ski_rwrec_t, ipNextHopIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, ipNextHopIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, ingressInterface);
    PRINT_OFFSET(pos, ski_rwrec_t, egressInterface);
    PRINT_OFFSET(pos, ski_rwrec_t, packetDeltaCount);
    PRINT_OFFSET(pos, ski_rwrec_t, octetDeltaCount);
    PRINT_OFFSET(pos, ski_rwrec_t, protocolIdentifier);
    PRINT_OFFSET(pos, ski_rwrec_t, silkFlowType);
    PRINT_OFFSET(pos, ski_rwrec_t, silkFlowSensor);
    PRINT_OFFSET(pos, ski_rwrec_t, tcpControlBits);
    PRINT_OFFSET(pos, ski_rwrec_t, initialTCPFlags);
    PRINT_OFFSET(pos, ski_rwrec_t, unionTCPFlags);
    PRINT_OFFSET(pos, ski_rwrec_t, silkTCPState);
    PRINT_OFFSET(pos, ski_rwrec_t, silkAppLabel);
#if SKI_FLOW_SPEC_PADDING != 0
    PRINT_OFFSET(pos, ski_rwrec_t, pad);
#endif

    pos = 0;
    PRINT_TITLE(ski_extrwrec_t);
    PRINT_OFFSET(pos, ski_extrwrec_t, rw);
    PRINT_OFFSET(pos, ski_extrwrec_t, packetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, octetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reversePacketDeltaCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseOctetDeltaCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reversePacketTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseOctetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartSeconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndSeconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowDurationMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowDurationMilliseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartDeltaMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndDeltaMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseFlowDeltaMilliseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseTcpControlBits);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseInitialTCPFlags);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseUnionTCPFlags);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndReason);
    PRINT_OFFSET(pos, ski_extrwrec_t, vlanId);
    PRINT_OFFSET(pos, ski_extrwrec_t, postVlanId);
#if SKI_EXTFLOW_SPEC_PADDING != 0
    PRINT_OFFSET(pos, ski_extrwrec_t, pad);
#endif
}


fbListener_t *skiCreateListener(
    fbConnSpec_t           *spec,
    fbListenerAppInit_fn    appinit,
    fbListenerAppFree_fn    appfree,
    GError                **err)
{
    fbInfoModel_t   *model = skiInfoModel();
    fbSession_t     *session = NULL;
    fbTemplate_t    *tmpl = NULL;
    fbListener_t    *listener = NULL;

    /* Allocate a session */
    session = fbSessionAlloc(model);

    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_FLOW_TID, tmpl, err)) {
        goto err;
    }

    /* Add the extended record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto err;
    }
    if (!fbTemplateAppendSpecArray(tmpl, ski_extflow_spec, 0, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_EXTFLOW_TID, tmpl, err)) {
        goto err;
    }

    /* Allocate a listener */
    listener = fbListenerAlloc(spec, session, appinit, appfree, err);
    if (NULL == listener) {
        goto err;
    }

    return listener;

  err:
    fbTemplateFreeUnused(tmpl);
    if (session) {
        fbSessionFree(session);
    }
    return NULL;
}


fBuf_t *skiCreateReadBuffer(
    fbCollector_t  *collector,
    GError        **err)
{
    fbInfoModel_t   *model = skiInfoModel();
    fbSession_t     *session = NULL;
    fbTemplate_t    *tmpl = NULL;
    fBuf_t          *fbuf = NULL;

    /* Allocate a session */
    session = fbSessionAlloc(model);

    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_FLOW_TID, tmpl, err)) {
        goto err;
    }

    /* Add the extended record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto err;
    }
    if (!fbTemplateAppendSpecArray(tmpl, ski_extflow_spec, 0, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_EXTFLOW_TID, tmpl, err)) {
        goto err;
    }

    /* Create a buffer with the session and supplied collector */
    fbuf = fBufAllocForCollection(session, collector);

    /* done */
    return fbuf;

  err:
    if (fbuf) {
        fBufFree(fbuf);
    } else {
        fbTemplateFreeUnused(tmpl);
        if (session) {
            fbSessionFree(session);
        }
    }
    return NULL;
}


fBuf_t *skiCreateReadBufferForFP(
    FILE           *fp,
    GError        **err)
{
    return skiCreateReadBuffer(fbCollectorAllocFP(NULL, fp), err);
}


fBuf_t *skiCreateWriteBuffer(
    fbExporter_t   *exporter,
    uint32_t        domain,
    GError        **err)
{
    fbInfoModel_t   *model = skiInfoModel();
    fbSession_t     *session = NULL;
    fbTemplate_t    *tmpl = NULL;
    fBuf_t          *fbuf = NULL;

    /* Allocate a session */
    session = fbSessionAlloc(model);

    /* set observation domain */
    fbSessionSetDomain(session, domain);

    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_FLOW_TID, tmpl, err)) {
        goto err;
    }
    if (!fbSessionAddTemplate(session, FALSE, SKI_FLOW_TID, tmpl, err)) {
        goto err;
    }

    /* Create a buffer with the session and supplied exporter */
    fbuf = fBufAllocForExport(session, exporter);

    /* write RW base flow template */
    if (!fbSessionExportTemplates(session, err)) {
        goto err;
    }

    /* set default templates */
    if (!fBufSetInternalTemplate(fbuf, SKI_FLOW_TID, err)) {
        goto err;
    }
    if (!fBufSetExportTemplate(fbuf, SKI_FLOW_TID, err)) {
        goto err;
    }

    /* done */
    return fbuf;

  err:
    if (fbuf) {
        fBufFree(fbuf);
    } else {
        fbTemplateFreeUnused(tmpl);
        if (session) {
            fbSessionFree(session);
        }
    }
    return NULL;
}


fBuf_t *skiCreateWriteBufferForFP(
    FILE           *fp,
    uint32_t        domain,
    GError        **err)
{
    return skiCreateWriteBuffer(fbExporterAllocFP(fp), domain, err);
}


void skiTeardown(void)
{
    skiInfoModelFree();
}


/* Convert an NTP timestamp (RFC1305) to epoch millisecond */
static uint64_t skiNTPDecode(
    uint64_t        ntp)
{
    double          dntp;
    uint64_t        millis;

    if (!ntp) {
        return 0;
    }

    dntp = (ntp & UINT64_C(0xFFFFFFFF00000000)) >> 32;
    dntp += ((ntp & UINT64_C(0x00000000FFFFFFFF)) * 1.0) / (UINT64_C(2) << 32);
    millis = dntp * 1000;
    return millis;
}


/* Get a record from the libfixbuf buffer 'fbuf' and fill in the
 * forward and reverse flow records, 'rec' and 'revRec'. */
gboolean skiRwNextRecord(
    fBuf_t     *fbuf,
    rwRec      *rec,
    rwRec      *revRec,
    GError    **err,
    int         vlan_interfaces)
{
    fbTemplate_t       *tmpl = NULL;
    ski_extrwrec_t      fixrec;
    size_t              len;
    uint64_t            sTime, eTime;
    uint64_t            pkts, bytes;

    /* Set internal template to read an extended flow record */
    if (!fBufSetInternalTemplate(fbuf, SKI_EXTFLOW_TID, err)) {
        return FALSE;
    }

    /* Clear out the record */
    RWREC_CLEAR(rec);

    /* Read a record */
    for (;;) {
        len = sizeof(fixrec);
        if (!fBufNext(fbuf, (uint8_t *)&fixrec, &len, err)) {
            return FALSE;
        }

#if !SK_ENABLE_IPV6
        if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
            || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
        {
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
            skAppPrintErr("Ignoring IPv6 flow");
#endif
            continue;
        }
#endif  /* !SK_ENABLE_IPV6 */

        if ((fixrec.flowEndReason & SKI_END_MASK)
            == SKI_END_YAF_INTERMEDIATE_FLOW)
        {
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
            skAppPrintErr("Ignoring YAF intermediate uniflows");
#endif
            continue;
        }

        if ((fixrec.rw.packetDeltaCount || fixrec.packetTotalCount)
            && (fixrec.rw.octetDeltaCount || fixrec.octetTotalCount))
        {
            /* have a forward record with volume */
            break;
        }

        if ((fixrec.reversePacketDeltaCount || fixrec.reversePacketTotalCount)
            && (fixrec.reverseOctetDeltaCount || fixrec.reverseOctetTotalCount))
        {
            /* have a reverse record with volume */
            break;
        }

#if SK_ENABLE_ASA_ZERO_PACKET_HACK && SK_ENABLE_NETFLOW9
        /* attempt to handle NetFlowV9 records from an ASA router that
         * have no packet count; unfortunately, this check is not
         * limited to flows that originated as NetFlowV9  */
        if (fixrec.rw.sourceIPv4Address
            || !SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address))
        {
            /* flow record has a sIP, so it may be valid (or as valid
             * a flow as the ASA routers can produce) */
            int is_valid = 0;

            /* set forward packets if we have forward bytes */
            if ((fixrec.rw.octetDeltaCount || fixrec.octetTotalCount)
                && !(fixrec.rw.packetDeltaCount || fixrec.packetTotalCount))
            {
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
                skAppPrintErr("Setting forward packets to 1");
#endif
                fixrec.rw.packetDeltaCount = 1;
                is_valid = 1;
            }

            /* set reverse packets if we have reverse bytes */
            if ((fixrec.reverseOctetDeltaCount || fixrec.reverseOctetTotalCount)
                && !(fixrec.reversePacketDeltaCount
                     || fixrec.reversePacketTotalCount))
            {
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
                skAppPrintErr("Setting reverse packets to 1");
#endif
                fixrec.reversePacketDeltaCount = 1;
                is_valid = 1;
            }

            if (is_valid) {
                break;
            }
        }
#endif  /* SK_ENABLE_ASA_ZERO_PACKET_HACK && SK_ENABLE_NETFLOW9 */

#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
        skAppPrintErr("Ignoring record with no forward or reverse volume");
#endif
    }

    /* Get the template used for the last record */
    tmpl = fBufGetCollectionTemplate(fbuf, NULL);

    /* FIXME.  What if the record has a flowDirection field that is
     * set to egress (0x01)?  Shouldn't we handle that by reversing
     * the record?  Or has fixbuf done that for us? */

    /* Get the forward volumes.  Use either delta or total count for
     * bytes and packets */
    pkts = ((fixrec.rw.packetDeltaCount)
            ? fixrec.rw.packetDeltaCount
            : fixrec.packetTotalCount);
    bytes = ((fixrec.rw.octetDeltaCount)
             ? fixrec.rw.octetDeltaCount
             : fixrec.octetTotalCount);

    if (bytes) {
        /* We have forward information. */

#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
        skAppPrintErr("Read a forward record");
#endif

        /* Handle the IP addresses */
#if SK_ENABLE_IPV6
        if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
            || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
        {
            /* Values found in IPv6 addresses--use them */
            rwRecSetIPv6(rec);
            rwRecMemSetSIPv6(rec, &fixrec.rw.sourceIPv6Address);
            rwRecMemSetDIPv6(rec, &fixrec.rw.destinationIPv6Address);
            rwRecMemSetNhIPv6(rec, &fixrec.rw.ipNextHopIPv6Address);
        } else
#endif /* SK_ENABLE_IPV6 */
        {
            /* Take values from IPv4 */
            rwRecSetSIPv4(rec, fixrec.rw.sourceIPv4Address);
            rwRecSetDIPv4(rec, fixrec.rw.destinationIPv4Address);
            rwRecSetNhIPv4(rec, fixrec.rw.ipNextHopIPv4Address);
        }

        /* Handle the Ports */
        rwRecSetSPort(rec, fixrec.rw.sourceTransportPort);
        rwRecSetDPort(rec, fixrec.rw.destinationTransportPort);

        /* Handle the SNMP or VLAN interfaces */
        if (vlan_interfaces) {
            rwRecSetInput(rec, fixrec.vlanId);
            rwRecSetOutput(rec, fixrec.postVlanId);
        } else {
            rwRecSetInput(rec,
                          CLAMP_VAL(fixrec.rw.ingressInterface, UINT16_MAX));
            rwRecSetOutput(rec,
                           CLAMP_VAL(fixrec.rw.egressInterface, UINT16_MAX));
        }

        /* Store volume, clamping counts to 32 bits. */
        rwRecSetPkts(rec, CLAMP_VAL(pkts, UINT32_MAX));
        rwRecSetBytes(rec, CLAMP_VAL(bytes, UINT32_MAX));

        /* Check whether this is a bi-flow by getting the volume for
         * the reverse direction.  We will check this below */
        pkts = ((fixrec.reversePacketDeltaCount)
                ? fixrec.reversePacketDeltaCount
                : fixrec.reversePacketTotalCount);
        bytes = ((fixrec.reverseOctetDeltaCount)
                 ? fixrec.reverseOctetDeltaCount
                 : fixrec.reverseOctetTotalCount);

    } else {
        /* We have no forward information.  Write the source and dest
         * values from the IPFIX record to SiLK's dest and source
         * fields, respectively. */

#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
        skAppPrintErr("Read a reversed record");
#endif

        /* Handle the IP addresses */
#if SK_ENABLE_IPV6
        if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
            || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
        {
            /* Values found in IPv6 addresses--use them */
            rwRecSetIPv6(rec);
            rwRecMemSetSIPv6(rec, &fixrec.rw.destinationIPv6Address);
            rwRecMemSetDIPv6(rec, &fixrec.rw.sourceIPv6Address);
            rwRecMemSetNhIPv6(rec, &fixrec.rw.ipNextHopIPv6Address);
        } else
#endif /* SK_ENABLE_IPV6 */
        {
            /* Take values from IPv4 */
            rwRecSetSIPv4(rec, fixrec.rw.destinationIPv4Address);
            rwRecSetDIPv4(rec, fixrec.rw.sourceIPv4Address);
            rwRecSetNhIPv4(rec, fixrec.rw.ipNextHopIPv4Address);
        }

        /* Handle the Ports */
        rwRecSetSPort(rec, fixrec.rw.destinationTransportPort);
        rwRecSetDPort(rec, fixrec.rw.sourceTransportPort);

        /* Handle the SNMP or VLAN interfaces */
        if (vlan_interfaces) {
            /* If we have a single vlanId, set 'input' to that value;
             * otherwise, set 'input' to postVlanId and 'output' to
             * vlanId. */
            if (fbTemplateContainsElementByName(tmpl, &post_vlan_ies)) {
                rwRecSetInput(rec, fixrec.postVlanId);
                rwRecSetOutput(rec, fixrec.vlanId);
            } else {
                rwRecSetInput(rec, fixrec.vlanId);
            }
        } else {
            rwRecSetInput(rec,
                          CLAMP_VAL(fixrec.rw.egressInterface, UINT16_MAX));
            rwRecSetOutput(rec,
                           CLAMP_VAL(fixrec.rw.ingressInterface, UINT16_MAX));
        }

        /* Get the volume from the reverse fields */
        pkts = ((fixrec.reversePacketDeltaCount)
                ? fixrec.reversePacketDeltaCount
                : fixrec.reversePacketTotalCount);
        bytes = ((fixrec.reverseOctetDeltaCount)
                 ? fixrec.reverseOctetDeltaCount
                 : fixrec.reverseOctetTotalCount);

        /* Store volume, clamping counts to 32 bits. */
        rwRecSetPkts(rec, CLAMP_VAL(pkts, UINT32_MAX));
        rwRecSetBytes(rec, CLAMP_VAL(bytes, UINT32_MAX));

        /* This cannot be a bi-flow.  Clear the pkts and bytes variables. */
        pkts = bytes = 0;
    }


    /* Run the Gauntlet of Time - convert all the various ways an IPFIX
     * record's time could be represented into start and elapsed times. */
    if (fixrec.rw.flowStartMilliseconds) {
        /* Flow start time in epoch milliseconds */
        rwRecSetStartTime(rec, (sktime_t)fixrec.rw.flowStartMilliseconds);
        if (fixrec.rw.flowEndMilliseconds >= fixrec.rw.flowStartMilliseconds) {
            /* Flow end time in epoch milliseconds */
            rwRecSetElapsed(rec,(uint32_t)(fixrec.rw.flowEndMilliseconds
                                           -fixrec.rw.flowStartMilliseconds));
        } else {
            /* Flow duration in milliseconds */
            rwRecSetElapsed(rec, fixrec.flowDurationMilliseconds);
        }
    } else if (fixrec.flowStartMicroseconds) {
        /* Flow start time in NTP microseconds */
        sTime = skiNTPDecode(fixrec.flowStartMicroseconds);
        rwRecSetStartTime(rec, (sktime_t)sTime);
        if (fixrec.flowEndMicroseconds >= fixrec.flowStartMicroseconds) {
            /* Flow end time in NTP microseconds */
            rwRecSetElapsed(rec,
                            (uint32_t)(skiNTPDecode(fixrec.flowEndMicroseconds)
                                       - sTime));
        } else {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        }
    } else if (fixrec.flowStartSeconds) {
        /* Seconds? Sure, why not. */
        rwRecSetStartTime(rec, sktimeCreate(fixrec.flowStartSeconds, 0));
        rwRecSetElapsed(rec, ((uint32_t)1000 * (fixrec.flowEndSeconds
                                                - fixrec.flowStartSeconds)));
    } else if (fixrec.flowStartDeltaMicroseconds) {
        /* Flow start time in delta microseconds */
        sTime = (fBufGetExportTime(fbuf) * 1000
                 - fixrec.flowStartDeltaMicroseconds / 1000);
        rwRecSetStartTime(rec, (sktime_t)sTime);
        if (fixrec.flowEndDeltaMicroseconds
            && (fixrec.flowEndDeltaMicroseconds
                <= fixrec.flowStartDeltaMicroseconds))
        {
            /* Flow end time in delta microseconds */
            eTime = (fBufGetExportTime(fbuf) * 1000
                     - fixrec.flowEndDeltaMicroseconds / 1000);
            rwRecSetElapsed(rec, (uint32_t)(eTime - sTime));
        } else {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        }
    } else {
        /* No per-flow time information.
         * Assume message export is flow end time. */
        if (fixrec.flowDurationMilliseconds) {
            /* Flow duration in milliseconds */
            rwRecSetElapsed(rec, fixrec.flowDurationMilliseconds);
        } else if (fixrec.flowDurationMicroseconds) {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        } else {
            /* Presume zero duration flow */
            rwRecSetElapsed(rec, 0);
        }
        /* Set start time based on export and elapsed time */
        rwRecSetStartTime(rec, sktimeCreate((fBufGetExportTime(fbuf)
                                             - rwRecGetElapsed(rec)), 0));
    }

    /* Copy the remainder of the record */
    rwRecSetProto(rec, fixrec.rw.protocolIdentifier);
    rwRecSetFlowType(rec, fixrec.rw.silkFlowType);
    rwRecSetSensor(rec, fixrec.rw.silkFlowSensor);
    rwRecSetFlags(rec, fixrec.rw.tcpControlBits);
    rwRecSetInitFlags(rec, fixrec.rw.initialTCPFlags);
    rwRecSetRestFlags(rec, fixrec.rw.unionTCPFlags);
    rwRecSetTcpState(rec, fixrec.rw.silkTCPState);
    rwRecSetApplication(rec, fixrec.rw.silkAppLabel);

    /* Generate full flags from initial and rest if necessary */
    if (!rwRecGetFlags(rec)) {
        rwRecSetFlags(rec, rwRecGetInitFlags(rec) | rwRecGetRestFlags(rec));
    }

    /* Convert TCP state info */
    if (!rwRecGetTcpState(rec)) {
        /* Note expanded flags */
        uint8_t tcp_state = 0;
        if ((rwRecGetInitFlags(rec) | rwRecGetRestFlags(rec)) ||
            fbTemplateContainsElement(tmpl, &tcp_init_flags_ie))
        {
            tcp_state |= SK_TCPSTATE_EXPANDED;
        }
        /* Note active timeout */
        if ((fixrec.flowEndReason & SKI_END_MASK) == SKI_END_ACTIVE) {
            tcp_state |= SK_TCPSTATE_TIMEOUT_KILLED;
        }
        /* Note continuation */
        if (fixrec.flowEndReason & SKI_END_ISCONT) {
            tcp_state |= SK_TCPSTATE_TIMEOUT_STARTED;
        }
        rwRecSetTcpState(rec, tcp_state);
    }

    /* Handle the reverse record if the caller provided one */
    if (revRec) {
        /* We determined the bytes for the bi-flow record above. */
        if (0 == bytes) {
            /* No data for reverse direction; just clear the record. */
            RWREC_CLEAR(revRec);
        } else {
            /* We have data for reverse direction. */
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
            skAppPrintErr("Handling reverse side of bi-flow");
#endif

            /* Initialize the reverse record with the forward
             * record  */
#if 1
            RWREC_COPY(revRec, rec);
#else
            /* instead of copying the forward record and changing
             * nearly everything, we could just set these fields on
             * the reverse record. */
            rwRecSetProto(revRec, fixrec.rw.protocolIdentifier);
            rwRecSetFlowType(revRec, fixrec.rw.silkFlowType);
            rwRecSetSensor(revRec, fixrec.rw.silkFlowSensor);
            rwRecSetTcpState(revRec, fixrec.rw.silkTCPState);
            rwRecSetApplication(revRec, fixrec.rw.silkAppLabel);
            /* does using the forward nexthop IP for the reverse
             * record make any sense?  Shouldn't we check for a
             * reverse next hop address? */
#if SK_ENABLE_IPV6
            if (rwRecIsIPv6(rec)) {
                rwRecSetIPv6(revRec);
                rwRecMemSetNhIPv6(revRec, &fixrec.rw.ipNextHopIPv6Address);
            } else
#endif
            {
                rwRecSetNhIPv4(revRec, &fixrec.rw.ipNextHopIPv4Address);
            }
#endif  /* else clause of '#if 1' */

            /* Reverse the IPs */
#if SK_ENABLE_IPV6
            if (rwRecIsIPv6(rec)) {
                rwRecMemSetSIPv6(revRec, &fixrec.rw.destinationIPv6Address);
                rwRecMemSetDIPv6(revRec, &fixrec.rw.sourceIPv6Address);
            } else
#endif
            {
                rwRecSetSIPv4(revRec, fixrec.rw.destinationIPv4Address);
                rwRecSetDIPv4(revRec, fixrec.rw.sourceIPv4Address);
            }

            /* Reverse the ports */
            rwRecSetSPort(revRec, rwRecGetDPort(rec));
            rwRecSetDPort(revRec, rwRecGetSPort(rec));

            /* Reverse the SNMP or VLAN interfaces */
            if (vlan_interfaces &&
                !fbTemplateContainsElementByName(tmpl, &post_vlan_ies))
            {
                /* we have a single vlanId, so don't swap the values */
            } else {
                rwRecSetInput(revRec, rwRecGetOutput(rec));
                rwRecSetOutput(revRec, rwRecGetInput(rec));
            }

            /* Set volume.  We retrieved them above */
            rwRecSetPkts(revRec, CLAMP_VAL(pkts, UINT32_MAX));
            rwRecSetBytes(revRec, CLAMP_VAL(bytes, UINT32_MAX));

            /* Calculate reverse start time from reverse RTT */

            /* Reverse flow's start time must be increased and its
             * duration decreased by its offset from the forward
             * record  */
            rwRecSetStartTime(revRec, (rwRecGetStartTime(rec)
                                       + fixrec.reverseFlowDeltaMilliseconds));
            rwRecSetElapsed(revRec, (rwRecGetElapsed(rec)
                                     - fixrec.reverseFlowDeltaMilliseconds));

            /* Get reverse TCP flags from the IPFIX record if they are
             * available.  Otherwise, leave the flags unchanged (using
             * those from the forward direction). */
            if (fbTemplateContainsElementByName(tmpl, &reverse_tcp_ctrl_ies)
                || fbTemplateContainsElementByName(tmpl,&reverse_tcp_init_ies))
            {
#if (SKIPFIX_DEBUG & SKI_TRACE_REC_DECODE)
                skAppPrintErr("Using reverse TCP flags");
#endif
                rwRecSetFlags(revRec, fixrec.reverseTcpControlBits);
                rwRecSetInitFlags(revRec, fixrec.reverseInitialTCPFlags);
                rwRecSetRestFlags(revRec, fixrec.reverseUnionTCPFlags);

                /* Generate full flags from initial and rest if necessary */
                if (!rwRecGetFlags(revRec)) {
                    rwRecSetFlags(revRec, (rwRecGetInitFlags(revRec)
                                           | rwRecGetRestFlags(revRec)));
                }
            }
        }
    }

    /* all done */
    return TRUE;
}


/* Append SiLK Flow 'rec' to the libfixbuf buffer 'fbuf' */
gboolean skiRwAppendRecord(
    fBuf_t      *fbuf,
    const rwRec *rec,
    GError     **err)
{
    ski_rwrec_t fixrec;

    /* Convert time from start/elapsed to start and end epoch millis. */
    fixrec.flowStartMilliseconds = (uint64_t)rwRecGetStartTime(rec);
    fixrec.flowEndMilliseconds = ((uint64_t)fixrec.flowStartMilliseconds
                                  + rwRecGetElapsed(rec));

    /* Handle IP addresses */
#if SK_ENABLE_IPV6
    if (rwRecIsIPv6(rec)) {
        rwRecMemGetSIPv6(rec, fixrec.sourceIPv6Address);
        rwRecMemGetDIPv6(rec, fixrec.destinationIPv6Address);
        rwRecMemGetNhIPv6(rec, fixrec.ipNextHopIPv6Address);
        fixrec.sourceIPv4Address = 0;
        fixrec.destinationIPv4Address = 0;
        fixrec.ipNextHopIPv4Address = 0;
    } else
#endif
    {
        memset(fixrec.sourceIPv6Address, 0,
               sizeof(fixrec.sourceIPv6Address));
        memset(fixrec.destinationIPv6Address, 0,
               sizeof(fixrec.destinationIPv6Address));
        memset(fixrec.ipNextHopIPv6Address, 0,
               sizeof(fixrec.ipNextHopIPv6Address));
        fixrec.sourceIPv4Address = rwRecGetSIPv4(rec);
        fixrec.destinationIPv4Address = rwRecGetDIPv4(rec);
        fixrec.ipNextHopIPv4Address = rwRecGetNhIPv4(rec);
    }

    /* Copy rest of record */
    fixrec.sourceTransportPort = rwRecGetSPort(rec);
    fixrec.destinationTransportPort = rwRecGetDPort(rec);
    fixrec.ingressInterface = rwRecGetInput(rec);
    fixrec.egressInterface = rwRecGetOutput(rec);
    fixrec.packetDeltaCount = rwRecGetPkts(rec);
    fixrec.octetDeltaCount = rwRecGetBytes(rec);
    fixrec.protocolIdentifier = rwRecGetProto(rec);
    fixrec.silkFlowType = rwRecGetFlowType(rec);
    fixrec.silkFlowSensor = rwRecGetSensor(rec);
    fixrec.tcpControlBits = rwRecGetFlags(rec);
    fixrec.initialTCPFlags = rwRecGetInitFlags(rec);
    fixrec.unionTCPFlags = rwRecGetRestFlags(rec);
    fixrec.silkTCPState = rwRecGetTcpState(rec);
    fixrec.silkAppLabel = rwRecGetApplication(rec);

#if SKI_FLOW_SPEC_PADDING != 0
    /* According to RFC5102, the value of the paddingOctets
     * Information Element "is always a sequence of 0x00 values." */
    memset(fixrec.pad, 0, SKI_FLOW_SPEC_PADDING);
#endif

    /* Append the record to the buffer */
    if (!fBufAppend(fbuf, (uint8_t *)&fixrec, sizeof(fixrec), err)) {
        return FALSE;
    }

    /* all done */
    return TRUE;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
