/*
** Copyright (C) 2006-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  SiLK message functions
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: skmsg.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skdeque.h>
#include <silk/multiqueue.h>
#include "intdict.h"
#include <silk/sklog.h>
#include <silk/skmsg.h>
#include <poll.h>
#if SK_ENABLE_GNUTLS
#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <gnutls/pkcs12.h>
#include <gcrypt.h>
#endif /* SK_ENABLE_GNUTLS */

/* #define SKTHREAD_DEBUG_MUTEX 1 */
#include <silk/skthread.h>


/* Define Constants */

/* Keepalive timeout for the control channel */
#define SKMSG_CONTROL_KEEPALIVE_TIMEOUT 60 /* seconds */

/* Time used by connections without keepalive times to determine how
 * whether the connection is stagnant. */
#define SKMSG_DEFAULT_STAGNANT_TIMEOUT (2 * SKMSG_CONTROL_KEEPALIVE_TIMEOUT)

/* Time used to determine how often to check to see if the connection
 * is still alive. */
#define SKMSG_MINIMUM_READ_SELECT_TIMEOUT SKMSG_CONTROL_KEEPALIVE_TIMEOUT


/* Read and write sides of control pipes */
#define READ 0
#define WRITE 1

#define SKMERR_MEMORY  -1
#define SKMERR_PIPE    -2
#define SKMERR_MUTEX   -3
#define SKMERR_PTHREAD -4
#define SKMERR_ERROR   -5
#define SKMERR_ERRNO   -6
#define SKMERR_CLOSED  -7
#define SKMERR_SHORT   -8

#define LISTENQ 5

#define SKMSG_CTL_CHANNEL_ANNOUNCE  0xFFFE
#define SKMSG_CTL_CHANNEL_REPLY     0xFFFD
#define SKMSG_CTL_CHANNEL_KILL      0xFFFC
#define SKMSG_CTL_CHANNEL_KEEPALIVE 0xFFFA
#define SKMSG_WRITER_UNBLOCKER      0xFFFB

#define SKMSG_MINIMUM_SYSTEM_CTL_CHANNEL 0xFFFA

/* Diffie-Hellman bits for GnuTLS */
#define DH_BITS 1024

/* TLS read timeout, in milliseconds */
#define TLS_POLL_TIMEOUT 1000

/* IO thread check timeout, in milliseconds*/
#define SKMSG_IO_POLL_TIMEOUT 1000

/* Define Macros */

/* Turns on DEBUG_PRINT macros if defined. */
/* #define DEBUGGING_OUTPUT 1 */

/* Turns on function entry/exit debug print macros if defined. */
/* #define DEBUG_FUNCTION_ENTRY_EXIT 1 */


#ifdef DEBUGGING_OUTPUT

/* Macros to print debug messages, including line number and thread
   information. */
#define DEBUG_PRINT1(x)           SKTHREAD_DEBUG_PRINT1(x)
#define DEBUG_PRINT2(x, y)        SKTHREAD_DEBUG_PRINT2(x, y)
#define DEBUG_PRINT3(x, y, z)     SKTHREAD_DEBUG_PRINT3(x, y, z)
#define DEBUG_PRINT4(x, y, z, zz) SKTHREAD_DEBUG_PRINT4(x, y, z, zz)

#else  /* DEBUGGING_OUTPUT */

#define DEBUG_PRINT1(x)
#define DEBUG_PRINT2(x, y)
#define DEBUG_PRINT3(x, y, z)
#define DEBUG_PRINT4(x, y, z, zz)

#endif  /* DEBUGGING_OUTPUT */


/* Should be at the beginning of functions */
#define XDEBUG_ENTER_FUNC DEBUG_PRINT2("Entering %s", __func__)

/* Should be used instead of return */
#define XRETURN(x)                              \
    do {                                        \
        DEBUG_PRINT2("Exiting %s", __func__);   \
        return x;                               \
    } while (0)

#define XRETURN_VOID                            \
    do {                                        \
        DEBUG_PRINT2("Exiting %s", __func__);   \
        return;                                 \
    } while (0)

#ifdef DEBUG_FUNCTION_ENTRY_EXIT

#  define RETURN(x) XRETURN(x)
#  define DEBUG_ENTER_FUNC XDEBUG_ENTER_FUNC
#  define RETURN_VOID XRETURN_VOID

#else  /* DEBUG_FUNCTION_ENTRY_EXIT */

#  define DEBUG_ENTER_FUNC
#  define RETURN(x)   return x
#  define RETURN_VOID return

#endif  /* DEBUG_FUNCTION_ENTRY_EXIT */


#if SK_ENABLE_GNUTLS
#  define UNUSED_IF_NOTLS(x) x
#else
#  define UNUSED_IF_NOTLS(x) UNUSED(x)
#endif /* SK_ENABLE_GNUTLS */


/* Mutex accessor */
#define QUEUE_MUTEX(q) (&(q)->root->mutex)

/* Queue lock */
#define QUEUE_LOCK(q) MUTEX_LOCK(QUEUE_MUTEX(q))

/* Queue unlock */
#define QUEUE_UNLOCK(q) MUTEX_UNLOCK(QUEUE_MUTEX(q))

/* Queue wait */
#define QUEUE_WAIT(cond, q) MUTEX_WAIT(cond, QUEUE_MUTEX(q))

/* Queue locked assert */
#define ASSERT_QUEUE_LOCK(q)                                    \
    assert(pthread_mutex_trylock(QUEUE_MUTEX(q)) == EBUSY)


/* Any place an XASSERT occurs could be replaced with better error
   handling at some point in time. */
#define XASSERT(x) do {                                                 \
    if (!(x)) {                                                         \
        CRITMSG("Unhandled error at " __FILE__ ":%" PRIu32 " \"" #x "\"", \
                __LINE__);                                              \
        skAbort();                                                      \
    }                                                                   \
    } while (0)
#define MEM_ASSERT(x) do {                                              \
    if (!(x)) {                                                         \
        CRITMSG(("Memory allocation error creating \"" #x "\" at "      \
                 __FILE__ ":%" PRIu32), __LINE__);                      \
        abort();                                                        \
    }                                                                   \
    } while (0)

/* Macros to deal with thread creation and destruction */
#define QUEUE_TINFO(q) ((q)->root->tinfo)

#define THREAD_INFO_INIT(q)                             \
    do {                                                \
        pthread_cond_init(&QUEUE_TINFO(q).cond, NULL);  \
        QUEUE_TINFO(q).count = 0;                       \
    } while (0)

#define THREAD_INFO_DESTROY(q) pthread_cond_destroy(&QUEUE_TINFO(q).cond)


#define THREAD_START(name, rv, q, loc, fn, arg)                 \
    do {                                                        \
        DEBUG_PRINT1("THREAD_START");                           \
        ASSERT_QUEUE_LOCK(q);                                   \
        QUEUE_TINFO(q).count++;                                 \
        (rv) = skthread_create(name, (loc), (fn), (arg));       \
        if ((rv) != 0) {                                        \
            QUEUE_TINFO(q).count--;                             \
        }                                                       \
    } while (0)

#define THREAD_END(q)                                           \
    do {                                                        \
        DEBUG_PRINT1("THREAD_END");                             \
        ASSERT_QUEUE_LOCK(q);                                   \
        assert(QUEUE_TINFO(q).count != 0);                      \
        QUEUE_TINFO(q).count--;                                 \
        DEBUG_PRINT2("THREAD END COUNT decremented to %d",      \
                     QUEUE_TINFO(q).count);                     \
        MUTEX_BROADCAST(&QUEUE_TINFO(q).cond);                  \
    } while (0)

#define THREAD_WAIT_END(q, state)                               \
    do {                                                        \
        DEBUG_PRINT1("WAITING FOR THREAD_END");                 \
        ASSERT_QUEUE_LOCK(q);                                   \
        while ((state) != SKM_THREAD_ENDED) {                   \
            QUEUE_WAIT(&QUEUE_TINFO(q).cond, q);                \
        }                                                       \
        DEBUG_PRINT1("FINISHED WAITING FOR THREAD_END");        \
    } while (0)

#define THREAD_WAIT_ALL_END(q)                                  \
    do {                                                        \
        DEBUG_PRINT1("WAITING FOR ALL THREAD_END");             \
        ASSERT_QUEUE_LOCK(q);                                   \
        while (QUEUE_TINFO(q).count != 0) {                     \
            DEBUG_PRINT2("THREAD ALL END WAIT COUNT == %d",     \
                         QUEUE_TINFO(q).count);                 \
            QUEUE_WAIT(&QUEUE_TINFO(q).cond, q);                \
        }                                                       \
        DEBUG_PRINT1("FINISHED WAITING FOR ALL THREAD_END");    \
    } while (0)

#define CONNECTION_STAGNANT(conn, t)                    \
    (difftime(t, (conn)->last_recv)                     \
     > ((conn)->keepalive ? (2.0 * (conn)->keepalive) : \
        SKMSG_DEFAULT_STAGNANT_TIMEOUT))


/*** Local types ***/

/* The type of message headers */
typedef struct sk_msg_hdr_st {
    skm_channel_t channel;
    skm_type_t    type;
    skm_len_t     size;
} sk_msg_hdr_t;

/* The type of messages */
struct sk_msg_st {
    sk_msg_hdr_t hdr;
    void       (*free_fn)(uint16_t, struct iovec *);
    void       (*simple_free)(void *);
    uint16_t     segments;
    struct iovec segment[1];
};

/* Forward declaration for sk_msg_conn_queue */
struct sk_msg_conn_queue_st;

/* Buffer for reading sk_msg_t's.   */
typedef struct sk_msg_buffer_st {
    sk_msg_t *msg;              /* The message being read */
    uint8_t  *loc;              /* Location to read into */
    uint16_t  count;            /* Number of bytes still to read */
} sk_msg_buffer_t;

/* Protocol specific transport functions */
typedef struct sk_msg_transport_fn_st {
    int  (*send) (struct sk_msg_conn_queue_st  *conn,
                  sk_msg_t                     *msg);
    int  (*recv) (struct sk_msg_conn_queue_st  *conn,
                  sk_msg_t                    **msg);
} sk_msg_transport_fn_t;


/* Allow tracking of thread entance and exits. */
typedef struct sk_thread_info_st {
    pthread_cond_t  cond;
    uint32_t        count;
} sk_thread_info_t;

typedef enum {
    SKM_THREAD_BEFORE,
    SKM_THREAD_RUNNING,
    SKM_THREAD_SHUTTING_DOWN,
    SKM_THREAD_ENDED
} sk_thread_state_t;


/* The type of a message queue root */
typedef struct sk_msg_root_st {
    pthread_mutex_t     mutex;  /* Global mutex for message queue */

    skm_channel_t       next_channel; /* The next channel number to
                                         try to allocate */

    sk_thread_info_t    tinfo;  /* Information about active threads */

    int_dict_t         *channel; /* Map of channel-id to channels */
    int_dict_t         *connection; /* Map of read socket to connections */
    int_dict_t         *groups;     /* Map of channel-ids to queues */

    /* The information for binding and listening for connections */
    struct sockaddr_in  bind_addr;
    int                 bind_type;
    int                 listen_sock;
    pthread_t           listener;
    sk_thread_state_t   listener_state; /* The listener state */
    pthread_cond_t      listener_cond;  /* Listener condition variable */

    sk_msg_queue_t     *shutdownqueue;

#if SK_ENABLE_GNUTLS
    gnutls_certificate_credentials_t cred; /* Auth/Encryption credentials */
#endif

    unsigned            shuttingdown: 1;
#if SK_ENABLE_GNUTLS
    unsigned            cred_set: 1;
    unsigned            bind_tls: 1;
#endif
} sk_msg_root_t;


/* The type of a message queue */
/* typedef struct sk_msg_queue_st sk_msg_queue_t; */
struct sk_msg_queue_st {
    sk_msg_root_t      *root;
    int_dict_t         *channel;   /* Map of channels */
    mq_multi_t         *group;     /* Queue group for all channels */
    pthread_cond_t      shutdowncond;
    unsigned            shuttingdown: 1;
};

struct sk_msg_channel_queue_st;
typedef struct sk_msg_channel_queue_st sk_msg_channel_queue_t;

/* States for connections and channels */
typedef enum {
    SKM_CREATED,
    SKM_CONNECTING,
    SKM_CONNECTED,
    SKM_CLOSED
} sk_msg_state_t;

/* States for threads */
typedef enum {
    SKM_SEND_INTERNAL,
    SKM_SEND_REMOTE,
    SKM_SEND_CONTROL
} sk_send_type_t;

/* What type of connection to use */
typedef enum {
    CONN_TCP,
    CONN_TLS
} skm_conn_t;

/* TLS connections */
typedef enum {
    SKM_TLS_NONE,
    SKM_TLS_CLIENT,
    SKM_TLS_SERVER
} skm_tls_type_t;

/* Represents a connected socket or pipe */
typedef struct sk_msg_conn_queue_st {
    int                     rsocket; /* Read socket */
    int                     wsocket; /* Write socket */

    sk_sockaddr_t          *addr; /* Address of connection */

    sk_msg_transport_fn_t   fn; /* send and receive functions */

    int_dict_t             *channelmap; /* Channel map */
    uint16_t                refcount; /* channel refcount */

    sk_msg_state_t          state; /* Current state of connection */

    skDeque_t               queue; /* outgoing write queue */
    pthread_t               writer; /* Writer thread handle */
    sk_thread_state_t       writer_state; /* State */
    pthread_cond_t          writer_cond;  /* Condition variable for thread */

    pthread_t               reader;       /* The reader thread */
    sk_thread_state_t       reader_state; /* The reader state */
    pthread_cond_t          reader_cond;  /* Reader condition variable */

    uint16_t                keepalive; /* Keepalive timeout */
    time_t                  last_recv; /* Time of last received data */

    sk_msg_buffer_t         msg_buffer; /* A place to buffer incoming
                                         * messages */

    sk_msg_channel_queue_t *first_channel; /* Pre-connected initial channel */

#if SK_ENABLE_GNUTLS
    gnutls_session_t        session;
    unsigned                use_tls : 1;
#endif
} sk_msg_conn_queue_t;


/* Represents a channel */
struct sk_msg_channel_queue_st {
    mq_queue_t            *queue;    /* Channel's queue */
    skm_channel_t          channel;  /* local channel ID */
    skm_channel_t          rchannel; /* remote channel ID */
    sk_msg_state_t         state;    /* channel state */
    sk_msg_conn_queue_t   *conn;     /* associated connection */
    sk_msg_queue_t        *group;    /* group associated with this channel */
    pthread_cond_t         pending;  /* pending condition variable */
    unsigned               is_pending: 1; /* Whether we are waiting for
                                             connection */
    unsigned               flushing: 1;
};

/* Used for passing data to a writer thread */
typedef struct sk_queue_and_conn_st {
    sk_msg_queue_t         *q;
    sk_msg_conn_queue_t    *conn;
} sk_queue_and_conn_t;


/* Used to represent a local/remote channel pair */
typedef struct sk_channel_pair_st {
    skm_channel_t   lchannel;
    skm_channel_t   rchannel;
} sk_channel_pair_t;


/*** Local function prototypes ***/

static int tcp_send(sk_msg_conn_queue_t *conn, sk_msg_t  *msg);
static int tcp_recv(sk_msg_conn_queue_t *conn, sk_msg_t **msg);

#if SK_ENABLE_GNUTLS

static int tls_send(sk_msg_conn_queue_t *conn, sk_msg_t  *msg);
static int tls_recv(sk_msg_conn_queue_t *conn, sk_msg_t **msg);

#endif /* SK_ENABLE_GNUTLS */

static void *reader_thread(void *);
static void *writer_thread(void *);
static void *listener_thread(void *);

static void destroy_connection(
    sk_msg_queue_t         *q,
    sk_msg_conn_queue_t    *conn);

static int send_message(
    sk_msg_queue_t *q,
    skm_channel_t   lchannel,
    skm_type_t      type,
    void           *message,
    skm_len_t       length,
    sk_send_type_t  send_type,
    int             no_copy,
    void          (*free_fn)(void *));

static int send_message_internal(
    sk_msg_channel_queue_t *chan,
    sk_msg_t               *msg,
    sk_send_type_t          send_type);


/*** Local variables ***/

static sk_msg_transport_fn_t tcp_transport_fns = {
    tcp_send,
    tcp_recv
};


#if SK_ENABLE_GNUTLS

static sk_msg_transport_fn_t tls_transport_fns = {
    tls_send,
    tls_recv
};


pthread_mutex_t sk_msg_gnutls_mutex = PTHREAD_MUTEX_INITIALIZER;
static int sk_msg_gnutls_initialized = 0;
static gnutls_dh_params_t dh_params;

#endif /* SK_ENABLE_GNUTLS */



/* Utility functions */


static sk_msg_channel_queue_t *find_channel(
    sk_msg_queue_t *q,
    skm_channel_t   channel)
{
    sk_msg_channel_queue_t **chan;

    DEBUG_ENTER_FUNC;

    chan = (sk_msg_channel_queue_t **)int_dict_get(q->root->channel,
                                                   channel, NULL);

    RETURN(chan ? *chan : NULL);
}


static void msg_simple_free(uint16_t n, struct iovec *iov)
{
    uint16_t i;
    for (i = 0; i < n; i++) {
        free(iov[i].iov_base);
    }
}


static void sk_destroy_report_message(void *vmsg)
{
    sk_msg_t *msg = (sk_msg_t *)vmsg;

    DEBUG_ENTER_FUNC;

    DEBUG_PRINT3("Queue (destroy): chan=%#x type=%#x",
                 msg->hdr.channel, msg->hdr.type);

    skMsgDestroy(msg);

    RETURN_VOID;
}


static void set_nonblock(int fd)
{
    int flags, rv;

    DEBUG_ENTER_FUNC;

    flags = fcntl(fd, F_GETFL, 0);
    XASSERT(flags != -1);
    flags |= O_NONBLOCK;
    rv = fcntl(fd, F_SETFL, flags);
    XASSERT(rv != -1);

    RETURN_VOID;
}

#if SK_ENABLE_GNUTLS
/*
 *  status = check_cert_times(x509_cert, generic_file, file_path);
 *
 *    Check the activation and expiry times on the certificate
 *    'x509_cert'.  If the times are valid, return 0.  Otherwise,
 *    print an error about an error in 'generic_file' (which
 *    references the switch used to load the file) and 'file_path'
 *    (which is the path to the file), and return -1.
 */
static int check_cert_times(
    gnutls_x509_crt_t   cert,
    const char         *file_generic,
    const char         *file_path)
{
    time_t now, t;
    int rv = 0;

    DEBUG_ENTER_FUNC;

    time(&now);

    t = gnutls_x509_crt_get_activation_time(cert);
    if (t == (time_t)-1) {
        INFOMSG("Error loading %s file '%s': Unable to get activation time",
                file_generic, file_path);
        rv = -1;
    } else if (now < t) {
        INFOMSG("Error loading %s file '%s': Certificate is not yet valid",
                file_generic, file_path);
        rv = -1;
    }

    t = gnutls_x509_crt_get_expiration_time(cert);
    if (t == (time_t)-1) {
        INFOMSG("Error loading %s file '%s': Unable to get expiration time",
                file_generic, file_path);
        rv = -1;
    } else if (now > t) {
        INFOMSG("Error loading %s file '%s': Certificate has expired",
                file_generic, file_path);
        rv = -1;
    }

    RETURN(rv);
}


/*
 *  status = read_trust_file(cred, ca_filename, x509_format);
 *
 *    Read the CA list from 'ca_filename', which is in the
 *    'x509_format', and store the CAs in the credential 'cred'.
 */
static int read_trust_file(
    gnutls_certificate_credentials_t    cred,
    const char                         *ca_filename,
    gnutls_x509_crt_fmt_t               x509_format)
{
    int rv;

    DEBUG_ENTER_FUNC;

    /* add the trusted CAs from 'ca_filename'.  Returns number of
     * CERTs processed or a negative error code. */
    rv = gnutls_certificate_set_x509_trust_file(cred, ca_filename,
                                                x509_format);
    if (rv < 0) {
        INFOMSG("Error loading x509 CA trust file '%s': %s",
                ca_filename, gnutls_strerror(rv));
        goto END;
    }
    rv = 0;

#ifdef SK_HAVE_GNUTLS_CERTIFICATE_GET_X509_CAS
    {
        gnutls_x509_crt_t *ca_list;
        unsigned int ca_list_len;

        gnutls_certificate_get_x509_cas(cred, &ca_list, &ca_list_len);
        while (ca_list_len) {
            if (check_cert_times(*ca_list, "x509 CA trust", ca_filename)) {
                rv = -1;
            }
            ++ca_list;
            --ca_list_len;
        }
    }
#endif  /* SK_HAVE_GNUTLS_CERTIFICATE_GET_X509_CAS */

  END:
    RETURN((0 == rv) ? 0 : -1);
}


/*
 *  status = read_key_file(cred, cert_filename, key_filename, x509_format);
 *
 *    Read the certificate from 'cert_filename' and the key from
 *    'key_filename', where both files are in in the 'x509_format'.
 *    Store the certificate and key in the credential 'cred'.
 *
 *    In addition, check the activation and expiration times on the
 *    certificates.
 *
 *    This function is similar to the following direct GnuTLS call,
 *    except our function checks times while the GnuTLS function does
 *    not.
 *
 *    rv = gnutls_certificate_set_x509_key_file(cred, cert_filename,
 *                                              key_filename, x509_format);
 *    if (rv < 0) { ERRROR; }
 */
static int read_key_file(
    gnutls_certificate_credentials_t    cred,
    const char                         *cert_filename,
    const char                         *key_filename,
    gnutls_x509_crt_fmt_t               x509_format)
{
    skstream_t *stream = NULL;
    gnutls_x509_crt_t cert = NULL;
    gnutls_x509_privkey_t key = NULL;
    gnutls_datum_t datum;
    ssize_t file_size = 0;
    int rv = -1;

    DEBUG_ENTER_FUNC;

    /* initialize certificate and key */
    rv = gnutls_x509_crt_init(&cert);
    if (rv != GNUTLS_E_SUCCESS) {
        goto ERROR;
    }
    rv = gnutls_x509_privkey_init(&key);
    if (rv != GNUTLS_E_SUCCESS) {
        goto ERROR;
    }

    /* Read certificate file */
    if ((rv = skStreamCreate(&stream, SK_IO_READ, SK_CONTENT_OTHERBINARY))
        || (rv = skStreamBind(stream, cert_filename))
        || (rv = skStreamOpen(stream)))
    {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    datum.data = skStreamReadToEndOfFile(stream, &file_size);
    datum.size = file_size;
    if (NULL == datum.data) {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    skStreamDestroy(&stream);

    /* Parse certificate */
    rv = gnutls_x509_crt_import(cert, &datum, x509_format);
    free(datum.data);
    if (rv != GNUTLS_E_SUCCESS) {
        goto ERROR;
    }

    /* Read key file */
    if ((rv = skStreamCreate(&stream, SK_IO_READ, SK_CONTENT_OTHERBINARY))
        || (rv = skStreamBind(stream, key_filename))
        || (rv = skStreamOpen(stream)))
    {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    datum.data = skStreamReadToEndOfFile(stream, &file_size);
    datum.size = file_size;
    if (NULL == datum.data) {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    skStreamDestroy(&stream);

    /* Parse key */
    rv = gnutls_x509_privkey_import(key, &datum, x509_format);
    free(datum.data);
    if (rv != GNUTLS_E_SUCCESS) {
        goto ERROR;
    }

    /* Put cert and key onto the credential */
    rv = gnutls_certificate_set_x509_key(cred, &cert, 1, key);
    if (rv != GNUTLS_E_SUCCESS) {
        goto ERROR;
    }

    /* Check activation/expire times on the certifcate */
    rv = check_cert_times(cert, "certificate", cert_filename);
    if (rv) {
        goto END;
    }

    rv = 0;

    /* Use "goto ERROR" to print a GnuTLS error; "goto END" otherwise */
  ERROR:
    if (0 != rv) {
        ERRMSG("Error loading certificate or key files '%s', '%s': %s",
               cert_filename, key_filename, gnutls_strerror(rv));
    }
  END:
    skStreamDestroy(&stream);
    if (cert) {
        gnutls_x509_crt_deinit(cert);
    }
    if (key) {
        gnutls_x509_privkey_deinit(key);
    }
    RETURN((0 == rv) ? 0 : -1);
}


/*
 *  status = read_pkcs12_file(cred, cert_filename, x509_format, password);
 *
 *    Read the PKCS12 certificates and keys from 'cert_filename',
 *    where the file is in in the 'x509_format'.  Store the
 *    certificate and key in the credential 'cred'.  If the file is
 *    password protected, the 'password' is the password; otherwise it
 *    is NULL.
 *
 *    This function is similar to the following direct GnuTLS call,
 *    except our function checks times while the GnuTLS function does
 *    not.
 *
 *    rv = gnutls_certificate_set_x509_simple_pkcs12_file(cred, cert_filename,
 *                                                        x509_format,
 *                                                        password);
 *    if (rv < 0) { ERROR; }
 */
static int read_check_pkcs12(
    gnutls_certificate_credentials_t    cred,
    const char                         *cert_filename,
    gnutls_x509_crt_fmt_t               x509_format,
    const char                         *password)
{
    /* To verify the PKCS12 cert, we basically need to inline
     * everything that the GnuTLS functions
     * gnutls_certificate_set_x509_simple_pkcs12_mem() and
     * parse_pkcs12() do. Whee! */

    skstream_t *stream = NULL;
    gnutls_datum_t p12blob;
    ssize_t file_size = 0;
    int rv;

    gnutls_pkcs12_t p12 = NULL;
    gnutls_x509_privkey_t key = NULL;
    gnutls_x509_crt_t cert = NULL;
    gnutls_pkcs12_bag_t bag = NULL;
    int idx = 0;
    size_t cert_id_size = 0;
    size_t key_id_size = 0;
    unsigned char cert_id[20];
    unsigned char key_id[20];
    int elements_in_bag;
    int i;
    int bag_type;
    gnutls_datum_t data;

     DEBUG_ENTER_FUNC;

    /* Read pkcs12 file */
    if ((rv = skStreamCreate(&stream, SK_IO_READ, SK_CONTENT_OTHERBINARY))
        || (rv = skStreamBind(stream, cert_filename))
        || (rv = skStreamOpen(stream)))
    {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    p12blob.data = skStreamReadToEndOfFile(stream, &file_size);
    p12blob.size = file_size;
    if (NULL == p12blob.data) {
        skStreamPrintLastErr(stream, rv, &ERRMSG);
        goto END;
    }
    skStreamDestroy(&stream);

    /* Initialize PKCS#12 certificate */
    rv = gnutls_pkcs12_init(&p12);
    if (rv < 0) {
        free(p12blob.data);
        p12 = NULL;
        goto ERROR;
    }

    /* Parse PKCS#12 certificate */
    rv = gnutls_pkcs12_import(p12, &p12blob, x509_format, 0);
    free(p12blob.data);
    if (rv < 0)    {
        goto ERROR;
    }
    if (password) {
        rv = gnutls_pkcs12_verify_mac(p12, password);
        if (rv < 0) {
            goto ERROR;
        }
    }

    /* Following taken from parse_pkcs12() in GnuTLS 2.8.6 sources */

    /* find the first private key */
    for (idx = 0; NULL == key; ++idx) {
        rv = gnutls_pkcs12_bag_init(&bag);
        if (rv < 0) {
            bag = NULL;
            goto ERROR;
        }
        rv = gnutls_pkcs12_get_bag(p12, idx, bag);
        if (rv < 0) {
            /* either error or no more bags in pkcs structure */
            goto ERROR;
        }

        rv = gnutls_pkcs12_bag_get_type(bag, 0);
        if (rv < 0) {
            goto ERROR;
        }
        if (rv == GNUTLS_BAG_ENCRYPTED) {
            rv = gnutls_pkcs12_bag_decrypt(bag, password);
            if (rv < 0) {
                goto ERROR;
            }
        }

        elements_in_bag = gnutls_pkcs12_bag_get_count(bag);
        for (i = 0; NULL == key && i < elements_in_bag; ++i) {
            bag_type = gnutls_pkcs12_bag_get_type(bag, i);
            rv = gnutls_pkcs12_bag_get_data(bag, i, &data);

            switch (bag_type) {
              case GNUTLS_BAG_PKCS8_ENCRYPTED_KEY:
              case GNUTLS_BAG_PKCS8_KEY:
                rv = gnutls_x509_privkey_init(&key);
                if (rv < 0) {
                    key = NULL;
                    goto ERROR;
                }
                rv = gnutls_x509_privkey_import_pkcs8(
                    key, &data, GNUTLS_X509_FMT_DER, password,
                    ((bag_type==GNUTLS_BAG_PKCS8_KEY) ? GNUTLS_PKCS_PLAIN :0));
                if (rv < 0) {
                    goto ERROR;
                }
                /* get key_id to compare to cert_id below */
                key_id_size = sizeof(key_id);
                rv = gnutls_x509_privkey_get_key_id(key, 0, key_id,
                                                    &key_id_size);
                if (rv < 0) {
                    goto ERROR;
                }
                break;

              default:
                break;
            }
        }

        gnutls_pkcs12_bag_deinit(bag);
    }

    bag = NULL;

    /* now find the corresponding certificate */
    for (idx = 0; NULL == cert; ++idx) {
        rv = gnutls_pkcs12_bag_init(&bag);
        if (rv < 0) {
            bag = NULL;
            goto ERROR;
        }
        rv = gnutls_pkcs12_get_bag(p12, idx, bag);
        if (rv < 0) {
            /* either error or no more bags in pkcs structure */
            goto ERROR;
        }

        rv = gnutls_pkcs12_bag_get_type(bag, 0);
        if (rv < 0) {
            goto ERROR;
        }
        if (rv == GNUTLS_BAG_ENCRYPTED) {
            rv = gnutls_pkcs12_bag_decrypt(bag, password);
            if (rv < 0) {
                goto ERROR;
            }
        }

        elements_in_bag = gnutls_pkcs12_bag_get_count(bag);
        for (i = 0; NULL == cert && i < elements_in_bag; ++i) {
            bag_type = gnutls_pkcs12_bag_get_type(bag, i);
            rv = gnutls_pkcs12_bag_get_data(bag, i, &data);

            switch (bag_type) {
              case GNUTLS_BAG_CERTIFICATE:
                rv = gnutls_x509_crt_init(&cert);
                if (rv < 0) {
                    cert = NULL;
                    goto ERROR;
                }
                rv = gnutls_x509_crt_import(cert, &data,
                                            GNUTLS_X509_FMT_DER);
                if (rv < 0) {
                    goto ERROR;
                }
                /* check if the key id match */
                cert_id_size = sizeof(cert_id);
                rv = gnutls_x509_crt_get_key_id(cert, 0, cert_id,
                                                &cert_id_size);
                if (rv < 0) {
                    goto ERROR;
                }
                if (memcmp(cert_id, key_id, cert_id_size) != 0) {
                    /* they don't match - skip the certificate */
                    gnutls_x509_crt_deinit(cert);
                    cert = NULL;
                }
                break;

              case GNUTLS_BAG_CRL:
              case GNUTLS_BAG_ENCRYPTED:
              case GNUTLS_BAG_EMPTY:
              default:
                break;
            }
        }

        gnutls_pkcs12_bag_deinit(bag);
    }

    bag = NULL;

    assert(key);
    assert(cert);

    /* Put cert and key onto the credential */
    rv = gnutls_certificate_set_x509_key(cred, &cert, 1, key);
    if (rv < 0) {
        goto ERROR;
    }

    /* Check activation/expire times on the certifcate */
    rv = check_cert_times(cert, "PKCS#12", cert_filename);
    if (rv) {
        goto END;
    }

    rv = 0;

    /* Use "goto ERROR" to print a GnuTLS error; "goto END" otherwise */
  ERROR:
    if (rv != 0) {
        ERRMSG("Error getting PKCS#12 certificate from file '%s': %s",
               cert_filename, gnutls_strerror(rv));
    }
  END:
    skStreamDestroy(&stream);
    if (bag) {
        gnutls_pkcs12_bag_deinit(bag);
    }
    if (p12) {
        gnutls_pkcs12_deinit(p12);
    }
    if (cert) {
        gnutls_x509_crt_deinit(cert);
    }
    if (key) {
        gnutls_x509_privkey_deinit(key);
    }
    RETURN((0 == rv) ? 0 : -1);
}

#endif  /* SK_ENABLE_GNUTLS */


/*** TCP functions ***/

static int tcp_send(
    sk_msg_conn_queue_t *conn,
    sk_msg_t            *msg)
{
    ssize_t rv;
    ssize_t msg_size;

    DEBUG_ENTER_FUNC;

    assert(msg);
    assert(conn);
    assert(msg->segments);
    assert(msg->segment[0].iov_base == &msg->hdr);
    assert(msg->segment[0].iov_len  == sizeof(msg->hdr));

    msg_size = sizeof(msg->hdr) + msg->hdr.size;

    /* Convert data to network byte order */
    DEBUG_PRINT3("Sending chan = %#x type = %#x",
                 msg->hdr.channel, msg->hdr.type);
    msg->hdr.channel = htons(msg->hdr.channel);
    msg->hdr.type    = htons(msg->hdr.type);
    msg->hdr.size    = htons(msg->hdr.size);

  retry:
    /* Write the message */
    rv = writev(conn->wsocket, msg->segment, msg->segments);
    if (rv == -1) {
        int err;
        if (errno == EINTR) {
            goto retry;
        }
        if (errno == EPIPE || errno == ECONNRESET) {
            RETURN(SKMERR_CLOSED);
        }
        err = errno;
        DEBUG_PRINT3("send: System error %d [%s]", errno, strerror(errno));
        errno = err;
        RETURN(SKMERR_ERRNO);
    } else if (rv == 0) {
        DEBUG_PRINT1("send: Connection closed");
        RETURN(SKMERR_CLOSED);
    } else if (rv != msg_size) {
        DEBUG_PRINT2("send: Short write (%d)", rv);
        RETURN(SKMERR_SHORT);
    }

    RETURN(0);
}


static int tcp_recv(
    sk_msg_conn_queue_t  *conn,
    sk_msg_t            **message)
{
    ssize_t          rv;
    sk_msg_buffer_t *buffer;
    int              retval;

    DEBUG_ENTER_FUNC;

    assert(message);
    assert(conn);

    buffer = &conn->msg_buffer;

    if (buffer->msg == NULL) {
        sk_msg_hdr_t    *hdr;
        sk_msg_t        *msg;

        /* Create a message structure */
        buffer->msg = malloc(sizeof(*msg) + sizeof(struct iovec));
        msg = buffer->msg;
        MEM_ASSERT(msg != NULL);
        msg->segments = 1;
        msg->simple_free = NULL;
        msg->free_fn = msg_simple_free;
        hdr = &msg->hdr;
        msg->segment[0].iov_base = hdr;
        msg->segment[0].iov_len = sizeof(*hdr);
        memset(hdr, 0, sizeof(hdr));

        /* Read a header */
        rv = skreadn(conn->rsocket, hdr, sizeof(*hdr));
        if (rv == -1) {
            DEBUG_PRINT3("recv: System error %d [%s]", errno, strerror(errno));
            retval = SKMERR_ERRNO;
            goto error;
        } else if (rv != sizeof(*hdr)) {
            DEBUG_PRINT2("recv: Short read (%d)", rv);
            retval = SKMERR_SHORT;
            goto error;
        }

        /* Convert network byte order to host byte order */
        hdr->channel = ntohs(hdr->channel);
        hdr->type    = ntohs(hdr->type);
        hdr->size    = ntohs(hdr->size);

        DEBUG_PRINT3("Receiving chan = %#x type = %#x",
                     hdr->channel, hdr->type);

        if (hdr->size == 0) {
            /* If the size is zero, the message is complete */
            *message = msg;
            buffer->msg = NULL;
        } else {
            /* Allocate space for the body of the message */
            msg->segment[1].iov_base = malloc(hdr->size);
            MEM_ASSERT(msg->segment[1].iov_base);
            msg->segment[1].iov_len = hdr->size;
            msg->segments++;

            /* Maintain state for re-entrant call */
            buffer->count = hdr->size;
            buffer->loc = msg->segment[1].iov_base;
        }
        RETURN(0);
    }

    /* We are to continue reading a message */
    assert(buffer->count);
    rv = read(conn->rsocket, buffer->loc, buffer->count);

    if (rv == -1) {
        DEBUG_PRINT2("read failure: [%s]", strerror(errno));
        retval = SKMERR_ERRNO;
        goto error;
    } else if (rv == 0) {
        DEBUG_PRINT1("read ended: [EOF]");
        retval = SKMERR_CLOSED;
        goto error;
    }

    buffer->count -= rv;
    buffer->loc   += rv;

    if (buffer->count == 0) {
        /* Message is complete */
        *message = buffer->msg;
        buffer->msg = NULL;
    }

    RETURN(0);

  error:
    if (buffer->msg) {
        skMsgDestroy(buffer->msg);
        buffer->msg = NULL;
    }
    RETURN(retval);
}


#if SK_ENABLE_GNUTLS

/*** TLS functions ***/

static int tls_send(
    sk_msg_conn_queue_t    *conn,
    sk_msg_t               *msg)
{
    ssize_t rv;
    uint16_t i;

    DEBUG_ENTER_FUNC;

    assert(msg);
    assert(conn);
    assert(conn->use_tls);
    assert(msg->segments);
    assert(msg->segment[0].iov_base == &msg->hdr);
    assert(msg->segment[0].iov_len  == sizeof(msg->hdr));

    /* Convert data to network byte order */
    DEBUG_PRINT3("Sending chan = %#x type = %#x",
                 msg->hdr.channel, msg->hdr.type);
    msg->hdr.channel = htons(msg->hdr.channel);
    msg->hdr.type    = htons(msg->hdr.type);
    msg->hdr.size    = htons(msg->hdr.size);


    for (i = 0; i < msg->segments; i++) {
        size_t remaining = msg->segment[i].iov_len;
        uint8_t *loc = msg->segment[i].iov_base;

        while (remaining) {
          retry:
            /* Write the message */
            DEBUG_PRINT2("calling gnutls_record_send (%d)",
                         (skm_len_t)msg->segment[i].iov_len);
            rv = gnutls_record_send(conn->session, loc, remaining);
            DEBUG_PRINT2("gnutls_record_send -> %d", rv);
            if (rv < 0) {
                if (rv == GNUTLS_E_INTERRUPTED || rv == GNUTLS_E_AGAIN) {
                    goto retry;
                }
                if (rv == GNUTLS_E_EXPIRED || rv == GNUTLS_E_INTERRUPTED ||
                    (rv == GNUTLS_E_PUSH_ERROR &&
                     (errno == EPIPE || errno == ECONNRESET)))
                {
                    RETURN(SKMERR_CLOSED);
                }
                RETURN(SKMERR_ERRNO);
            } else if (rv == 0) {
                RETURN(SKMERR_CLOSED);
            } else {
                remaining -= rv;
                loc += rv;
            }
        }
    }

    RETURN(0);
}


static int tls_recv(
    sk_msg_conn_queue_t    *conn,
    sk_msg_t              **message)
{
    ssize_t          rv;
    sk_msg_buffer_t *buffer;
    int              retval;

    DEBUG_ENTER_FUNC;

    assert(message);
    assert(conn);

    buffer = &conn->msg_buffer;

    if (buffer->msg == NULL) {
        sk_msg_hdr_t    *hdr;
        sk_msg_t        *msg;

        /* Create a message structure */
        buffer->msg = malloc(sizeof(*msg) + sizeof(struct iovec));
        msg = buffer->msg;
        MEM_ASSERT(msg != NULL);
        msg->segments = 1;
        msg->simple_free = NULL;
        msg->free_fn = msg_simple_free;
        hdr = &msg->hdr;
        msg->segment[0].iov_base = hdr;
        msg->segment[0].iov_len = sizeof(*hdr);
        memset(hdr, 0, sizeof(hdr));

      retry:
        /* Read a header */
        DEBUG_PRINT2("calling gnutls_record_recv (%d)", sizeof(*hdr));
        rv = gnutls_record_recv(conn->session, hdr, sizeof(*hdr));
        DEBUG_PRINT2("gnutls_record_recv -> %d", rv);
        if (rv < 0) {
            if (rv == GNUTLS_E_INTERRUPTED || rv == GNUTLS_E_AGAIN) {
                goto retry;
            }
            retval = SKMERR_ERRNO;
            goto error;
        } else if (rv == 0) {
            retval = SKMERR_CLOSED;
            goto error;
        } else if (rv != sizeof(*hdr)) {
            retval = SKMERR_SHORT;
            goto error;
        }

        /* Convert network byte order to host byte order */
        hdr->channel = ntohs(hdr->channel);
        hdr->type    = ntohs(hdr->type);
        hdr->size    = ntohs(hdr->size);

        DEBUG_PRINT3("Receiving chan = %#x type = %#x",
                     hdr->channel, hdr->type);

        if (hdr->size == 0) {
            /* If the size is zero, the message is complete */
            *message = msg;
            buffer->msg = NULL;
        } else {
            /* Allocate space for the body of the message */
            msg->segment[1].iov_base = malloc(hdr->size);
            MEM_ASSERT(msg->segment[1].iov_base);
            msg->segment[1].iov_len = hdr->size;
            msg->segments++;

            /* Maintain state for re-entrant call */
            buffer->count = hdr->size;
            buffer->loc = msg->segment[1].iov_base;
        }
        RETURN(0);
    }


    /* We are to continue reading a message */
    assert(buffer->count);

  retry2:
    DEBUG_PRINT2("calling gnutls_record_recv (%d)", buffer->count);
    rv = gnutls_record_recv(conn->session, buffer->loc, buffer->count);
    DEBUG_PRINT2("gnutls_record_recv -> %d", rv);
    if (rv < 0) {
        if (rv == GNUTLS_E_INTERRUPTED || rv == GNUTLS_E_AGAIN) {
            goto retry2;
        }
        DEBUG_PRINT2("read failure: [%s]", gnutls_strerror(rv));
        retval = SKMERR_ERRNO;
        goto error;
    } else if (rv == 0) {
        DEBUG_PRINT1("read ended: [EOF]");
        retval = SKMERR_CLOSED;
        goto error;
    }

    buffer->count -= rv;
    buffer->loc   += rv;

    if (buffer->count == 0) {
        /* Message is complete */
        *message = buffer->msg;
        buffer->msg = NULL;
    }

    RETURN(0);

  error:
    if (buffer->msg) {
        skMsgDestroy(buffer->msg);
        buffer->msg = NULL;
    }
    RETURN(retval);
}

#endif /* SK_ENABLE_GNUTLS */


/***********************************************************************/


#if SK_ENABLE_GNUTLS

GCRY_THREAD_OPTION_PTHREAD_IMPL;

/*
 *    Initialize GnuTLS.  The caller should have the
 *    sk_msg_gnutls_mutex before calling this function.
 */
static int skMsgGnuTLSInit(void)
{
    DEBUG_ENTER_FUNC;

    if (!sk_msg_gnutls_initialized) {
        int rv;

        rv = gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_pthread);
        if (rv == 0) {
            rv = gnutls_global_init();
        }
        if (rv >= 0) {
            rv = gnutls_dh_params_init(&dh_params);
        }
        if (rv >= 0) {
            rv = gnutls_dh_params_generate2(dh_params, DH_BITS);
        }
        if (rv >= 0) {
            sk_msg_gnutls_initialized = 1;
        }
        RETURN(rv);

    }
    RETURN(0);
}


int skMsgQueueAddCA(
    sk_msg_queue_t *queue,
    const char     *ca_filename)
{
    int rv;

    DEBUG_ENTER_FUNC;

    assert(queue);
    assert(queue->root);
    assert(ca_filename);

    pthread_mutex_lock(&sk_msg_gnutls_mutex);
    if (!sk_msg_gnutls_initialized) {
        rv = skMsgGnuTLSInit();
        if (rv != 0) {
            goto END;
        }
    }
    /* allocate credentials and set Diffie-Hellman parameters */
    if (!queue->root->cred_set) {
        rv = gnutls_certificate_allocate_credentials(&queue->root->cred);
        if (rv < 0) {
            INFOMSG("Unable to allocate credentials: %s",
                    gnutls_strerror(rv));
            goto END;
        }
        gnutls_certificate_set_dh_params(queue->root->cred, dh_params);
    }

    /* add the trusted CAs from 'ca_filename'; the file should
     * be in PEM format. */
    rv = read_trust_file(queue->root->cred, ca_filename, GNUTLS_X509_FMT_PEM);
    if (rv < 0) {
        goto END;
    }

    if (!queue->root->cred_set) {
        queue->root->cred_set = 1;
        sk_msg_gnutls_initialized++;
    }
    rv = 0;

  END:
    if (0 != rv) {
        rv = -1;
        if (queue->root->cred_set) {
            gnutls_certificate_free_credentials(queue->root->cred);
            queue->root->cred_set = 0;
        }
    }
    pthread_mutex_unlock(&sk_msg_gnutls_mutex);
    RETURN(rv);
}

int skMsgQueueAddCert(
    sk_msg_queue_t *queue,
    const char     *cert_filename,
    const char     *key_filename)
{
    int rv;

    assert(queue);
    assert(queue->root);
    assert(cert_filename);
    assert(key_filename);

    DEBUG_ENTER_FUNC;

    pthread_mutex_lock(&sk_msg_gnutls_mutex);
    if (!sk_msg_gnutls_initialized) {
        rv = skMsgGnuTLSInit();
        if (rv != 0) {
            goto END;
        }
    }

    /* allocate credentials and set Diffie-Hellman parameters */
    if (!queue->root->cred_set) {
        rv = gnutls_certificate_allocate_credentials(&queue->root->cred);
        if (rv < 0) {
            INFOMSG("Unable to allocate credentials: %s",
                    gnutls_strerror(rv));
            goto END;
        }
        gnutls_certificate_set_dh_params(queue->root->cred, dh_params);
    }

    /* set a certificate/private-key pair in the credential from
     * separate certificate and key files, each in PEM format. */
    rv = read_key_file(queue->root->cred, cert_filename,
                       key_filename, GNUTLS_X509_FMT_PEM);
    if (rv < 0) {
        goto END;
    }

    if (!queue->root->cred_set) {
        queue->root->cred_set = 1;
        sk_msg_gnutls_initialized++;
    }

    rv = 0;

  END:
    if (0 != rv) {
        rv = -1;
        if (queue->root->cred_set) {
            gnutls_certificate_free_credentials(queue->root->cred);
            queue->root->cred_set = 0;
        }
    }
    pthread_mutex_unlock(&sk_msg_gnutls_mutex);
    RETURN(rv);
}


int skMsgQueueAddPKCS12(
    sk_msg_queue_t *queue,
    const char     *cert_filename,
    const char     *password)
{
    int rv;

    assert(queue);
    assert(queue->root);
    assert(cert_filename);

    DEBUG_ENTER_FUNC;

    pthread_mutex_lock(&sk_msg_gnutls_mutex);
    if (!sk_msg_gnutls_initialized) {
        rv = skMsgGnuTLSInit();
        if (rv != 0) {
            goto END;
        }
    }

    /* allocate credentials and set Diffie-Hellman parameters */
    if (!queue->root->cred_set) {
        rv = gnutls_certificate_allocate_credentials(&queue->root->cred);
        if (rv < 0) {
            INFOMSG("Unable to allocate credentials: %s",
                    gnutls_strerror(rv));
            goto END;
        }
        gnutls_certificate_set_dh_params(queue->root->cred, dh_params);
    }

    rv = read_check_pkcs12(queue->root->cred, cert_filename,
                           GNUTLS_X509_FMT_DER, password);
    if (rv < 0) {
        goto END;
    }

    if (!queue->root->cred_set) {
        queue->root->cred_set = 1;
        sk_msg_gnutls_initialized++;
    }

    rv = 0;

  END:
    if (0 != rv) {
        rv = -1;
        if (queue->root->cred_set) {
            gnutls_certificate_free_credentials(queue->root->cred);
            queue->root->cred_set = 0;
        }
    }
    pthread_mutex_unlock(&sk_msg_gnutls_mutex);
    RETURN(rv);
}

#endif /* SK_ENABLE_GNUTLS */


/* Create a channel within a message queue. */
static sk_msg_channel_queue_t *create_channel(
    sk_msg_queue_t *q)
{
    sk_msg_channel_queue_t *chan;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    ASSERT_QUEUE_LOCK(q);

    /* Allocate space for a new channel */
    chan = (sk_msg_channel_queue_t *)calloc(1, sizeof(*chan));
    MEM_ASSERT(chan != NULL);

    chan->queue = mqCreateQueue(q->group);
    MEM_ASSERT(chan->queue != NULL);

    /* Assign a local channel number and add the channel to the
       message queue */
    do {
        chan->channel = q->root->next_channel++;
        rv = int_dict_set(q->root->channel, chan->channel, &chan);
    } while (rv == 1);
    MEM_ASSERT(rv == 0);

    /* Channel is created, rchannel is unset */
    chan->state = SKM_CREATED;
    chan->rchannel = SKMSG_CHANNEL_CONTROL;
    rv = pthread_cond_init(&chan->pending, NULL);
    XASSERT(rv == 0);
    chan->is_pending = 0;

    rv = int_dict_set(q->root->groups, chan->channel, &q);
    MEM_ASSERT(rv == 0);
    rv = int_dict_set(q->channel, chan->channel, &chan);
    MEM_ASSERT(rv == 0);
    chan->group = q;

    DEBUG_PRINT2("create_channel() == %d", chan->channel);

    RETURN(chan);
}


/* Attach a channel to a connection object. */
static int set_channel_connecting(
    sk_msg_queue_t          UNUSED(*q),
    sk_msg_channel_queue_t *chan,
    sk_msg_conn_queue_t    *conn)
{
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(chan);
    assert(conn);
    ASSERT_QUEUE_LOCK(q);
    assert(chan->state == SKM_CREATED);
    assert(conn->state != SKM_CLOSED);

    DEBUG_PRINT2("set_channel_connecting(%d)", chan->channel);

    /* Set the channel's communication stream, set it to
       half-connected, and up the refcount on the connection
       object. */
    chan->conn = conn;
    chan->state = SKM_CONNECTING;

    /* Add an entry in the connections's channel map for the
       channel. */
    rv = int_dict_set(conn->channelmap, chan->channel, &chan);
    MEM_ASSERT(rv != -1);
    assert(rv == 0);

    conn->state = SKM_CONNECTED;
    conn->refcount++;

    RETURN(0);
}


/* Return 1 if the connection was destroyed, zero otherwise */
static int set_channel_closed(
    sk_msg_queue_t         *q,
    sk_msg_channel_queue_t *chan,
    int                     no_destroy)
{
    int rv;
    sk_msg_conn_queue_t *conn;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(chan);
    ASSERT_QUEUE_LOCK(q);
    assert(chan->state == SKM_CONNECTING ||
           chan->state == SKM_CONNECTED);
    assert(chan->conn);
    assert(chan->conn->refcount > 0);

    DEBUG_PRINT2("set_channel_closed(%d)", chan->channel);

    conn = chan->conn;

    if (chan->state == SKM_CONNECTED &&
        chan->channel != SKMSG_CHANNEL_CONTROL)
    {
        skm_channel_t lchannel = htons(chan->channel);
        DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_DIED (Internal)");
        rv = send_message(q, SKMSG_CHANNEL_CONTROL,
                          SKMSG_CTL_CHANNEL_DIED,
                          &lchannel, sizeof(lchannel), SKM_SEND_INTERNAL,
                          0, NULL);
    }

    rv = int_dict_del(conn->channelmap, chan->channel);
    assert(rv == 0);

    chan->state = SKM_CLOSED;
    conn->refcount--;

    /* Notify people waiting on this channel to complete connecting
       that it is dead. */
    MUTEX_BROADCAST(&chan->pending);

    if (conn->refcount == 0 && !no_destroy) {
        destroy_connection(q, conn);
        RETURN(1);
    }

    RETURN(0);
}


static int set_channel_connected(
    sk_msg_queue_t          UNUSED(*q),
    sk_msg_channel_queue_t *chan,
    skm_channel_t           rchannel)
{
    DEBUG_ENTER_FUNC;

    assert(q);
    assert(chan);
    ASSERT_QUEUE_LOCK(q);
    assert(chan->state == SKM_CONNECTING);

    DEBUG_PRINT2("set_channel_connected(%d)", chan->channel);

    chan->rchannel = rchannel;

    chan->state = SKM_CONNECTED;
    RETURN(0);
}


static void destroy_channel(
    sk_msg_queue_t         *q,
    sk_msg_channel_queue_t *chan)
{
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(chan);
    ASSERT_QUEUE_LOCK(q);

    DEBUG_PRINT2("destroy_channel(%d)", chan->channel);

    if (chan->state == SKM_CONNECTED &&
        chan->channel != SKMSG_CHANNEL_CONTROL)
    {
        skm_channel_t rchannel = htons(chan->rchannel);
        DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_KILL (Ext-control)");
        rv = send_message(q, chan->channel, SKMSG_CTL_CHANNEL_KILL,
                          &rchannel, sizeof(rchannel), SKM_SEND_CONTROL,
                          0, NULL);
    }
    if (chan->state == SKM_CONNECTED || chan->state == SKM_CONNECTING) {
        set_channel_closed(q, chan, 0);
    }

    assert(chan->state == SKM_CLOSED);

    rv = int_dict_del(q->root->channel, chan->channel);
    assert(rv == 0);

    rv = int_dict_del(q->root->groups, chan->channel);
    assert(rv == 0);
    rv = int_dict_del(chan->group->channel, chan->channel);
    assert(rv == 0);

    rv = pthread_cond_destroy(&chan->pending);
    assert(rv == 0);

    /* Disable adding to the queue (it will be destroyed when the
       group is destroyed) */
    mqQueueDisable(chan->queue, MQ_ADD);

    free(chan);

    RETURN_VOID;
}


#if SK_ENABLE_GNUTLS
static ssize_t tls_pull(
    gnutls_transport_ptr_t fd,
    void *buf,
    size_t len)
{
    struct pollfd pfd;
    int           rv;
    ssize_t       rlen;

    pfd.fd = (int)fd;
    pfd.events = POLLIN | POLLERR | POLLNVAL;

    rv = poll(&pfd, 1, TLS_POLL_TIMEOUT);
    if (rv != 1) {
        return rv;
    }

    rlen = read(pfd.fd, buf, len);

    return rlen;
}


static ssize_t tls_push(
    gnutls_transport_ptr_t fd,
    const void *buf,
    size_t len)
{
    struct pollfd pfd;
    int           rv;
    ssize_t       wlen;

    pfd.fd = (int)fd;
    pfd.events = POLLOUT | POLLERR | POLLNVAL;

    rv = poll(&pfd, 1, TLS_POLL_TIMEOUT);
    if (rv != 1) {
        return rv;
    }

    wlen = write(pfd.fd, buf, len);

    return wlen;
}


static int setup_tls(
    sk_msg_queue_t       *q,
    sk_msg_conn_queue_t  *conn,
    int                   rsocket,
    int                   wsocket,
    skm_tls_type_t        tls)
{
    int rv;
    unsigned int status;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(conn);
    assert(q->root->cred_set);

    /* initialize the TLS session object depending on whether it is
     * the client or the server */
    switch (tls) {
      case SKM_TLS_CLIENT:
        rv = gnutls_init(&conn->session, GNUTLS_CLIENT);
        break;
      case SKM_TLS_SERVER:
        rv = gnutls_init(&conn->session, GNUTLS_SERVER);
        break;
      default:
        skAbortBadCase(tls);
    }

    if (rv < 0) {
        ERRMSG("Failed TLS init: %s", gnutls_strerror(rv));
        RETURN(-1);
    }

    /* use "NORMAL" priority */
    rv = gnutls_set_default_priority(conn->session);
    XASSERT(rv >= 0);

    /* tell the session to use the public/private keys loaded earlier */
    rv = gnutls_credentials_set(conn->session, GNUTLS_CRD_CERTIFICATE,
                                q->root->cred);
    XASSERT(rv >= 0);

    /* tell TLS to use our read and write functions (tls_pull,
     * tls_push) instead of the defaults.  The call to set_ptr2()
     * controls what will be passed to tls_pull() and tls_push().  The
     * set_lowat() call sets the low water value. */
    gnutls_transport_set_ptr2(conn->session,
                              (gnutls_transport_ptr_t)rsocket,
                              (gnutls_transport_ptr_t)wsocket);
    gnutls_transport_set_pull_function(conn->session, tls_pull);
    gnutls_transport_set_push_function(conn->session, tls_push);
    gnutls_transport_set_lowat(conn->session, 0);
    set_nonblock(rsocket);

    /* force the client to send its certificate to the server */
    if (tls == SKM_TLS_SERVER) {
        gnutls_certificate_server_set_request(conn->session,
                                              GNUTLS_CERT_REQUIRE);
    }

  again1:
    DEBUG_PRINT1("Attempting TLS handshake");
    rv = gnutls_handshake(conn->session);
    if (rv < 0) {
        if (rv == GNUTLS_E_AGAIN || rv == GNUTLS_E_INTERRUPTED) {
            goto again1;
        }
        if (rv == GNUTLS_E_PUSH_ERROR) {
            NOTICEMSG("Remote side disconnected during TLS handshake.");
        } else {
            NOTICEMSG("TLS handshake failed: %s", gnutls_strerror(rv));
        }
        gnutls_deinit(conn->session);
        RETURN(-1);
    }
    DEBUG_PRINT1("TLS handshake succeeded");

    status = 0;
    rv = gnutls_certificate_verify_peers2(conn->session, &status);
    if (rv < 0) {
        const char *reason = "Unknown reason";

        NOTICEMSG("Certificate verification failed: %s",
                  gnutls_strerror(rv));

        if (status & GNUTLS_CERT_REVOKED) {
            reason = "Certificate was revoked";
        } else if (status & GNUTLS_CERT_INVALID) {
            if (status & GNUTLS_CERT_SIGNER_NOT_FOUND) {
                reason = "Certificate issuer unknown";
            } else if (status & GNUTLS_CERT_SIGNER_NOT_CA) {
                reason = "Certificate signer is not a CA";
            } else if (status & GNUTLS_CERT_INSECURE_ALGORITHM) {
                reason = "Insecure algorithm";
#if defined SK_HAVE_DECL_GNUTLS_CERT_EXPIRED && SK_HAVE_DECL_GNUTLS_CERT_EXPIRED
            } else if (status & GNUTLS_CERT_NOT_ACTIVATED) {
                reason = "Certificate is not yet activated";
            } else if (status & GNUTLS_CERT_EXPIRED) {
                reason = "Certificate has expired";
#endif
            }
        }

        NOTICEMSG("Certificate verification failed: %s", reason);

        /* teardown connection */
      again2:
        rv = gnutls_bye(conn->session, GNUTLS_SHUT_RDWR);
        if (rv == GNUTLS_E_AGAIN || rv == GNUTLS_E_INTERRUPTED) {
            goto again2;
        }
        gnutls_deinit(conn->session);
        RETURN(-1);
    }

    conn->use_tls = 1;

    RETURN(0);
}
#endif /* SK_ENABLE_GNUTLS */


static int create_connection(
    sk_msg_queue_t       *q,
    int                   rsocket,
    int                   wsocket,
    sk_sockaddr_t        *addr,
    sk_msg_conn_queue_t **rconn,
    skm_tls_type_t        UNUSED_IF_NOTLS(tls))
{
    sk_msg_conn_queue_t *conn;
    sk_queue_and_conn_t *qac;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(rconn);
    ASSERT_QUEUE_LOCK(q);

    DEBUG_PRINT3("create_connection() = %d, %d", rsocket, wsocket);

    /* Allocate space for the connection */
    conn = (sk_msg_conn_queue_t *)calloc(1, sizeof(*conn));
    MEM_ASSERT(conn != NULL);

#if SK_ENABLE_GNUTLS
    if (tls != SKM_TLS_NONE) {
        rv = setup_tls(q, conn, rsocket, wsocket, tls);
        if (rv != 0) {
            free(conn);
            RETURN(-1);
        }
    }
#endif /* SK_ENABLE_GNUTLS */

    /* Set the read and write sockets */
    conn->rsocket = rsocket;
    conn->wsocket = wsocket;

    /* And the address */
    conn->addr = addr;

    /* Set the transport functions */
#if SK_ENABLE_GNUTLS
    if (tls == SKM_TLS_NONE) {
        conn->fn = tcp_transport_fns;
    } else {
        conn->fn = tls_transport_fns;
    }
#else
    conn->fn = tcp_transport_fns;
#endif /* SK_ENABLE_GNUTLS */

    /* Set up the channel queue and refcount */
    conn->channelmap = int_dict_create(sizeof(sk_msg_channel_queue_t *));
    MEM_ASSERT(conn->channelmap != NULL);
    conn->refcount = 0;

    /* Set the connections initial state */
    conn->state = SKM_CREATED;

    /* Set up the write queue */
    conn->queue = skDequeCreate();
    XASSERT(conn->queue != NULL);

    /* Initialize the writer thread start state */
    pthread_cond_init(&conn->writer_cond, NULL);
    conn->writer_state = SKM_THREAD_BEFORE;

    /* Initialize the reader thread start state */
    pthread_cond_init(&conn->reader_cond, NULL);
    conn->reader_state = SKM_THREAD_BEFORE;

    /* Set up and start the writer thread */
    qac = (sk_queue_and_conn_t *)malloc(sizeof(*qac));
    MEM_ASSERT(qac != NULL);
    qac->q = q;
    qac->conn = conn;
    THREAD_START("skmsg_writer", rv, q, &conn->writer, writer_thread, qac);
    XASSERT(rv == 0);

    /* Wait for the thread to begin. */
    while (conn->writer_state == SKM_THREAD_BEFORE) {
        QUEUE_WAIT(&conn->writer_cond, q);
    }
    assert(conn->writer_state == SKM_THREAD_RUNNING);

    /* Start the reader thread, */
    qac = (sk_queue_and_conn_t *)malloc(sizeof(*qac));
    MEM_ASSERT(qac != NULL);
    qac->q = q;
    qac->conn = conn;
    THREAD_START("skmsg_reader", rv, q, &conn->reader, reader_thread, qac);
    XASSERT(rv == 0);

    /* Wait for the thread to begin. */
    while (conn->reader_state == SKM_THREAD_BEFORE) {
        QUEUE_WAIT(&conn->reader_cond, q);
    }

    *rconn = conn;
    RETURN(0);
}

static void unblock_connection(
    sk_msg_queue_t       UNUSED(*q),
    sk_msg_conn_queue_t *conn)
{
    static sk_msg_t unblocker = {{SKMSG_CHANNEL_CONTROL,
                                  SKMSG_WRITER_UNBLOCKER, 0},
                                 NULL, NULL, 1, {{NULL, 0}}};
    skDQErr_t err;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(conn);
    ASSERT_QUEUE_LOCK(q);

    /* Add a special messaage to the writers queue to guarantee it
       will unblock */
    DEBUG_PRINT1("Sending SKMSG_WRITER_UNBLOCKER message");
    err = skDequePushBack(conn->queue, &unblocker);
    XASSERT(err == SKDQ_SUCCESS);

    RETURN_VOID;
}


static void destroy_connection(
    sk_msg_queue_t         *q,
    sk_msg_conn_queue_t    *conn)
{
    sk_msg_channel_queue_t *chan;
    void *cont;
    int rv;
    sk_msg_t *msg;
    skDQErr_t err;
    pthread_t self;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(conn);
    ASSERT_QUEUE_LOCK(q);

    DEBUG_PRINT3("destroy_connection() = %d, %d",
                 conn->rsocket, conn->wsocket);

    /* Check to see if this connection is already being shut down */
    if (conn->state == SKM_CLOSED) {
        RETURN_VOID;
    }

    /* Okay, start closing */
    conn->state = SKM_CLOSED;
    conn->writer_state = SKM_THREAD_SHUTTING_DOWN;
    conn->reader_state = SKM_THREAD_SHUTTING_DOWN;
    unblock_connection(q, conn);

    self = pthread_self();
    if (!pthread_equal(self, conn->writer)) {
        /* Wait for the writer thread to end */
        THREAD_WAIT_END(q, conn->writer_state);
        pthread_join(conn->writer, NULL);
    }
    if (!pthread_equal(self, conn->reader)) {
        THREAD_WAIT_END(q, conn->reader_state);
        pthread_join(conn->reader, NULL);
    }

    /* Empty the queue */
    while ((err = skDequePopBackNB(conn->queue, (void *)&msg)) ==
           SKDQ_SUCCESS) {
        skMsgDestroy(msg);
    }
    assert(err == SKDQ_EMPTY);

    /* Shut down the queue */
    err = skDequeUnblock(conn->queue);
    assert(err == SKDQ_SUCCESS);

    /* Mark all channels using this connection as closed. */
    if (conn->first_channel) {
        assert(conn->first_channel->state == SKM_CREATED);
        conn->first_channel->state = SKM_CLOSED;
        destroy_channel(q, conn->first_channel);
        conn->first_channel = NULL;
    }
    cont = int_dict_get_first(conn->channelmap, &chan);
    while (cont != NULL) {
        intkey_t channel = chan->channel;
        if ((chan->state == SKM_CONNECTING ||
             chan->state == SKM_CONNECTED))
        {
            set_channel_closed(q, chan, 1);
        }

        cont = int_dict_get_next(conn->channelmap, channel, &chan);
    }
    assert(conn->refcount == 0);

    if (pthread_equal(self, conn->reader)
        || pthread_equal(self, conn->writer))
    {
        DEBUG_PRINT1("Detaching self");
        pthread_detach(self);
    }

    /* Destroy the channelmap */
    int_dict_destroy(conn->channelmap);

    /* Close the socket(s) */
    close(conn->rsocket);
    if (conn->rsocket != conn->wsocket) {
        close(conn->wsocket);
    }

    /* Destroy the queue */
    err = skDequeDestroy(conn->queue);
    assert(err == SKDQ_SUCCESS);

#if SK_ENABLE_GNUTLS
    /* Destroy the session */
    if (conn->use_tls) {
      again:
        rv = gnutls_bye(conn->session, GNUTLS_SHUT_RDWR);
        if (rv == GNUTLS_E_AGAIN || rv == GNUTLS_E_INTERRUPTED) {
            goto again;
        }
        gnutls_deinit(conn->session);
    }
#endif /* SK_ENABLE_GNUTLS */

    /* Destroy the condition variables */
    rv = pthread_cond_destroy(&conn->writer_cond);
    assert(rv == 0);
    rv = pthread_cond_destroy(&conn->reader_cond);
    assert(rv == 0);

    /* Destroy the address */
    if (conn->addr != NULL) {
        free(conn->addr);
    }

    /* Remove any incomplete buffers */
    if (conn->msg_buffer.msg) {
        skMsgDestroy(conn->msg_buffer.msg);
    }

    /* Finally, free the connection object */
    free(conn);

    RETURN_VOID;
}


static int accept_connection(sk_msg_queue_t *q)
{
    int fd;
    sk_msg_conn_queue_t *conn;
    int rv;
    struct sockaddr_in addr;
    sk_sockaddr_t *addr_copy;
    socklen_t addrlen = sizeof(addr);

    DEBUG_ENTER_FUNC;

    assert(q);
    ASSERT_QUEUE_LOCK(q);

  retry_accept:
    fd = accept(q->root->listen_sock, (struct sockaddr *)&addr, &addrlen);
    if (fd == -1) {
        if (errno == EAGAIN || errno == EWOULDBLOCK) {
            DEBUG_PRINT1("Properly handling EAGAIN/EWOULDBLOCK");
            RETURN(1);
        }
        DEBUGMSG("accept() [%s]", strerror(errno));
        if (errno == EINTR) {
            goto retry_accept;
        }
        if (errno != EBADF) {
            CRITMSG("Unexpected accept() error: %s", strerror(errno));
            XASSERT(0);
            skAbort();
        }
        RETURN(-1);
    }

    /* Create the queue and both references */
    addr_copy = (sk_sockaddr_t *)malloc(sizeof(*addr_copy));
    if (addr_copy != NULL) {
        memcpy(addr_copy, &addr, addrlen);
    }
#if SK_ENABLE_GNUTLS
    rv = create_connection(q, fd, fd, addr_copy, &conn,
                           q->root->bind_tls ? SKM_TLS_SERVER : SKM_TLS_NONE);
#else
    rv = create_connection(q, fd, fd, addr_copy, &conn, SKM_TLS_NONE);
#endif /* SK_ENABLE_GNUTLS */

    if (rv != 0) {
        close(fd);
        free(addr_copy);
        RETURN(-1);
    }

    conn->first_channel = create_channel(q);

    RETURN(0);
}

static int handle_system_control_message(
    sk_msg_queue_t      *q,
    sk_msg_conn_queue_t *conn,
    sk_msg_t            *msg)
{
    int rv;
    int retval = 0;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(msg);
    ASSERT_QUEUE_LOCK(q);

    switch (msg->hdr.type) {
      case SKMSG_CTL_CHANNEL_ANNOUNCE:
        /* Handle the announcement of a connection */
        {
            skm_channel_t rchannel;
            skm_channel_t lchannel;
            sk_msg_channel_queue_t *chan;
            sk_channel_pair_t pair;
            sk_new_channel_info_t info;

            DEBUG_PRINT1("Handling SKMSG_CTL_CHANNEL_ANNOUNCE");
            assert(msg->hdr.size == sizeof(rchannel));
            assert(msg->segments == 2);

            /* Decode the remote channel */
            rchannel = SKMSG_CTL_MSG_GET_CHANNEL(msg);

            /* Create a local channel */
            if (conn->first_channel) {
                chan = conn->first_channel;
                conn->first_channel = NULL;
            } else {
                chan = create_channel(q);
            }
            lchannel = chan->channel;

            /* Attach the channel to the connection */
            rv = set_channel_connecting(q, chan, conn);
            assert(rv == 0);

            /* Set the remote channel */
            rv = set_channel_connected(q, chan, rchannel);
            assert(rv == 0);

            /* Respond to the announcement with the channel pair */
            pair.rchannel = htons(rchannel);
            pair.lchannel = htons(lchannel);
            DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_REPLY (Ext-control)");
            rv = send_message(q, lchannel,
                              SKMSG_CTL_CHANNEL_REPLY,
                              &pair, sizeof(pair), SKM_SEND_CONTROL,
                              0, NULL);
            if (rv != 0) {
                DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_REPLY failed");
                retval = -11;
                break;
            }

            /* Announce the new channel internally */
            info.channel = pair.lchannel;
            if (conn->addr != NULL) {
                memcpy(&info.addr, conn->addr, sizeof(info.addr));
                info.known = 1;
            } else {
                info.known = 0;
            }
            DEBUG_PRINT1("Sending SKMSG_CTL_NEW_CONNECTION (Internal)");
            rv = send_message(q, SKMSG_CHANNEL_CONTROL,
                              SKMSG_CTL_NEW_CONNECTION,
                              &info, sizeof(info),
                              SKM_SEND_INTERNAL, 0, NULL);
            XASSERT(rv == 0);
        }
        break;

      case SKMSG_CTL_CHANNEL_REPLY:
        /* Handle the reply to a channel announcement */
        {
            sk_channel_pair_t *pair;
            sk_msg_channel_queue_t *chan;
            skm_channel_t rchannel;
            skm_channel_t lchannel;

            DEBUG_PRINT1("Handling SKMSG_CTL_CHANNEL_REPLY");
            assert(sizeof(*pair) == msg->hdr.size);
            assert(2 == msg->segments);

            /* Decode the channels: Reversed directionality is on
               purpose. */
            pair = (sk_channel_pair_t *)msg->segment[1].iov_base;
            rchannel = ntohs(pair->lchannel);
            lchannel = ntohs(pair->rchannel);

            /* Get the channel object */
            chan = find_channel(q, lchannel);
            XASSERT(chan != NULL);

            /* Set the remote channel */
            rv = set_channel_connected(q, chan, rchannel);
            assert(rv == 0);

            chan->conn->state = SKM_CONNECTED;

            /* Complete the connection */
            assert(chan->state != SKM_CONNECTING);
            assert(chan->is_pending);
            MUTEX_BROADCAST(&chan->pending);
        }
        break;

      case SKMSG_CTL_CHANNEL_KILL:
        /* Handle the death of a remote channel */
        {
            skm_channel_t channel;
            sk_msg_channel_queue_t *chan;

            DEBUG_PRINT1("Handling SKMSG_CTL_CHANNEL_KILL");

            assert(msg->hdr.size == sizeof(channel));
            assert(msg->segments == 2);

            /* Decode the channel. */
            channel = SKMSG_CTL_MSG_GET_CHANNEL(msg);

            /* Get the channel object. */
            chan = find_channel(q, channel);
            XASSERT(chan != NULL);

            /* Close the channel. */
            retval = set_channel_closed(q, chan, 0);
        }
        break;

      case SKMSG_CTL_CHANNEL_KEEPALIVE:
        DEBUG_PRINT1("Handling SKMSG_CTL_CHANNEL_KEEPALIVE");
        assert(msg->hdr.size == 0);
        /* Do nothing on KEEPALIVE*/
        break;

      default:
        skAbortBadCase(msg->hdr.type);
    }

    skMsgDestroy(msg);

    RETURN(retval);
}

static void *listener_thread(void *vq)
{
    sk_msg_queue_t *q = (sk_msg_queue_t *)vq;

    DEBUG_ENTER_FUNC;

    DEBUG_PRINT1("Started listener_thread");

    assert(q);

    QUEUE_LOCK(q);
    MUTEX_BROADCAST(&q->root->listener_cond);
    q->root->listener_state = SKM_THREAD_RUNNING;
    QUEUE_UNLOCK(q);

    while (q->root->listener_state == SKM_THREAD_RUNNING) {
        int rv;
        struct pollfd pfd;

        pfd.fd = q->root->listen_sock;
        pfd.events = POLLIN | POLLERR | POLLNVAL;

        /* Call poll */
        rv = poll(&pfd, 1, SKMSG_IO_POLL_TIMEOUT);
        if (rv == -1) {
            if (errno == EINTR || errno == EBADF) {
                DEBUG_PRINT2("Ignoring expected poll() error: %s",
                             strerror(errno));
                continue;
            }
            CRITMSG("Unexpected poll() error: %s", strerror(errno));
            skAbort();
        }
        if (rv == 0) {
            continue;
        }
        if (!(pfd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
            QUEUE_LOCK(q);
            if (q->root->listener_state == SKM_THREAD_RUNNING) {
                DEBUG_PRINT1("Accepting connection: trying");
                rv = accept_connection(q);
                if (rv == 0) {
                    DEBUG_PRINT1("Accepting connection: succeeded");
                } else {
                    DEBUG_PRINT1("Accepting connection: failed");
                }
            }
            QUEUE_UNLOCK(q);
        } else {
            DEBUG_PRINT3("Poll returned %d, but revents was %d", rv,
                         pfd.revents);
        }
    }

    QUEUE_LOCK(q);
    q->root->listener_state = SKM_THREAD_ENDED;
    THREAD_END(q);
    QUEUE_UNLOCK(q);

    DEBUG_PRINT1("STOPPED listener_thread");

    RETURN(NULL);
}


static void *reader_thread(void *vconn)
{
    sk_queue_and_conn_t *both = (sk_queue_and_conn_t *)vconn;
    sk_msg_conn_queue_t *conn = both->conn;
    sk_msg_queue_t *q         = both->q;
    int destroyed = 0;

    DEBUG_ENTER_FUNC;

    DEBUG_PRINT1("STARTED reader_thread");

    assert(conn);
    assert(q);

    free(both);
    both = NULL;

    QUEUE_LOCK(q);
    conn->reader_state = SKM_THREAD_RUNNING;
    MUTEX_BROADCAST(&conn->reader_cond);
    QUEUE_UNLOCK(q);

    /* Add the current time to the connection */
    conn->last_recv = time(NULL);

    while (!destroyed && conn->state != SKM_CLOSED
           && conn->reader_state == SKM_THREAD_RUNNING)
    {
        int           rv;
        struct pollfd pfd;
        time_t        current_time;
        sk_msg_t *message = NULL;

        pfd.fd = conn->rsocket;
        pfd.events = POLLIN | POLLERR | POLLNVAL;

        /* Call poll */
        rv = poll(&pfd, 1, SKMSG_IO_POLL_TIMEOUT);
        if (rv == -1) {
            if (errno == EINTR || errno == EBADF) {
                continue;
            }
            CRITMSG("Unexpected poll() error: %s", strerror(rv));
            skAbort();
        }

        current_time = time(NULL);
        if (rv == 0) {
            if (CONNECTION_STAGNANT(conn, current_time)) {
                /* It's been too long since we have heard something.
                 * Assume the connection died. */
                QUEUE_LOCK(q);
                destroy_connection(q, conn);
                QUEUE_UNLOCK(q);
                destroyed = 1;
                break;
            }
            continue;
        }

        conn->last_recv = time(NULL);

        /* Read a message */
        DEBUG_PRINT1("Calling recv");
        rv = conn->fn.recv(conn, &message);
        if (rv != 0) {
            /* Treat the connection as closed */
            QUEUE_LOCK(q);
            destroy_connection(q, conn);
            QUEUE_UNLOCK(q);
            destroyed = 1;
            break;
        }

        if (message) {
            sk_msg_channel_queue_t *chan;

            /* Handle control messages */
            if (message->hdr.channel == SKMSG_CHANNEL_CONTROL &&
                message->hdr.type >= SKMSG_MINIMUM_SYSTEM_CTL_CHANNEL)
            {
                QUEUE_LOCK(q);
                rv = handle_system_control_message(q, conn, message);
                QUEUE_UNLOCK(q);
                if (rv) {
                    destroyed = 1;
                }
                continue;
            }

            /* Handle ordinary messages */
            chan = find_channel(q, message->hdr.channel);
            if (chan == NULL) {
                skMsgDestroy(message);
            } else {
                /* Put the message on the queue */
                DEBUG_PRINT3("Enqueue: chan=%#x type=%#x",
                             message->hdr.channel, message->hdr.type);
                DEBUG_PRINT2("From reader: %p", (void *)message);
                rv = mqQueueAdd(chan->queue, message);
                if (rv != 0) {
                    XASSERT(conn->state == SKM_CLOSED ||
                            conn->reader_state != SKM_THREAD_RUNNING);
                    skMsgDestroy(message);
                }
            }
        }
    }


    QUEUE_LOCK(q);
    conn->reader_state = SKM_THREAD_ENDED;
    THREAD_END(q);
    QUEUE_UNLOCK(q);

    DEBUG_PRINT1("STOPPED reader_thread");

    RETURN(NULL);
}


static void *writer_thread(void *vconn)
{
    sk_queue_and_conn_t *both  = (sk_queue_and_conn_t *)vconn;
    sk_msg_conn_queue_t *conn  = both->conn;
    sk_msg_queue_t      *q     = both->q;
    int                  again = 0;
    sk_msg_t            *msg   = NULL;

    DEBUG_ENTER_FUNC;

    DEBUG_PRINT1("STARTED writer_thread");

    assert(conn);
    assert(q);

    free(both);

    QUEUE_LOCK(q);
    conn->writer_state = SKM_THREAD_RUNNING;
    MUTEX_BROADCAST(&conn->writer_cond);
    QUEUE_UNLOCK(q);

    while (conn->writer_state == SKM_THREAD_RUNNING) {
        int           rv;
        skDQErr_t     err;
        struct pollfd pfd;
        int           block = (conn->state != SKM_CLOSED);

        if (!again) {
            if (!block) {
                /* If the connection is closed, don't block in the deque */
                err = skDequePopBackNB(conn->queue, (void *)&msg);
            } else {
                if (conn->keepalive == 0) {
                    err = skDequePopBack(conn->queue, (void *)&msg);
                } else {
                    err = skDequePopBackTimed(conn->queue, (void *)&msg,
                                              conn->keepalive);
                    if (err == SKDQ_TIMEDOUT) {
                        /* Create a keepalive message */
                        msg = calloc(1, sizeof(*msg));
                        MEM_ASSERT(msg);
                        msg->segments = 1;
                        msg->segment[0].iov_base = &msg->hdr;
                        msg->segment[0].iov_len = sizeof(msg->hdr);
                        msg->hdr.channel = SKMSG_CHANNEL_CONTROL;
                        msg->hdr.type = SKMSG_CTL_CHANNEL_KEEPALIVE;
                        /* Pretend it came from the queue */
                        err = SKDQ_SUCCESS;
                        DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_KEEPALIVE");
                    }
                }
            }
            if (err != SKDQ_SUCCESS) {
                assert(err == SKDQ_UNBLOCKED || err == SKDQ_DESTROYED
                       || err == SKDQ_EMPTY);
                break;
            }
            if (msg->hdr.channel == SKMSG_CHANNEL_CONTROL
                && msg->hdr.type == SKMSG_WRITER_UNBLOCKER)
            {
                /* Do not destroy message, as this is a special static
                   message. */
                msg = NULL;
                DEBUG_PRINT1("Handling SKMSG_WRITER_UNBLOCKER message");
                continue;
            }
            again = 1;
        }

        pfd.fd = conn->wsocket;
        pfd.events = POLLOUT | POLLERR | POLLNVAL;

        rv = poll(&pfd, 1, SKMSG_IO_POLL_TIMEOUT);
        if (rv == -1) {
            if (errno == EINTR || errno == EBADF) {
                continue;
            }
        }
        if (rv == 0) {
            continue;
        }

        rv = conn->fn.send(conn, msg);
        again = 0;
        skMsgDestroy(msg);
        msg = NULL;
        if (rv != 0) {
            /* Treat the connection as closed */
            DEBUGMSG("Closing connection due to failed write.");
            QUEUE_LOCK(q);
            destroy_connection(q, conn);
            QUEUE_UNLOCK(q);
            continue;
        }
    }

    if (msg) {
        skMsgDestroy(msg);
    }

    QUEUE_LOCK(q);
    conn->writer_state = SKM_THREAD_ENDED;
    THREAD_END(q);
    QUEUE_UNLOCK(q);

    DEBUG_PRINT1("STOPPED writer_thread");

    RETURN(NULL);
}


int skMsgQueueCreate(sk_msg_queue_t **queue)
{
    sk_msg_queue_t *q;
    int retval = 0;
    int fd[2];
    int rv;
    sk_msg_conn_queue_t *conn;
    sk_msg_channel_queue_t *chan;

    DEBUG_ENTER_FUNC;

    q = calloc(1, sizeof(*q));
    if (q == NULL) {
        RETURN(SKMERR_MEMORY);
    }

    q->root = calloc(1, sizeof(*q->root));
    if (q->root == NULL) {
        free(q);
        RETURN(SKMERR_MEMORY);
    }

    THREAD_INFO_INIT(q);

    q->root->channel = int_dict_create(sizeof(sk_msg_channel_queue_t *));
    if (q->root->channel == NULL) {
        retval = SKMERR_MEMORY;
        goto error;
    }
    q->root->groups = int_dict_create(sizeof(sk_msg_queue_t *));
    if (q->root->groups == NULL) {
        retval = SKMERR_MEMORY;
        goto error;
    }
    q->channel = int_dict_create(sizeof(sk_msg_channel_queue_t *));
    if (q->channel == NULL) {
        retval = SKMERR_MEMORY;
        goto error;
    }

    rv = pthread_mutex_init(QUEUE_MUTEX(q), NULL);
    if (rv != 0) {
        retval = SKMERR_MUTEX;
        goto error;
    }

    rv = pthread_cond_init(&q->shutdowncond, NULL);
    if (rv != 0) {
        retval = SKMERR_MUTEX;
        goto error;
    }

    q->group = mqCreateFair(sk_destroy_report_message);
    if (q->group == NULL) {
        goto error;
    }

    rv = pipe(fd);
    if (rv == -1) {
        retval = SKMERR_PIPE;
        goto error;
    }

    /* Initialize the listener thread start state */
    pthread_cond_init(&q->root->listener_cond, NULL);
    q->root->listener_state = SKM_THREAD_BEFORE;

    /* Lock the mutex to satisfy preconditions for mucking with
       connections and channels. */
    QUEUE_LOCK(q);

    /* Create an internal connection for the control channel. */
    rv = create_connection(q, fd[READ], fd[WRITE], NULL, &conn, SKM_TLS_NONE);
    conn->keepalive = SKMSG_CONTROL_KEEPALIVE_TIMEOUT;
    unblock_connection(q, conn);
    XASSERT(rv == 0);

    /* Create a channel for the control channel. */
    q->root->next_channel = SKMSG_CHANNEL_CONTROL;
    chan = create_channel(q);

    /* Attach the internal connection to the control channel. */
    rv = set_channel_connecting(q, chan, conn);
    assert(rv == 0);

    /* And let's completely connect it. */
    rv = set_channel_connected(q, chan, SKMSG_CHANNEL_CONTROL);
    conn->state = SKM_CONNECTED;

    /* Unlock the mutex */
    QUEUE_UNLOCK(q);

    *queue = q;
    RETURN(0);

  error:
    XASSERT(0);
    skMsgQueueDestroy(q);
    RETURN(retval);
}

static void sk_msg_queue_shutdown(
    sk_msg_queue_t *q)
{
    sk_msg_channel_queue_t *chan;
    void *cont;

    DEBUG_ENTER_FUNC;

    assert(q);
    ASSERT_QUEUE_LOCK(q);

    if (q->shuttingdown) {
        RETURN_VOID;
    }

    q->shuttingdown = 1;

    /* Shut down a queue by shutting down all its channels. */
    cont = int_dict_get_first(q->channel, &chan);
    while (cont != NULL) {
        intkey_t key = chan->channel;
        if (chan->state == SKM_CONNECTED || chan->state == SKM_CONNECTING) {
            set_channel_closed(q, chan, 0);
        }
        cont = int_dict_get_next(q->channel, key, &chan);
    }

    /* And then shutting down the multiqueue */
    mqShutdown(q->group);

    q->shuttingdown = 0;

    MUTEX_BROADCAST(&q->shutdowncond);

    RETURN_VOID;
}


void skMsgQueueShutdown(
    sk_msg_queue_t *q)
{
    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    sk_msg_queue_shutdown(q);

    QUEUE_UNLOCK(q);
    RETURN_VOID;
}


void skMsgQueueShutdownAll(
    sk_msg_queue_t *q)
{
    sk_msg_channel_queue_t *chan;
    void *cont;

    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    if (q->root->shuttingdown) {
        QUEUE_UNLOCK(q);
        RETURN_VOID;
    }

    q->root->shuttingdown = 1;
    q->root->shutdownqueue = q;

    q->root->listener_state = SKM_THREAD_SHUTTING_DOWN;

    /* Shut down all channels */
    cont = int_dict_get_first(q->root->channel, &chan);
    while (cont != NULL) {
        intkey_t key = chan->channel;
        sk_msg_queue_shutdown(chan->group);
        cont = int_dict_get_next(q->root->channel, key, &chan);
    }

    if (q->root->listen_sock != 0) {
        close(q->root->listen_sock);
    }

    THREAD_WAIT_ALL_END(q);

    if (q->root->listen_sock) {
        pthread_join(q->root->listener, NULL);
    }
    q->root->listen_sock = 0;

    q->root->shuttingdown = 0;

    MUTEX_BROADCAST(&q->shutdowncond);

    QUEUE_UNLOCK(q);

    RETURN_VOID;
}


static void skMsgQueueDestroyAll(
    sk_msg_queue_t *q)
{
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    ASSERT_QUEUE_LOCK(q);

    /* Verify that all channels have been destroyed */
    assert(int_dict_get_first(q->root->channel, NULL) == NULL);

    int_dict_destroy(q->root->channel);
    int_dict_destroy(q->root->groups);

    QUEUE_UNLOCK(q);

    THREAD_INFO_DESTROY(q);

    rv = pthread_cond_destroy(&q->root->listener_cond);
    assert(rv == 0);

    rv = pthread_mutex_destroy(QUEUE_MUTEX(q));
    assert(rv == 0);

    free(q->root);
    free(q);

    RETURN_VOID;
}


void skMsgQueueDestroy(
    sk_msg_queue_t *q)
{
    sk_msg_root_t *root;
    sk_msg_channel_queue_t *chan;
    void *cont;

    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    root = q->root;

    while (q->shuttingdown
           || (root->shuttingdown && root->shutdownqueue == q))
    {
        QUEUE_WAIT(&q->shutdowncond, q);
    }

    /* Destroy the channels */
    cont = int_dict_get_first(q->channel, &chan);
    while (cont != NULL) {
        intkey_t channel = chan->channel;
        destroy_channel(q, chan);
        cont = int_dict_get_next(q->channel, channel, &chan);
    }

    mqShutdown(q->group);
    mqDestroy(q->group);

    int_dict_destroy(q->channel);

    if (int_dict_get_first(q->root->groups, NULL) == NULL) {
        skMsgQueueDestroyAll(q);
        RETURN_VOID;
    }

    free(q);

    MUTEX_UNLOCK(&root->mutex);

    RETURN_VOID;
}

static int skMsgQueueBind(
    sk_msg_queue_t     *q,
    struct sockaddr_in *addr,
    int UNUSED_IF_NOTLS(conn_type))
{
    static int on = 1;
    int sock;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);

    if (q->root->listen_sock != 0) {
        RETURN(-1);
    }

    /* Bind a socket to the address*/
    sock = socket(AF_INET, SOCK_STREAM, 0);
    XASSERT(sock != -1);
    rv = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
    XASSERT(rv != -1);
    rv = bind(sock, (struct sockaddr *)addr, sizeof(*addr));
    if (rv != 0) {
        return -1;
    }
    rv = listen(sock, LISTENQ);
    XASSERT(rv != -1);

    QUEUE_LOCK(q);

    /* Set the listen sock for the queue. */
    set_nonblock(sock);
    q->root->listen_sock = sock;
    q->root->bind_type = SOCK_STREAM;
#if SK_ENABLE_GNUTLS
    q->root->bind_tls = (conn_type == CONN_TLS);
#endif
    memcpy(&q->root->bind_addr, addr, sizeof(*addr));

    THREAD_START("skmsg_listener", rv, q, &q->root->listener,
                 listener_thread, q);
    XASSERT(rv == 0);

    while (q->root->listener_state == SKM_THREAD_BEFORE) {
        QUEUE_WAIT(&q->root->listener_cond, q);
    }
    assert(q->root->listener_state == SKM_THREAD_RUNNING);

    QUEUE_UNLOCK(q);

    RETURN(0);
}

/* Start a listener */
int skMsgQueueBindTCP(
    sk_msg_queue_t     *queue,
    struct sockaddr_in *addr)
{
    return skMsgQueueBind(queue, addr, CONN_TCP);
}

#if SK_ENABLE_GNUTLS
/* Start a listener */
int skMsgQueueBindTLS(
    sk_msg_queue_t     *queue,
    struct sockaddr_in *addr)
{
    return skMsgQueueBind(queue, addr, CONN_TLS);
}
#endif /* SK_ENABLE_GNUTLS */


static int skMsgQueueConnect(
    sk_msg_queue_t     *q,
    struct sockaddr_in *addr,
    skm_channel_t      *channel,
    skm_conn_t          tls)
{
    int rv;
    int sock;
    sk_msg_conn_queue_t *conn;
    sk_msg_channel_queue_t *chan;
    int retval;
    skm_channel_t lchannel;
    sk_sockaddr_t *copy;

    DEBUG_ENTER_FUNC;

    assert(q);
    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock == -1) {
        RETURN(-1);
    }

    /* Connect to the remote side */
    rv = connect(sock, (struct sockaddr *)addr, sizeof(*addr));
    if (rv == -1) {
        close(sock);
        RETURN(-1);
    }

    /* Create a channel and connection, and bind them */
    QUEUE_LOCK(q);
    if (q->shuttingdown) {
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }
    copy = (sk_sockaddr_t *)malloc(sizeof(*copy));
    if (copy != NULL) {
        memcpy(copy, addr, sizeof(*addr));
    }
    rv = create_connection(q, sock, sock, copy, &conn,
                           (tls == CONN_TLS) ? SKM_TLS_CLIENT : SKM_TLS_NONE);
    if (rv == -1) {
        close(sock);
        free(copy);
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }

    chan = create_channel(q);
    rv = set_channel_connecting(q, chan, conn);
    XASSERT(rv == 0);

    /* Announce the channel id to the remote queue */
    lchannel = htons(chan->channel);
    DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_ANNOUNCE (Ext-control)");
    rv = send_message(q, chan->channel,
                      SKMSG_CTL_CHANNEL_ANNOUNCE,
                      &lchannel, sizeof(lchannel), SKM_SEND_CONTROL, 0, NULL);
    if (rv != 0) {
        DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_ANNOUNCE failed");
        destroy_connection(q, conn);
        close(sock);
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }

    /* Wait for a reply */
    chan->is_pending = 1;
    while (chan->is_pending && chan->state == SKM_CONNECTING) {
        QUEUE_WAIT(&chan->pending, q);
    }
    chan->is_pending = 0;

    if (chan->state == SKM_CLOSED) {
        destroy_channel(q, chan);
        retval = -1;
    } else {
        retval = 0;
        *channel = chan->channel;
    }

    QUEUE_UNLOCK(q);

    RETURN(retval);
}

int skMsgQueueConnectTCP(
    sk_msg_queue_t     *q,
    struct sockaddr_in *addr,
    skm_channel_t      *channel)
{
    return skMsgQueueConnect(q, addr, channel, CONN_TCP);
}

#if SK_ENABLE_GNUTLS
int skMsgQueueConnectTLS(
    sk_msg_queue_t     *q,
    struct sockaddr_in *addr,
    skm_channel_t      *channel)
{
    return skMsgQueueConnect(q, addr, channel, CONN_TLS);
}
#endif /* SK_ENABLE_GNUTLS */

int skMsgChannelNew(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_channel_t  *new_channel)
{
    sk_msg_channel_queue_t *chan;
    sk_msg_channel_queue_t *newchan;
    skm_channel_t lchannel;
    int retval;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(new_channel);

    QUEUE_LOCK(q);

    if (q->shuttingdown) {
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }

    chan = find_channel(q, channel);
    XASSERT(chan != NULL);
    XASSERT(chan->state == SKM_CONNECTED);
    assert(chan->conn != NULL);

    /* Create a channel and connection, and bind it to the connection */
    newchan = create_channel(q);
    rv = set_channel_connecting(q, newchan, chan->conn);
    assert(rv == 0);

    lchannel = htons(newchan->channel);

    /* Announce the channel id to the remote queue */
    DEBUG_PRINT1("Sending SKMSG_CTL_CHANNEL_ANNOUNCE (Ext-control))");
    rv = send_message(q, newchan->channel,
                      SKMSG_CTL_CHANNEL_ANNOUNCE,
                      &lchannel, sizeof(lchannel), SKM_SEND_CONTROL, 0, NULL);
    if (rv != 0) {
        destroy_channel(q, newchan);
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }

    /* Wait for a response */
    newchan->is_pending = 1;
    while (newchan->is_pending && newchan->state == SKM_CONNECTING) {
        QUEUE_WAIT(&newchan->pending, q);
    }
    newchan->is_pending = 0;

    if (newchan->state == SKM_CLOSED) {
        retval = -1;
        destroy_channel(q, newchan);
    } else {
        retval = 0;
        *new_channel = newchan->channel;
    }

    QUEUE_UNLOCK(q);

    RETURN(retval);
}


int skMsgChannelSplit(
    sk_msg_queue_t  *q,
    skm_channel_t    channel,
    sk_msg_queue_t **new_queue)
{
    sk_msg_queue_t *new_q;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);

    new_q = (sk_msg_queue_t *)calloc(1, sizeof(*new_q));
    if (new_q == NULL) {
        return -1;
    }

    rv = pthread_cond_init(&new_q->shutdowncond, NULL);
    if (rv != 0) {
        free(new_q);
        return -1;
    }

    new_q->channel = int_dict_create(sizeof(sk_msg_channel_queue_t *));
    if (new_q->channel == NULL) {
        free(new_q);
        return -1;
    }

    new_q->group = mqCreateFair(sk_destroy_report_message);
    if (new_q->group == NULL) {
        int_dict_destroy(new_q->channel);
        free(new_q);
        return -1;
    }

    new_q->root = q->root;

    rv = skMsgChannelMove(channel, new_q);
    if (rv != 0) {
        skMsgQueueDestroy(new_q);
    } else {
        *new_queue = new_q;
    }

    return rv;
}


int skMsgChannelMove(
    skm_channel_t    channel,
    sk_msg_queue_t  *q)
{
    sk_msg_channel_queue_t *chan;
    int rv;
    int retval = 0;

    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    chan = find_channel(q, channel);
    if (chan == NULL) {
        retval = -1;
        goto end;
    }

    rv = mqQueueMove(q->group, chan->queue);
    assert(rv == 0);
    rv = int_dict_del(chan->group->channel, channel);
    assert(rv == 0);
    rv = int_dict_set(q->channel, channel, &chan);
    assert(rv == 0);
    rv = int_dict_set(q->root->groups, channel, &q);
    assert(rv == 0);

    chan->group = q;

  end:
    QUEUE_UNLOCK(q);

    RETURN(retval);
}


int skMsgChannelKill(
    sk_msg_queue_t *q,
    skm_channel_t   channel)
{
    sk_msg_channel_queue_t *chan;

    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    if (!q->shuttingdown) {
        chan = find_channel(q, channel);
        XASSERT(chan != NULL);

        destroy_channel(q, chan);
    }

    QUEUE_UNLOCK(q);

    RETURN(0);
}


static int send_message(
    sk_msg_queue_t *q,
    skm_channel_t   lchannel,
    skm_type_t      type,
    void           *message,
    skm_len_t       length,
    sk_send_type_t  send_type,
    int             no_copy,
    void          (*free_fn)(void *))
{
    sk_msg_channel_queue_t *chan;
    sk_msg_t *msg;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(message || length == 0);
    ASSERT_QUEUE_LOCK(q);

    if (int_dict_get(q->root->channel, lchannel, &chan) == NULL) {
        RETURN(-1);
    }

    if (chan->state == SKM_CLOSED && send_type != SKM_SEND_INTERNAL) {
        RETURN(0);
    }

    msg = malloc(sizeof(*msg) + (length ? sizeof(struct iovec) : 0));
    MEM_ASSERT(msg);

    msg->free_fn = msg_simple_free;
    msg->simple_free = NULL;

    msg->segment[0].iov_base = &msg->hdr;
    msg->segment[0].iov_len = sizeof(msg->hdr);

    if (!length) {
        msg->segments = 1;
    } else {
        msg->segments = 2;
        msg->segment[1].iov_len = length;
        if (no_copy) {
            msg->simple_free = free_fn;
            msg->segment[1].iov_base = message;
        } else {
            msg->segment[1].iov_base = malloc(length);
            if (msg->segment[1].iov_base == NULL) {
                free(msg);
                RETURN(-1);
            }
            memcpy(msg->segment[1].iov_base, message, length);
        }
    }

    msg->hdr.type = type;
    msg->hdr.size = length;

    rv = send_message_internal(chan, msg, send_type);
    if (rv != 0) {
        skMsgDestroy(msg);
        RETURN(-1);
    }

    RETURN(0);
}


static int send_message_internal(
    sk_msg_channel_queue_t *chan,
    sk_msg_t               *msg,
    sk_send_type_t          send_type)
{
    skDQErr_t err;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(chan);
    assert(msg);

    switch (send_type) {
      case SKM_SEND_INTERNAL:
        msg->hdr.channel = chan->channel;
        DEBUG_PRINT3("Enqueue: chan=%#x type=%#x",
                     msg->hdr.channel, msg->hdr.type);
        rv = mqQueueAdd(chan->queue, msg);
        if (rv != 0) {
            RETURN(-1);
        }
        break;
      case SKM_SEND_REMOTE:
        msg->hdr.channel = chan->rchannel;
        err = skDequePushFront(chan->conn->queue, msg);
        if (err != SKDQ_SUCCESS) {
            RETURN(-1);
        }
        break;
      case SKM_SEND_CONTROL:
        msg->hdr.channel = SKMSG_CHANNEL_CONTROL;
        err = skDequePushFront(chan->conn->queue, msg);
        if (err != SKDQ_SUCCESS) {
            RETURN(-1);
        }
        break;
      default:
        skAbortBadCase(send_type);
    }

    RETURN(0);
}


int skMsgQueueSendMessage(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_type_t      type,
    const void     *message,
    skm_len_t       length)
{
    int rv;

    DEBUG_ENTER_FUNC;

    QUEUE_LOCK(q);
    rv = send_message(q, channel, type, (void *)message,
                      length, SKM_SEND_REMOTE, 0, NULL);
    QUEUE_UNLOCK(q);
    RETURN(rv);
}


int skMsgQueueInjectMessage(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_type_t      type,
    const void     *message,
    skm_len_t       length)
{
    int rv;

    DEBUG_ENTER_FUNC;

    QUEUE_LOCK(q);
    rv = send_message(q, channel, type, (void *)message,
                      length, SKM_SEND_INTERNAL, 0, NULL);
    QUEUE_UNLOCK(q);
    RETURN(rv);
}

int skMsgQueueSendMessageNoCopy(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_type_t      type,
    void           *message,
    skm_len_t       length,
    void          (*free_fn)(void *))
{
    int rv;

    DEBUG_ENTER_FUNC;

    QUEUE_LOCK(q);
    rv = send_message(q, channel, type, message, length, SKM_SEND_REMOTE,
                      1, free_fn);
    QUEUE_UNLOCK(q);
    RETURN(rv);
}

int skMsgQueueScatterSendMessageNoCopy(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_type_t      type,
    uint16_t        num_segments,
    struct iovec   *segments,
    void          (*free_fn)(uint16_t, struct iovec *))
{
    sk_msg_channel_queue_t *chan;
    sk_msg_t               *msg;
    size_t                  size;
    uint16_t                i;
    int                     rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert((num_segments && segments) || (!num_segments && !segments));

    QUEUE_LOCK(q);

    if (int_dict_get(q->root->channel, channel, &chan) == NULL) {
        rv = -1;
        goto end;
    }

    if (chan->state == SKM_CLOSED) {
        rv = 0;
        goto end;
    }

    msg = malloc(sizeof(*msg) + sizeof(struct iovec) * num_segments);
    MEM_ASSERT(msg);

    msg->free_fn = free_fn;
    msg->simple_free = NULL;

    msg->segment[0].iov_base = &msg->hdr;
    msg->segment[0].iov_len = sizeof(msg->hdr);
    msg->segments = 1;

    msg->hdr.type = type;
    size = 0;

    for (i = 0; i < num_segments; i++) {
        msg->segment[i + 1] = segments[i];
        size += segments[i].iov_len;
        msg->segments++;
        if (size > UINT16_MAX) {
            skMsgDestroy(msg);
            rv = -1;
            goto end;
        }
    }

    msg->hdr.size = size;

    rv = send_message_internal(chan, msg, SKM_SEND_REMOTE);
    if (rv != 0) {
        skMsgDestroy(msg);
    }

  end:
    QUEUE_UNLOCK(q);

    RETURN(rv);
}


int skMsgQueueInjectMessageNoCopy(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    skm_type_t      type,
    void           *message,
    skm_len_t       length,
    void          (*free_fn)(void *))
{
    int rv;

    DEBUG_ENTER_FUNC;

    QUEUE_LOCK(q);
    rv = send_message(q, channel, type, message, length, SKM_SEND_INTERNAL,
                      1, free_fn);
    QUEUE_UNLOCK(q);
    RETURN(rv);
}


int skMsgQueueGetMessage(
    sk_msg_queue_t *q,
    sk_msg_t      **message)
{
    sk_msg_t *msg;
    sk_msg_channel_queue_t *chan;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(message);

    do {
        rv = mqGet(q->group, (void **)&msg);
        if (rv != 0) {
            RETURN(-1);
        }
        DEBUG_PRINT2("From GetMessage: %p", (void *)msg);
        DEBUG_PRINT4("Deque: chan=%#x type=%#x size=%d",
                     msg->hdr.channel, msg->hdr.type, msg->hdr.size);

        chan = find_channel(q, msg->hdr.channel);
    } while (chan == NULL);

    *message = msg;

    RETURN(0);
}


int skMsgQueueGetMessageFromChannel(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    sk_msg_t      **message)
{
    sk_msg_t *msg;
    sk_msg_channel_queue_t *chan;
    int rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(message);

    chan = find_channel(q, channel);

    if (chan == NULL) {
        RETURN(-1);
    }

    rv = mqQueueGet(chan->queue, (void **)&msg);
    if (rv != 0) {
        RETURN(-1);
    }
    DEBUG_PRINT4("Deque: chan=%#x type=%#x size=%d",
                 msg->hdr.channel, msg->hdr.type, msg->hdr.size);

    assert(msg->hdr.channel == channel);

    chan = find_channel(q, msg->hdr.channel);

    if (chan == NULL) {
        RETURN(-1);
    }

    *message = msg;

    RETURN(0);
}

int skMsgGetRemoteChannelID(
    sk_msg_queue_t *q,
    skm_channel_t   lchannel,
    skm_channel_t  *rchannel)
{
    sk_msg_channel_queue_t *chan;
    int                     retval;

    DEBUG_ENTER_FUNC;

    assert(q);

    chan = find_channel(q, lchannel);
    if (chan == NULL) {
        retval = -1;
    } else {
        *rchannel = chan->rchannel;
        retval = 0;
    }

    return retval;
}


int skMsgSetKeepalive(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    uint16_t        keepalive)
{
    sk_msg_channel_queue_t *chan;
    int                     retval;

    DEBUG_ENTER_FUNC;

    assert(q);

    QUEUE_LOCK(q);

    chan = find_channel(q, channel);
    if (chan == NULL || chan->state != SKM_CONNECTED) {
        retval = -1;
    } else {
        chan->conn->keepalive = keepalive;
        unblock_connection(q, chan->conn);
        retval = 0;
    }
    QUEUE_UNLOCK(q);

    return retval;
}


int skMsgGetConnectionInformation(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    char           *buffer,
    size_t          buffer_size)
{
    sk_msg_channel_queue_t *chan;
    sk_msg_conn_queue_t    *conn;
    int                     rv;

    DEBUG_ENTER_FUNC;

    assert(q);
    assert(buffer);

    QUEUE_LOCK(q);
    chan = find_channel(q, channel);
    if (chan == NULL) {
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }
    conn = chan->conn;
    if (conn == NULL) {
        QUEUE_UNLOCK(q);
        RETURN(-1);
    }

#if SK_ENABLE_GNUTLS
    if (conn->use_tls) {
        const char *protocol;
        const char *encryption;

        protocol = gnutls_protocol_get_name(
            gnutls_protocol_get_version(conn->session));
        encryption = gnutls_cipher_get_name(gnutls_cipher_get(conn->session));
        QUEUE_UNLOCK(q);

        rv = snprintf(buffer, buffer_size, "TCP, %s, %s",
                      protocol, encryption);
        RETURN(rv);
    }
#endif  /* SK_ENABLE_GNUTLS */

    QUEUE_UNLOCK(q);
    rv = snprintf(buffer, buffer_size, "TCP");
    RETURN(rv);
}


void skMsgDestroy(
    sk_msg_t *msg)
{
    DEBUG_ENTER_FUNC;

    assert(msg);

    if (msg->segments == 2 && msg->simple_free) {
        msg->simple_free(msg->segment[1].iov_base);
    } else if (msg->segments > 1 && msg->free_fn) {
        msg->free_fn(msg->segments - 1, &msg->segment[1]);
    }
    /* We use a static message to unblock the writer queue.  It should
     * not be freed. */
    if (msg->hdr.channel != SKMSG_CHANNEL_CONTROL ||
        msg->hdr.type != SKMSG_WRITER_UNBLOCKER)
    {
        free(msg);
    }

    RETURN_VOID;
}


skm_channel_t skMsgChannel(
    const sk_msg_t *msg)
{
    DEBUG_ENTER_FUNC;

    assert(msg);
    RETURN(msg->hdr.channel);
}


skm_type_t skMsgType(
    const sk_msg_t *msg)
{
    DEBUG_ENTER_FUNC;

    assert(msg);
    RETURN(msg->hdr.type);
}


skm_len_t skMsgLength(
    const sk_msg_t *msg)
{
    DEBUG_ENTER_FUNC;

    assert(msg);
    RETURN(msg->hdr.size);
}


const void *skMsgMessage(
    const sk_msg_t *msg)
{
    DEBUG_ENTER_FUNC;

    assert(msg);
    if (msg->segments == 0) {
        RETURN(NULL);
    } else {
        RETURN(msg->segment[1].iov_base);
    }
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
